%plotScreenSimulations A script to create the figures in Katzin et al., Radiometric properties of screens and energy saving in greenhouses
%
% This script file should be run from its directory, i.e., the active directory in
% MATLAB should be ...\Scripts\

% David Katzin, Wageningen University and Research, July 2024
% david.katzin@wur.nl


%% Default settings

% Uncomment the next line if you want to close the figures generted by MATLAB
% keepFigsOpen = false;

% Uncomment the next line to prevent MATLAB from saving the generated figures and data
% saveFigs = false;

if ~exist('keepFigsOpen', 'var') || isempty(keepFigsOpen)
    keepFigsOpen = true;
end

if ~exist('saveFigs', 'var') || isempty(saveFigs)
    saveFigs = true;
end


%% Set up directories

% Find the full path of the this file's directory.
currentFilename = mfilename();
fullFilePath = mfilename('fullpath');
currentFileDir = strrep(fullFilePath, currentFilename, '');

if isempty(currentFileDir) || contains(currentFileDir, 'Temp\Editor') % This script is run from the console
    currentFileDir = [cd() '\']; % the directory from which this script is running
end

origDirSlashes = strfind(currentFileDir,'\');
upDir = currentFileDir(1:origDirSlashes(end-1)); % Up one level from currentFileDir

% Location of simulation results: upDir\Simulation results\
simResultsDir =  [upDir 'Simulation results\'];

% Location of figures directory: upDir\Figures\
figuresDir =  [upDir 'Figures\'];

if ~exist(figuresDir, 'dir')
    mkdir(figuresDir);
end

% Create subdirectories: png, fig, csv
if ~exist([figuresDir 'png\'], 'dir')
    mkdir([figuresDir 'png\']);
end
if ~exist([figuresDir 'fig\'], 'dir')
    mkdir([figuresDir 'fig\']);
end
if ~exist([figuresDir 'csv\'], 'dir')
    mkdir([figuresDir 'csv\']);
end

%% Figure settings and useful variables
fontSize = 10;
lineWidth = 0.8;
figSize = [20 10]; % [width height]    

% Default color scheme of MATLAB
cc = lines();

%% Create a cell with all relevant file names
fileNames = dir(simResultsDir);
fileNames = fileNames(3:end);
fileNames = struct2cell(fileNames);
fileNames = fileNames(1,:);

fieldNames = readmatrix([simResultsDir fileNames{1}], ...
    'range','1:1','outputtype','char');

f = struct; % numerical reference to fields
for n=1:numel(fieldNames)
    fieldName = fieldNames{n};
    fieldName = strrep(fieldName, '%', '');
    
    f.(fieldNames{n}) = n;
end

scenarioNames = [];
scrTauPar = [];
scrTauNir = [];


%% Load simulation data
% 1. Reference simulation - standard screen
simName = 'pepper_standard';
scenarioNames = [scenarioNames "Standard"];
scrTauPar = [scrTauPar 0.7]; % PAR transmissivity of the screen
scrTauNir = [scrTauNir 0.7]; % NIR transmissivity of the screen

csvRead = readmatrix([simResultsDir simName '_output.csv']);

% output(iiScen, iiHour, outputChannel)
% is the value of outputChannel at time iiHour in scenario iiScen
output = zeros(1, size(csvRead,1), size(csvRead,2));
scen = 1;
output(scen,:,:) = csvRead;
scen = scen+1;

% 2. NIR reflecting 
% TauPAR 0.6, tauNIR 0.2, rhoNIR 0.7
simName = 'NIR_reflecting';
scenarioNames = [scenarioNames "NIR reflecting"];
scrTauPar = [scrTauPar 0.6]; % PAR transmissivity of the screen
scrTauNir = [scrTauNir 0.2]; % NIR transmissivity of the screen
csvRead = readmatrix([simResultsDir simName '_output.csv']);
output(scen,:,:) = csvRead;
scen = scen+1;

% 3. NIR reflecting with same PAR transmissivity
% TauPAR 0.7, tauNIR 0.2, rhoNIR 0.7
simName = 'NIR_reflecting_sameTauPAR';
scenarioNames = [scenarioNames "NIR ref., PAR tr."];
scrTauPar = [scrTauPar 0.7]; % PAR transmissivity of the screen
scrTauNir = [scrTauNir 0.2]; % NIR transmissivity of the screen
csvRead = readmatrix([simResultsDir simName '_output.csv']);
output(scen,:,:) = csvRead;
scen = scen+1;

% 4. NIR reflecting with control adjusted for equivalent SWR
simName = 'NIR_reflecting_adjSWR';
scenarioNames = [scenarioNames "NIR ref., less hours"];
scrTauPar = [scrTauPar 0.6]; % PAR transmissivity of the screen
scrTauNir = [scrTauNir 0.2]; % NIR transmissivity of the screen
csvRead = readmatrix([simResultsDir simName '_output.csv']);
output(scen,:,:) = csvRead;
scen = scen+1;

% 5. NIR reflecting, same tauPAR with control adjusted for equivalent SWR
simName = 'NIR_reflecting_sameTauPAR_adjSWR';
scenarioNames = [scenarioNames "NIR ref., PAR tr., less hours"];
scrTauPar = [scrTauPar 0.7]; % PAR transmissivity of the screen
scrTauNir = [scrTauNir 0.2]; % NIR transmissivity of the screen
csvRead = readmatrix([simResultsDir simName '_output.csv']);
output(scen,:,:) = csvRead;
scen = scen+1;

% 6. No screen
simName = 'no_screen';
scenarioNames = [scenarioNames "No screen"];
scrTauPar = [scrTauPar 1]; % PAR transmissivity of the screen
scrTauNir = [scrTauNir 1]; % NIR transmissivity of the screen
csvRead = readmatrix([simResultsDir simName '_output.csv']);
output(scen,:,:) = csvRead;
scen = scen+1;

%% General time and transmissivity variables

% timestamps of the data
ts = datetime(output(1,:,1), 'ConvertFrom', 'excel');

% Timestamps of the days in this simulation
% For use when plotting day sums or averages
tsDay = dateshift(ts(1:24:end),'start','day');

% Greenhouse SWR transmissivity
ghTau = 0.76; 

fracSunPar = .45; % Fraction of solar radiation that is PAR
fracSunNir = .50; % Fraction of solar radiation that is NIR

%% Daily indoor temperature
% File name for saving the figure
fileName = 'fig1dailyIndoorTemperature';

% Create a new figure with white background
fig = figure('InvertHardcopy','off','Color',[1 1 1]);

% Variable for holding data to be stored in CSV file
csvData = [];
hold on

chosenScens = [1 2 6];
nScens = numel(chosenScens);

leg = [];
lineColor = 1;
for iiScen = chosenScens
    % blockmean (without further arguments) calculates the average of each
    % 24-cell block. In standard use the output is hourly 
    % (simset.@loginterval is 3600). 
    % In this case, blockmean calculates daily averages
    p = plot(tsDay, blockmean(output(iiScen,:,f.tIn)), 'LineWidth', lineWidth, 'Color', cc(lineColor,:)); 
    csvData = [csvData ...
                [strcat(scenarioNames(iiScen), " average daily indoor air temperature (°C)"); p.YData']];

    p = plot(blockmean(squeeze(output(iiScen,:,f.rtrTarget))'), ...
        ':', 'LineWidth', lineWidth+1,'Color',cc(lineColor,:)); 
    csvData = [csvData ...
            [strcat(scenarioNames(iiScen), "target temperature (°C)"); p.YData']];

    leg = [leg strcat(scenarioNames(iiScen), " screen: achieved temperature") ...
        strcat(scenarioNames(iiScen), " screen: target temperature")];
    lineColor = lineColor+1;
end
leg = strrep(leg, 'screen screen', 'screen');
legend(leg,'FontSize',fontSize-4, 'Location', 'nw');
title('Average daily indoor temperature','FontSize',fontSize);
ylabel('°C')
ylim([19 25])
labels = strrep(strrep(xticklabels,'2021',''),'2022','');
xlim([ts(48) ts(end-24*5)])
xticks([ts(48) xticks (ts(end-24*5))])
xticklabels(labels);
grid
set(gca,'FontSize',fontSize-2)

fig.Units = 'centimeters';
fig.OuterPosition(3)=figSize(1);
fig.OuterPosition(4)=figSize(2);

% Save figure to png, fig, and csv
if saveFigs
    exportgraphics(fig,[figuresDir 'png\' fileName '.png']);
    savefig(fig,[figuresDir 'fig\' fileName '.fig'],'compact');
    writematrix(csvData, [figuresDir 'csv\' fileName '.csv']);
end
    
if ~keepFigsOpen
    close(fig);
end

%% Screen hours
fileName = 'tab1screen_hours';
fig = figure('InvertHardcopy','off','Color',[1 1 1]);
csvData = [];

chosenScens = 1:5;
nScens = numel(chosenScens);

screenHours = zeros(nScens,1);

row = 1;    
for iiScen = chosenScens
    screenHours(row) = sum(output(iiScen,:,f.scrPos)); % Screen use, hours/year
    row = row+1;
end

csvData = [["hours/year" "Screen use"]; [scenarioNames(chosenScens)' screenHours]];

b = bar(screenHours);

xticks(1:nScens);
labels = [];
k = 1;
for iiScen=chosenScens
    labels = [labels strcat(num2str(k), '. ', scenarioNames(iiScen))];
    k = k+1;
end
xticklabels(labels);xlim([0.5 nScens+0.5]);
title('Yearly screen use','FontSize',fontSize);
ylabel('hours screen use per year')
grid
set(gca,'FontSize',fontSize-2)

fig.Units = 'centimeters';
fig.OuterPosition(3)=figSize(1);
fig.OuterPosition(4)=figSize(2);

if saveFigs
    exportgraphics(fig,[figuresDir 'png\' fileName '.png']);
    savefig(fig,[figuresDir 'fig\' fileName '.fig'],'compact');
    writematrix(csvData, [figuresDir 'csv\' fileName '.csv']);
end
    
if ~keepFigsOpen
    close(fig);
end


%% SWR above the crop - bar chart
fileName = 'fig2AswrCrop_barchart';
fig = figure('InvertHardcopy','off','Color',[1 1 1]);
csvData = [];
hold on

csvData = [];

chosenScens = 1:6;
nScens = numel(chosenScens);

swrSum = zeros(2,nScens);

row = 1;
for iiScen = chosenScens
    % Effective PAR transmissivity of GH+screen
    tauPar = ghTau*(scrTauPar(iiScen)*output(iiScen,:,f.scrPos) ...
             +1-output(iiScen,:,f.scrPos));

    % Effective NIR transmissivity of GH+screen
    tauNir = ghTau*(scrTauNir(iiScen)*output(iiScen,:,f.scrPos) ...
             +1-output(iiScen,:,f.scrPos));

    if sum(isnan(tauPar)) > 0 % No screen was installed
        tauPar = ghTau;
        tauNir = ghTau;
    end
    
    % PAR above the crop, MJ/m2/s
    swrSum(1,row) = 3600e-6*sum(fracSunPar*tauPar.*output(iiScen,:,f.iGlobOut));
    
    % NIR above the crop, MJ/m2/s
    swrSum(2,row) = 3600e-6*sum(fracSunNir*tauNir.*output(iiScen,:,f.iGlobOut));
    row = row+1;
end


b = bar(swrSum','stacked');

csvData = [scenarioNames(chosenScens); swrSum];
xticks(1:nScens);
xticklabels([]);
xlim([0.5 nScens+0.5])
title('A. Total short wave radiation at the crop level','FontSize',fontSize);
ylabel('MJ m^{-2} year^{-1}')
legend('PAR','NIR',fontSize=fontSize-2,Location='nw')
grid
set(gca,'FontSize',fontSize-2)

fig.Units = 'centimeters';
fig.OuterPosition(3)=figSize(1);
fig.OuterPosition(4)=figSize(2);

if saveFigs
    exportgraphics(fig,[figuresDir 'png\' fileName '.png']);
    savefig(fig,[figuresDir 'fig\' fileName '.fig'],'compact');
    writematrix(csvData, [figuresDir 'csv\' fileName '.csv']);
end
    
if ~keepFigsOpen
    close(fig);
end


%% Water use - bar chart
fileName = 'fig2BwaterUse_barchart';
fig = figure('InvertHardcopy','off','Color',[1 1 1]);
csvData = [];

chosenScens = 1:6;
nScens = numel(chosenScens);

waterUse = zeros(nScens,2);

row = 1;    
for iiScen = chosenScens
    waterUse(row,1) = 3600*sum(output(iiScen,:,f.padFanWaterUse)); % Pad and fan, liter/m2/year
    waterUse(row,2) = 1.1*3600*sum(output(iiScen,:,f.transpiration)); % Irrigation, liter/m2/year
    row = row+1;
end

csvData = [["liter/m2/year" "Pad and fan" "Irrigation"]; [scenarioNames(chosenScens)' waterUse]];

b = bar(waterUse, 'stacked');

ylim([0 3500]);
xticks(1:nScens);
xticklabels([]);
xlim([0.5 nScens+0.5]);
title('B. Yearly water use','FontSize',fontSize);
legend('Cooling', 'Irrigation', FontSize = fontSize-2, Location='sw')
ylabel('liter m^{-2} year^{-1}')
grid
set(gca,'FontSize',fontSize-2)

fig.Units = 'centimeters';
fig.OuterPosition(3)=figSize(1);
fig.OuterPosition(4)=figSize(2);

if saveFigs
    exportgraphics(fig,[figuresDir 'png\' fileName '.png']);
    savefig(fig,[figuresDir 'fig\' fileName '.fig'],'compact');
    writematrix(csvData, [figuresDir 'csv\' fileName '.csv']);
end
    
if ~keepFigsOpen
    close(fig);
end

%% Water use efficiency - PAR/water
fileName = 'fig2CwaterUseEfficiency';
fig = figure('InvertHardcopy','off','Color',[1 1 1]);
csvData = [];
hold on

chosenScens = 1:6;
nScens = numel(chosenScens);

wue = zeros(2,nScens);
row = 1;

for iiScen = chosenScens
    wue(1,row) =  3600*sum(output(iiScen,:,f.padFanWaterUse)) ...
                    / (3600e-6*sum(output(iiScen,:,f.parAboveCrop)));
    wue(2,row) = 1.1*3600*sum(output(iiScen,:,f.transpiration)) ...
                    / (3600e-6*sum(output(iiScen,:,f.parAboveCrop)));
    row = row+1;
end

b = bar(wue','stacked');
csvData = [scenarioNames(chosenScens); wue];
xticks(1:nScens);

labels = [];
k = 1;
for iiScen=chosenScens
    labels = [labels strcat(num2str(k), '. ', scenarioNames(iiScen))];
    k = k+1;
end
xticklabels(labels);
xlim([0.5 nScens+0.5])
title('C. Product water use (liters per mol PAR)','FontSize',fontSize);
ylabel('liter water per mol PAR')
legend('Cooling', 'Irrigation', FontSize = fontSize-2, Location='sw')
grid
set(gca,'FontSize',fontSize-2)

fig.Units = 'centimeters';
fig.OuterPosition(3)=figSize(1);
fig.OuterPosition(4)=figSize(2);

if saveFigs
    exportgraphics(fig,[figuresDir 'png\' fileName '.png']);
    savefig(fig,[figuresDir 'fig\' fileName '.fig'],'compact');
    writematrix(csvData, [figuresDir 'csv\' fileName '.csv']);
end
    
if ~keepFigsOpen
    close(fig);
end