#!/usr/bin/env python
# coding: utf-8

# In[1]:


import os
import numpy as np
import flopy
import matplotlib.pyplot as plt
import flopy.utils.binaryfile as bf
import pandas as pd


# In[8]:


def leftBoundary(level, amplitude, frequency, duration, plot):
    
    wave = np.linspace(0, duration - 1, duration)
    output = amplitude * np.sin(2 * np.pi * wave / frequency) + level
    left = np.insert(output, 0, level)
    
    if plot == True:
        month = np.linspace(1, duration + 1, duration + 1)
        plt.plot(month, left)
        
    return left


# In[9]:


def rightBoundary(name, model_ws, amplitude, frequency, period, plot):

    modelname = "Right_Boundary"
    model_ws = model_ws 

    L = 5200.0         #Model length
    W = 5200.0         #Model width
    ztop = 15.0       #Model top elevation
    zbot = 0.0        #Model bottom elevation
    nlay = 30         #Number of layers
    nrow = 1          #Number of rows
    ncol = 5200        #Number of columns
    delr = L / ncol   #Row discretization
    delc = W / nrow   #Width discretization
    delz = (ztop - zbot) / nlay                #Thickness discretization
    botm = np.linspace(ztop, zbot, nlay + 1)   #Bottom cells elevation  

    # Boundary conditions for basic package

    lefthead = 30    #Constant head at left boundary
    righthead = lefthead - (L * 0.0035)    #Constant head at right boundary

    rech = 0 #Recharge flux in m/day

    hk = 6.1            #Aquifer hydraulic conductivity (nlay, nrow, ncol)
    vka = 0.61          #Aquifer vertical hydraulic conductivity
    ss = 0.01              #Aquifer storage coefficient
    sy = 0.2              #Aquifer storage yield

    nper = 12 * period + 1      #Number of stress periods, 6 years plus 1 day
    duration = nper - 1

    len_sp = np.ones(nper) #assigning individual length of stress period as 1
    len_sp[1:] = 30        # 1 day for starting which is steady state and 30 days for transient
    perlen = len_sp        #Length of stress periods

    nstp = 4               #Number of time steps per stress period, 1 week
    nprs = 1               #Frequency of the output, 1: depends on timprs
    itmuni = 4             #Time unit {0: "undefined", 1: "seconds", 2: "minutes",3: "hours", 4: "days", 5: "years"}

    output_time = 12 * duration + 1              #weekly output time
    time_out = np.linspace(0, duration, output_time)
    time_out = time_out * 12 * 30             #Frequency of output

    def convert_to_boolean(a):
        return [bool(x == 1) for x in a]      #funtion to convert to boolean, steady if 1 day stress period length

    steady = convert_to_boolean(perlen)       #call function

    # Create flopy simulation object

    mf_sim = flopy.modflow.Modflow(
        modelname=modelname,
        exe_name="mf2005",
        model_ws = model_ws,
    )

    # Create flopy discretization (DIS) package

    dis = flopy.modflow.ModflowDis(
        mf_sim, nlay=nlay, nrow=nrow, ncol=ncol, 
        delr=delr, delc=delc, top=ztop, botm=botm[1:],
        perlen=perlen, nstp=nstp, itmuni=itmuni,
        nper=nper, steady=steady,
    )

    # Create basic (BAS) package
    ibound = np.ones((nlay, nrow, ncol), dtype=np.int32)
    ibound[:, :, -1] = -1      #Cell set as constant head at right boundary

    strt = np.ones((nlay, nrow, ncol), dtype=np.float32)
    strt[:, :, -1] = righthead #Assign value at the right boundary

    bas = flopy.modflow.ModflowBas(
        mf_sim, ibound=ibound, strt=strt,
        ichflg=True,)

    # Create layer property flow (LPF) package

    lpf = flopy.modflow.ModflowLpf(
        mf_sim,
        hk=hk, 
        vka=vka, ss=ss, sy=sy,
        ipakcb = 53,
    )
    
    left = leftBoundary(lefthead, amplitude, frequency, duration, False)
    
    stageright = righthead
    bound = []
    i = 1
    j = 1

    # Create list of names and make each items a variable
    while i <= nper:
        bound.append("bound_sp"+str(i))
        locals()["bound_sp"+str(i)] = []
        i = i+1

    # Append arrays to each variables    
    while j <= nper:
        stageleft = left[j-1]
        for il in range(nlay):
            condleft = hk * (stageleft - zbot) * delr
            condright = hk * (stageright - zbot) * delr
            for ir in range(nrow):
                locals()["bound_sp"+str(j)].append([il, ir, 0, stageleft, condleft])
                locals()["bound_sp"+str(j)].append([il, ir, ncol - 1, stageright, condright])

        #print("Adding ", len(locals()["bound_sp"+str(j)]), "GHBs for stress period " + str(j))
        j = j + 1

    rch = flopy.modflow.ModflowRch(
        mf_sim,
        rech=rech,
    )

    stress_period_data = {}
    for kper in range(nper):
        stress_period_data[kper] = locals()[bound[kper]]

    ghb = flopy.modflow.ModflowGhb(mf_sim, stress_period_data=stress_period_data)
    pcg = flopy.modflow.ModflowPcg(mf_sim)

    stress_period_data = {}
    for kper in range(nper):
        for kstp in range(nstp):
            stress_period_data[(kper, kstp)] = [
                "save head",
                "save drawdown",
                "save budget",
                "print head",
                "print budget",
            ]
    oc = flopy.modflow.ModflowOc(
        mf_sim, stress_period_data=stress_period_data, compact=True
    )

    spd = {}
    stress_period = nper
    time_step = np.linspace(0,9, nstp).astype(int)

    x = 0

    while x < stress_period:
        for i in time_step:
            spd[x ,i] = ["print head", "print budget", "save head", "save budget"]
        x = x + 1

    mf_sim.write_input()        
    
    if os.path.isfile(f"{model_ws}\\{modelname}.hds"):
    
        headobj = bf.HeadFile(f"{model_ws}\\{modelname}.hds")
        times_head = headobj.get_times()
        head = headobj.get_data(totim=times_head[-1])  
        
    else:
        # Run model
        success, mfoutput = mf_sim.run_model(silent=False, pause=False)
        if not success:
            raise Exception("MODFLOW did not terminate normally.")
        
        headobj = bf.HeadFile(f"{model_ws}\\{modelname}.hds")
        times_head = headobj.get_times()
        head = headobj.get_data(totim=times_head[-1])  
        
    if plot == True: 

        print('Head statistics')
        print('  min: ', head.min())
        print('  max: ', head.max())
        print('  std: ', head.std())

        contour = 0.5
        H = ztop - zbot

        h = headobj.get_data()
        x = np.linspace(0, L, h[0].size)
        y = np.linspace(0, H, len(head))
        y = y[::-1]

        vmin, vmax = righthead, lefthead
        contour_intervals = np.arange(0, 100, contour)
        
        #colored
        headobj = bf.HeadFile(f"{model_ws}\\{modelname}.hds")
        times = headobj.get_times()
       
        head = headobj.get_data(totim=times[-1])

        fig, ax = plt.subplots(1, 1, figsize=(10, 4), constrained_layout=True)
        ax.imshow(head[:,0,:], interpolation='nearest',          #colored graph
                  extent=(0, L , 0, ztop-zbot), aspect="auto")   #colored graph

        h = headobj.get_data(kstpkper=(0, 0)) #timestep, stress period
        c = ax.contour(x, y, h[:, 0, :], contour_intervals, colors="black")
        plt.clabel(c, fmt="%1.1f");

        ax.set_title('head')

        #plt.savefig(f"{model_ws}\\Sample"+str(1))
        plt.show()

    #location of wells from left boundary
    w1_loc = 0             
    w2_loc = 300           
    d1 = 0 

    i = 0

    #Define list for head

    head_w1_d1 = []       # well 1 depth 1
    head_w2_d1 = []       # well 2 depth 1

    while i < len(times_head):

        head = headobj.get_data(totim = times_head[i])

        head_w1_d1.append(head[d1, 0, w1_loc])   #(nlay, nrow, ncol) 
        head_w2_d1.append(head[d1, 0, w2_loc])   #(nlay, nrow, ncol)

        i = i + 1

    headleft = map(lambda x: x - head_w1_d1[0], head_w1_d1)
    headright = map(lambda x: x - head_w2_d1[0], head_w2_d1)

    x = 0
    listofhead = []
    while x < len(times_head):
        if (times_head[x] % 30) == 1:
          listofhead.append(times_head.index(times_head[x])) 
        x += 1

    arr = np.array(head_w2_d1) - head_w2_d1[0]
    rightFluctuation = arr[listofhead]
    rightFluctuation = np.array(rightFluctuation)
    
    #np.save("righthead_" + str(name), x)
    if plot == True: 
        #plot of changes in heads
        fig, ax = plt.subplots(figsize=(12, 6), dpi=80)
        month = np.linspace(1, duration + 1, duration + 1)

        plt.plot(month, left - lefthead, color='#184A45FF', ls='-', marker='.')
        plt.plot(month, rightFluctuation, color='#FC766AFF', ls='-', marker='.')

        font1 = {'family':'times new roman','color':'black','size':15}

        plt.legend(("Left boundary", "Right boundary"), ncol = 2
                   , bbox_to_anchor = (0.5,1), loc = "center", framealpha = 1)

        plt.xlabel("Month", fontdict = font1)
        plt.ylabel("Change in water level", fontdict = font1)

        plt.xlim(1,duration+1)

        plt.show()
    
    return rightFluctuation


# In[ ]:




