% Script for moving load transition
clc
clear
% Time step
dt = 0.001;
TS = [num2str(dt) 's'];
TS(find(TS=='.')) = 'p';
% Load speed
V = 60;
VS = [num2str(V) 'mps'];
Tmax = 2;

%% Track properties
% Rail
Track.a = 2;
Track.EI = 2*6.044e6;
Track.m = 2*60;
Track.DampRail = 0.0001;
% Sleepers
Track.Width = 0.3;
Track.Spacing = 0.6;
Track.Ms = 315;
Track.Js = Track.Ms/6*Track.Width^2;
% Rail pads
Track.Kv = 100e6;
Track.Kt = Track.Kv*Track.Width^2/4;
Track.DampRP = 0.001;
% Under sleeper pads
Track.DampUSP = 0.001;
Track.Ku = 45e9;
Track.Kuh = Track.Ku*1e2*0;
%% Ballast and soil - stiffness properties for full track (2m cross section)
% Test box stifness
D = 0.03;
K0 = 300e6; % Pa/m
WidthLoad = 1;
LxLoad = 0.25;
Lz = 0.3;
K0 = K0*LxLoad*WidthLoad;
Ratio = 1/8;
nXLoad = round(LxLoad/D);
nZ = round(Lz/D);
[Kn,Ks,Kd,Fn,Fs,Fd] = FindStiffnesses(K0,Ratio,nXLoad,nZ);
Kn = Kn/WidthLoad*Track.a; % 7.4488e7
Ks = Ks/WidthLoad*Track.a; % 9.3110e6
Kd = Kd/WidthLoad*Track.a; % 3.2588e7
Ballast.Damp = 0.001;
Ballast.D = D;
Ballast.H = 0.3;
Ballast.Rho = 1800;
Ballast.Kn = Kn;
Ballast.Ks = Ks;
Ballast.Kd = Kd;
% Old values
% Ballast.Kn = 1.4222e+07;
% Ballast.Ks = 1.7778e+06;
% Ballast.Kd = 6.2222e+06;

% Example new
Soil.Left.FilenameDM = ['ExampleGouda/DM_' TS '.mat'];
Soil.Left.FilenameID = ['ExampleGouda/ID_' VS '_' TS '.mat'];
% Normal parameters
Soil.Left.Cs = 140*[1;1];
Soil.Left.Cp = Soil.Left.Cs*sqrt(2-0.5)/sqrt(1-0.5);
Soil.Left.Rho = 2290.8*[1;1];
Soil.Left.Damp = 6.6361e-4*[1;1];
Soil.Left.H = 0.6*[1;1];
Soil.Left.KvBot = 6.7150e7;
Soil.Left.KhBot = 1.3850e9;
Soil.Left.CvBot = 2.3987e4;
Soil.Left.ChBot = 3.7719e4;
% % Modified parameters
% Soil.Left.Cs = 140*[1;1];
% Soil.Left.Cp = 230*[1;1];
% Soil.Left.Rho = 18952*[1;1];
% Soil.Left.Damp = 1.5859e-3*[1;1];
% Soil.Left.H = 0.6*[1;1];
% Soil.Left.KvBot = 1.4348e8;
% Soil.Left.KhBot = 3.0572e9;
% Soil.Left.CvBot = 2.0328e4;
% Soil.Left.ChBot = 3.8462e4;

Soil.Right = Soil.Left;
Soil.Mid = Soil.Left;
Soil.Mid.Cs(end) = inf;
Soil.Mid.Cp(end) = inf;
Soil.Mid.KvBot = inf;
Soil.Mid.KhBot = inf;
Soil.Mid.CvBot = inf;
Soil.Mid.ChBot = inf;
%% Middle sections - change here
nSleeperSide = 5;
%                                     Left Mid Right
Soil.MidSections = [nSleeperSide      1   0     0 ; % Ten sleepers at beginning
                               7      1   0     0 ; % Seven sleepers at transition
                               4    0.0   1   0.0 ; % Culvert
                               7      0   0     1 ; % Seven sleepers at transition
                    nSleeperSide      0   0     1]; % Ten sleepers at end

%% Approach slab
% Left Approach slav
ApproachSlab(1).Cells = [nSleeperSide+1 nSleeperSide+7];
ApproachSlab(1).E = 20e9;
ApproachSlab(1).Thick = 0.3;
ApproachSlab(1).Rho = 2300;
ApproachSlab(1).Depth = 0.6;
ApproachSlab(1).RotSpringStart = 0;
ApproachSlab(1).RotSpringEnd = inf;
ApproachSlab(1).VertSpringStart = 0;
ApproachSlab(1).VertSpringEnd = inf;
% Right approach slab
ApproachSlab(2).Cells = [nSleeperSide+12 nSleeperSide+18];
ApproachSlab(2).E = 20e9;
ApproachSlab(2).Thick = 0.3;
ApproachSlab(2).Rho = 2300;
ApproachSlab(2).Depth = 0.6;
ApproachSlab(2).RotSpringStart = inf;
ApproachSlab(2).RotSpringEnd = 0;
ApproachSlab(2).VertSpringStart = inf;
ApproachSlab(2).VertSpringEnd = 0;
% If no approach slabs, uncomment the following instruction
% ApproachSlab = [];

% OutputFilename = ['ExampleGouda/HingeSlab_' num2str(nSleeperSide) 'Sleepers_' VS '_' TS '.mat'];
OutputFilename = ['ExampleGouda/ApproachSlab_' num2str(nSleeperSide) 'Sleepers_' VS '_' TS '.mat'];
% OutputFilename = ['ExampleGouda/Culvert_' num2str(nSleeperSide) 'Sleepers_' VS '_' TS '.mat'];
% OutputFilename = ['ExampleGouda/NoCulvert_' num2str(nSleeperSide) 'Sleepers_' VS '_' TS '.mat'];
CALCULATE = 0;

%% Inputs
Inputs.Track = Track;
Inputs.Ballast = Ballast;
Inputs.Soil = Soil;
Inputs.ApproachSlab = ApproachSlab;

%% Calculate response to moving load
W = [0.025:0.025:15 15.1:0.1:60 61:1:100 110:10:1000]*2*pi;

if CALCULATE
    OUTPUT = MovingLoadTransition(Inputs,dt,Tmax,V,W,0,0.001,0);
    save(OutputFilename, 'OUTPUT','-v7.3');
end
load(OutputFilename);
% Create video with response
% CreateVideo_ML(OUTPUT,OutputFilename(1:end-4),V,1);

SLEEPER = OUTPUT.SLEEPER;
RAIL = OUTPUT.RAIL;
BALLAST = OUTPUT.BALLAST;
BEAM = OUTPUT.BEAM;
SPRING = OUTPUT.SPRING;
NODE_TYPE = OUTPUT.NODE_TYPE;
ad2Connection = OUTPUT.ad2Connection;
T = OUTPUT.T;
ad2Displ = OUTPUT.ad2Displ;
ad2Vel = OUTPUT.ad2Vel;
return

%% Calculate energy transfer to ballast by each sleeper
close all
SLEEPER_NODES = find(NODE_TYPE(:,1)==SLEEPER);
nSleeper = length(SLEEPER_NODES);
SleeperEnergy = zeros(nSleeper,1);
for iSleeper = 1: 1: nSleeper
    Node = SLEEPER_NODES(iSleeper);
    xSleeper = NODE_TYPE(Node,2);
    TIME_STAPM = find((T>=xSleeper/V-0.1).*(T<=xSleeper/V+0.1));
    Connections = find((ad2Connection(:,2)==Node)+(ad2Connection(:,3)==Node));
    Connections = ad2Connection(Connections,:);
    nConnections = size(Connections,1);
    for iConnection = 1: 1: nConnections
        if Connections(iConnection,1) ~= SPRING, continue; end
        Node1 = Connections(iConnection,2);
        Node2 = Connections(iConnection,3);
        if NODE_TYPE(Node1,1) == RAIL || NODE_TYPE(Node2,1) == RAIL
            continue;
        end
        % Calculate force on spring
        dofs =  Connections(iConnection,4:7);
        K = Connections(iConnection,8);
        C = Connections(iConnection,9);
        K = K*(dofs'*dofs);
        C = C*(dofs'*dofs);
        F = K*ad2Displ([2*Node1-1; 2*Node1; 2*Node2-1; 2*Node2],:) + ...
            C*ad2Vel([2*Node1-1; 2*Node1; 2*Node2-1; 2*Node2],:);
        if NODE_TYPE(Node1,1) == SLEEPER && NODE_TYPE(Node2,1) == BALLAST
            SleeperEnergy(iSleeper) = SleeperEnergy(iSleeper) + ...
                sum(sum(-F(3:4,TIME_STAPM).*ad2Vel([2*Node2-1; 2*Node2],TIME_STAPM)))*(T(2)-T(1));
        elseif NODE_TYPE(Node1,1) == BALLAST && NODE_TYPE(Node2,1) == SLEEPER
            SleeperEnergy(iSleeper) = SleeperEnergy(iSleeper) + ...
                sum(sum(-F(1:2,TIME_STAPM).*ad2Vel([2*Node1-1; 2*Node1],TIME_STAPM)))*(T(2)-T(1));
        else
            fprintf(1, 'ERROR: something went wrong...\n');
            return;
        end
    end
end
plot((1:1:nSleeper)-nSleeperSide+5, SleeperEnergy, 'o');
xlabel('Sleeper number');
ylabel('Energy transmitted per sleeper (J/N^2)');
grid on;

return;
%% Calculate maximum force and power input
close all
SLEEPER_NODES = find(NODE_TYPE(:,1)==SLEEPER);
nSleeper = length(SLEEPER_NODES);
nCol = round(sqrt(nSleeper));
nRow = ceil(nSleeper/nCol);
MAX_FORCE = zeros(nSleeper,2);
figure;
for iSleeper = 1: 1: nSleeper
    SleeperPowerInput = zeros(1,size(ad2Displ,2));
    Node = SLEEPER_NODES(iSleeper);
    xSleeper = NODE_TYPE(Node,2);
    Connections = find((ad2Connection(:,2)==Node)+(ad2Connection(:,3)==Node));
    Connections = ad2Connection(Connections,:);
    nConnections = size(Connections,1);
    SleeperForce = zeros(1,size(ad2Displ,2));
    for iConnection = 1: 1: nConnections
        if Connections(iConnection,1) ~= SPRING, continue; end
        Node1 = Connections(iConnection,2);
        Node2 = Connections(iConnection,3);
        if NODE_TYPE(Node1,1) == RAIL || NODE_TYPE(Node2,1) == RAIL
            continue;
        end
        % Calculate force on spring
        dofs =  Connections(iConnection,4:7);
        K = Connections(iConnection,8);
        C = Connections(iConnection,9);
        K = K*(dofs'*dofs);
        C = C*(dofs'*dofs);
        F = K*ad2Displ([2*Node1-1; 2*Node1; 2*Node2-1; 2*Node2],:) + ...
            C*ad2Vel([2*Node1-1; 2*Node1; 2*Node2-1; 2*Node2],:);
        if NODE_TYPE(Node1,1) == SLEEPER && NODE_TYPE(Node2,1) == BALLAST
            SleeperPowerInput = SleeperPowerInput + ...
                sum(-F(3:4,:).*ad2Vel([2*Node2-1; 2*Node2],:),1);
            SleeperForce = SleeperForce - F(4,:);
        elseif NODE_TYPE(Node1,1) == BALLAST && NODE_TYPE(Node2,1) == SLEEPER
            SleeperPowerInput = SleeperPowerInput + ...
                sum(-F(1:2,:).*ad2Vel([2*Node1-1; 2*Node1],:),1);
            SleeperForce = SleeperForce - F(2,:);
        else
            fprintf(1, 'ERROR: something went wrong...\n');
            return;
        end
    end
    MAX_FORCE(iSleeper,1) = max(SleeperForce);
    MAX_FORCE(iSleeper,2) = -min(SleeperForce);
    subplot(nRow,nCol,iSleeper);
    plot(T,SleeperPowerInput); hold on;
    plot(xSleeper/V*[1 1], [min(SleeperPowerInput) max(SleeperPowerInput)], '--k');
    xlim([xSleeper/V-0.1 xSleeper/V+0.1]); hold off;
end
figure;
plot((1:1:nSleeper)-nSleeperSide+5, MAX_FORCE(:,1),'o',(1:1:nSleeper)-nSleeperSide+5, MAX_FORCE(:,2),'*');
xlabel('Sleeper number');
ylabel('Maximum transmitted force (N/N)');
grid on;

return;                


%% Calculate energy dissipation at ballast
clc
fprintf(1, '\n');
fprintf(1, 'Creating energy dissipation at ballast ...\n');
fprintf(1, '[0%%                                            100%%]\n');
fprintf(1, '[');
STARS = zeros(50,1); star = 0;

nNode = size(NODE_TYPE,1);
EnergyDissipated = zeros(nNode,1);
for iConnection = 1: 1: size(ad2Connection,1)
    if mod(iConnection,round(size(ad2Connection,1)/50))==0
        star = star + 1;
        if star <= 50 && STARS(star) == 0
            fprintf(1, '*');
            STARS(star) = 1;
        end
    end
    if ad2Connection(iConnection,1) ~= SPRING, continue; end
    Node1 = ad2Connection(iConnection,2); if Node1 < 1, continue; end
    Node2 = ad2Connection(iConnection,3); if Node2 < 1, continue; end
    if NODE_TYPE(Node1,1) ~= BALLAST || NODE_TYPE(Node2,1)  ~= BALLAST
        continue;
    end
    % Calculate force on spring
    dofs =  ad2Connection(iConnection,4:7);
    K = ad2Connection(iConnection,8);
    C = ad2Connection(iConnection,9);
    K = K*(dofs'*dofs);
    C = C*(dofs'*dofs);
    F = K*ad2Displ([2*Node1-1; 2*Node1; 2*Node2-1; 2*Node2],:) + ...
        C*ad2Vel([2*Node1-1; 2*Node1; 2*Node2-1; 2*Node2],:);
    Energy = sum(sum(F(1:2,:).*ad2Vel([2*Node1-1; 2*Node1],:)))*(T(2)-T(1)) + ...
             sum(sum(F(3:4,:).*ad2Vel([2*Node2-1; 2*Node2],:)))*(T(2)-T(1));
     EnergyDissipated([Node1 Node2]) = EnergyDissipated([Node2 Node2]) + Energy/2;
end
for iStar = star+1:1:50
    fprintf(1, '*');
end
fprintf(1, ']\n');

%% Plot things related to energy dissipated at ballast
close all
figure;
subplot(2,1,2);
% Plot rail nodes
RAIL_NODES = find(NODE_TYPE(:,1) == RAIL);
plot(NODE_TYPE(RAIL_NODES,2), ones(size(RAIL_NODES))*0.6, 'b'); hold on
plot(NODE_TYPE(RAIL_NODES,2), ones(size(RAIL_NODES))*0.3, 'b');
% Plot Sleepers
SLEEPER_NODES = find(NODE_TYPE(:,1) == SLEEPER);
for iSleeper = 1:1:length(SLEEPER_NODES)
    kNode = SLEEPER_NODES(iSleeper);
    x = NODE_TYPE(kNode,2);
    y = 0.15;
    r = 0.0;
    X = [x-0.15-0.15*r y+0.15-0.15*r;
         x-0.15+0.15*r y-0.15-0.15*r;
         x+0.15+0.15*r y-0.15+0.15*r;
         x+0.15-0.15*r y+0.15+0.15*r];
    plot(X([1 2 3 4 1],1), X([1 2 3 4 1],2), 'r');
end
% Plot energy ballast
BALLAST_NODES = find(NODE_TYPE(:,1) == BALLAST);
XX = unique(NODE_TYPE(BALLAST_NODES,2));
YY = unique(NODE_TYPE(BALLAST_NODES,3));
YY = YY(end:-1:1);
EnergyDissipatedBallast = (EnergyDissipated(BALLAST_NODES));
EnergyDissipatedBallast = reshape(EnergyDissipatedBallast, [length(YY), length(XX)]);

LEAVE = 20;
XX = XX(LEAVE+1:end-LEAVE);
EnergyDissipatedBallast = EnergyDissipatedBallast(:,LEAVE+1:end-LEAVE);
surface(XX,YY,zeros(size(EnergyDissipatedBallast)),(EnergyDissipatedBallast),'LineStyle','none');
caxis(([min(EnergyDissipatedBallast(:)) max(EnergyDissipatedBallast(:))]));
axis equal

% Calculate energy dissipation per column
subplot(2,2,1);
EnergyDisspSection = sum(EnergyDissipatedBallast,1);
plot(XX-(nSleeperSide-5)*Track.Spacing, EnergyDisspSection); hold on;
SLEEPER_NODES = find(NODE_TYPE(:,1) == SLEEPER);
for iSleeper = 1:1:length(SLEEPER_NODES)
    kNode = SLEEPER_NODES(iSleeper);
    x = NODE_TYPE(kNode,2);
    y = mean(EnergyDisspSection);
    r = 0.0;
    X = [x-0.15-0.15*r y*1.1-0.15*r;
         x-0.15+0.15*r y*0.9-0.15*r;
         x+0.15+0.15*r y*0.9+0.15*r;
         x+0.15-0.15*r y*1.1+0.15*r];
    plot(X([1 2 3 4 1],1)-(nSleeperSide-5)*Track.Spacing, X([1 2 3 4 1],2), 'r');
end

% Calculate Energy dissipation under each sleeper
subplot(2,2,2);
EnergyDissSleep = zeros(length(SLEEPER_NODES)-2,1);
for iSleeper = 2:1:length(SLEEPER_NODES)-1
    xl = mean(NODE_TYPE(SLEEPER_NODES(iSleeper+[-1 0]),2));
    xr = mean(NODE_TYPE(SLEEPER_NODES(iSleeper+[1 0]),2));
    for iX = 1:1:length(XX)
        if XX(iX) > xl && XX(iX) < xr
            EnergyDissSleep(iSleeper-1) = EnergyDissSleep(iSleeper-1) + EnergyDisspSection(iX);
        elseif XX(iX) == xl || XX(iX) == xr
            EnergyDissSleep(iSleeper-1) = EnergyDissSleep(iSleeper-1) + 0.5*EnergyDisspSection(iX);
        end
    end
end
plot((2:1:length(SLEEPER_NODES)-1)-(nSleeperSide-5), EnergyDissSleep);
%% Calculate theoretical response
CALC_THEOR = 0;
if CALC_THEOR
    dW = 2*pi/10/Tmax;
    Wf = 2*pi/0.001;
    W = [dW/100 dW:dW:Wf/100 Wf/100+10*dW:10*dW:Wf];
    OUTPUT_Theor = MovingLoadTheoretical(Inputs,1,0.001,Tmax,V,W,...
                                            [], 0);
    save('ExampleGouda/Theor.mat','OUTPUT_Theor');
end
%% Response of rail at connectons with sleepers
close all;
load ExampleGouda/Theor.mat
RAIL_NODES = find(NODE_TYPE(:,1)==RAIL);
[Min,kNodeStart] = min(NODE_TYPE(RAIL_NODES,2));
[Max,kNodeEnd] = max(NODE_TYPE(RAIL_NODES,2));
kNodeStart = RAIL_NODES(kNodeStart);
kNodeEnd = RAIL_NODES(kNodeEnd);
NODES_MID = find(NODE_TYPE(:,1) == SLEEPER);
NODES_MID = NODES_MID - 1;
COLOR = 'brgcm';
tenter = Min/V;
tleave = Max/V;
close all
for iNode = 1: 1: length(NODES_MID)
    figure(1);
    plot(T, -ad2Displ(2*NODES_MID(iNode)-1,:),COLOR(1+mod(iNode-1,length(COLOR))), 'LineWidth', 2); hold on;
    [MAX,kT] = max(ad2Displ(2*NODES_MID(iNode)-1,:));
    figure(2);
    plot(T-T(kT), -ad2Displ(2*NODES_MID(iNode)-1,:),COLOR(1+mod(iNode-1,length(COLOR))), 'LineWidth', 1); hold on
end
figure(1);
% plot(tenter*[1 1], -[min(min(ad2Displ(2*NODES_MID-1,:))) max(max(ad2Displ(2*NODES_MID-1,:)))], '--k', 'LineWidth', 2);
% plot(tleave*[1 1], -[min(min(ad2Displ(2*NODES_MID-1,:))) max(max(ad2Displ(2*NODES_MID-1,:)))], '--k', 'LineWidth', 2);
xlabel('Time (s)');
ylabel('u_r_a_i_l (m)');
title('a)');
grid on;
xlim([-0.2 0.6]);
figure(2);
UrailTheor = permute(OUTPUT_Theor.Urail(1,:,:), [3 1 2]);
[MAX,kT] = max(UrailTheor);
plot((OUTPUT_Theor.T-OUTPUT_Theor.T(kT)), -UrailTheor,'--k', 'LineWidth', 2);
xlabel('Time (s)');
ylabel('u_r_a_i_l (m)');
title('b)');
grid on;
return;

