function OUTPUT = MovingLoadTheoretical(Inputs,DampType,dt,Tmax,V,W,...
                                        I_out, Xout)
VISCOUS = 1;
HYSTERETIC = 0;
if nargin < 6, W = []; end
OUTPUT = [];
if isempty(W)
    dW = 2*pi/10/Tmax;
    Wf = 2*pi/dt;
    W = 0:dW:Wf;
end

%% Organize inputs
Track = Inputs.Track;
DampEI = Track.DampRail;
EI = Track.EI; EI = EI*(1+1i*DampEI);
m = Track.m;
a = Track.a;
% Sleepers
Width = Track.Width;
Spacing = Track.Spacing;
Ms = Track.Ms;
Js = Track.Js;
% Rail pads
DampRP = Track.DampRP;
Kv = Track.Kv; Kv = Kv*(1+1i*DampRP);
Kt = Track.Kt; Kt = Kt*(1+1i*DampRP);
% Under sleeper pads
DampUSP = Track.DampUSP;
Ku = Track.Ku; Ku = Ku*(1+1i*DampUSP);
Kuh = Track.Kuh; Kuh = Kuh*(1+1i*DampUSP);

%% Ballast properties
Ballast = Inputs.Ballast;
Dampb = Ballast.Damp;
D = Ballast.D;
Hb = Ballast.H;
Rhob = Ballast.Rho;
Knb = Ballast.Kn*(1+1i*Dampb);
Ksb = Ballast.Ks*(1+1i*Dampb);
Kdb = Ballast.Kd*(1+1i*Dampb);
Mb  = Rhob*D^2*a;
Nb = round(Hb/D);
%% Foundation properties - ballast and soil - Left side
Soil = Inputs.Soil.Left;
Damp = Soil.Damp;
H = Soil.H;
Rho = Soil.Rho;
Cs = Soil.Cs;
Cp = Soil.Cp;
KvBot = Soil.KvBot;
KhBot = Soil.KhBot;
CvBot = Soil.CvBot;
ChBot = Soil.ChBot;
% Generate left lattice vectors
nLayer = length(H);
Nlat = zeros(nLayer,1);
Mlat = zeros(nLayer,1);
Knlat = zeros(nLayer,1);
Kslat = zeros(nLayer,1);
Kdlat = zeros(nLayer,1);
for iLayer = 1: 1: nLayer
    cs = Cs(iLayer);
    cp = Cp(iLayer);
    rho = Rho(iLayer);
    G = cs^2*rho;
    Lambda = cp^2*rho - 2*G;
    Mat = [1/2 -3/2; 1/2 1/2];
    B = Mat^-1*[Lambda;G];
    Knlat(iLayer) = B(1);
    Kslat(iLayer) = B(2);
    Kdlat(iLayer) = (Knlat(iLayer) - Kslat(iLayer)) / 2;
    Mlat(iLayer) = rho*D^2;
    Nlat(iLayer) = round(H(iLayer)/D);
end
Knlat = Knlat.*(1+1i*Damp);
Kslat = Kslat.*(1+1i*Damp);
Kdlat = Kdlat.*(1+1i*Damp);
% Add ballast layer
Knlat = [Knb;Knlat];
Kslat = [Ksb;Kslat];
Kdlat = [Kdb;Kdlat];
Mlat = [Mb;Mlat];
Nlat = [Nb;Nlat];
% Add bottom layer
if ~isempty(KvBot)
    Knlat = [Knlat;(KvBot+1i*CvBot)*D];
    Kslat = [Kslat;(KhBot+1i*ChBot)*D];
    Kdlat = [Kdlat;0];
    Mlat = [Mlat;0];
    Nlat = [Nlat;0];
%     Knlat = [Knlat;(KvBot+1i*CvBot)*D];
%     Kslat = [Kslat;(KhBot+1i*ChBot)*D];
%     Kdlat = [Kdlat;((KvBot+1i*CvBot)-(KhBot+1i*ChBot))*D/2];
%     Mlat = [Mlat;Mlat(end)];
%     Nlat = [Nlat;1];
end
%% Geometrcal approximations
N = round(Width/D)+1;
M = round(Spacing/D); M = M - N;
Ku  = Ku/N;
Kuh = Kuh/N;
if Kuh == inf, Kuh = 100*Ku; end

%% Do calculation
T = -Tmax:dt:Tmax;
fprintf(1, '\n');
fprintf(1, 'Calculating displacements due to moding load on full space...\n');
fprintf(1, 'Solving for %d frequencies between %f and %f ...\n', length(W), W(1)/2/pi, W(end)/2/pi);
fprintf(1, '[0%%     20%%       40%%       60%%       80%%      100%%]\n');
fprintf(1, '[');
STARS = zeros(50,1); star = 0;
for iW = 1: 1: length(W)
    if mod(iW,round(length(W)/50))==0
        star = star + 1;
        if star <= 50 && STARS(star) == 0
            fprintf(1, '*');
            STARS(star) = 1;
        end
    end
    w = W(iW);
    wo = w;
    if w == 0, w = W(2)/100; end
%% Part 1 - calculate interaction forces in the frequency domain
    [ForcesP, ForcesM, UP, UM, UlatP, UlatM] = CalculateFrequencyDomainForces(...
            EI,m,Kv,Kt,Ms,Js,Ku,Kuh,D,Mlat,Knlat,Kslat,Nlat,N,M,V,0,w,DampType);
    if Nlat(end) ~= 0
        UlatP = UlatP(1:end-2,:);
    end
%% Part 2 - calculate displacements in the frequency domain of rail
%           and lattice for positions at interfaces and adjacent ones
    % Lattice
    Ulat = zeros(size(UlatP,1),length(I_out));
    for iM = 0: 1: N+M-1
        km = w/V + 2*pi*iM/(N+M)/D;
        for iOut = 1: 1: length(I_out)
            Ulat(:,iOut) = Ulat(:,iOut) + UlatP(:,iM+1) * exp(-1i*km*D*I_out(iOut));
        end
    end

    % Rail
    EPS = 1e-5*0;
    ForceRP = ForcesP([N+1 N+2]);
    Urail = zeros(2,length(Xout));
    Srail = zeros(2,length(Xout));
    if DampType == VISCOUS
        EIr = real(EI) + 1i*w*imag(EI);
    else
        EIr = real(EI) + 1i*imag(EI);
    end
    km = w/V;
    kr = (w^2*m/EIr)^.25;
    for iOut = 1: 1: length(Xout)
        [c1,d1] = CalcCD(1i*kr,km,-Xout(iOut),(N+M)*D);
        [c2,d2] = CalcCD(kr,km,-Xout(iOut),(N+M)*D);
        % Contribution of moving load
        Urail(:,iOut) = exp(-1i*km*Xout(iOut))/(V*(km^4*EIr-w^2*m))*[1; -1i*km];
        Srail(:,iOut) = EIr*exp(-1i*km*Xout(iOut))/(V*(km^4*EIr-w^2*m))*[-km^2; 1i*km^3];
        % Contribution of rail pads
        Urail(:,iOut) = Urail(:,iOut) - 1/(4*EIr) * ...
            [(-1i*c1-c2)/kr^3 (d1-d2)/kr^2; (-d1+d2)/kr^2 (-1i*c1+c2)/kr] * ForceRP;
        Srail(:,iOut) = Srail(:,iOut) - kr^2/4 * ...
            [(1i*c1-c2)/kr^3 (-d1-d2)/kr^2; (d1+d2)/kr^2 (1i*c1+c2)/kr] * ForceRP;
    end
    
%% Part 3 - Convert displacements to time domain
    if iW == 1
        Urail_t = zeros([size(Urail) length(T)]);
        Ulat_t = zeros([size(Ulat) length(T)]);
        UrailPrev = Urail;
        UlatPrev = Ulat;
        continue;
    end
    dw = W(iW)-W(iW-1);
    %% Real component of integral
    COS1 = cos(W(iW-1)*T)./T.^2/pi;
    SIN1 = sin(W(iW-1)*T)./T/pi;
    COS2 = cos(W(iW)*T)./T.^2/pi;
    SIN2 = sin(W(iW)*T)./T/pi;
    SIN1(T==0) = -dw/(2*pi);
    SIN2(T==0) = dw/(2*pi);
    COS1(T==0) = 0;
    COS2(T==0) = 0;
    % Rail part
    A = real(UrailPrev);
    C = real(Urail);
    B = (C-A)/dw;
    for iT = 1: 1: length(T)
        Urail_t(:,:,iT)  = Urail_t(:,:,iT)  + C*SIN2(iT) - A*SIN1(iT) + B*(COS2(iT)-COS1(iT));
    end
    % Lattice part
    A = real(UlatPrev);
    C = real(Ulat);
    B = (C-A)/dw;
    for iT = 1: 1: length(T)
        Ulat_t(:,:,iT)  = Ulat_t(:,:,iT)  + C*SIN2(iT) - A*SIN1(iT) + B*(COS2(iT)-COS1(iT));
    end
    %% Imaginary component of integral
    COS1 = cos(W(iW-1)*T)./T/pi;
    SIN1 = sin(W(iW-1)*T)./T.^2/pi;
    COS2 = cos(W(iW)*T)./T/pi;
    SIN2 = sin(W(iW)*T)./T.^2/pi;
    SIN1(T==0) = 0;
    SIN2(T==0) = 0;
    COS1(T==0) = 0;
    COS2(T==0) = 0;
    % Rail part
    A = imag(UrailPrev);
    C = imag(Urail);
    B = (C-A)/dw;
    for iT = 1: 1: length(T)
        Urail_t(:,:,iT)  = Urail_t(:,:,iT)  + C*COS2(iT) - A*COS1(iT) - B*(SIN2(iT)-SIN1(iT));
    end
    % Lattice part
    A = imag(UlatPrev);
    C = imag(Ulat);
    B = (C-A)/dw;
    for iT = 1: 1: length(T)
        Ulat_t(:,:,iT)  = Ulat_t(:,:,iT)  + C*COS2(iT) - A*COS1(iT) - B*(SIN2(iT)-SIN1(iT));
    end
    
    UrailPrev = Urail;
    UlatPrev = Ulat;
end
for iStar = star+1:1:50
    fprintf(1, '*');
end
fprintf(1, ']\n');

OUTPUT.T = T;
OUTPUT.Urail = Urail_t;
OUTPUT.Ulat = Ulat_t;

end

function [c,d] = CalcCD(k1,k2,x,L)
    X = floor(x/L);
    exp1 = exp(1i*k2*L*(X+1));
    exp2 = exp(-k1*(x-(X+1)*L));
    exp3 = 1/exp2;
    exp4 = exp((1i*k2+k1)*L);
    exp5 = exp((1i*k2-k1)*L);
    c = exp1*(exp2/(-1+exp4)-exp3/(-1+exp5));
    d = exp1*(-exp2/(-1+exp4)-exp3/(-1+exp5));
end

function VALID = CheckInputs(sInputFile,EI,m,Kv,Kt,Ms,Js,Ku,Kuh,D,Mlat,Knlat,Kslat,Kdlat,Nlat,N,M,DampType,dt,Tmax,V,W)
VALID = 0;
if EI ~= sInputFile.EI, return; end
if m ~= sInputFile.m, return; end
if Kv ~= sInputFile.Kv, return; end
if Kt ~= sInputFile.Kt, return; end
if Ms ~= sInputFile.Ms, return; end
if Js ~= sInputFile.Js, return; end
if Ku ~= sInputFile.Ku, return; end
if Kuh ~= sInputFile.Kuh, return; end
if D ~= sInputFile.D, return; end
if length(Mlat) ~= length(sInputFile.Mlat), return; end
if length(Knlat) ~= length(sInputFile.Knlat), return; end
if length(Kslat) ~= length(sInputFile.Kslat), return; end
if length(Kdlat) ~= length(sInputFile.Kdlat), return; end
if length(Nlat) ~= length(sInputFile.Nlat), return; end
for iLayer = 1: 1: length(Nlat)
    if Mlat(iLayer) ~= sInputFile.Mlat(iLayer), return; end
    if Knlat(iLayer) ~= sInputFile.Knlat(iLayer), return; end
    if Kslat(iLayer) ~= sInputFile.Kslat(iLayer), return; end
    if Kdlat(iLayer) ~= sInputFile.Kdlat(iLayer), return; end
    if Nlat(iLayer) ~= sInputFile.Nlat(iLayer), return; end
end
if N ~= sInputFile.N, return; end
if M ~= sInputFile.M, return; end
if DampType ~= sInputFile.DampType, return; end
if dt ~= sInputFile.dt, return; end
if Tmax > sInputFile.Tmax, return; end
if V ~= sInputFile.V, return; end
if length(W) ~= length(sInputFile.W), return; end
for iW = 1: 1: length(W)
    if W(iW) ~= sInputFile.W(iW), return; end
end
VALID = 1;
end