% Script for moving load transition
clc
clear
% Time step
dt = 0.001;
TS = [num2str(dt) 's'];
TS(find(TS=='.')) = 'p';

%% Track properties
% Rail
Track.a = 2;
Track.EI = 2*6.044e6;
Track.m = 2*60;
Track.DampRail = 0.0001;
% Sleepers
Track.Width = 0.3;
Track.Spacing = 0.6;
Track.Ms = 315;
Track.Js = Track.Ms/6*Track.Width^2;
% Rail pads
Track.Kv = 100e6;
Track.Kt = Track.Kv*Track.Width^2/4;
Track.DampRP = 0.001;
% Under sleeper pads
Track.DampUSP = 0.001;
Track.Ku = 45e9;
Track.Kuh = Track.Ku*1e2*0;
%% Ballast and soil - stiffness properties for full track (2m cross section)
% Test box stifness
D = 0.03;
K0 = 300e6; % Pa/m
WidthLoad = 1;
LxLoad = 0.25;
Lz = 0.3;
K0 = K0*LxLoad*WidthLoad;
Ratio = 1/8;
nXLoad = round(LxLoad/D);
nZ = round(Lz/D);
[Kn,Ks,Kd,Fn,Fs,Fd] = FindStiffnesses(K0,Ratio,nXLoad,nZ);
Kn = Kn/WidthLoad*Track.a; % 7.4488e7
Ks = Ks/WidthLoad*Track.a; % 9.3110e6
Kd = Kd/WidthLoad*Track.a; % 3.2588e7
Ballast.Damp = 0.001;
Ballast.D = D;
Ballast.H = 0.3;
Ballast.Rho = 1800;
Ballast.Kn = Kn;
Ballast.Ks = Ks;
Ballast.Kd = Kd;
% Old values
% Ballast.Kn = 1.4222e+07;
% Ballast.Ks = 1.7778e+06;
% Ballast.Kd = 6.2222e+06;

% Example new
Soil.Left.FilenameDM = ['ExampleGouda/DM_' TS '.mat'];
Soil.Left.FilenameID = ['ExampleGouda/ID_' TS '.mat'];
% Normal parameters
Soil.Left.Cs = 140*[1;1];
Soil.Left.Cp = 262*[1;1];
Soil.Left.Rho = 18952*[1;1];
Soil.Left.Damp = 1.5859e-3*[1;1];
Soil.Left.H = 0.6*[1;1];
Soil.Left.KvBot = 1.4348e8;
Soil.Left.KhBot = 3.0572e9;
Soil.Left.CvBot = 2.0328e4;
Soil.Left.ChBot = 3.8462e4;
% Modified parameters
Soil.Left.Cs = 140*[1;1];
Soil.Left.Cp = 230*[1;1];
Soil.Left.Rho = 18952*[1;1];
Soil.Left.Damp = 1.5859e-3*[1;1];
Soil.Left.H = 0.6*[1;1];
Soil.Left.KvBot = 1.4348e8;
Soil.Left.KhBot = 3.0572e9;
Soil.Left.CvBot = 2.0328e4;
Soil.Left.ChBot = 3.8462e4;
% Soil.Left.KvBot = [];
% Soil.Left.KhBot = [];
% Soil.Left.CvBot = [];
% Soil.Left.ChBot = [];
Soil.Right = Soil.Left;
Soil.Mid = Soil.Left;
Soil.Mid.Cs(end) = inf;
Soil.Mid.Cp(end) = inf;
Soil.Mid.KvBot = inf;
Soil.Mid.KhBot = inf;
Soil.Mid.CvBot = inf;
Soil.Mid.ChBot = inf;
%                       Left Mid Right
Soil.MidSections = [5      1   0     0 ; % Five sleepers at beginning
                    7      1   0     0 ; % Seven sleepers at transition
                    4      0   1     0 ; % Culvert
                    7      0   0     1 ; % Seven sleepers at transition
                    5      0   0     1]; % Five sleepers at end

%% Approach slab
% Left Approach slav
ApproachSlab(1).Cells = [6 12];
ApproachSlab(1).E = 20e9;
ApproachSlab(1).Thick = 0.3;
ApproachSlab(1).Rho = 2300;
ApproachSlab(1).Depth = 0.6;
ApproachSlab(1).RotSpringStart = 0;
ApproachSlab(1).RotSpringEnd = inf;
ApproachSlab(1).VertSpringStart = 0;
ApproachSlab(1).VertSpringEnd = inf;
% Right approach slab
ApproachSlab(2).Cells = [17 23];
ApproachSlab(2).E = 20e9;
ApproachSlab(2).Thick = 0.3;
ApproachSlab(2).Rho = 2300;
ApproachSlab(2).Depth = 0.6;
ApproachSlab(2).RotSpringStart = inf;
ApproachSlab(2).RotSpringEnd = 0;
ApproachSlab(2).VertSpringStart = inf;
ApproachSlab(2).VertSpringEnd = 0;
% If no approach slabs, uncomment the following instruction
% ApproachSlab = [];

OutputFilename = ['ExampleGouda/Culvert_ApproachSlab_' TS '.mat'];
% OutputFilename = ['ExampleGouda/Culvert_' TS '.mat'];
% OutputFilename = ['ExampleGouda/NoCulvert_' TS '.mat'];
CALCULATE = 0;

%% Inputs
Inputs.Track = Track;
Inputs.Ballast = Ballast;
Inputs.Soil = Soil;
Inputs.ApproachSlab = ApproachSlab;

%% Calculate response to moving load
Tmax = 2;
V = 60;
W = [0.025:0.025:15 15.1:0.1:60 61:1:100 110:10:1000]*2*pi;

if CALCULATE
    OUTPUT = MovingLoadTransition(Inputs,dt,Tmax,V,W);
    save(OutputFilename, 'OUTPUT','-v7.3');
end
load(OutputFilename);

SLEEPER = OUTPUT.SLEEPER;
RAIL = OUTPUT.RAIL;
BALLAST = OUTPUT.BALLAST;
BEAM = OUTPUT.BEAM;
SPRING = OUTPUT.SPRING;
NODE_TYPE = OUTPUT.NODE_TYPE;
ad2Connection = OUTPUT.ad2Connection;
T = OUTPUT.T;
ad2Displ = OUTPUT.ad2Displ;
ad2Vel = OUTPUT.ad2Vel;

%% Calculate energy transfer to ballast by each sleeper
SLEEPER_NODES = find(NODE_TYPE(:,1)==SLEEPER);
nSleeper = length(SLEEPER_NODES);
SleeperEnergy = zeros(nSleeper,1);
for iSleeper = 1: 1: nSleeper
    Node = SLEEPER_NODES(iSleeper);
    Connections = find((ad2Connection(:,2)==Node)+(ad2Connection(:,3)==Node));
    Connections = ad2Connection(Connections,:);
    nConnections = size(Connections,1);
    for iConnection = 1: 1: nConnections
        if Connections(iConnection,1) ~= SPRING, continue; end
        Node1 = Connections(iConnection,2);
        Node2 = Connections(iConnection,3);
        if NODE_TYPE(Node1,1) == RAIL || NODE_TYPE(Node2,1) == RAIL
            continue;
        end
        % Calculate force on spring
        dofs =  Connections(iConnection,4:7);
        K = Connections(iConnection,8);
        C = imag(K); K = real(K);
        K = K*(dofs'*dofs);
        C = C*(dofs'*dofs);
        F = K*ad2Displ([2*Node1-1; 2*Node1; 2*Node2-1; 2*Node2],:) + ...
            C*ad2Vel([2*Node1-1; 2*Node1; 2*Node2-1; 2*Node2],:);
        if NODE_TYPE(Node1,1) == SLEEPER && NODE_TYPE(Node2,1) == BALLAST
            SleeperEnergy(iSleeper) = SleeperEnergy(iSleeper) + ...
                sum(sum(-F(3:4).*ad2Vel([2*Node2-1; 2*Node2])))*dt;
        elseif NODE_TYPE(Node1,1) == BALLAST && NODE_TYPE(Node2,1) == SLEEPER
            SleeperEnergy(iSleeper) = SleeperEnergy(iSleeper) + ...
                sum(sum(-F(1:2,:).*ad2Vel([2*Node1-1; 2*Node1],:)))*dt;
        else
            fprintf(1, 'ERROR: something went wrong...\n');
            return;
        end
    end
end
plot(1:1:nSleeper, SleeperEnergy, 'o');
return;                

%% Calculate theoretical response
CALC_THEOR = 0;
if CALC_THEOR
    dW = 2*pi/10/Tmax;
    Wf = 2*pi/0.001;
    W = [dW/100 dW:dW:Wf/100 Wf/100+10*dW:10*dW:Wf];
    OUTPUT_Theor = MovingLoadTheoretical(Inputs,1,0.001,Tmax,V,W,...
                                            [], 0);
    save('ExampleGouda/Theor.mat','OUTPUT_Theor');
end
%% Response of rail at connectons with sleepers
close all;
load ExampleGouda/Theor.mat
RAIL_NODES = find(NODE_TYPE(:,1)==RAIL);
[Min,kNodeStart] = min(NODE_TYPE(RAIL_NODES,2));
[Max,kNodeEnd] = max(NODE_TYPE(RAIL_NODES,2));
kNodeStart = RAIL_NODES(kNodeStart);
kNodeEnd = RAIL_NODES(kNodeEnd);
NODES_MID = find(NODE_TYPE(:,1) == SLEEPER);
NODES_MID = NODES_MID - 1;
COLOR = 'brgcm';
tenter = Min/V;
tleave = Max/V;
close all
for iNode = 1: 1: length(NODES_MID)
    figure(1);
    plot(T, ad2Displ(2*NODES_MID(iNode)-1,:),COLOR(1+mod(iNode-1,length(COLOR))), 'LineWidth', 2); hold on;
    plot(tenter*[1 1], [min(ad2Displ(2*NODES_MID(iNode)-1,:)) max(ad2Displ(2*NODES_MID(iNode)-1,:))], '--k');
    plot(tleave*[1 1], [min(ad2Displ(2*NODES_MID(iNode)-1,:)) max(ad2Displ(2*NODES_MID(iNode)-1,:))], '--k');
    [MAX,kT] = max(ad2Displ(2*NODES_MID(iNode)-1,:));
    figure(2);
    plot(T-T(kT), ad2Displ(2*NODES_MID(iNode)-1,:),COLOR(1+mod(iNode-1,length(COLOR))), 'LineWidth', 1); hold on
end
figure(1);
xlabel('Time [s]');
ylabel('u_r_a_i_l [m]');
title('a)');
grid on;
figure(2);
UrailTheor = permute(OUTPUT_Theor.Urail(1,:,:), [3 1 2]);
[MAX,kT] = max(UrailTheor);
plot((OUTPUT_Theor.T-OUTPUT_Theor.T(kT)), UrailTheor,'--k', 'LineWidth', 2);
xlabel('Time [s]');
ylabel('u_r_a_i_l [m]');
title('b)');
grid on;
return;

%% Time domain incident displacements and duhamel matrices
COMPARE_1 = 0;
if COMPARE_1
    CALC_TD = 1;
    if CALC_TD
        OUTPUT_TD = IncidentDisplacement_TD(Inputs,dt,Tmax,V,10);
        save('ID_DM_TD.mat', 'OUTPUT_TD');
    end
    load ID_DM_TD.mat
end

%% Compare Duhamel matrices
close all
COMPARE_1 = 0; if COMPARE_1, load('ID_DM_TD.mat'); end
load(Soil.Left.FilenameDM);
W = OUTPUT_Freq.W;
Tlim = Tmax;
nDof = size(OUTPUT_Freq.Mats(1).KRight,1);
FlexRight = zeros(nDof,nDof,length(W));
FlexLeft = zeros(nDof,nDof,length(W));
for iW = 1: 1: length(W)
    FlexRight(:,:,iW) = inv(OUTPUT_Freq.Mats(iW).KRight);
    FlexLeft(:,:,iW) = inv(OUTPUT_Freq.Mats(iW).KLeft);
end

for iDof = 1: 1: nDof
    for jDof = 1: 1: nDof
        subplot(2,2,1);
        plot(Tout,permute(Fright(iDof,jDof,:),[3 1 2]),'b', 'LineWidth', 2); hold on;
        if COMPARE_1
            plot(OUTPUT_TD.T,permute(OUTPUT_TD.IR_L(iDof,jDof,:),[3 1 2]),'--r', 'LineWidth', 2);
        end
        title(num2str(iDof));
        hold off;
        subplot(2,2,2);
        plot(Tout,permute(Fleft(iDof,jDof,:),[3 1 2]),'b', 'LineWidth', 2); hold on;
        if COMPARE_1
            plot(OUTPUT_TD.T,permute(OUTPUT_TD.IR_R(iDof,jDof,:),[3 1 2]),'--r', 'LineWidth', 2);
        end
        title(num2str(jDof));
        hold off;
        subplot(2,2,3);
        plot(W/2/pi,real(permute(FlexRight(iDof,jDof,:),[3 1 2])),'b', ...
             W/2/pi,imag(permute(FlexRight(iDof,jDof,:),[3 1 2])),'r', 'LineWidth', 2);
        subplot(2,2,4);
        plot(W/2/pi,real(permute(FlexLeft(iDof,jDof,:),[3 1 2])),'b', ...
             W/2/pi,imag(permute(FlexLeft(iDof,jDof,:),[3 1 2])),'r', 'LineWidth', 2);
        pause;
    end
end
return;
%% Compare Incident displacements
close all
COMPARE_1 = 0; if COMPARE_1, load('ID_DM_TD.mat'); end
load(Soil.Left.FilenameID);
W = sInputFile.W;
Tlim = Tmax;
for iDof = 1: 1: nDof
    subplot(2,2,1);
    plot(T,Uleft(iDof,:),'b','LineWidth',2); hold on;
    if COMPARE_1
        plot(OUTPUT_TD.Tl,OUTPUT_TD.ULeft(iDof,:),'--r','LineWidth',2);
    end
    title(num2str(iDof));
    hold off;
    subplot(2,2,2);
    plot(T,Uright(iDof,:),'b','LineWidth',2); hold on;
    if COMPARE_1
        plot(OUTPUT_TD.Tl,OUTPUT_TD.URight(iDof,:),'--r','LineWidth',2);
    end
    hold off;
    subplot(2,2,3);
    plot(W(2:end)/2/pi,real(Uleft_W(iDof,2:end)),'b',W(2:end)/2/pi,imag(Uleft_W(iDof,2:end)),'r','LineWidth',2);
    subplot(2,2,4);
    plot(W(2:end)/2/pi,real(Uright_W(iDof,2:end)),'b',W(2:end)/2/pi,imag(Uright_W(iDof,2:end)),'r','LineWidth',2);
    pause;
end
