# -*- coding: utf-8 -*-
### Created by Jaho Koo, IHE Delft, TU Delft, K-water

from pyomo.environ import (AbstractModel, Constraint, Reals, NonNegativeIntegers, Boolean, Param, RangeSet, Var,
                           NonNegativeReals, Objective, Set, minimize, Binary)
# ================================================================================

MT = 264  # the maximun discharge via turbines
MSP = int(11680)  # the maximum discharge via spillway gates
LWL = 60.0
FWL = 80.0
NHWL = 76.5
LWS = 385583000
FWS = 1408538000
NHWS = 1170402000




def MPC_formula():  # etc 1
    model = AbstractModel()

    model.t = Set()
    model.w = Set()
    model.wp = Set()
    model.wm = Set()
    model.QIN = Param(model.t, model.w)  # inflow input
    model.gpf = Param(model.t, model.wp)
    model.mhf = Param(model.t, model.wm)
    model.P_QIN = Param(model.w)
    model.P_gpf = Param(model.wp)
    model.P_mhf = Param(model.wm)
    model.N_QIN = Param()
    model.N_gpf = Param()
    model.N_mhf = Param()
    model.etc_ = Param()
    model.I_G = Param()

    model.Md = Param(model.t)  # demand input
    model.I_RWS = Param()
    model.SO_P = Param(model.t)
    model.SP_P = Param(model.t)
    model.beta1 = Param()
    model.beta2 = Param()
    model.delta_SO_1 = Param()

    model.W1 = Param()
    model.W2 = Param()
    model.W3 = Param()
    model.W4 = Param()
    model.W5 = Param()
    model.W6 = Param()
    model.TWS_N = Param()
    model.TWS_L = Param()
    model.TWS_F = Param()
    model.N = Param()
    model.max_INF = Param()
    model.FWS = Param()
    model.LWS = Param()
    model.NHWS = Param()
    model.TWS_U = Param()
    model.TWS_H = Param()
    model.DCW_H = Param()


    model.SO = Var(model.t, domain=NonNegativeReals, bounds=(0, (MT + MSP)), initialize=model.SO_P)
    model.GWL = Var(model.t, model.wp, model.wm, within=Reals, initialize=0)
    model.ST = Var(model.t, domain=NonNegativeReals, bounds=(0, MT), initialize=MT)
    model.SP = Var(model.t, domain=NonNegativeReals, bounds=(0, MSP), initialize=model.SP_P)
    model.S = Var(model.t, model.w, domain=NonNegativeReals, bounds=(LWS, FWS), initialize=NHWS)
    model.d_S1 = Var(model.t, model.w, within=NonNegativeReals, initialize=0)
    model.d_S2 = Var(model.t, model.w, within=NonNegativeReals, initialize=0)
    model.d_S1t = Var(model.t, within=NonNegativeReals, initialize=0)
    model.d_S2t = Var(model.t, within=NonNegativeReals, initialize=0)
    model.d_ST = Var(model.t, within=NonNegativeReals, initialize=0)

    model.d_SO0_1 = Var(within=NonNegativeReals, initialize=0)
    model.d_SO0_2 = Var(within=NonNegativeReals, initialize=0)
    model.d_SO1 = Var(model.t, within=NonNegativeReals, initialize=0)
    model.d_SO2 = Var(model.t, within=NonNegativeReals, initialize=0)
    model.max_SP = Var(within=NonNegativeReals)
    model.d_SI = Var(model.t, within=NonNegativeReals, initialize=0)

    model.alpha_r1 = Var(model.t, within=NonNegativeReals, initialize=0)
    model.alpha_r2 = Var(model.t, within=NonNegativeReals, initialize=0)
    model.z_r1 = Var(model.t, model.w, within=NonNegativeReals, initialize=0)
    model.z_r2 = Var(model.t, model.w, within=NonNegativeReals, initialize=0)
    model.CVaR_r1 = Var(model.t, within=NonNegativeReals, initialize=0)
    model.CVaR_r2 = Var(model.t, within=NonNegativeReals, initialize=0)

    model.d_cvar_r1 = Var(model.t, within=NonNegativeReals, initialize=0)
    model.d_cvar_r2 = Var(model.t, within=NonNegativeReals, initialize=0)

    model.alpha_dl1 = Var(model.t, within=NonNegativeReals, initialize=0)
    model.z_dl1 = Var(model.t, model.wp, model.wm, within=NonNegativeReals, initialize=0)
    model.CVaR_dl1 = Var(model.t, within=NonNegativeReals, initialize=0)

    model.d_cvar_dl1 = Var(model.t, within=NonNegativeReals, initialize=0)

    model.Obj_v = Var(model.t, within=NonNegativeReals, initialize=0)

    # # ======================================= essential parts for all MPC formulations ==================

    # # ======================================= system dynamic and default constraints ==================
    # System dynamics
    def _S1(model, t, w):
        if t == 0:
            return model.S[t, w] - model.I_RWS - (model.QIN[t, w] - model.SO[t]) * 3600 == 0
        return model.S[t, w] - model.S[t - 1, w] - (model.QIN[t, w] - model.SO[t]) * 3600 == 0

    model.storage1 = Constraint(model.t, model.w, rule=_S1)

    # Hard constraint on RWL to be between LWS and FWS (same as the bound)
    def _S2(model, t, w):
        return (LWS, model.S[t, w], FWS)

    model.storage2 = Constraint(model.t, model.w, rule=_S2)

    # #definition of QT
    def SO_PT(model, t):
        return model.SO[t] - model.SP[t] - model.ST[t] == 0

    model.SO_PT = Constraint(model.t, rule=SO_PT)

    # #definition of QT
    def ST_PT(model, t):
        return model.SO[t] - MT + model.d_ST[t] - model.SP[t] == 0

    model.ST_PT = Constraint(model.t, rule=ST_PT)

    # Hard constraint that turbine and spill flow must be above demand:
    def Demand_meet(model, t):
        return model.SO[t] - model.Md[t] >= 0

    model.demanddown = Constraint(model.t, rule=Demand_meet)

    # Hard constraint to stick previously determined QSP and QT
    def Stick_first_QTO(model, t):
        if t==0:
            return (model.delta_SO_1*-1, model.SO[0] - model.SO_P[1], model.delta_SO_1)
        else:
            return model.SO[t] >= 0

    model.stick_first_qsp_ = Constraint(model.t, rule=Stick_first_QTO)


    # # ======================================= different parts for each MPC formulation ==================

    # # ============  relating to changes in outflows

    # minimize the changes between outflow schedules
    def SO_1(model, t):  # min model.d_QSP1[t]
        if t == 0:
            return model.d_SO1[t] + model.d_SO2[t] == 0
        elif t == 1:
            return (model.SO[t-1] - model.SO_P[t]) / (int(t)*0.5) + model.d_SO1[t] - model.d_SO2[t] == 0
        else:
            return (model.SO[t-1] - model.SO_P[t]) / (int(t)-1) + model.d_SO1[t] - model.d_SO2[t] == 0
    model.SO_1 = Constraint(model.t, rule=SO_1)


    # minimize the peak spillway outflow
    def SP_max(model, t):  # min model.max_QO
        return model.SP[t] - model.max_SP <= 0

    model.SP_max = Constraint(model.t, rule=SP_max)


    # # ======================================= The formulation of the 2-types of a CVaR constraint for uncertain inflow ==================

    # # # ============  relating to RWL
    #
    # keep RWL following TWL as musch as possible
    def Gap_S_1(model, t, w):  # min model.d_S1[t]
        return model.S[t, w] - model.TWS_N - model.d_S1[t, w] <= 0

    model.Gap_S_1 = Constraint(model.t, model.w, rule=Gap_S_1)
    #
    def d_S1_(model, t):  # min model.d_S1[t]
        return sum(model.d_S1[t, i] for i in model.w) / model.N_QIN == model.d_S1t[t]

    model.d_S1_ = Constraint(model.t, rule=d_S1_)
    #
    def Gap_S_2(model, t, w):  # min model.d_S2[t]
        return model.TWS_L - model.S[t, w] - model.d_S2[t, w] <= 0

    model.Gap_S_2 = Constraint(model.t, model.w, rule=Gap_S_2)

    def d_S2_(model, t):  # min model.d_S1[t]
        return sum(model.d_S2[t, i] for i in model.w) / model.N_QIN == model.d_S2t[t]

    model.d_S2_ = Constraint(model.t, rule=d_S2_)


    # # ======================================= The formulation of a CVaR constraint for reservoir water level ==================

    def Z_r1(model, t, w):
        return model.S[t, w] - model.alpha_r1[t] - model.z_r1[t, w] <= 0
    model.Z_r1 = Constraint(model.t, model.w, rule=Z_r1)

    def Z_r2(model, t, w):
        return model.alpha_r2[t] - model.S[t, w] - model.z_r2[t, w] <= 0
    model.Z_r2 = Constraint(model.t, model.w, rule=Z_r2)

    def CVaR_r1_(model, t):
        return model.CVaR_r1[t] == model.alpha_r1[t] + (1 / model.N_QIN) * sum(model.P_QIN[i] * model.z_r1[t, i] for i in model.w) / (1-model.beta1)
    model.CVaR_r1_ = Constraint(model.t, rule=CVaR_r1_)

    def CVaR_r2_(model, t):
        return model.CVaR_r2[t] == model.alpha_r2[t] - (1 / model.N_QIN) * sum(model.P_QIN[i] * model.z_r2[t, i] for i in model.w) / (1-model.beta1)
    model.CVaR_r2_ = Constraint(model.t, rule=CVaR_r2_)

    def CVaR_r1_FWL(model, t):
        # return (model.TWS_N, model.CVaR_r1[t], model.TWS_H)
        return model.CVaR_r1[t] - model.TWS_H - model.d_cvar_r1[t] <= 0
    model.CVaR_r1_FWL = Constraint(model.t, rule=CVaR_r1_FWL)

    def CVaR_r2_LWL(model, t):
        # return (model.TWS_U, model.CVaR_r2[t], model.TWS_L)
        return model.TWS_U - model.d_cvar_r2[t] - model.CVaR_r2[t] <= 0
    model.CVaR_r2_LWL = Constraint(model.t, rule=CVaR_r2_LWL)


    def dwl_(model, t, wp, wm):
        # return model.GWL[t, w] == +1.17952818979681E-05*model.SO[t]+0.0959532830510608*model.gpf[t, w]+0.122602351047384*model.mhf[t, w]-0.10091590348334+model.etc_[t]
        if t == 0:
            return model.GWL[t, wp, wm] == 2.72017E-05 * model.SO_P[t] + 0.904003508 * model.I_G + 0.029571729 * model.gpf[t, wp] + 0.060278127 * model.mhf[t, wm]
        else:
            return model.GWL[t, wp, wm] == 2.72017E-05 * model.SO[t-1] + 0.904003508 * model.GWL[t-1, wp, wm] + 0.029571729 * model.gpf[t, wp] + 0.060278127 * model.mhf[t, wm]

    model.dwl_1 = Constraint(model.t, model.wp, model.wm, rule=dwl_)


    def Z_dl1(model, t, wp, wm):
        return model.GWL[t, wp, wm] - model.alpha_dl1[t] - model.z_dl1[t, wp, wm] <= 0
    model.Z_dl1 = Constraint(model.t, model.wp, model.wm, rule=Z_dl1)

    def CVaR_dl1_(model, t):
        return model.CVaR_dl1[t] == model.alpha_dl1[t] + (1 / (model.N_gpf*model.N_mhf)) * sum(model.P_gpf[i] * model.P_mhf[j] * model.z_dl1[t, i, j] for i in model.wp for j in model.wm) / (1-model.beta2)
    model.CVaR_dl1_ = Constraint(model.t, rule=CVaR_dl1_)

    def CVaR_dl1_DCW(model, t):
        # return (model.TWS_N, model.CVaR_r1[t], model.TWS_H)
        return model.CVaR_dl1[t] - model.d_cvar_dl1[t] - model.DCW_H <= 0
    model.CVaR_dl1_DCW = Constraint(model.t, rule=CVaR_dl1_DCW)


    def Obj_var(model, t):
        return model.Obj_v[t] == (model.max_SP) * model.W1 / MSP + \
                                 (model.d_cvar_r1[t] * model.W2 + model.d_cvar_r2[t] * model.W3) / (FWS - model.TWS_H) + model.d_cvar_dl1[t] * model.W4 / (9.5 - model.DCW_H) + \
                                 (model.d_S1t[t]+model.d_S2t[t]) * model.W5 / (FWS - NHWS) + \
                                 (model.d_SO1[t] + model.d_SO2[t]) * model.W6 / MSP + \
                                 (model.d_ST[t]) * 1 * 1000 / MSP

    model.Obj_var = Constraint(model.t, rule=Obj_var)

    def obj_f(model):
        return sum(model.Obj_v[i] for i in model.t)

    model.obj = Objective(rule=obj_f, sense=minimize)

    return model

