% MATLAB script built by Joost de Winter and Pavlo Bazilinksyy

% Belonging to the paper
% Sripada, A., Bazilinskyy, P., & De Winter, J. C. F. (2021). Automated vehicles that communicate implicitly: Examining the use of lateral position within the lane. Ergonomics

clear all;close all;clc; %#ok<*CLALL>
video_info = xlsread('mapping.xlsx','B2:G57'); % Read Excel file with information about the 56 videos
read_raw % Read raw experiment data
savepng=0; % if 1 then save .png files of figures

%% Transform to RT matrix (No of participants x 56 videos x 1000 RTs per participant) to RTf0, containing the keypress data in 0.1-s increments
RTf = NaN(size(RT,1),size(RT,2),1000);
NumberofVideosPlayed = zeros(size(RT,1),size(RT,2)); % set counter to 0
for i = 1:size(RT, 1) % loop over participants
    for j = 1:size(RT, 2) % loop over videos (56)
        if ok1(i,j) && ok2(i,j) && ok3(i,j) && TrialDurationEstimated(i,j)/1000 < 12 && NumberofVideosPlayed(i,StimulusIDs(i,j)) <=1 % Determine if trial is valid
            rts = squeeze(RT(i,j,:))/1000; % Vector RTs of person in seconds. The vector is filled with 0s after the last RT.
            rts(isnan(rts))=[]; % remove NaNs
            
            % filling the 0.5-s gaps
            if length(rts)>=2 % fill data gap for first half a second (0.5 s) of holding the key
                if  rts(1)<=0.54 % if the first key press is 'exactly' at 0.5 seconds, suggesting that participants held the key
                    rts=[transpose(0.04:0.04:rts(1));rts(1);rts(2:end)];
                end
            end
            if length(rts)==1 % fill data gap for first half a second (0.5 s) of holding the key
                if  rts(1)<=0.54 % if the first key press is 'exactly' at 0.5 seconds, suggesting that participants held the key
                    rts=[transpose(0.04:0.04:rts(1));rts(1)];
                end
            end
            gapindexes=1+find(diff(rts)>=0.42 & diff(rts)<=0.54); % indexes where the gap is 'exactly' 0.5 seconds, suggesting that participants held the key
            for k=length(gapindexes):-1:1 % loop over all gaps (backward order)
                rts=[rts(1:gapindexes(k)-1);transpose(rts(gapindexes(k)-1)+0.04:0.04:rts(gapindexes(k)));rts(gapindexes(k):end)]; % fill with key press data
            end
            % end of filling the 0.5-s gaps
            
            rts=10*rts./(TrialDurationEstimated(i,j)/1000); % normalize the keypress data to a range from 0 s to 10 s (it is known that each video is 10 s long)
            rts(rts>100)=[]; % remove keypress times above 100 seconda as they do not fit the data matrix
            
            O = 10 * ceil(rts*10)/10; % Indices where key is pressed (in 0.1-second increments)
            
            NumberofVideosPlayed(i,StimulusIDs(i,j))=NumberofVideosPlayed(i,StimulusIDs(i,j))+1;
            
            if isnan(RTf(i,StimulusIDs(i,j),1)) % if RTf is empty
                RTf(i,StimulusIDs(i,j),:) = 0; % fill RTf with zeros
                O(O==0) = [];
                RTf(i,StimulusIDs(i,j),O) = RTf(i,StimulusIDs(i,j),O) + 1;
            else
                O(O==0) = [];
                RTf(i,StimulusIDs(i,j),O) = RTf(i,StimulusIDs(i,j),O) + 1;
            end
            
        end
        
    end
end

instructionsf1 =  [find(X(:,20)==0); find(X(:,20)==2)]; % Informed group
instructionsf2 =  [find(X(:,20)==1); find(X(:,20)==3)]; % Uninformed group
deviationgroup1 = [find(X(:,20)==0); find(X(:,20)==1)]; % Group who watched Videos 1-28 (yielding = towards)
deviationgroup2 = [find(X(:,20)==2); find(X(:,20)==3)]; % Group who watched Videos 29-56 (yielding = away)
NumberofVideosPlayed(deviationgroup1,29:56)=NaN;
NumberofVideosPlayed(deviationgroup2,1:28)=NaN;

disp([num2str(sum(nansum(NumberofVideosPlayed))) ' observed number of trials, out of a maximum possible of ' num2str(size(X,1)*num_stimuli)]) %#ok<*NANSUM>

% Divide keypresses by the number of times the video was played
for i = 1:size(RT, 1) % loop over participants
    for j = 1:size(RT, 2) % loop over videos (56)
        RTf(i,j,:)=RTf(i,j,:)./NumberofVideosPlayed(i,j);
    end
end
%% Slider question data
IntuitvenessRating = NaN(size(RP,1),size(RT,2));
NumberofIntuitivenessQuestions = zeros(size(RT,1),size(RT,2)); % set counter to 0

for i = 1:size(RP, 1) % loop over participants
    for j = 1:size(RP, 2) % loop over videos
        if ok1(i,j)==1 && ok2(i,j)==1 && ok3(i,j)==1 && TrialDurationEstimated(i,j)/1000 < 12 && NumberofIntuitivenessQuestions(i,StimulusIDs(i,j))<=1 % Determine if trial is valid
            if isnan(IntuitvenessRating(i,StimulusIDs(i,j)))
                IntuitvenessRating(i,StimulusIDs(i,j)) = RP(i,j);
                NumberofIntuitivenessQuestions(i,StimulusIDs(i,j))=1;
            else
                IntuitvenessRating(i,StimulusIDs(i,j)) = IntuitvenessRating(i,StimulusIDs(i,j)) + RP(i,j);
                NumberofIntuitivenessQuestions(i,StimulusIDs(i,j))=2;
            end
        end
    end
end
for i = 1:size(RP, 1) % loop over participants
    for j = 1:size(RP, 2) % loop over videos
        IntuitvenessRating(i,j)=IntuitvenessRating(i,j)./NumberofIntuitivenessQuestions(i,j);
    end
end
%% Most common countries (after filtering)
[~, ~, ub] = unique(Country);
test2counts = histcounts(ub, 'BinMethod','integers');
[B,I] = maxk(test2counts,10);
country_unique = unique(Country);
if (size(country_unique,1) > 0)
    disp('Most common countries (after filtering) = ')
    disp(country_unique(I)')
    disp(B)
end
%% Variables
% 1. VehicleDeviation (0 = no deviation, 1 = 0.4 m, 2 = 0.8 m, 3 = 1.2 m)
% 2. DeviationDirection (0 = no deviation, 1 = away from pedestrian, 2 = towards pedestrian)
% Videos 1-28: Away from pedestrian for non-yielding, towards pedestrian for yielding (deviation group 1)
% Videos 29-56: Towards pedestrian for non-yielding, away from pedestrian for yielding (deviation group 2)
% 3. BrakingBehaviour (1 = non-yielding, 2 = yielding)
% 4. Vehicle Speed (1 = 39 km/h)
% 5. DeviationOnset (0 = no deviation, 1 = starts at 30 m, 2 = starts at 50 m)
% 6. Indicator (1 = Off, 2 = On)

time = (0.1:0.1:size(RTf,3)/10)'; % Time vector

Country_TowardsYielding=Country(deviationgroup1);
Country_AwayYielding=Country(deviationgroup2);

PPinclude_1=find(X(:,20)==0); % Informed   (videos 1-28; yielding = towards)
PPinclude_2=find(X(:,20)==1); % Uninformed (videos 1-28; yielding = towards)
PPinclude_3=find(X(:,20)==2); % Informed   (videos 29-56; yielding = away)
PPinclude_4=find(X(:,20)==3); % Uninformed (videos 29-56; yielding = away)
disp(['Number of participants in group Informed & Towards: ' num2str(size(PPinclude_1,1))]);
disp(['Number of participants in group Uninformed & Towards: ' num2str(size(PPinclude_2,1))]);
disp(['Number of participants in group Informed & Away: ' num2str(size(PPinclude_3,1))]);
disp(['Number of participants in group Uninformed & Away: ' num2str(size(PPinclude_4,1))]);
%%
%% Figure 1. Top: Lateral position in non-yielding trials. Middle: Lateral position in yielding trials. Bottom: Vehicle speed in yielding and non-yielding trials.
figure
opengl hardware
filePattern = fullfile('*Yield.txt');
Files = dir(filePattern);
pos_x=NaN(500,13);pos_z=NaN(500,13);plots=NaN(13,1);

subplot(3,1,1)
linecolor1=['r','r','b','b','g','g','r','r','b','b','g','g','k'];
for p = 1 : length(Files)
    baseFileName = Files(p).name;
    data=readtable(baseFileName,'Delimiter',';');
    vehicle_x=2.5+table2array(data(:,2));
    vehicle_z=table2array(data(:,4))-10;
    pos_x(1:length(vehicle_x),p)=vehicle_x;
    pos_z(1:length(vehicle_z),p)=vehicle_z;
    pos_x(2:end+1,:)=pos_x;
    pos_x(1,:)=pos_x(2,:);
    pos_z(2:end+1,:)=pos_z;
    pos_z(1,:)=50;
    plots(p)= plot(pos_z(:,p),pos_x(:,p),linecolor1(p),'LineWidth',3);
    hold on
end

title('Lateral position - Yielding vehicles','FontSize',14)
xlabel('Distance from pedestrian (m)','FontSize',14)
ylabel('Lateral position (m)','FontSize',14)
h1=legend(plots([13 1 3 5]),'No deviation','0.4 m','0.8 m','1.2 m','location','northeast');
set(gca,'xdir','reverse','xlim',[0 50],'ylim',[-1.65 1.65],'pos',[0.045 0.76 0.94 0.20],'ticklength',[.003 .003],'ytick',-1.5:0.5:1.5)
grid on;box on;

filePattern = fullfile('*NY.txt');
Files = dir(filePattern);
pos_x=NaN(500,13);pos_z=NaN(500,13);plots=NaN(13,1);
subplot(3,1,2)
for p = 1 : length(Files)
    baseFileName = Files(p).name;
    data=readtable(baseFileName,'Delimiter',';');
    vehicle_x=2.5+table2array(data(:,2));
    vehicle_z=table2array(data(:,4))-10;
    pos_x(1:length(vehicle_x),p)=vehicle_x;
    pos_z(1:length(vehicle_z),p)=vehicle_z;
    pos_x(2:end+1,:)=pos_x;
    pos_x(1,:)=pos_x(2,:);
    pos_z(2:end+1,:)=pos_z;
    pos_z(1,:)=50;
    plots(p)= plot(pos_z(:,p),pos_x(:,p),linecolor1(p),'LineWidth',3);
    hold on
end
title('Lateral position - Non-yielding vehicles','FontSize',14)
xlabel('Distance from pedestrian (m)','FontSize',14)
ylabel('Lateral position (m)','FontSize',14)
h2=legend(plots([13 1 3 5]),'No deviation','0.4 m','0.8 m','1.2 m','location','northeast');
set(gca,'xdir','reverse','xlim',[0 50],'ylim',[-1.65 1.65],'pos',[0.045 0.415 0.94 0.20],'ticklength',[.003 .003],'ytick',-1.5:0.5:1.5)
grid on;box on;

subplot(3,1,3)
for p = 1 : length(Files)
    baseFileName = Files(p).name;
    data=readtable(baseFileName,'Delimiter',';');
    vehicle_x=2.5+table2array(data(:,2));
    vehicle_z=table2array(data(:,4))-10;
    pos_x(1:length(vehicle_x),p)=vehicle_x;
    pos_z(1:length(vehicle_z),p)=vehicle_z;
    hold on
end

plot([95 0],[39.1 39.1],'k-.','LineWidth',3)
v2=real(3.6*sqrt((39.1/3.6)^2+2*-1.54*(0:0.01:30)));
plot(30-(0:0.01:30),v2,'k:','LineWidth',3)
plot([95 30],[39.1 39.1],'k','LineWidth',3)
v=real(3.6*sqrt((39.1/3.6)^2+2*-2.74*(0:0.01:30)));
plot(30-(0:0.01:30),v,'k','LineWidth',3)
title('Vehicle speed','FontSize',14)
ylabel('Speed (km/h)','FontSize',14)
h3=legend('Non-yielding (Indicator Off)','Non-yielding (Indicator On)','Yielding','location','southwest');
set(gca,'xdir','reverse','xlim',[0 50],'ylim',[0 42],'pos',[0.045 0.08 0.94 0.20],'ticklength',[.003 .003])
xlabel('Distance from pedestrian (m)','FontSize',14)
grid on;box on
h=findobj('FontName','Helvetica'); set(h,'FontSize',18,'Fontname','Arial')
set(h1,'pos',[0.0501736131910649  0.873794707192073        0.0963541645867123 0.122014534436282])
set(h2,'pos',[0.0501736131910649  0.873794707192073-0.345  0.0963541645867123 0.122014534436282])
if savepng==1;set(gcf,'WindowState', 'maximized');print(gcf,'Figure1.png','-dpng','-r300');end
%% Figure 3. Percentage of trials in which the response key was pressed, for the Informed and Uninformed groups in yielding trials.

% 1. VehicleDeviation (0 = no deviation, 1 = 0.4m, 2 = 0.8m, 3 = 1.2m)
% 2. DeviationDirection (0 = no deviation, 1 = towards road centre, 2 = towards curb)
% Videos 1-28: Towards center for non-yielding, towards curb for yielding (deviation group 1)
% Videos 29-56: Towards curb for non-yielding, towards center for yielding (deviation group 2)
% 3. BrakingBehaviour (1 = non-yielding, 2 = yielding)

v1=find(video_info(:,3)==2 & video_info(:,1)>0 & video_info(:,2)==2);  % Yielding vehicles, Towards deviation
v2=find(video_info(:,3)==2 & video_info(:,1)>0 & video_info(:,2)==1);  % Yielding vehicles, Away deviation
v3=find(video_info(:,3)==2 & video_info(:,1)==0); % Yielding vehicles, Without deviation

figure;hold on;grid on;box on
title('Yielding vehicles')
V1 = 100 * squeeze(nanmean(nanmean(RTf([PPinclude_1;PPinclude_3],v1,:),2),1)); %#ok<*NANMEAN> % Yielding vehicles, With deviation, Informed
V2 = 100 * squeeze(nanmean(nanmean(RTf([PPinclude_2;PPinclude_4],v1,:),2),1)); % Yielding vehicles, With deviation, Uninformed
V3 = 100 * squeeze(nanmean(nanmean(RTf([PPinclude_1;PPinclude_3],v2,:),2),1)); % Yielding vehicles, Without deviation, Informed
V4 = 100 * squeeze(nanmean(nanmean(RTf([PPinclude_2;PPinclude_4],v2,:),2),1)); % Yielding vehicles, Without deviation, Uninformed
V5 = 100 * squeeze(nanmean(nanmean(RTf([PPinclude_1;PPinclude_3],v3,:),2),1)); % Yielding vehicles, Without deviation, Informed
V6 = 100 * squeeze(nanmean(nanmean(RTf([PPinclude_2;PPinclude_4],v3,:),2),1)); % Yielding vehicles, Without deviation, Uninformed

v1p=100 * nanmean(nanmean(RTf([PPinclude_1;PPinclude_3],v1,76:85),3),2);
v2p=100 * nanmean(nanmean(RTf([PPinclude_2;PPinclude_4],v1,76:85),3),2);
v3p=100 * nanmean(nanmean(RTf([PPinclude_1;PPinclude_3],v2,76:85),3),2);
v4p=100 * nanmean(nanmean(RTf([PPinclude_2;PPinclude_4],v2,76:85),3),2);
v5p=100 * nanmean(nanmean(RTf([PPinclude_1;PPinclude_3],v3,76:85),3),2);
v6p=100 * nanmean(nanmean(RTf([PPinclude_2;PPinclude_4],v3,76:85),3),2);

v1p(isnan(v1p))=[];
v2p(isnan(v2p))=[];
v3p(isnan(v3p))=[];
v4p(isnan(v4p))=[];
v5p(isnan(v5p))=[];
v6p(isnan(v6p))=[];

plot(time,V1,'-','Linewidth',5,'color',[0 1 0]);
plot(time,V2,':','Linewidth',5,'color',[0 1 0]);
plot(time,V3,'-','Linewidth',5,'color',[1 0 0]);
plot(time,V4,':','Linewidth',5,'color',[1 0 0]);
plot(time,V5,'-','Linewidth',2,'color',[.5 .5 .5]);
plot(time,V6,':','Linewidth',2,'color',[.5 .5 .5]);

xlabel('Time (s)')
ylabel('Percentage of trials with response key pressed')
set(gca,'LooseInset',[0.01 0.01 0.01 0.01],'xlim',[0 10],'xtick',1:50,'ylim',[0 70])
legend(['Yielding - Towards deviation - Informed group (' num2str(length(v1p)) ' participants, ' num2str(length(v1)) ' videos per participant, \itM\rm = ' sprintf('%0.1f',nanmean(v1p)) '%, \itSD\rm = ' sprintf('%0.1f',nanstd(v1p)) '%)'],...
       ['Yielding - Towards deviation - Uninformed group (' num2str(length(v2p)) ' participants, ' num2str(length(v1)) ' videos per participant, \itM\rm = ' sprintf('%0.1f',nanmean(v2p)) '%, \itSD\rm = ' sprintf('%0.1f',nanstd(v2p)) '%)'],...
       ['Yielding - Away deviation - Informed group (' num2str(length(v3p)) ' participants, ' num2str(length(v2)) ' videos per participant, \itM\rm = ' sprintf('%0.1f',nanmean(v3p)) '%, \itSD\rm = ' sprintf('%0.1f',nanstd(v3p)) '%)'],...
       ['Yielding - Away deviation - Uninformed group (' num2str(length(v4p)) ' participants, ' num2str(length(v2)) ' videos per participant, \itM\rm = ' sprintf('%0.1f',nanmean(v4p)) '%, \itSD\rm = ' sprintf('%0.1f',nanstd(v4p)) '%)'],...
       ['Yielding - No deviation - Informed group (' num2str(length(v5p)) ' participants, ' num2str(length(v3)/2) ' videos per participant, \itM\rm = ' sprintf('%0.1f',nanmean(v5p)) '%, \itSD\rm = ' sprintf('%0.1f',nanstd(v5p)) '%)'],...
       ['Yielding - No deviation - Uninformed group (' num2str(length(v6p)) ' participants, ' num2str(length(v3)/2) ' videos per participant, \itM\rm = ' sprintf('%0.1f',nanmean(v6p)) '%, \itSD\rm = ' sprintf('%0.1f',nanstd(v6p)) '%)'],...
    'autoupdate','off','location','southwest'); %#ok<*NANSTD>
h=findobj('FontName','Helvetica'); set(h,'FontSize',18,'Fontname','Arial')
rectangle('position',[0 0 1 100],'facecolor',[127 127 127 20]/255,'Linewidth',.5);

if savepng==1;set(gcf,'WindowState', 'maximized');print(gcf,'Figure3.png','-dpng','-r300');end
%% Figure 4. Percentage of trials in which the response key was pressed for the Informed and Uninformed groups in non-yielding trials.

% 1. VehicleDeviation (0 = no deviation, 1 = 0.4 m, 2 = 0.8 m, 3 = 1.2 m)
% 2. DeviationDirection (0 = no deviation, 1 = towards road centre, 2 = towards curb)
% Videos 1-28: Towards center for non-yielding, towards curb for yielding (deviation group 1)
% Videos 29-56: Towards curb for non-yielding, towards center for yielding (deviation group 2)
% 3. BrakingBehaviour (1 = non-yielding, 2 = yielding)

v1=find(video_info(:,3)==1 & video_info(:,1)>0 & video_info(:,2)==2);  % Yielding vehicles, Towards deviation
v2=find(video_info(:,3)==1 & video_info(:,1)>0 & video_info(:,2)==1);  % Yielding vehicles, Away deviation
v3=find(video_info(:,3)==1 & video_info(:,1)==0); % Yielding vehicles, Without deviation

figure;hold on;grid on;box on
title('Non-yielding vehicles')
V1 = 100 * squeeze(nanmean(nanmean(RTf([PPinclude_1;PPinclude_3],v1,:),2),1)); % Yielding vehicles, With deviation, Informed
V2 = 100 * squeeze(nanmean(nanmean(RTf([PPinclude_2;PPinclude_4],v1,:),2),1)); % Yielding vehicles, With deviation, Uninformed
V3 = 100 * squeeze(nanmean(nanmean(RTf([PPinclude_1;PPinclude_3],v2,:),2),1)); % Yielding vehicles, Without deviation, Informed
V4 = 100 * squeeze(nanmean(nanmean(RTf([PPinclude_2;PPinclude_4],v2,:),2),1)); % Yielding vehicles, Without deviation, Uninformed
V5 = 100 * squeeze(nanmean(nanmean(RTf([PPinclude_1;PPinclude_3],v3,:),2),1)); % Yielding vehicles, Without deviation, Informed
V6 = 100 * squeeze(nanmean(nanmean(RTf([PPinclude_2;PPinclude_4],v3,:),2),1)); % Yielding vehicles, Without deviation, Uninformed
v1p=100 * nanmean(nanmean(RTf([PPinclude_1;PPinclude_3],v1,76:85),3),2);
v2p=100 * nanmean(nanmean(RTf([PPinclude_2;PPinclude_4],v1,76:85),3),2);
v3p=100 * nanmean(nanmean(RTf([PPinclude_1;PPinclude_3],v2,76:85),3),2);
v4p=100 * nanmean(nanmean(RTf([PPinclude_2;PPinclude_4],v2,76:85),3),2);
v5p=100 * nanmean(nanmean(RTf([PPinclude_1;PPinclude_3],v3,76:85),3),2);
v6p=100 * nanmean(nanmean(RTf([PPinclude_2;PPinclude_4],v3,76:85),3),2);

v1p(isnan(v1p))=[];
v2p(isnan(v2p))=[];
v3p(isnan(v3p))=[];
v4p(isnan(v4p))=[];
v5p(isnan(v5p))=[];
v6p(isnan(v6p))=[];

plot(time,V1,'-','Linewidth',5,'color',[0 1 0]);
plot(time,V2,':','Linewidth',5,'color',[0 1 0]);
plot(time,V3,'-','Linewidth',5,'color',[1 0 0]);
plot(time,V4,':','Linewidth',5,'color',[1 0 0]);
plot(time,V5,'-','Linewidth',2,'color',[.5 .5 .5]);
plot(time,V6,':','Linewidth',2,'color',[.5 .5 .5]);

xlabel('Time (s)')
ylabel('Percentage of trials with response key pressed')
set(gca,'LooseInset',[0.01 0.01 0.01 0.01],'xlim',[0 10],'xtick',1:50,'ylim',[0 70])
legend(['Non-yielding - Towards deviation - Informed group (' num2str(length(v1p)) ' participants, ' num2str(length(v1)) ' videos per participant, \itM\rm = ' sprintf('%0.1f',nanmean(v1p)) '%, \itSD\rm = ' sprintf('%0.1f',nanstd(v1p)) '%)'],...
       ['Non-Yielding - Towards deviation - Uninformed group (' num2str(length(v2p)) ' participants, ' num2str(length(v1)) ' videos per participant, \itM\rm = ' sprintf('%0.1f',nanmean(v2p)) '%, \itSD\rm = ' sprintf('%0.1f',nanstd(v2p)) '%)'],...
       ['Non-Yielding - Away deviation - Informed group (' num2str(length(v3p)) ' participants, ' num2str(length(v2)) ' videos per participant, \itM\rm = ' sprintf('%0.1f',nanmean(v3p)) '%, \itSD\rm = ' sprintf('%0.1f',nanstd(v3p)) '%)'],...
       ['Non-Yielding - Away deviation - Uninformed group (' num2str(length(v4p)) ' participants, ' num2str(length(v2)) ' videos per participant, \itM\rm = ' sprintf('%0.1f',nanmean(v4p)) '%, \itSD\rm = ' sprintf('%0.1f',nanstd(v4p)) '%)'],...
       ['Non-Yielding - No deviation - Informed group (' num2str(length(v5p)) ' participants, ' num2str(length(v3)/2) ' videos per participant, \itM\rm = ' sprintf('%0.1f',nanmean(v5p)) '%, \itSD\rm = ' sprintf('%0.1f',nanstd(v5p)) '%)'],...
       ['Non-Yielding - No deviation - Uninformed group (' num2str(length(v6p)) ' participants, ' num2str(length(v3)/2) ' videos per participant, \itM\rm = ' sprintf('%0.1f',nanmean(v6p)) '%, \itSD\rm = ' sprintf('%0.1f',nanstd(v6p)) '%)'],...
    'autoupdate','off','location','southwest');
h=findobj('FontName','Helvetica'); set(h,'FontSize',18,'Fontname','Arial')
rectangle('position',[0 0 1 100],'facecolor',[127 127 127 20]/255,'Linewidth',.5);
if savepng==1;set(gcf,'WindowState', 'maximized');print(gcf,'Figure4.png','-dpng','-r300');end
%% KeyPressPercentage ratio computed as: Ratio = time pressed / total time.
KeyPressPercentage = NaN(size(RTf,1),size(RTf,2));
for nPar = 1 : size(RTf,1)
    for nVid = 1 :  size(RTf,2)
        KeyPressPercentage(nPar,nVid) = 100*mean(RTf(nPar,nVid,76:85)); % only use the time between 7.5 and 8.5 seconds
    end
end
%%
FS_Yielding_Matrix_TowardsYielding=KeyPressPercentage(deviationgroup1,[2 4 17:28]);   % Keypress percentage, Yielding vehicles, Mapping: Towards = yielding
FS_Yielding_Matrix_AwayYielding=KeyPressPercentage(deviationgroup2,[30 32 33:44]);    % Keypress percentage, Yielding vehicles, Mapping: Away = yielding
FS_NonYielding_Matrix_TowardsYielding=KeyPressPercentage(deviationgroup1,[1 3 5:16]); % Keypress percentage, Non-yielding vehicles, Mapping: Towards = yielding
FS_NonYielding_Matrix_AwayYielding=KeyPressPercentage(deviationgroup2,[29 31 45:56]); % Keypress percentage, Non-yielding vehicles, Mapping: Away = yielding

FS_Yielding_TowardsYielding_Deviation=[nanmean(FS_Yielding_Matrix_TowardsYielding(:,[1 2]),2) nanmean(FS_Yielding_Matrix_TowardsYielding(:,3:3:end),2) nanmean(FS_Yielding_Matrix_TowardsYielding(:,4:3:end),2) nanmean(FS_Yielding_Matrix_TowardsYielding(:,5:3:end),2)];
FS_Yielding_AwayYielding_Deviation=[nanmean(FS_Yielding_Matrix_AwayYielding(:,[1 2]),2) nanmean(FS_Yielding_Matrix_AwayYielding(:,3:3:end),2) nanmean(FS_Yielding_Matrix_AwayYielding(:,4:3:end),2) nanmean(FS_Yielding_Matrix_AwayYielding(:,5:3:end),2)];

IR_Yielding_Matrix_TowardsYielding=IntuitvenessRating(deviationgroup1,[2 4 17:28]);   % Intuitiveness rating, Yielding vehicles, Mapping: Towards = yielding
IR_Yielding_Matrix_AwayYielding=IntuitvenessRating(deviationgroup2,[30 32 33:44]);    % Intuitiveness rating, Yielding vehicles, Mapping: Away = yielding
IR_NonYielding_Matrix_TowardsYielding=IntuitvenessRating(deviationgroup1,[1 3 5:16]); % Intuitiveness rating, Non-yielding vehicles, Mapping: Towards = yielding
IR_NonYielding_Matrix_AwayYielding=IntuitvenessRating(deviationgroup2,[29 31 45:56]); % Intuitiveness rating, Non-yielding vehicles, Mapping: Away = yielding

IR_Yielding_TowardsYielding_Deviation=[nanmean(IR_Yielding_Matrix_TowardsYielding(:,[1 2]),2) nanmean(IR_Yielding_Matrix_TowardsYielding(:,3:3:end),2) nanmean(IR_Yielding_Matrix_TowardsYielding(:,4:3:end),2) nanmean(IR_Yielding_Matrix_TowardsYielding(:,5:3:end),2)];
IR_Yielding_AwayYielding_Deviation=[nanmean(IR_Yielding_Matrix_AwayYielding(:,[1 2]),2) nanmean(IR_Yielding_Matrix_AwayYielding(:,3:3:end),2) nanmean(IR_Yielding_Matrix_AwayYielding(:,4:3:end),2) nanmean(IR_Yielding_Matrix_AwayYielding(:,5:3:end),2)];

[~,VideoTitles] = xlsread('mapping.xlsx','I2:I57');    % Import Excel file with FigureEight data (crowd study)
confidenceinterval1=within_subject_ci(FS_Yielding_Matrix_TowardsYielding);
confidenceinterval2=within_subject_ci(FS_Yielding_Matrix_AwayYielding);
confidenceinterval3=within_subject_ci(FS_NonYielding_Matrix_TowardsYielding);
confidenceinterval4=within_subject_ci(FS_NonYielding_Matrix_AwayYielding);
confidenceinterval5=within_subject_ci(IR_Yielding_Matrix_TowardsYielding);
confidenceinterval6=within_subject_ci(IR_Yielding_Matrix_AwayYielding);
confidenceinterval7=within_subject_ci(IR_NonYielding_Matrix_TowardsYielding);
confidenceinterval8=within_subject_ci(IR_NonYielding_Matrix_AwayYielding);

%% Perform independent-samples t-test
% Percentage of trials in which participants pressed the response key between 7.5 and 8.5 s in the yielding trials for the Towards
% Informed group vs. Uninformed group
v1=nanmean(KeyPressPercentage(instructionsf1,video_info(:,2)==2 & video_info(:,3)==2 & video_info(:,1)>0),2); % Informed group (Towards, Yielding, Deviation)
v2=nanmean(KeyPressPercentage(instructionsf2,video_info(:,2)==2 & video_info(:,3)==2 & video_info(:,1)>0),2); % Uninformed group (Towards, Yielding, Deviation)

[~,p,~,stats]=ttest2(v1,v2);
disp('Stats for independent-samples t-test')
disp([nanmean(v1) nanmean(v2) sum(~isnan(v1)) sum(~isnan(v2)) stats.df stats.tstat p])

%% Figure 6. Keypress percentage in the 7.5–8.5 s interval key for yielding and non-yielding vehicles deviating towards and away from the participant.

figure
% Subplot for Yielding Vehicles, Mapping: Towards = Yielding
subplot(2,2,1)
ydt=nanmean(FS_Yielding_Matrix_TowardsYielding);
ydts=nanstd(FS_Yielding_Matrix_TowardsYielding);
barh(ydt,'facecolor',[.5 .5 .5]);box off;hold on
Range=[22 45];
for i=1:14;text(Range(1)+diff(Range)/100,i,[sprintf('%0.1f',ydt(i)) ' (' sprintf('%0.1f',ydts(i)) ')'],'color',[1 1 1]);end
VideoTitlesYielding=VideoTitles([2 4 17:28]);
errorbar(ydt, 1:length(ydt), confidenceinterval1, '.b','horizontal','handlevisibility','off','linewidth',1)
set(gca,'ytick',1:14,'yticklabel',VideoTitlesYielding, 'xlim',Range,'tickdir','out','ticklength',[0.01 0.00],'xtick',0:2:100,'ydir','reverse')
title('Yielding Vehicles - Towards deviation')
fig=gcf;set(findall(fig,'-property','FontName'),'FontName','Arial','Fontsize',14)
xlabel('Response key pressed (%)')
set(gca,'pos',[0.12 0.58 0.35 0.39])

% Subplot for Yielding Vehicles, Mapping: Towards = Non-yielding
subplot(2,2,2)
ydt=nanmean(FS_Yielding_Matrix_AwayYielding);
ydts=nanstd(FS_Yielding_Matrix_AwayYielding);
barh(ydt,'facecolor',[.5 .5 .5]);box off;hold on
for i=1:14;text(Range(1)+diff(Range)/100,i,[sprintf('%0.1f',ydt(i)) ' (' sprintf('%0.1f',ydts(i)) ')'],'color',[1 1 1]);end
VideoTitlesYielding=VideoTitles([2 4 17:28]);
errorbar(ydt, 1:length(ydt), confidenceinterval2, '.b','horizontal','handlevisibility','off','linewidth',1)
set(gca,'ytick',1:14,'yticklabel',VideoTitlesYielding, 'xlim',Range,'tickdir','out','ticklength',[0.01 0.00],'xtick',0:2:100,'ydir','reverse')
title('Yielding Vehicles - Away deviation')
fig=gcf;set(findall(fig,'-property','FontName'),'FontName','Arial','Fontsize',14)
xlabel('Response key pressed (%)')
set(gca,'pos',[0.63 0.58 0.35 0.39])

% Subplot for Non-yielding Vehicles, Mapping: Towards = Yielding
subplot(2,2,3)
ydt=nanmean(FS_NonYielding_Matrix_TowardsYielding);
ydts=nanstd(FS_NonYielding_Matrix_TowardsYielding);
barh(ydt,'facecolor',[.5 .5 .5]);box off;hold on
Range=[8 25];
for i=1:14;text(Range(1)+diff(Range)/100,i,[sprintf('%0.1f',ydt(i)) ' (' sprintf('%0.1f',ydts(i)) ')'],'color',[1 1 1]);end
VideoTitlesYielding=VideoTitles([2 4 17:28]);
errorbar(ydt, 1:length(ydt), confidenceinterval3, '.b','horizontal','handlevisibility','off','linewidth',1)
set(gca,'ytick',1:14,'yticklabel',VideoTitlesYielding, 'xlim',Range,'tickdir','out','ticklength',[0.01 0.00],'xtick',0:2:100,'ydir','reverse')
title('Non-yielding vehicles - Away deviation')
fig=gcf;set(findall(fig,'-property','FontName'),'FontName','Arial','Fontsize',14)
xlabel('Response key pressed (%)')
set(gca,'pos',[0.12 0.065 0.35 0.39])

% Subplot for Non-yielding Vehicles, Mapping: Towards = Non-yielding
subplot(2,2,4)
ydt=nanmean(FS_NonYielding_Matrix_AwayYielding);
ydts=nanstd(FS_NonYielding_Matrix_AwayYielding);
barh(ydt,'facecolor',[.5 .5 .5]);box off;hold on
for i=1:14;text(Range(1)+diff(Range)/100,i,[sprintf('%0.1f',ydt(i)) ' (' sprintf('%0.1f',ydts(i)) ')'],'color',[1 1 1]);end
VideoTitlesYielding=VideoTitles([2 4 17:28]);
errorbar(ydt, 1:length(ydt), confidenceinterval4, '.b','horizontal','handlevisibility','off','linewidth',1)
set(gca,'ytick',1:14,'yticklabel',VideoTitlesYielding, 'xlim',Range,'tickdir','out','ticklength',[0.01 0.00],'xtick',0:2:100,'ydir','reverse')
title('Non-yielding vehicles - Towards deviation')
fig=gcf;set(findall(fig,'-property','FontName'),'FontName','Arial','Fontsize',14)
xlabel('Response key pressed (%)')
set(gca,'pos',[0.63 0.065 0.35 0.39])
if savepng==1;set(gcf,'WindowState', 'maximized');print(gcf,'Figure6.png','-dpng','-r300');end
%% Figure 5. Mean intuitiveness ratings for yielding and non-yielding vehicles deviating towards and away from the participant.

figure
% Subplot for Yielding Vehicles, Mapping: Towards = Yielding
subplot(2,2,1)
ydt=nanmean(IR_Yielding_Matrix_TowardsYielding);
ydts=nanstd(IR_Yielding_Matrix_TowardsYielding);
barh(ydt,'facecolor',[.5 .5 .5]);box off;hold on
Range=[54 79];
for i=1:14;text(Range(1)+diff(Range)/100,i,[sprintf('%0.1f',ydt(i)) ' (' sprintf('%0.1f',ydts(i)) ')'],'color',[1 1 1]);end
VideoTitlesYielding=VideoTitles([2 4 17:28]);
errorbar(ydt, 1:length(ydt), confidenceinterval5, '.b','horizontal','handlevisibility','off','linewidth',1)
set(gca,'ytick',1:14,'yticklabel',VideoTitlesYielding, 'xlim',Range,'tickdir','out','ticklength',[0.01 0.00],'xtick',0:2:100,'ydir','reverse')
title('Yielding vehicles - Towards deviation')
fig=gcf;set(findall(fig,'-property','FontName'),'FontName','Arial','Fontsize',14)
xlabel('Intuitiveness rating (%)')
set(gca,'pos',[0.12 0.58 0.35 0.39])

% Subplot for Yielding Vehicles, Mapping: Towards = Non-yielding
subplot(2,2,2)
ydt=nanmean(IR_Yielding_Matrix_AwayYielding);
ydts=nanstd(IR_Yielding_Matrix_AwayYielding);
barh(ydt,'facecolor',[.5 .5 .5]);box off;hold on
for i=1:14;text(Range(1)+diff(Range)/100,i,[sprintf('%0.1f',ydt(i)) ' (' sprintf('%0.1f',ydts(i)) ')'],'color',[1 1 1]);end
VideoTitlesYielding=VideoTitles([2 4 17:28]);
errorbar(ydt, 1:length(ydt), confidenceinterval6, '.b','horizontal','handlevisibility','off','linewidth',1)
set(gca,'ytick',1:14,'yticklabel',VideoTitlesYielding, 'xlim',Range,'tickdir','out','ticklength',[0.01 0.00],'xtick',0:2:100,'ydir','reverse')
title('Yielding vehicles - Away deviation')
fig=gcf;set(findall(fig,'-property','FontName'),'FontName','Arial','Fontsize',14)
xlabel('Intuitiveness rating (%)')
set(gca,'pos',[0.63 0.58 0.35 0.39])

% Subplot for Non-yielding Vehicles, Mapping: Towards = Yielding
subplot(2,2,3)
ydt=nanmean(IR_NonYielding_Matrix_TowardsYielding);
ydts=nanstd(IR_NonYielding_Matrix_TowardsYielding);
barh(ydt,'facecolor',[.5 .5 .5]);box off;hold on
Range=[35 52];
for i=1:14;text(Range(1)+diff(Range)/100,i,[sprintf('%0.1f',ydt(i)) ' (' sprintf('%0.1f',ydts(i)) ')'],'color',[1 1 1]);end
VideoTitlesYielding=VideoTitles([2 4 17:28]);
errorbar(ydt, 1:length(ydt), confidenceinterval7, '.b','horizontal','handlevisibility','off','linewidth',1)
set(gca,'ytick',1:14,'yticklabel',VideoTitlesYielding, 'xlim',Range,'tickdir','out','ticklength',[0.01 0.00],'xtick',0:2:100,'ydir','reverse')
title('Non-yielding vehicles - Away deviation')
fig=gcf;set(findall(fig,'-property','FontName'),'FontName','Arial','Fontsize',14)
xlabel('Intuitiveness rating (%)')
set(gca,'pos',[0.12 0.065 0.35 0.39])

% Subplot for Non-yielding Vehicles, Mapping: Towards = Non-yielding
subplot(2,2,4)
ydt=nanmean(IR_NonYielding_Matrix_AwayYielding);
ydts=nanmean(IR_NonYielding_Matrix_AwayYielding);
barh(ydt,'facecolor',[.5 .5 .5]);box off;hold on
for i=1:14;text(Range(1)+diff(Range)/100,i,[sprintf('%0.1f',ydt(i)) ' (' sprintf('%0.1f',ydts(i)) ')'],'color',[1 1 1]);end
VideoTitlesYielding=VideoTitles([2 4 17:28]);
errorbar(ydt, 1:length(ydt), confidenceinterval8, '.b','horizontal','handlevisibility','off','linewidth',1)
set(gca,'ytick',1:14,'yticklabel',VideoTitlesYielding, 'xlim',Range,'tickdir','out','ticklength',[0.01 0.00],'xtick',0:2:100,'ydir','reverse')
title('Non-yielding vehicles - Towards deviation')
fig=gcf;set(findall(fig,'-property','FontName'),'FontName','Arial','Fontsize',14)
xlabel('Intuitiveness rating (%)')
set(gca,'pos',[0.63 0.065 0.35 0.39])

if savepng==1;set(gcf,'WindowState', 'maximized');print(gcf,'Figure5.png','-dpng','-r300');end
%% Figure 7. Mean intuitiveness rating versus the keypress percentage for yielding vehicles in the ‘Towards yielding’ condition, for four levels of lateral deviation (mean of 2 videos for no Deviation, mean of 4 videos for the other deviation levels).

figure;hold on;grid on
v1=nanmean(FS_Yielding_TowardsYielding_Deviation);
v2=nanmean(IR_Yielding_TowardsYielding_Deviation);
v1ci=within_subject_ci(FS_Yielding_TowardsYielding_Deviation); % calculate within-subject confidence interval, for response key pressed
v2ci=within_subject_ci(IR_Yielding_TowardsYielding_Deviation); % calculate within-subject confidence interval, for intuitiveness rating
h1=NaN(7,1);
for i=1:4;h1(i)=plot(v1(i),v2(i),'ks','Markersize',20,'Linewidth',i*2);end
for i=1:4;plot([v1(i)-v1ci(i) v1(i)+v1ci(i)],[nanmean(v2(i)) nanmean(v2(i))],'k-o','markersize',8,'markerfacecolor','k');end
for i=1:4;plot([v1(i) v1(i)],[nanmean(v2(i))-v2ci(i) nanmean(v2(i))+v2ci(i)],'k-o','markersize',8,'markerfacecolor','k');end
disp('Correlation matrix between lateral deviation, key press percentage, and mean intuitiveness rating')
disp(round(corr([[0 0.4 0.8 1.2]' v1' v2']),3))

v1=nanmean(FS_Yielding_TowardsYielding_Deviation(ismember(Country_TowardsYielding,'VEN'),:));
v2=nanmean(IR_Yielding_TowardsYielding_Deviation(ismember(Country_TowardsYielding,'VEN'),:));
for i=1:4
    plot(v1(i),v2(i),'ms','Markersize',20,'Linewidth',i*2);
    if i==4
        h1(5)=plot(v1(i),v2(i),'ms','Markersize',20,'Linewidth',i*2);
    end
end

v1=nanmean(FS_Yielding_TowardsYielding_Deviation(ismember(Country_TowardsYielding,'USA'),:));
v2=nanmean(IR_Yielding_TowardsYielding_Deviation(ismember(Country_TowardsYielding,'USA'),:));
for i=1:4
    plot(v1(i),v2(i),'rs','Markersize',20,'Linewidth',i*2);
    if i==4
        h1(6)=plot(v1(i),v2(i),'rs','Markersize',20,'Linewidth',i*2);
    end
end

v1=nanmean(FS_Yielding_TowardsYielding_Deviation(ismember(Country_TowardsYielding,'IND'),:));
v2=nanmean(IR_Yielding_TowardsYielding_Deviation(ismember(Country_TowardsYielding,'IND'),:));
for i=1:4
    plot(v1(i),v2(i),'bs','Markersize',20,'Linewidth',i*2);
    if i==4
        h1(7)=plot(v1(i),v2(i),'bs','Markersize',20,'Linewidth',i*2);
    end
end

legend(h1,'No deviation, all countries','0.4 m, all countries','0.8 m, all countries','1.2 m, all countries','1.2 m, Venezuela','1.2 m, USA','1.2 m, India','location','southeast')
xlabel('Response key pressed (%)')
ylabel('Mean intuitiveness rating (%)')
set(gca,'pos',[0.08 0.09 0.9 0.89])
set(gcf,'position',[1 41 1000 963])
fig=gcf;set(findall(fig,'-property','FontName'),'FontName','Arial','Fontsize',22)
disp('Sample sizes')
disp([length(FS_Yielding_TowardsYielding_Deviation) length(FS_Yielding_TowardsYielding_Deviation(ismember(Country_TowardsYielding,'VEN'))) length(FS_Yielding_TowardsYielding_Deviation(ismember(Country_TowardsYielding,'USA'))) length(FS_Yielding_TowardsYielding_Deviation(ismember(Country_TowardsYielding,'IND')))])

if savepng==1;print(gcf,'Figure7.png','-dpng','-r300');end
%% Mapping question
% X(:,20)==0  -  Informed   (videos 1-28; yielding = towards)
% X(:,20)==1  -  Uninformed (videos 1-28; yielding = towards)
% X(:,20)==2  -  Informed   (videos 29-56; yielding = away)
% X(:,20)==3  -  Uninformed (videos 29-56; yielding = away)
% X(:,22)==0  -  I prefer the mapping 'towards me when stopping and away from me when the car continues driving'
% X(:,22)==1  -  I prefer the mapping 'away from me when stopping and towards me when the car continues driving'

mapping=[nansum(X(X(:,20)==0, 22)==0) nansum(X(X(:,20)==1, 22)==0) nansum(X(X(:,20)==2, 22)==0) nansum(X(X(:,20)==3, 22)==0);
    nansum(X(X(:,20)==0, 22)==1) nansum(X(X(:,20)==1, 22)==1) nansum(X(X(:,20)==2, 22)==1) nansum(X(X(:,20)==3, 22)==1)];

disp('Percentage of participants choosing the ‘Towards’ mapping for stopping, for the four respective groups:')
disp(round(100*mapping(1,:)./(mapping(1,:)+mapping(2,:)),1))
%% Figure S1

% 1. VehicleDeviation (0 = no deviation, 1 = 0.4m, 2 = 0.8m, 3 = 1.2m)
% 2. DeviationDirection (0 = no deviation, 1 = towards road centre, 2 = towards curb)
% Videos 1-28: Towards center for non-yielding, towards curb for yielding (deviation group 1)
% Videos 29-56: Towards curb for non-yielding, towards center for yielding (deviation group 2)
% 3. BrakingBehaviour (1 = non-yielding, 2 = yielding)

v1=find(video_info(:,3)==2 & video_info(:,1)>0 & video_info(:,2)==2 & video_info(:,5)==2);  % Yielding vehicles, Towards deviation, Early
v2=find(video_info(:,3)==2 & video_info(:,1)>0 & video_info(:,2)==2 & video_info(:,5)==1);  % Yielding vehicles, Away deviation, Late
v3=find(video_info(:,3)==2 & video_info(:,1)==0); % Yielding vehicles, Without deviation

figure;hold on;grid on;box on
title('Yielding vehicles')
V1 = 100 * squeeze(nanmean(nanmean(RTf(:,v1,:),2),1)); % Yielding vehicles, With deviation
V2 = 100 * squeeze(nanmean(nanmean(RTf(:,v2,:),2),1)); % Yielding vehicles, Without deviation
V3 = 100 * squeeze(nanmean(nanmean(RTf(:,v3,:),2),1)); % Yielding vehicles, Without deviation

v1p=100 * nanmean(nanmean(RTf(:,v1,66:75),3),2);
v2p=100 * nanmean(nanmean(RTf(:,v2,66:75),3),2);
v3p=100 * nanmean(nanmean(RTf(:,v3,66:75),3),2);

v1p(isnan(v1p))=[];
v2p(isnan(v2p))=[];
v3p(isnan(v3p))=[];

plot(time,V1,'-','Linewidth',5,'color',[0 1 0]);
plot(time,V2,'-','Linewidth',5,'color',[1 0 0]);
plot(time,V3,'-','Linewidth',5,'color',[.5 .5 .5]);

xlabel('Time (s)')
ylabel('Percentage of trials with response key pressed')
set(gca,'LooseInset',[0.01 0.01 0.01 0.01],'xlim',[0 10],'xtick',1:50,'ylim',[0 70])
legend(['Yielding - Towards deviation - Early deviation (' num2str(length(v1p)) ' participants, ' num2str(length(v1)) ' videos per participant, \itM\rm = ' sprintf('%0.1f',nanmean(v1p)) '%, \itSD\rm = ' sprintf('%0.1f',nanstd(v1p)) '%)'],...
       ['Yielding - Towards deviation - Late deviation (' num2str(length(v2p)) ' participants, ' num2str(length(v2)) ' videos per participant, \itM\rm = ' sprintf('%0.1f',nanmean(v2p)) '%, \itSD\rm = ' sprintf('%0.1f',nanstd(v2p)) '%)'],...
       ['Yielding - No deviation (' num2str(length(v3p)) ' participants,' num2str(length(v3)) ' videos per participant, \itM\rm = ' sprintf('%0.1f',nanmean(v3p)) '%, \itSD\rm = ' sprintf('%0.1f',nanstd(v3p)) '%)'],...
    'autoupdate','off','location','southwest');
h=findobj('FontName','Helvetica'); set(h,'FontSize',18,'Fontname','Arial')
rectangle('position',[0 0 1 100],'facecolor',[127 127 127 20]/255,'Linewidth',.5);
if savepng==1;set(gcf,'WindowState', 'maximized');print(gcf,'FigureS1.png','-dpng','-r300');end
%%
UniqueComments=unique(Comments_coding(:,2));
UniqueComments(end+1)={'technical weakness/problem/response to pedestrian'};
UniqueComments(end+1)={'driver aberrant behaviour/problem/response to pedestrian'};
UniqueComments(end+1)={'driver aberrant behaviour/technical problem'};
UniqueComments(end+1)={'obstacle avoidance/technical weakness/problem'};
UniqueComments(end+1)={'technical weakness/problem/other'};

CommentsCounts=NaN(length(UniqueComments),5);
CommentsCountsPilot=NaN(length(UniqueComments),5);
for i=1:length(UniqueComments)
    for i2=0:3
        CommentsCounts(i,i2+1)=100*sum(strcmp(Comments_coding(:,2),UniqueComments{i})&X(:,20)==i2)./sum(X(:,20)==i2);
    end
        CommentsCounts(i,5)=100*sum(strcmp(Comments_coding(:,2),UniqueComments{i}))./sum(X(:,20)>=0);
end
CommentsCounts=round(CommentsCounts);

T = table(UniqueComments,CommentsCounts(:,1),CommentsCounts(:,2),CommentsCounts(:,3),CommentsCounts(:,4),CommentsCounts(:,5));
T.Properties.VariableNames = {'Category','Informed - Yielding Towards','Uninformed - Yielding Towards','Informed - Yielding Away','Uninformed - Yielding Away','All 4 groups'};
disp('Responses to: In the experiment, the car sometimes steered to the left or right. Why did the car do that? Please elaborate.')
disp(T)
for i2=1:4
Comments_coding_pilot=readtable('Comments coding pilot.xlsx', 'ReadVariableNames', false,'Sheet',i2);
Comments_coding_pilot=table2cell(Comments_coding_pilot);
for i=1:length(UniqueComments)
        CommentsCountsPilot(i,i2)=sum(strcmp(Comments_coding_pilot(:,2),UniqueComments{i}));
end
end
CommentsCountsPilot(:,5)=sum(CommentsCountsPilot(:,1:4),2); % compute total
CommentsCountsPilot=CommentsCountsPilot./repmat(sum(CommentsCountsPilot),length(UniqueComments),1); % compute percentages
CommentsCountsPilot=round(100*CommentsCountsPilot);
T = table(UniqueComments,CommentsCountsPilot(:,1),CommentsCountsPilot(:,2),CommentsCountsPilot(:,3),CommentsCountsPilot(:,4),CommentsCountsPilot(:,5));
T.Properties.VariableNames = {'Category','Informed - Yielding Towards','Uninformed - Yielding Towards','Informed - Yielding Away','Uninformed - Yielding Away','All 4 groups'};
disp('Responses to: In the experiment, the car sometimes steered to the left or right. Why did the car do that? Please elaborate. (Pilot test)')
disp(T)

%% Morey, R. D. (2008). Confidence intervals from normalized data: A correction to Cousineau (2005). Reason, 4, 61-64.
%  CI is half of the confidence interval; so it should be plotted from mean-CI to mean+CI

function CI = within_subject_ci(X)
X=X-repmat(nanmean(X,2),1,size(X,2)); % subtract participant mean
[~,~,ci]=ttest(X); % calculate confidence interval assuming a normal distribution
CI=diff(ci)/2*sqrt(size(X,2)/(size(X,2)-1)); % Correction factor as described in Morey (2008)
end

