classdef Logger < handle
    % LOGGER  Handles structured logging.

    properties
        level (1, 1) {mustBeInteger} = 0;
    end


    methods
        function header(obj, format, varargin)
            % HEADER  Prints [format] with [varargin] with the current preamble and
            % increases the indentation level for subsequent messages.
            %
            % If [format] is the empty string, nothing is printed.

            arguments% (Input)
                obj (1, 1) Logger;
                format (1, 1) {mustBeText} = "";
            end
            arguments (Repeating)
                varargin;
            end

            if format ~= ""
                obj.write(sprintf("%s" + format, obj.get_preamble("<"), varargin{:}));
            end

            obj.level = obj.level + 1;
        end

        function footer(obj, format, varargin)
            % FOOTER  Decreases the indentation level for this and subsequent messages and
            % prints [format] with [varargin] with the updated preamble.
            %
            % If [format] is the empty string, nothing is printed.

            arguments% (Input)
                obj (1, 1) Logger;
                format (1, 1) {mustBeText} = "";
            end
            arguments (Repeating)
                varargin;
            end

            obj.level = obj.level - 1;

            if format ~= ""
                obj.write(sprintf("%s" + format, obj.get_preamble(">"), varargin{:}));
            end
        end

        function print(obj, format, varargin)
            % PRINT  Prints [format] with [varargin] the current preamble.

            arguments% (Input)
                obj (1, 1) Logger;
                format (1, 1) {mustBeText} = "";
            end
            arguments (Repeating)
                varargin;
            end

            obj.write(sprintf("%s" + format, obj.get_preamble(), varargin{:}));
        end

        function print_each(obj, lines)
            % PRINT_EACH  Prints each line in [lines] separately, without formatting.

            arguments% (Input)
                obj (1, 1) Logger;
                lines (:, 1) {mustBeText};
            end

            arrayfun(@(it) obj.print("%s", it), lines);
        end

        function append(obj, format, varargin)
            % APPEND  Prints [format] with [varargin] without preamble.

            arguments% (Input)
                obj (1, 1) Logger;
                format (1, 1) {mustBeText} = "";
            end
            arguments (Repeating)
                varargin;
            end

            obj.write(sprintf(format, varargin{:}));
        end
    end

    methods (Access = protected)
        function write(~, text)
            % WRITE  Determines how and where messages are written.

            arguments% (Input)
                ~;
                text (1, 1) {mustBeText};
            end

            fprintf("%s", text);
        end
    end

    methods (Access = private)
        function out = get_preamble(obj, suffix)
            % PREFIX  Returns the current preamble, finalised by the prefix' [suffix]
            % (unless [suffix] is an empty string).

            arguments% (Input)
                obj (1, 1) Logger;
                suffix (1, 1) {mustBeText} = "";
            end
            % arguments (Output)
            %     out (1, 1) {mustBeText};
            % end

            parts = [];

            % Thread ID
            worker = getCurrentWorker();
            if ~isempty(worker)
                parts = [parts, sprintf("[%d]", worker.ProcessId)];
            end

            % Time
            parts = [parts, sprintf("[%s]", datetime("now", Format = "HH:mm:ss"))];

            % Indentation and suffix
            if obj.level ~= 0 || suffix ~= ""
                parts = [parts, sprintf("%s%s", strjoin(repmat("#", [1, obj.level]), ""), suffix)];
            end

            % Combine
            if isempty(parts)
                out = "";
            else
                out = sprintf("%s ", strjoin(parts, " "));
            end
        end
    end
end
