classdef Graphs
    % GRAPHS  Contains static functions for creating and manipulating graphs.

    methods (Static)
        function G = generate_erdos_renyi(n, p)
            % GENERATE_ERDOS_RENYI  Outputs an (n, p) Erdős-Renyi graph, with [n] nodes and
            % each edge having a probability [p] of being included.

            arguments% (Input)
                n (1, 1) {mustBeInteger, mustBePositive};
                p (1, 1) {mustBeInRange(p, 0, 1)};
            end
            % arguments (Output)
            %     G (1, 1) graph;
            % end

            A = zeros(n);
            [X, Y] = meshgrid(1:n);
            A(X > Y) = rand([(n * (n - 1)) / 2, 1]) < p;
            G = graph(A, "upper");
        end

        function G = generate_watts_strogatz(n, k, p)
            % GENERATE_WATTS_STROGATZ  Outputs a Watts-Strogatz graph with [n] nodes,
            % [n]*[k] edges, mean node degree 2*[k], and rewiring probability [p].
            %
            % [p] = 0 creates a ring lattice, and [p] = 1 is a random graph.
            %
            % Code taken from
            % https://mathworks.com/help/matlab/math/build-watts-strogatz-small-world-graph-model.html.

            arguments% (Input)
                n (1, 1) {mustBeInteger, mustBePositive};
                k (1, 1) {mustBeInteger, mustBeInRange(k, 0, n)};
                p (1, 1) {mustBeFloat, mustBeInRange(p, 0, 1)};
            end
            % arguments (Output)
            %     G (1, 1) graph;
            % end

            % Connect each node to its K next and previous neighbors. This constructs
            % indices for a ring lattice.
            s = repelem((1:n)', 1, k);
            t = s + repmat(1:k, n, 1);
            t = mod(t - 1, n) + 1;

            % Rewire the target node of each edge with probability beta
            for source = 1:n
                switch_edge = rand(k, 1) < p;

                new_targets = rand(n, 1);
                new_targets(source) = 0;
                new_targets(s(t == source)) = 0;
                new_targets(t(source, ~switch_edge)) = 0;

                [~, ind] = sort(new_targets, 'descend');
                t(source, switch_edge) = ind(1:nnz(switch_edge));
            end

            G = graph(s,t);
        end

        function G = generate_barabasi_albert(n, m)
            % GENERATE_BARABASI_ALBERT  Outputs a Barabási-Albert graph with [n] nodes,
            % using preferential attachment parameter [m].

            arguments% (Input)
                n (1, 1) {mustBeInteger, mustBePositive};
                m (1, 1) {mustBeInteger, mustBeLessThan(m, n)};
            end
            % arguments (Output)
            %     G (1, 1) graph;
            % end

            m0 = m + 1;
            A = zeros([n, n]);

            % Use first `m0` nodes to create star graph
            A(2:m0, 1) = 1;
            A(1, 2:m0) = 1;

            % Populate graph
            for idx = (m0 + 1):n
                selected = randsample(1:n, m, true, sum(A, 2));

                A(selected, idx) = 1;
                A(idx, selected) = 1;
            end

            G = graph(A);
        end

        function G = generate_complete(n)
            % GENERATE_COMPLETE  Outputs a complete graph with [n] nodes.

            arguments% (Input)
                n (1, 1) {mustBeInteger, mustBePositive};
            end
            % arguments (Output)
            %     G (1, 1) graph;
            % end

            A = ones([n, n]);
            A = A - diag(diag(A));
            G = graph(A);
        end

        function G = generate_empty(n)
            % GENERATE_EMPTY  Outputs a graph with [n] nodes and 0 edges.

            arguments% (Input)
                n (1, 1) {mustBeInteger, mustBePositive};
            end
            % arguments (Output)
            %     G (1, 1) graph;
            % end

            G = graph(zeros([n, n]));
        end


        function girth = calc_girth(G)
            % CALC_GIRTH  Returns the length of the shortest cycle in [G], or `-1` if [G]
            % has no cycles.

            if isempty(allcycles(G, MaxNumCycles = 1))
                girth = Inf;
                return;
            end

            for girth = 2:numnodes(G)
                [~, found_cycles] = allcycles(G, MaxCycleLength = girth, MaxNumCycles = 1);

                if ~isempty(found_cycles); return; end
            end
        end

        function G = stretched(G, args)
            % STRETCHED  Removes edges from [G] according to [args.method] such that the
            % shortest cycle has at least length [args.min_girth].
            %
            % The girth of a graph is the length of the shortest cycle. If a graph has no
            % cycles, the girth is infinite.
            %
            % If [args.min_girth] is negative, all cycles are removed. If [args.min_girth]
            % is 0, 1, or 2, no cycles are removed, because simple graphs always have a
            % girth of at least 3.
            %
            % The [args.method] specifies how the edges to be removed  are selected. The
            % removal process is repeated until the girth is as desired. The following
            % methods are supported:
            %
            % * "any": An arbitrary edge of an arbitrary cycle of length less than
            %          [args.min_girth] is removed.
            % * "basis": An arbitrary edge of an arbitrary cycle in the fundamental cycle
            %            basis of length less than [args.min_girth] is removed.

            arguments% (Input)
                G (1, 1) graph;
                args.min_girth (1, 1) {mustBeInteger} = -1;
                args.method (1, 1) {mustBeText, mustBeMember(args.method, ["any", "basis"])} = "basis";
            end
            % arguments (Output)
            %     G (1, 1) graph;
            % end

            if ismember(args.min_girth, 0:2)
                return;
            end

            if args.min_girth < 0
                args.min_girth = numnodes(G) + 1;
            end

            while true
                if args.method == "any"
                    [~, found_cycles] = allcycles(G, MaxCycleLength = args.min_girth - 1, MaxNumCycles = 1);

                    if isempty(found_cycles); break; end
                elseif args.method == "basis"
                    [~, cycles] = cyclebasis(G);
                    found_cycles = cycles(cellfun(@(it) numel(it) < args.min_girth, cycles));

                    if isempty(found_cycles)
                        args.method = "any";
                        continue;
                    end
                else
                    error("Unknown method '%s'.", args.method);
                end

                edges = found_cycles{1}(1);
                while true
                    found_cycles = found_cycles(cellfun(@(it) ~any(ismember(edges, it)), found_cycles));
                    if isempty(found_cycles); break; end

                    edges = [edges; found_cycles{1}(1)];  %#ok<AGROW> % Size is not known beforehand
                end

                G = rmedge(G, edges);
            end

            if ~isempty(allcycles(G, MaxCycleLength = args.min_girth - 1, MaxNumCycles = 1))
                error("Failed to stretch girth.");
            end
        end
    end
end
