%%
% clear memory and previously allocated variables
clc;clear all;close all;


%% Load Grain Boundary Analysis Preprocessing (gbap) data

run ColorIceToolInput

%create name and find location for data file and load file
load(fullfile([pwd,subfolder],'ImageStackData.mat'),'gbap')
clear subfolder


%% Settings

gbap.cropscale = cropscale;
gbap.threshfacVec = threshfacVec;
gbap.plotFigureFlagSegmenter = plotFigureFlagSegmenter;
gbap.fontSize = fontSize;
gbap.markerSize = markerSize;
clear cropscale threshfacVec plotFigureFlagSegmenter fontSize markerSize


%% Interpolate thickness array to dimensions of image

% gbap.thicknessArray = [0.63 0.93;1.10 1.23]; %remove this once placed in preprocessor!!!!!!!

[col,row] = meshgrid(...
    1:size(gbap.imageStack{1,1},1)-1:size(gbap.imageStack{1,1},1),...
    1:size(gbap.imageStack{1,1},2)-1:size(gbap.imageStack{1,1},2));

[gbap.thicknessColumns,gbap.thicknessRows] = meshgrid(...
    1:1:size(gbap.imageStack{1,1},1),...
    1:1:size(gbap.imageStack{1,1},2));

gbap.thicknessArrayInterp = interp2(col,row,gbap.thicknessArray,gbap.thicknessColumns,gbap.thicknessRows,'linear');

% figure
% surf(gbap.thicknessArrayInterp,'EdgeColor','none','FaceColor','interp');
% xlabel('column')
% ylabel('row')


%% Crop image stack to central part

figure;montage(gbap.imageStack(:,1), 'Size', [2 ceil(size(gbap.imageStack,1)/2)])

gbap.winCrop = centerCropWindow2d(size(gbap.imageStack{1,1}),ceil(size(gbap.imageStack{1,1})*gbap.cropscale));
for k = 1:size(gbap.imageStack,1)
    gbap.imageStack{k,2} = imcrop(gbap.imageStack{k,1},gbap.winCrop);
end

%crop ice thickness array by the image cropping
gbap.thicknessArrayInterpCrop = gbap.thicknessArrayInterp(...
    gbap.winCrop.XLimits(1):gbap.winCrop.XLimits(end),...
    gbap.winCrop.YLimits(1):gbap.winCrop.YLimits(end));

figure;montage(gbap.imageStack(:,2), 'Size', [2 ceil(size(gbap.imageStack,1)/2)])
% NOTE: use second column of imageStack for final cropped images


%% Segmentation of image stacks

gbap.kernelVal = 3; gbap.Nmedfilt = 1; gbap.SE0 = strel('disk',3); gbap.SE = strel('square',3);

optA = struct(); %initialize structure

if parforopt == 1
    
    delete(gcp('nocreate'))
    
    gbapParfor = gbap; %create new structure for parallel for-loop
    % convert to constant arrays in parallel pool
    poolconst = parallel.pool.Constant(gbapParfor);
    
    tic
    parpool('local',6);
    parfor k = 1:size(gbap.threshfacVec,1)
        
        % redefine parallel pool constant with original input array name for brevity
        gbapParfor = poolconst.Value;
        
        disp(k)
        %     tic
        [optA(k).B,optA(k).L,optA(k).N] = ...
            imagestacksegmenter(gbapParfor.imageStack(:,2),...
            gbapParfor.kernelVal,gbapParfor.Nmedfilt,gbapParfor.SE0,...
            gbapParfor.SE,gbapParfor.threshfacVec(k),gbapParfor.threshfacVec(k),...
            gbapParfor.threshfacVec(k),gbapParfor.plotFigureFlagSegmenter);
        %     toc
    end
    toc
    delete(gcp('nocreate'))
    gbap.optA = optA;
    clear poolconst optA
    
else
    
    
    gbapParfor = gbap; %create new structure for parallel for-loop
    % convert to constant arrays in parallel pool
    % poolconst = parallel.pool.Constant(gbapParfor);
    
    tic
    %     parpool('local',6);
    for k = 1:size(gbap.threshfacVec,1)
        
        % redefine parallel pool constant with original input array name for brevity
        %     gbapParfor = poolconst.Value;
        
        disp(k)
        %     tic
        [optA(k).B,optA(k).L,optA(k).N] = ...
            imagestacksegmenter(gbapParfor.imageStack(:,2),...
            gbapParfor.kernelVal,gbapParfor.Nmedfilt,gbapParfor.SE0,...
            gbapParfor.SE,gbapParfor.threshfacVec(k),gbapParfor.threshfacVec(k),...
            gbapParfor.threshfacVec(k),gbapParfor.plotFigureFlagSegmenter);
        %     toc
    end
    toc
    % delete(gcp('nocreate'))
    gbap.optA = optA;
    clear poolconst optA
    clear optA
    
end


%% High resolution grain boundary segmentation stack

gbap.CM = hsv(size(gbap.threshfacVec,1));

if plotFigureHighResGB == 1
    
    figure;
    imshow(gbap.imageStack{1,2})
    % image(imageStack{1})
    for q = 1:size(gbap.threshfacVec,1)
        hold on
        %     for k = 1:optA(q).N
        %         boundary = optA(q).B{k};
        %         plot(boundary(:,2), boundary(:,1), 'k', 'LineWidth', 0.5)
        %     end
        
        for k = gbap.optA(q).N+1:length(gbap.optA(q).B)
            boundary = gbap.optA(q).B{k};
            if k == length(gbap.optA(q).B)
                plot(boundary(1:end-1,2), boundary(1:end-1,1),'Color',gbap.CM(q,:),'LineWidth', 0.5)
                gbap.optA(q).hh = plot(boundary(end,2), boundary(end,1),'Color',gbap.CM(q,:),'LineWidth', 0.5,'DisplayName',num2str(gbap.threshfacVec(q)));
            else
                plot(boundary(:,2), boundary(:,1),'Color',gbap.CM(q,:),'LineWidth', 0.5)
            end
        end
        hold off
    end
    lgd = legend([gbap.optA(:).hh],'location','bestoutside',...
        'Interpreter','LaTeX','FontName','Times New Roman','FontSize',gbap.fontSize);
    set(gcf,'color','w')
end

%% Low resolution grain boundary segmentation stack

% figure
% for q = 1:size(gbap.threshfacVec,1)
%     subplot(2,ceil(size(gbap.threshfacVec,1))/2,q)
%     baseImage = rgb2gray(gbap.imageStack{1,2});
%     mask = boundarymask(gbap.optA(q).L);
%     baseImage = imoverlay(baseImage,mask,gbap.CM(q,:));
%     imshow(baseImage)
% end

% figure
scrsz = get(groot,'ScreenSize');
figure('Position',...
    [scrsz(3)*0.02 scrsz(4)*0.15 scrsz(3)*0.95 scrsz(4)*0.70]);
% 'Position' definition: [left bottom width height]

tiledlayout(1+(size(gbap.threshfacVec,1)>1),ceil(size(gbap.threshfacVec,1)/2),'TileSpacing','tight','Padding','compact');
for q = 1:size(gbap.threshfacVec,1)
    nexttile
    baseImage = rgb2gray(gbap.imageStack{1,2});
    mask = boundarymask(gbap.optA(q).L);
    baseImage = imoverlay(baseImage,mask,gbap.CM(q,:));
    imshow(baseImage)
    title(['threshold=',num2str(gbap.threshfacVec(q),2)])
end


%% Calculate measurements and statistics for image

gbap.stats = struct();

for q = 1:size(gbap.threshfacVec,1)
    gbap.stats(q).regionProps = regionprops('table',gbap.optA(q).L,'Area','Centroid','Perimeter',...
        'MajorAxisLength','MinorAxisLength','Orientation','EquivDiameter');
    
    gbap.stats(q).Area = gbap.stats(q).regionProps.Area(gbap.optA(q).N+1:end)*gbap.mmPerPixel^2;
    gbap.stats(q).meanArea = mean(gbap.stats(q).Area);
    gbap.stats(q).medianArea = median(gbap.stats(q).Area);
    gbap.stats(q).areaFraction = sum(gbap.stats(q).regionProps.Area(gbap.optA(q).N+1:end))/numel(gbap.imageStack{1,2}(:,:,1));
    
    gbap.stats(q).equivDia = gbap.stats(q).regionProps.EquivDiameter(gbap.optA(q).N+1:end)*gbap.mmPerPixel;
    gbap.stats(q).meanEquivDia = mean(gbap.stats(q).equivDia);
    gbap.stats(q).medianEquivDia = median(gbap.stats(q).equivDia);
    gbap.stats(q).numGrains = numel(gbap.stats(q).regionProps.EquivDiameter(gbap.optA(q).N+1:end));
end


%% Plot statistical trends

axset = struct();

% figure
scrsz = get(groot,'ScreenSize');
figure('Position',...
    [scrsz(3)*0.005 scrsz(4)*0.2 scrsz(3)*0.99 scrsz(4)*0.60]);
% 'Position' definition: [left bottom width height]
for q = 1:size(gbap.threshfacVec,1)
    %     figure
    %     subplot(1,2,1)
    axset(q).ax1 = subplot(2,size(gbap.threshfacVec,1),q);
    histogram(gbap.stats(q).Area,'FaceColor',gbap.CM(q,:),'FaceAlpha',1)
    %     title([num2str(threshfacVec(q),'%.1f'),': ','$\tilde{A}$=',...
    %         num2str(median(stats(q).regionProps.Area(optA(q).N+1:end)*mmPerPixel^2),'%.1f'),'mm$\textsuperscript{2}$',...
    %         ', $A_{f}$=',num2str(sum(stats(q).regionProps.Area(optA(q).N+1:end))/numel(imageStack{1})*100,'%.1f'),'$\%$'],...
    %         'Interpreter','LaTeX','FontName','Times New Roman','FontSize',fontSize)
    xlabel('grain area [mm$\textsuperscript{2}$]',...
        'Interpreter','LaTeX','FontName','Times New Roman','FontSize',gbap.fontSize)
    if q == 1
        ylabel('frequency [-]',...
            'Interpreter','LaTeX','FontName','Times New Roman','FontSize',gbap.fontSize)
    end
    axis tight
    grid on
    set(gca,'FontName','Times New Roman')
    set(gca,'FontSize',gbap.fontSize)
    title([num2str(gbap.threshfacVec(q),'%.1f'),': ','$\bar{A}$=',...
        num2str(gbap.stats(q).meanArea,'%.1f'),'mm$\textsuperscript{2}$',...
        ', $A_{f}$=',num2str(gbap.stats(q).areaFraction,'%.2f')],...
        'Interpreter','LaTeX','FontName','Times New Roman','FontSize',gbap.fontSize-1)
    
    %     subplot(1,2,2)
    axset(q).ax2 = subplot(2,size(gbap.threshfacVec,1),q+size(gbap.threshfacVec,1));
    histogram(gbap.stats(q).equivDia,'FaceColor',gbap.CM(q,:),'FaceAlpha',1)
    %     title([num2str(threshfacVec(q),'%.1f'),': ','$\tilde{d}$=',...
    %         num2str(median(stats(q).regionProps.EquivDiameter(optA(q).N+1:end)*mmPerPixel),'%.1f'),'mm',...
    %         ', N=',num2str(length(stats(q).regionProps.EquivDiameter(optA(q).N+1:end)))],...
    %         'Interpreter','LaTeX','FontName','Times New Roman','FontSize',fontSize)
    xlabel('equiv. grain dia. [mm]',...
        'Interpreter','LaTeX','FontName','Times New Roman','FontSize',gbap.fontSize)
    if q == 1
        ylabel('frequency [-]',...
            'Interpreter','LaTeX','FontName','Times New Roman','FontSize',gbap.fontSize)
    end
    axis tight
    grid on
    set(gca,'FontName','Times New Roman')
    set(gca,'FontSize',gbap.fontSize)
    title([num2str(gbap.threshfacVec(q),'%.1f'),': ','$\bar{d}$=',...
        num2str(gbap.stats(q).meanEquivDia,'%.1f'),'mm',...
        ', N=',num2str(gbap.stats(q).numGrains)],...
        'Interpreter','LaTeX','FontName','Times New Roman','FontSize',gbap.fontSize-1)
end

% linkaxes([axset(:).ax1],'xy');linkaxes([axset(:).ax2],'xy');
set(gcf,'color','w')


%% Thresholding trends

% figure
scrsz = get(groot,'ScreenSize');
figure('Position',...
    [scrsz(3)*0.3 scrsz(4)*0.1 scrsz(3)*0.50 scrsz(4)*0.80]);
% 'Position' definition: [left bottom width height]

subplot(4,1,1)
hold on
for q=1:size(gbap.threshfacVec,1)
    %     plot(threshfacVec(q),[stats(q).areaFraction]','o','MarkerEdgeColor',CM(q,:),'MarkerFaceColor',CM(q,:),'MarkerSize',4)
    plot(gbap.threshfacVec(q),[gbap.stats(q).areaFraction]','o','MarkerEdgeColor','k','MarkerFaceColor',gbap.CM(q,:),'MarkerSize',gbap.markerSize)
end
% plot(threshfacVec,[stats(:).areaFraction]')
hold off
% xlabel('threshold [-]',...
%     'Interpreter','LaTeX','FontName','Times New Roman','FontSize',fontSize)
ylabel('area fraction [-]',...
    'Interpreter','LaTeX','FontName','Times New Roman','FontSize',gbap.fontSize)
% axis tight
grid on
set(gca,'FontName','Times New Roman')
set(gca,'FontSize',gbap.fontSize)

subplot(4,1,2)
hold on
for q=1:size(gbap.threshfacVec,1)
    %     plot(threshfacVec(q),[stats(q).numGrains]','o','MarkerEdgeColor',CM(q,:),'MarkerFaceColor',CM(q,:),'MarkerSize',4)
    plot(gbap.threshfacVec(q),[gbap.stats(q).numGrains]','o','MarkerEdgeColor','k','MarkerFaceColor',gbap.CM(q,:),'MarkerSize',gbap.markerSize)
end
% plot(threshfacVec,[stats(:).numGrains]')
hold off
% xlabel('threshold scale [-]',...
%     'Interpreter','LaTeX','FontName','Times New Roman','FontSize',fontSize)
ylabel('number of grains [-]',...
    'Interpreter','LaTeX','FontName','Times New Roman','FontSize',gbap.fontSize)
% axis tight
grid on
set(gca,'FontName','Times New Roman')
set(gca,'FontSize',gbap.fontSize)

subplot(4,1,3)
hold on
for q=1:size(gbap.threshfacVec,1)
    %     plot(threshfacVec(q),[stats(q).meanArea]','o','MarkerEdgeColor',CM(q,:),'MarkerFaceColor',CM(q,:),'MarkerSize',4)
    plot(gbap.threshfacVec(q),[gbap.stats(q).meanArea]','o','MarkerEdgeColor','k','MarkerFaceColor',gbap.CM(q,:),'MarkerSize',gbap.markerSize)
end
% plot(threshfacVec,[stats(:).meanArea]')
hold off
% xlabel('threshold [-]',...
%     'Interpreter','LaTeX','FontName','Times New Roman','FontSize',fontSize)
ylabel('mean grain area [mm$\textsuperscript{2}$]',...
    'Interpreter','LaTeX','FontName','Times New Roman','FontSize',gbap.fontSize)
% axis tight
grid on
set(gca,'FontName','Times New Roman')
set(gca,'FontSize',gbap.fontSize)

subplot(4,1,4)
hold on
for q=1:size(gbap.threshfacVec,1)
    %     plot(threshfacVec(q),[stats(q).meanEquivDia]','o','MarkerEdgeColor',CM(q,:),'MarkerFaceColor',CM(q,:),'MarkerSize',4)
    plot(gbap.threshfacVec(q),[gbap.stats(q).meanEquivDia]','o','MarkerEdgeColor','k','MarkerFaceColor',gbap.CM(q,:),'MarkerSize',gbap.markerSize)
end
% plot(threshfacVec,[stats(:).meanEquivDia]')
hold off
xlabel('threshold [-]',...
    'Interpreter','LaTeX','FontName','Times New Roman','FontSize',gbap.fontSize)
ylabel('mean equiv. grain dia. [mm]',...
    'Interpreter','LaTeX','FontName','Times New Roman','FontSize',gbap.fontSize)
% axis tight
grid on
set(gca,'FontName','Times New Roman')
set(gca,'FontSize',gbap.fontSize)

set(gcf,'color','w')


%% Determine mode of grayscale pixel value for each grain

% clear grainAzimuth
% grainAzimuth = zeros(size(gbap.imageStack,1),length(gbap.optA(q).B)-(gbap.optA(q).N + 0),3);
% grainAzimuthArray = {};

disp('')
disp('calculating grain azimuth array')
disp('')

if size(gbap.imageStack{1,2},1) > pixNumLim %image pixel limit for using parfor
    
    gbapParfor = gbap; %create new structure for parallel for-loop
    % convert to constant arrays in parallel pool
    poolconst = parallel.pool.Constant(gbapParfor);
    
    parpool('local',6);
    tic
    parfor q = 1:size(gbap.threshfacVec,1)
        
        % redefine parallel pool constant with original input array name for brevity
        gbapParfor = poolconst.Value;
        
        disp(q)
        
        for k = 1:size(gbapParfor.imageStack,1)
            %     for kk = optA(q).N+1:length(optA(q).B)
            for kk = 1:(length(gbapParfor.optA(q).B)-(gbapParfor.optA(q).N + 0))
                qq = kk + gbapParfor.optA(q).N + 0;
                for kkk = 1:3 %3:RGB color channels
                    imageMonochrome = (gbapParfor.imageStack{k,2}(:,:,kkk));
                    gbapParfor.grainMode(k,kk,kkk) = im2double(mode(imageMonochrome(gbapParfor.optA(q).L == qq),'all'));
                    gbapParfor.grainMean(k,kk,kkk) = im2double(mean2(imageMonochrome(gbapParfor.optA(q).L == qq)))./255;
                    gbapParfor.grainMedian(k,kk,kkk) = im2double(median(imageMonochrome(gbapParfor.optA(q).L == qq),'all'));
                end
            end
        end
        grainModeArray{q} = gbapParfor.grainMode;
        grainMeanArray{q} = gbapParfor.grainMean;
        grainMedianArray{q} = gbapParfor.grainMedian;
    end
    toc
    delete(gcp('nocreate'))
    gbap.grainModeArray = grainModeArray;
    gbap.grainMeanArray = grainMeanArray;
    gbap.grainMedianArray = grainMedianArray;
    clear poolconst gbapParfor grainAzimuthArray
    
else
    grainMode = [];
    grainMean = [];
    grainMedian = [];
    tic
    for q = 1:size(gbap.threshfacVec,1)
        for k = 1:size(gbap.imageStack,1)
            %     for kk = optA(q).N+1:length(optA(q).B)
            for kk = 1:(length(gbap.optA(q).B)-(gbap.optA(q).N + 0))
                qq = kk + gbap.optA(q).N + 0;
                for kkk = 1:3 %3:RGB color channels
                    imageMonochrome = (gbap.imageStack{k,2}(:,:,kkk));
                    grainMode(k,kk,kkk) = im2double(mode(imageMonochrome(gbap.optA(q).L == qq),'all'));
                    grainMean(k,kk,kkk) = im2double(mean2(imageMonochrome(gbap.optA(q).L == qq)))./255;
                    grainMedian(k,kk,kkk) = im2double(median(imageMonochrome(gbap.optA(q).L == qq),'all'));
                end
            end
        end
        gbap.grainModeArray{q} = grainMode;
        gbap.grainMeanArray{q} = grainMean;
        gbap.grainMedianArray{q} = grainMedian;
        %     grainAzimuth = [];
    end
    toc
    % gbap.grainAzimuthArray = grainAzimuthArray;
    
end


%% Determine extinction and max brightness angles for each grain

% disp('calculating grain azimuth extinction')
% tic
% grainExtinction = zeros(size(gbap.threshfacVec,1),length(gbap.optA(q).B)-(gbap.optA(q).N + 0));
% grainExtinction = [];
% grainExtinction = {};
for q = 1:size(gbap.threshfacVec,1)
    for kk = 1:(length(gbap.optA(q).B)-(gbap.optA(q).N + 0))
        %         [profileMinMatValue,profileMinMatIndex] = min(rgb2gray(squeeze(grainAzimuth(:,kk,:,q))));
        %         [profileMinMatValue,profileMinMatIndex] = min(rgb2gray((grainAzimuth(:,kk,:,q))));
        
        colorDistance = sqrt(...
            double(gbap.grainModeArray{q}(:,kk,1)).^2 + ...
            double(gbap.grainModeArray{q}(:,kk,2)).^2 + ...
            double(gbap.grainModeArray{q}(:,kk,3)).^2);
        
        [profileMinMatValue,profileMinMatIndex] = min(colorDistance);
        [profileMaxMatValue,profileMaxMatIndex] = max(colorDistance);
        
        %         [profileMinMatValue,profileMinMatIndex] = min(rgb2gray(uint8(gbap.grainAzimuthArray{q}(:,kk,:))));
        %         grainExtinction(q,kk) = gbap.angleVec(profileMinMatIndex);
        gbap.grainExtinction{q,kk} = gbap.angleVec(profileMinMatIndex);
        
        %define brightest grain angle as 45deg from extinction
        if gbap.angleVec(end) > 0 %positive angles
            if abs(gbap.grainExtinction{q,kk}) > (85 - 45)
                gbap.grainMaximum{q,kk} = gbap.grainExtinction{q,kk} - 45;
            else
                gbap.grainMaximum{q,kk} = gbap.grainExtinction{q,kk} + 45;
            end
        elseif gbap.angleVec(end) < 0 %negative angles
            if abs(gbap.grainExtinction{q,kk}) > (85 - 45)
                gbap.grainMaximum{q,kk} = -1*(abs(gbap.grainExtinction{q,kk}) - 45);
            else
                gbap.grainMaximum{q,kk} = -1*(abs(gbap.grainExtinction{q,kk}) + 45);
            end
        end
        
        %         gbap.grainMaximum{q,kk} = gbap.angleVec(profileMaxMatIndex);
    end
end
% toc
clear colorDistance


%% Generate birefringence color look-up table (CLUT)

%ice cross section thickness [m]
% thickness_vec = ((min(gbap.thicknessArrayInterpCrop(:))-0.1:0.05:max(gbap.thicknessArrayInterpCrop(:))+0.1)*10^-3)';
thickness_vec = ((min(gbap.thicknessArrayInterpCrop(:))-0.1:0.01:max(gbap.thicknessArrayInterpCrop(:))+0.1)*10^-3)';

%inclination/zenith angle between incident ray and c-axis [rad]
% theta_vec = (0:0.5:90)'*pi/180;
theta_vec = (0:0.1:90)'*pi/180;
% theta_vec = (23.0:0.5:90)'*pi/180;
% theta_vec = (45.0:0.5:90)'*pi/180;

%azimuth angle between polarization planes of incident ray and extraordinary ray [rad]
% alfa_vec = (44:0.5:46)'*pi/180;
alfa_vec = (40.5:0.5:49.5)'*pi/180;
% alfa_vec = (45)'*pi/180;

%chromatic adaptation target color temperature [k]
% colorTempChrom_vec = (6500:1000:12500)';
colorTempChrom_vec = (7500)';

tic
% [RGBnonlinCLUT] = birefringenceColorLookUpTable(thickness_vec,theta_vec,alfa_vec);
lambdaFlag = 0; %flag for wave plate inserted (0=no,1=quarter,2=full)
polFlag = 1; %flag for crossed- or plane-polarization (1 or 0) (only functions without wave plate)
colorTemp = 4000; %light source color temperature [2700,3200,5500,6500,7500]K
% colorTempChrom = 7500; %chromatic adaption color temperature [K] (6500 target)
% colorTempChrom = 9500; %chromatic adaption color temperature [K] (6500 target)
for ii = 1:length(colorTempChrom_vec)
    gbap.colorTempChrom = colorTempChrom_vec(ii);
    [~,RGBnonlinCLUT{ii},~] = ...
        birefringenceColorLookUpTableV01(thickness_vec,theta_vec,alfa_vec,lambdaFlag,polFlag,colorTemp,gbap.colorTempChrom);
end
toc

%% Determine inclination angles (theta) for each grain for each threshold

tic
for q = 1:size(gbap.threshfacVec,1)
    for kk = 1:(length(gbap.optA(q).B)-(gbap.optA(q).N + 0))
        
        qq = kk + gbap.optA(q).N;
        
        
        
        %thickness from interpolated array, using centroid values
        %(x(1)=column, y(2)=row)
        thickness = gbap.thicknessArrayInterpCrop(...
            round(gbap.stats(q).regionProps.Centroid(qq,2)),...
            round(gbap.stats(q).regionProps.Centroid(qq,1)))*10^-3;
        %         text(gbap.stats(q).regionProps.Centroid(qq,1),gbap.stats(q).regionProps.Centroid(qq,2)
        
        %find index of thickness corresponding to centroid of grain
        thicknessIndex_grain = find(thickness_vec >= thickness,1)-1;
        
        for ii = 1:length(colorTempChrom_vec)
            
            %limit color look-up table to thickness of centroid of grain with tolerance
            RGBnonlinCLUT_grain = RGBnonlinCLUT{ii}(:,thicknessIndex_grain-1:thicknessIndex_grain+1,:,:);
            
            maximumIndex = find(gbap.angleVec == gbap.grainMaximum{q,kk},1);
            colorTarget = squeeze(gbap.grainModeArray{q}(maximumIndex,kk,:));
            
            % RGBnonlinCLUT_grain
%             minError = zeros(length(colorTempChrom_vec),size(RGBnonlinCLUT_grain,2),size(RGBnonlinCLUT_grain,3),size(RGBnonlinCLUT_grain,4));
            
            %Find closest value of color to target
            % minError = zeros(length(thickness_vec),length(theta_vec),length(alfa_vec));
            
            for r=1:size(RGBnonlinCLUT_grain,2)
                for rr=1:size(RGBnonlinCLUT_grain,3)
                    for rrr=1:size(RGBnonlinCLUT_grain,4)
                        %                     minError(r,rr,rrr) = sum(abs((RGBnonlinCLUT_grain(:,r,rr,rrr))-colorTarget));
                        minError(ii,r,rr,rrr) = sqrt(sum(((RGBnonlinCLUT_grain(:,r,rr,rrr))-colorTarget).^2));
                    end
                end
            end
        end
        [minValue,closestIndex] = min(minError,[],'all','linear');
        [colorTempChromIndex,thicknessIndex,thetaIndex,alfaIndex] = ind2sub(size(minError),closestIndex);
%         disp(colorTempChrom_vec(colorTempChromIndex))
        gbap.grainTheta{q,kk} = theta_vec(thetaIndex)*180/pi;
    end
end
toc


%% Create vibrant colormaps for schmidt plots

for q = 1:size(gbap.threshfacVec,1)
    %     nexttile
    %     hold on
    for kk = 1:(length(gbap.optA(q).B)-(gbap.optA(q).N + 0))
        
        %extinction color
        minimumIndex = find(gbap.angleVec == gbap.grainExtinction{q,kk},1);
        %         colorTarget = squeeze(gbap.grainModeArray{q}(minimumIndex,kk,:));
        gbap.grainExtinctionModeColormap{q,kk} = squeeze(gbap.grainModeArray{q}(minimumIndex,kk,:));
        
        %brightest color
        maximumIndex = find(gbap.angleVec == gbap.grainMaximum{q,kk},1);
        %         colorTarget = squeeze(gbap.grainModeArray{q}(maximumIndex,kk,:));
        gbap.grainMaximumModeColormap{q,kk} = squeeze(gbap.grainModeArray{q}(maximumIndex,kk,:));
    end
end


%% Schmidt plot

% figure
scrsz = get(groot,'ScreenSize');
figure('Position',...
    [scrsz(3)*0.02 scrsz(4)*0.15 scrsz(3)*0.95 scrsz(4)*0.70]);
% 'Position' definition: [left bottom width height]

% tiledlayout(1+(size(gbap.threshfacVec,1)>1),ceil(size(gbap.threshfacVec,1)/2),'TileSpacing','loose','Padding','loose');
tiledlayout(1+(size(gbap.threshfacVec,1)>1),ceil(size(gbap.threshfacVec,1)/2));
for q = 1:size(gbap.threshfacVec,1)
    nexttile
    ps = polarscatter(deg2rad([gbap.grainExtinction{q,:}]),[gbap.grainTheta{q,:}],[],[gbap.grainExtinctionModeColormap{q,:}]','filled');
    ps.SizeData = 10; %marker size (default=36)
    %     polarscatter(deg2rad([gbap.grainMaximum{q,:}]),[gbap.grainTheta{q,:}],[],[gbap.grainMaximumModeColormap{q,:}]','filled')
    
    title(['threshold=',num2str(gbap.threshfacVec(q),2)],...
        'Interpreter','Latex','FontName','Times New Roman','FontSize',gbap.fontSize+1)
    %     hold off
    ax = gca;
    ax.RLim = [0 90];
    set(gca,'FontName','Times New Roman')
    set(gca,'FontSize',8)
    set(gcf,'color','w')
end


%% Plot c-axes for each grain for each threshold

%limits plot to labeling at most 149 grains per threshold value
if (length(gbap.optA(1).B)-(gbap.optA(1).N + 0)) < 150
    
    % figure
    scrsz = get(groot,'ScreenSize');
    figure('Position',...
        [scrsz(3)*0.02 scrsz(4)*0.15 scrsz(3)*0.95 scrsz(4)*0.70]);
    % 'Position' definition: [left bottom width height]
    
    tiledlayout(1+(size(gbap.threshfacVec,1)>1),ceil(size(gbap.threshfacVec,1)/2),'TileSpacing','tight','Padding','compact');
    for q = 1:size(gbap.threshfacVec,1)
        nexttile
        baseImage = rgb2gray(gbap.imageStack{1,2});
        mask = boundarymask(gbap.optA(q).L);
        baseImage = imoverlay(baseImage,mask,gbap.CM(q,:));
        imshow(baseImage)
        
        for kk = 1:(length(gbap.optA(q).B)-(gbap.optA(q).N + 0))
            hold on
            % stats(q).regionProps.Centroid
            qq = kk + gbap.optA(q).N;
            text(gbap.stats(q).regionProps.Centroid(qq,1),gbap.stats(q).regionProps.Centroid(qq,2),['\leftarrow',num2str(gbap.grainExtinction{q,kk})],'Color','w')
        end
        hold off
        
        title(['threshold=',num2str(gbap.threshfacVec(q),2)])
    end
    
end

% for kk = 1:(length(gbap.optA(q).B)-(gbap.optA(q).N + 0))
%     hold on
%     % stats(q).regionProps.Centroid
%     qq = kk + gbap.optA(q).N;
%     text(stats(q).regionProps.Centroid(qq,1),stats(q).regionProps.Centroid(qq,2),['\leftarrow',num2str(grainExtinction(kk))],'Color','w')
% end
% hold off
% title(['threshold=',num2str(gbap.threshfacVec(q),2)])


%% C-axis azimuth distribution per grain
% figure;polarscatter(deg2rad(grainExtinction(:)),ones(length(grainExtinction(q,:)),1),1)

% figure
scrsz = get(groot,'ScreenSize');
figure('Position',...
    [scrsz(3)*0.02 scrsz(4)*0.15 scrsz(3)*0.95 scrsz(4)*0.70]);
% 'Position' definition: [left bottom width height]

tiledlayout(1+(size(gbap.threshfacVec,1)>1),ceil(size(gbap.threshfacVec,1)/2),'TileSpacing','loose','Padding','loose');
for q = 1:size(gbap.threshfacVec,1)
    nexttile
    polarhistogram(deg2rad([gbap.grainExtinction{q,:}])',length(gbap.angleVec)+0)
    title(['threshold=',num2str(gbap.threshfacVec(q),2)],'FontName','Times New Roman','FontSize',gbap.fontSize)
end
dim = [0.50 0.98 0.00 0.00]; %[x y w h]
str = 'Distribution of c-axis azimuth by grain';
annotation('textbox',dim,'String',str,'FitBoxToText','on',...
    'EdgeColor','none','HorizontalAlignment','center',...
    'VerticalAlignment','middle',...
    'FontName','Times New Roman','FontWeight','bold','FontSize',gbap.fontSize+1);
set(gcf,'color','w')
% title('Distribution of c-axis azimuth by grain')


%% Determine circular polarized image from image stack

imageCirPol = im2double(rgb2gray(gbap.imageStack{1,2}));
for k = 2:size(gbap.imageStack,1)
    imageCirPol = imadd(imageCirPol,im2double(rgb2gray(gbap.imageStack{k,2})));
end
gbap.imageCirPoleq = histeq(im2uint8(imageCirPol/size(gbap.imageStack,1)));
gbap.imageCirPol = (im2uint8(imageCirPol/size(gbap.imageStack,1)));

% figure;imshow(imageCirPol)
figure;imshowpair(gbap.imageCirPoleq,gbap.imageCirPol,'montage')
title('Digital circular polarization')


%% Determine grayscale value profile for a selected pixel

profileArray = zeros(size(gbap.imageStack{1,2},1),size(gbap.imageStack{1,2},2),size(gbap.imageStack,1));

for k = 1:size(gbap.imageStack,1)
    
    %     profileArray(:,:,k) = im2double(rgb2gray(imageStack{k,1}));
    profileArray(:,:,k) = (rgb2gray(gbap.imageStack{k,2}));
end


%% Determine minimum values from grayscale profiles for each pixel

[profileMinMatValue,profileMinMatIndex] = min(profileArray,[],3);

% figure;imshow(profileMinMatValue)
% figure;imshow(profileMinMatIndex)

gbap.imageAzimuthAngle = zeros(size(profileMinMatIndex));
for k = 1:size(profileMinMatIndex,1)
    for kk = 1:size(profileMinMatIndex,2)
        gbap.imageAzimuthAngle(k,kk) = gbap.angleVec(profileMinMatIndex(k,kk));
    end
end

% figure
scrsz = get(groot,'ScreenSize');
figure('Position',...
    [scrsz(3)*0.02 scrsz(4)*0.15 scrsz(3)*0.95 scrsz(4)*0.70]);
% 'Position' definition: [left bottom width height]

subplot(1,2,1)
% imhist(imageCirPol)
% figure;polarscatter(deg2rad(azimuthAngle(:)),90*ones(numel(azimuthAngle),1))
polarscatter(deg2rad(gbap.imageAzimuthAngle(:)),gbap.imageCirPol(:),1)
ax = gca;
ax.RLim = [0 255];
title('CirPol c-axis pixel azimuth and grayscale values')

subplot(1,2,2)
polarhistogram(deg2rad(gbap.imageAzimuthAngle(:)),length(gbap.angleVec)+0)
title('CirPol distribution of c-axis azimuth by pixel')


%% Extinction composite image

% figure
scrsz = get(groot,'ScreenSize');
figure('Position',...
    [scrsz(3)*0.02 scrsz(4)*0.15 scrsz(3)*0.95 scrsz(4)*0.70]);
%'Position' definition: [left bottom width height]

tic
tiledlayout(1+(size(gbap.threshfacVec,1)>1),ceil(size(gbap.threshfacVec,1)/2),'TileSpacing','tight','Padding','compact');
for q = 1:size(gbap.threshfacVec,1)
    nexttile
    %     baseImage = (gbap.imageStack{1,2});
    imageExtinctionComposite = gbap.imageStack{1,2};
    for kk = 1:(length(gbap.optA(q).B)-(gbap.optA(q).N + 0))
        qq = kk + gbap.optA(q).N + 0;
        extinctionIndex = find(gbap.angleVec == gbap.grainExtinction{q,kk},1);
        exinctionImage = gbap.imageStack{extinctionIndex,2};
        imageExtinctionComposite(repmat(gbap.optA(q).L == qq,[1 1 3])) = ...
            exinctionImage(repmat(gbap.optA(q).L == qq,[1 1 3]));
    end
    %     imshow(imageExtinctionComposite)
    mask = boundarymask(gbap.optA(q).L);
    %     imageExtinctionComposite = imoverlay(imageExtinctionComposite,mask,gbap.CM(q,:));
    gbap.imageExtinctionComposite{q} = imoverlay(imageExtinctionComposite,mask,'w');
    imshow(gbap.imageExtinctionComposite{q})
    
    title(['threshold=',num2str(gbap.threshfacVec(q),2)],...
        'Interpreter','Latex','FontName','Times New Roman','FontSize',gbap.fontSize+1)
end
% set(gca,'FontName','Times New Roman') set(gca,'FontSize',gbap.fontSize)
dim = [0.50 0.98 0.00 0.00]; %[x y w h]
str = 'Extinction composite';
annotation('textbox',dim,'String',str,'FitBoxToText','on',...
    'EdgeColor','none','HorizontalAlignment','center',...
    'VerticalAlignment','middle',...
    'FontName','Times New Roman','FontWeight','bold','FontSize',gbap.fontSize+1);
set(gcf,'color','w')
toc

%% Maximum composite image

% figure
scrsz = get(groot,'ScreenSize');
figure('Position',...
    [scrsz(3)*0.02 scrsz(4)*0.15 scrsz(3)*0.95 scrsz(4)*0.70]);
%'Position' definition: [left bottom width height]

tic
tiledlayout(1+(size(gbap.threshfacVec,1)>1),ceil(size(gbap.threshfacVec,1)/2),'TileSpacing','tight','Padding','compact');
for q = 1:size(gbap.threshfacVec,1)
    nexttile
    %     baseImage = (gbap.imageStack{1,2});
    imageMaximumComposite = gbap.imageStack{1,2};
    for kk = 1:(length(gbap.optA(q).B)-(gbap.optA(q).N + 0))
        qq = kk + gbap.optA(q).N + 0;
        maximumIndex = find(gbap.angleVec == gbap.grainMaximum{q,kk},1);
        maximumImage = gbap.imageStack{maximumIndex,2};
        imageMaximumComposite(repmat(gbap.optA(q).L == qq,[1 1 3])) = ...
            maximumImage(repmat(gbap.optA(q).L == qq,[1 1 3]));
    end
    %     imshow(imageMaximumComposite)
    
    mask = boundarymask(gbap.optA(q).L);
    %     imageMaximumComposite = imoverlay(imageMaximumComposite,mask,gbap.CM(q,:));
    gbap.imageMaximumComposite{q} = imoverlay(imageMaximumComposite,mask,'w');
    imshow(gbap.imageMaximumComposite{q})
    
    title(['threshold=',num2str(gbap.threshfacVec(q),2)],...
        'Interpreter','Latex','FontName','Times New Roman','FontSize',gbap.fontSize+1)
end
% set(gca,'FontName','Times New Roman') set(gca,'FontSize',gbap.fontSize)
dim = [0.50 0.98 0.00 0.00]; %[x y w h]
str = 'Brightest composite';
annotation('textbox',dim,'String',str,'FitBoxToText','on',...
    'EdgeColor','none','HorizontalAlignment','center',...
    'VerticalAlignment','middle',...
    'FontName','Times New Roman','FontWeight','bold','FontSize',gbap.fontSize+1);
set(gcf,'color','w')
toc


%% Maximum composite image with mode values

% figure
scrsz = get(groot,'ScreenSize');
figure('Position',...
    [scrsz(3)*0.02 scrsz(4)*0.15 scrsz(3)*0.95 scrsz(4)*0.70]);
%'Position' definition: [left bottom width height]

tic
tiledlayout(1+(size(gbap.threshfacVec,1)>1),ceil(size(gbap.threshfacVec,1)/2),'TileSpacing','tight','Padding','compact');
for q = 1:size(gbap.threshfacVec,1)
    nexttile
    %     baseImage = (gbap.imageStack{1,2});
    imageMaximumModeComposite = gbap.imageStack{1,2};
    for kk = 1:(length(gbap.optA(q).B)-(gbap.optA(q).N + 0))
        qq = kk + gbap.optA(q).N + 0;
        
        maximumIndex = find(gbap.angleVec == gbap.grainMaximum{q,kk},1);
        maximumImage = gbap.imageStack{maximumIndex,2};
        %         maximumImage = imoverlay(maximumImage,gbap.optA(q).L == qq,((squeeze(im2double(gbap.grainModeArray{q}(maximumIndex,kk,:))))./1'));
        maximumImage = imoverlay(maximumImage,gbap.optA(q).L == qq,((squeeze(gbap.grainModeArray{q}(maximumIndex,kk,:)))./1'));
        imageMaximumModeComposite(repmat(gbap.optA(q).L == qq,[1 1 3])) = maximumImage(repmat(gbap.optA(q).L == qq,[1 1 3]));
    end
    
    mask = boundarymask(gbap.optA(q).L);
    %     imageMaximumModeComposite = imoverlay(imageMaximumModeComposite,mask,gbap.CM(q,:));
    gbap.imageMaximumModeComposite{q} = imoverlay(imageMaximumModeComposite,mask,'w');
    imshow(gbap.imageMaximumModeComposite{q})
    
    title(['threshold=',num2str(gbap.threshfacVec(q),2)],...
        'Interpreter','Latex','FontName','Times New Roman','FontSize',gbap.fontSize+1)
end
% set(gca,'FontName','Times New Roman') set(gca,'FontSize',gbap.fontSize)
dim = [0.50 0.98 0.00 0.00]; %[x y w h]
str = 'Mode composite';
annotation('textbox',dim,'String',str,'FitBoxToText','on',...
    'EdgeColor','none','HorizontalAlignment','center',...
    'VerticalAlignment','middle',...
    'FontName','Times New Roman','FontWeight','bold','FontSize',gbap.fontSize+1);
set(gcf,'color','w')
toc



%% Maximum composite image with mean values

% figure
scrsz = get(groot,'ScreenSize');
figure('Position',...
    [scrsz(3)*0.02 scrsz(4)*0.15 scrsz(3)*0.95 scrsz(4)*0.70]);
%'Position' definition: [left bottom width height]

tic
tiledlayout(1+(size(gbap.threshfacVec,1)>1),ceil(size(gbap.threshfacVec,1)/2),'TileSpacing','tight','Padding','compact');
for q = 1:size(gbap.threshfacVec,1)
    nexttile
    %     baseImage = (gbap.imageStack{1,2});
    imageMaximumMeanComposite = gbap.imageStack{1,2};
    for kk = 1:(length(gbap.optA(q).B)-(gbap.optA(q).N + 0))
        qq = kk + gbap.optA(q).N + 0;
        maximumIndex = find(gbap.angleVec == gbap.grainMaximum{q,kk},1);
        maximumImage = gbap.imageStack{maximumIndex,2};
        % maximumImage = imoverlay(maximumImage,gbap.optA(q).L ==
        % qq,rescale(squeeze(im2double(gbap.grainMeanArray{q}(maximumIndex,kk,:))),0,1)');
        %         maximumImage = imoverlay(maximumImage,gbap.optA(q).L == qq,((squeeze(im2double(gbap.grainMeanArray{q}(maximumIndex,kk,:))))./1)');
        maximumImage = imoverlay(maximumImage,gbap.optA(q).L == qq,((squeeze((gbap.grainMeanArray{q}(maximumIndex,kk,:))))./1)');
        imageMaximumMeanComposite(repmat(gbap.optA(q).L == qq,[1 1 3])) = maximumImage(repmat(gbap.optA(q).L == qq,[1 1 3]));
    end
    mask = boundarymask(gbap.optA(q).L);
    %     imageMaximumMeanComposite = imoverlay(imageMaximumMeanComposite,mask,gbap.CM(q,:));
    gbap.imageMaximumMeanComposite{q} = imoverlay(imageMaximumMeanComposite,mask,'w');
    imshow(gbap.imageMaximumMeanComposite{q})
    
    title(['threshold=',num2str(gbap.threshfacVec(q),2)],...
        'Interpreter','Latex','FontName','Times New Roman','FontSize',gbap.fontSize+1)
end
% set(gca,'FontName','Times New Roman') set(gca,'FontSize',gbap.fontSize)
dim = [0.50 0.98 0.00 0.00]; %[x y w h]
str = 'Mean composite';
annotation('textbox',dim,'String',str,'FitBoxToText','on',...
    'EdgeColor','none','HorizontalAlignment','center',...
    'VerticalAlignment','middle',...
    'FontName','Times New Roman','FontWeight','bold','FontSize',gbap.fontSize+1);
set(gcf,'color','w')
toc


%% Maximum composite image with median values

% figure
scrsz = get(groot,'ScreenSize');
figure('Position',...
    [scrsz(3)*0.02 scrsz(4)*0.15 scrsz(3)*0.95 scrsz(4)*0.70]);
%'Position' definition: [left bottom width height]

tic
tiledlayout(1+(size(gbap.threshfacVec,1)>1),ceil(size(gbap.threshfacVec,1)/2),'TileSpacing','tight','Padding','compact');
for q = 1:size(gbap.threshfacVec,1)
    nexttile
    %     baseImage = (gbap.imageStack{1,2});
    imageMaximumMedianComposite = gbap.imageStack{1,2};
    for kk = 1:(length(gbap.optA(q).B)-(gbap.optA(q).N + 0))
        qq = kk + gbap.optA(q).N + 0;
        maximumIndex = find(gbap.angleVec == gbap.grainMaximum{q,kk},1);
        maximumImage = gbap.imageStack{maximumIndex,2};
        % maximumImage = imoverlay(maximumImage,gbap.optA(q).L ==
        % qq,rescale(squeeze(im2double(gbap.grainMeanArray{q}(maximumIndex,kk,:))),0,1)');
        %         maximumImage = imoverlay(maximumImage,gbap.optA(q).L == qq,((squeeze(im2double(gbap.grainMedianArray{q}(maximumIndex,kk,:))))./1)');
        maximumImage = imoverlay(maximumImage,gbap.optA(q).L == qq,((squeeze((gbap.grainMedianArray{q}(maximumIndex,kk,:))))./1)');
        imageMaximumMedianComposite(repmat(gbap.optA(q).L == qq,[1 1 3])) = maximumImage(repmat(gbap.optA(q).L == qq,[1 1 3]));
    end
    mask = boundarymask(gbap.optA(q).L);
    %     imageMaximumMedianComposite = imoverlay(imageMaximumMedianComposite,mask,gbap.CM(q,:));
    gbap.imageMaximumMedianComposite{q} = imoverlay(imageMaximumMedianComposite,mask,'w');
    imshow(gbap.imageMaximumMedianComposite{q})
    
    title(['threshold=',num2str(gbap.threshfacVec(q),2)],...
        'Interpreter','Latex','FontName','Times New Roman','FontSize',gbap.fontSize+1)
end
% set(gca,'FontName','Times New Roman') set(gca,'FontSize',gbap.fontSize)
dim = [0.50 0.98 0.00 0.00]; %[x y w h]
str = 'Median composite';
annotation('textbox',dim,'String',str,'FitBoxToText','on',...
    'EdgeColor','none','HorizontalAlignment','center',...
    'VerticalAlignment','middle',...
    'FontName','Times New Roman','FontWeight','bold','FontSize',gbap.fontSize+1);
set(gcf,'color','w')
toc


%% Save results

% load(fullfile([pwd,'/practice',subfolder],'ImageStackData.mat'),'gbap')
% fileNameData = fullfile([pwd,gbap.subfolder],['gbapResultsData_',num2str(gbap.cropscale),'.mat']);
save(fullfile([pwd,gbap.subfolder],['gbapResultsData_',num2str(gbap.cropscale,'%0.2f'),'.mat']),'gbap')

