%%
% clear memory and previously allocated variables
clc;clear all;close all;


%% Load Grain Boundary Analysis Preprocessing (gbap) data

run ColorIceInputFile


%% Make directory and select file to load

% record directory in output folder
directory = dir(fullfile([pwd,subfolder],['gbapResultsData_*','.mat']));
directoryNameList = {directory.name}';
clear directory

% list of relevant files
TOCs = cell(size(directoryNameList,1),1);
for k = 1:size(directoryNameList,1)
    TOCs{k} = [num2str(k),':',' ',directoryNameList{k}];
end
fprintf('Table of Contents for Files: \n')
disp(TOCs)
clear k TOCs

% input explicit number from list
fileInput = input('Type number from list to load: ');

clc
disp(horzcat('Loading ',directoryNameList{fileInput}))


%% Load data file

%create name and find location for data file and load 
% load(fullfile([pwd,'/practice',subfolder],['gbapResultsData_',num2str(cropscale,'%0.2f'),'.mat']),'gbap')
tic
load(fullfile([pwd,subfolder],directoryNameList{fileInput}),'gbap')
toc
clear subfolder clear cropscale

% %create name and find location for data file and load file
% load(fullfile([pwd,'/practice',subfolder],['gbapResultsData_',num2str(cropscale,'%0.2f'),'.mat']),'gbap')
% clear subfolder clear cropscale


%% Plot cropped color image stack

figure;montage(gbap.imageStack(:,2), 'Size', [2 ceil(size(gbap.imageStack,1)/2)])
% NOTE: use second column of imageStack for final cropped images



%% Low resolution grain boundary segmentation stack

% figure
scrsz = get(groot,'ScreenSize');
figure('Position',...
    [scrsz(3)*0.02 scrsz(4)*0.15 scrsz(3)*0.95 scrsz(4)*0.70]);
% 'Position' definition: [left bottom width height]

% tiledlayout(1+(size(gbap.threshfacVec,1)>1),ceil(size(gbap.threshfacVec,1)/2),'TileSpacing','tight','Padding','compact');
tLayout = tiledlayout(1+(size(gbap.threshfacVec,1)>1),ceil(size(gbap.threshfacVec,1)/2),'TileSpacing','tight','Padding','compact');
for q = 1:size(gbap.threshfacVec,1)
    ax1(q) = nexttile(tLayout);
    %     figure subplot()
    baseImage = rgb2gray(gbap.imageStack{1,2});
    mask = boundarymask(gbap.optA(q).L);
    baseImage = imoverlay(baseImage,mask,gbap.CM(q,:));
    imshow(baseImage)
    
    %     ax = nt; ax.Units = 'normalized'; annotationPosition =
    %     ax1(q).Position;
    % %     figure annotation('rectangle',annotationPosition,'Color','red')
    %     rectangle('Position',annotationPosition,'Curvature',0,'EdgeColor','r','FaceColor','r')
    
    
    %     dim = [0.50 0.98 0.00 0.00]; %[x y w h]
    % str = '20 mm';
    % annotation('textbox',dim,'String',str,'FitBoxToText','on',...
    %     'EdgeColor','w','HorizontalAlignment','center',...
    %     'VerticalAlignment','middle',... 'FontName','Times New
    %     Roman','FontWeight','bold','FontSize',gbap.fontSize+1);
    
    
    title(['threshold=',num2str(gbap.threshfacVec(q),2)],...
        'Interpreter','Latex','FontName','Times New Roman','FontSize',gbap.fontSize+1)
end
dim = [0.50 0.98 0.00 0.00]; %[x y w h]
str = 'Grain boundary segmentation';
annotation('textbox',dim,'String',str,'FitBoxToText','on',...
    'EdgeColor','none','HorizontalAlignment','center',...
    'VerticalAlignment','middle',...
    'FontName','Times New Roman','FontWeight','bold','FontSize',gbap.fontSize+1);
set(gcf,'color','w')

% for q = 1:size(gbap.threshfacVec,1) %         ax1(q) = nexttile(tLayout);
%             annotationPosition = ax1(q).Position; %four-element vector of
%             the form [x y w h] annotationPosition =
%             [ax1(q).Position(1)*1.01 ax1(q).Position(2)*1.01
%             ax1(q).Position(3)/10 ax1(q).Position(4)/20 ];
%
%
% %             annotationPosition = annotationPosition/10; %
% dimLabelPosition = annotationPosition/10;
% annotation('rectangle',annotationPosition,'Color','w','FaceColor','w','FaceAlpha',0.5)
% end


%% Plot frequency trends

axset = struct();

% figure
scrsz = get(groot,'ScreenSize');
figure('Position',...
    [scrsz(3)*0.005 scrsz(4)*0.2 scrsz(3)*0.99 scrsz(4)*0.60]);
% 'Position' definition: [left bottom width height]
for q = 1:size(gbap.threshfacVec,1)
    %     figure subplot(1,2,1)
    axset(q).ax1 = subplot(2,size(gbap.threshfacVec,1),q);
    histogram(gbap.stats(q).Area,'FaceColor',gbap.CM(q,:),'FaceAlpha',1)
    %     title([num2str(threshfacVec(q),'%.1f'),': ','$\tilde{A}$=',...
    %         num2str(median(stats(q).regionProps.Area(optA(q).N+1:end)*mmPerPixel^2),'%.1f'),'mm$\textsuperscript{2}$',...
    %         ',
    %         $A_{f}$=',num2str(sum(stats(q).regionProps.Area(optA(q).N+1:end))/numel(imageStack{1})*100,'%.1f'),'$\%$'],...
    %         'Interpreter','LaTeX','FontName','Times New
    %         Roman','FontSize',fontSize)
    xlabel('grain area [mm$\textsuperscript{2}$]',...
        'Interpreter','LaTeX','FontName','Times New Roman','FontSize',gbap.fontSize)
    if q == 1
        ylabel('frequency [-]',...
            'Interpreter','LaTeX','FontName','Times New Roman','FontSize',gbap.fontSize)
    end
    axis tight
    grid on
    set(gca,'FontName','Times New Roman')
    set(gca,'FontSize',gbap.fontSize)
    title([num2str(gbap.threshfacVec(q),'%.1f'),': ','$\bar{A}$=',...
        num2str(gbap.stats(q).meanArea,'%.1f'),'mm$\textsuperscript{2}$',...
        ', $A_{f}$=',num2str(gbap.stats(q).areaFraction,'%.2f')],...
        'Interpreter','LaTeX','FontName','Times New Roman','FontSize',gbap.fontSize-1)
    
    %     subplot(1,2,2)
    axset(q).ax2 = subplot(2,size(gbap.threshfacVec,1),q+size(gbap.threshfacVec,1));
    histogram(gbap.stats(q).equivDia,'FaceColor',gbap.CM(q,:),'FaceAlpha',1)
    %     title([num2str(threshfacVec(q),'%.1f'),': ','$\tilde{d}$=',...
    %         num2str(median(stats(q).regionProps.EquivDiameter(optA(q).N+1:end)*mmPerPixel),'%.1f'),'mm',...
    %         ',
    %         N=',num2str(length(stats(q).regionProps.EquivDiameter(optA(q).N+1:end)))],...
    %         'Interpreter','LaTeX','FontName','Times New
    %         Roman','FontSize',fontSize)
    xlabel('equiv. grain dia. [mm]',...
        'Interpreter','LaTeX','FontName','Times New Roman','FontSize',gbap.fontSize)
    if q == 1
        ylabel('frequency [-]',...
            'Interpreter','LaTeX','FontName','Times New Roman','FontSize',gbap.fontSize)
    end
    axis tight
    grid on
    set(gca,'FontName','Times New Roman')
    set(gca,'FontSize',gbap.fontSize)
    title([num2str(gbap.threshfacVec(q),'%.1f'),': ','$\bar{d}$=',...
        num2str(gbap.stats(q).meanEquivDia,'%.1f'),'mm',...
        ', N=',num2str(gbap.stats(q).numGrains)],...
        'Interpreter','LaTeX','FontName','Times New Roman','FontSize',gbap.fontSize-1)
end

% linkaxes([axset(:).ax1],'xy');linkaxes([axset(:).ax2],'xy');
set(gcf,'color','w')


%% Plot thresholding trends

% figure
scrsz = get(groot,'ScreenSize');
figure('Position',...
    [scrsz(3)*0.3 scrsz(4)*0.1 scrsz(3)*0.50 scrsz(4)*0.80]);
% 'Position' definition: [left bottom width height]

subplot(4,1,1)
hold on
for q=1:size(gbap.threshfacVec,1)
    %     plot(threshfacVec(q),[stats(q).areaFraction]','o','MarkerEdgeColor',CM(q,:),'MarkerFaceColor',CM(q,:),'MarkerSize',4)
    plot(gbap.threshfacVec(q),[gbap.stats(q).areaFraction]','o','MarkerEdgeColor','k','MarkerFaceColor',gbap.CM(q,:),'MarkerSize',gbap.markerSize)
end
% plot(threshfacVec,[stats(:).areaFraction]')
hold off
% xlabel('threshold [-]',...
%     'Interpreter','LaTeX','FontName','Times New
%     Roman','FontSize',fontSize)
ylabel('area fraction [-]',...
    'Interpreter','LaTeX','FontName','Times New Roman','FontSize',gbap.fontSize)
% axis tight
grid on
set(gca,'FontName','Times New Roman')
set(gca,'FontSize',gbap.fontSize)

subplot(4,1,2)
hold on
for q=1:size(gbap.threshfacVec,1)
    %     plot(threshfacVec(q),[stats(q).numGrains]','o','MarkerEdgeColor',CM(q,:),'MarkerFaceColor',CM(q,:),'MarkerSize',4)
    plot(gbap.threshfacVec(q),[gbap.stats(q).numGrains]','o','MarkerEdgeColor','k','MarkerFaceColor',gbap.CM(q,:),'MarkerSize',gbap.markerSize)
end
% plot(threshfacVec,[stats(:).numGrains]')
hold off
% xlabel('threshold scale [-]',...
%     'Interpreter','LaTeX','FontName','Times New
%     Roman','FontSize',fontSize)
ylabel('number of grains [-]',...
    'Interpreter','LaTeX','FontName','Times New Roman','FontSize',gbap.fontSize)
% axis tight
grid on
set(gca,'FontName','Times New Roman')
set(gca,'FontSize',gbap.fontSize)

subplot(4,1,3)
hold on
for q=1:size(gbap.threshfacVec,1)
    %     plot(threshfacVec(q),[stats(q).meanArea]','o','MarkerEdgeColor',CM(q,:),'MarkerFaceColor',CM(q,:),'MarkerSize',4)
    plot(gbap.threshfacVec(q),[gbap.stats(q).meanArea]','o','MarkerEdgeColor','k','MarkerFaceColor',gbap.CM(q,:),'MarkerSize',gbap.markerSize)
end
% plot(threshfacVec,[stats(:).meanArea]')
hold off
% xlabel('threshold [-]',...
%     'Interpreter','LaTeX','FontName','Times New
%     Roman','FontSize',fontSize)
ylabel('mean grain area [mm$\textsuperscript{2}$]',...
    'Interpreter','LaTeX','FontName','Times New Roman','FontSize',gbap.fontSize)
% axis tight
grid on
set(gca,'FontName','Times New Roman')
set(gca,'FontSize',gbap.fontSize)

subplot(4,1,4)
hold on
for q=1:size(gbap.threshfacVec,1)
    %     plot(threshfacVec(q),[stats(q).meanEquivDia]','o','MarkerEdgeColor',CM(q,:),'MarkerFaceColor',CM(q,:),'MarkerSize',4)
    plot(gbap.threshfacVec(q),[gbap.stats(q).meanEquivDia]','o','MarkerEdgeColor','k','MarkerFaceColor',gbap.CM(q,:),'MarkerSize',gbap.markerSize)
end
% plot(threshfacVec,[stats(:).meanEquivDia]')
hold off
xlabel('threshold [-]',...
    'Interpreter','LaTeX','FontName','Times New Roman','FontSize',gbap.fontSize)
ylabel('mean equiv. grain dia. [mm]',...
    'Interpreter','LaTeX','FontName','Times New Roman','FontSize',gbap.fontSize)
% axis tight
grid on
set(gca,'FontName','Times New Roman')
set(gca,'FontSize',gbap.fontSize)

set(gcf,'color','w')


%% Plot c-axes for each grain for each threshold

%limits plot to labeling at most 149 grains per threshold value
if (length(gbap.optA(1).B)-(gbap.optA(1).N + 0)) < 150
    
    % figure
    scrsz = get(groot,'ScreenSize');
    figure('Position',...
        [scrsz(3)*0.02 scrsz(4)*0.15 scrsz(3)*0.95 scrsz(4)*0.70]);
    % 'Position' definition: [left bottom width height]
    
    tiledlayout(1+(size(gbap.threshfacVec,1)>1),ceil(size(gbap.threshfacVec,1)/2),'TileSpacing','tight','Padding','compact');
    for q = 1:size(gbap.threshfacVec,1)
        nexttile
        baseImage = rgb2gray(gbap.imageStack{1,2});
        mask = boundarymask(gbap.optA(q).L);
        baseImage = imoverlay(baseImage,mask,gbap.CM(q,:));
        imshow(baseImage)
        
        for kk = 1:(length(gbap.optA(q).B)-(gbap.optA(q).N + 0))
            hold on
            % stats(q).regionProps.Centroid
            qq = kk + gbap.optA(q).N;
            text(gbap.stats(q).regionProps.Centroid(qq,1),gbap.stats(q).regionProps.Centroid(qq,2),['\leftarrow',num2str(gbap.grainExtinction{q,kk})],'Color','w')
        end
        hold off
        
        title(['threshold=',num2str(gbap.threshfacVec(q),2)])
    end
    
end


%% C-axis azimuth distribution per grain
% figure;polarscatter(deg2rad(grainExtinction(:)),ones(length(grainExtinction(q,:)),1),1)

% figure
scrsz = get(groot,'ScreenSize');
figure('Position',...
    [scrsz(3)*0.02 scrsz(4)*0.15 scrsz(3)*0.95 scrsz(4)*0.70]);
% 'Position' definition: [left bottom width height]

tiledlayout(1+(size(gbap.threshfacVec,1)>1),ceil(size(gbap.threshfacVec,1)/2),'TileSpacing','loose','Padding','loose');
for q = 1:size(gbap.threshfacVec,1)
    nexttile
    polarhistogram(deg2rad([gbap.grainExtinction{q,:}])',length(gbap.angleVec)+0)
    title(['threshold=',num2str(gbap.threshfacVec(q),2)],'FontName','Times New Roman','FontSize',gbap.fontSize)
end
dim = [0.50 0.98 0.00 0.00]; %[x y w h]
str = 'Distribution of c-axis azimuth by grain';
annotation('textbox',dim,'String',str,'FitBoxToText','on',...
    'EdgeColor','none','HorizontalAlignment','center',...
    'VerticalAlignment','middle',...
    'FontName','Times New Roman','FontWeight','bold','FontSize',gbap.fontSize+1);
set(gcf,'color','w')
% title('Distribution of c-axis azimuth by grain')


%% Determine circular polarized image from image stack

% imageCirPol = im2double(rgb2gray(gbap.imageStack{1,2})); for k =
% 2:size(gbap.imageStack,1)
%     imageCirPol =
%     imadd(imageCirPol,im2double(rgb2gray(gbap.imageStack{k,2})));
% end gbap.imageCirPoleq =
% histeq(im2uint8(imageCirPol/size(gbap.imageStack,1))); gbap.imageCirPol =
% (im2uint8(imageCirPol/size(gbap.imageStack,1)));

% figure;imshow(imageCirPol)
figure;imshowpair(gbap.imageCirPoleq,gbap.imageCirPol,'montage')
title('Digital circular polarization')


%% Determine minimum values from grayscale profiles for each pixel

% [profileMinMatValue,profileMinMatIndex] = min(profileArray,[],3);

% figure;imshow(profileMinMatValue) figure;imshow(profileMinMatIndex)

% gbap.imageAzimuthAngle = zeros(size(profileMinMatIndex)); for k =
% 1:size(profileMinMatIndex,1)
%     for kk = 1:size(profileMinMatIndex,2)
%         gbap.imageAzimuthAngle(k,kk) =
%         gbap.angleVec(profileMinMatIndex(k,kk));
%     end
% end

% figure
scrsz = get(groot,'ScreenSize');
figure('Position',...
    [scrsz(3)*0.02 scrsz(4)*0.15 scrsz(3)*0.95 scrsz(4)*0.70]);
% 'Position' definition: [left bottom width height]

subplot(1,2,1)
% imhist(imageCirPol)
% figure;polarscatter(deg2rad(azimuthAngle(:)),90*ones(numel(azimuthAngle),1))
polarscatter(deg2rad(gbap.imageAzimuthAngle(:)),gbap.imageCirPol(:),1)
ax = gca;
ax.RLim = [0 255];
title('CirPol c-axis pixel azimuth and grayscale values')

subplot(1,2,2)
polarhistogram(deg2rad(gbap.imageAzimuthAngle(:)),length(gbap.angleVec)+0)
title('CirPol distribution of c-axis azimuth by pixel')



%% Make video from image stack

if videoFlag == 1
    
%     video = VideoWriter(fullfile([pwd,gbap.subfolder],['CoolVideo_',num2str(gbap.cropscale,'%0.2f'),'.avi'])); %create the video object
%     video = VideoWriter(fullfile([pwd,gbap.subfolder],['CoolVideo_',num2str(gbap.cropscale,'%0.2f')]),'MPEG-4'); %create the video object
    video = VideoWriter(fullfile([pwd,gbap.subfolder],'results',['CoolVideo_',strrep(directoryNameList{fileInput},'.mat','')]),'MPEG-4'); %create the video object
    
    
    video.FrameRate = 2; %frame rate [frames/second]
    video.Quality = 90;
    open(video); %open the file for writing
    
    mask = boundarymask(gbap.optA(qIndex).L);
    
    for k = 1:size(gbap.imageStack,1)
        
        frame00 = (gbap.imageStack{k,2});
        frame = imoverlay(frame00,mask,'w');
        
        %       I = imread('the ith image.jpg'); %read the next image
        %   writeVideo(video,gbap.imageStack{k,2}); %write the image to
        %   file
        writeVideo(video,frame); %write the image frame to file
        
    end
    
    % for ii=1:N %where N is the number of images
    %   I = imread('the ith image.jpg'); %read the next image
    %   writeVideo(video,I); %write the image to file
    % end
    close(video); %close the file
    
end


%% Extinction composite image

% figure
scrsz = get(groot,'ScreenSize');
figure('Position',...
    [scrsz(3)*0.02 scrsz(4)*0.15 scrsz(3)*0.95 scrsz(4)*0.70]);
%'Position' definition: [left bottom width height]

tic
tiledlayout(1+(size(gbap.threshfacVec,1)>1),ceil(size(gbap.threshfacVec,1)/2),'TileSpacing','tight','Padding','compact');
for q = 1:size(gbap.threshfacVec,1)
    nexttile
    %     %     baseImage = (gbap.imageStack{1,2});
    %     imageExtinctionComposite = gbap.imageStack{1,2};
    %     for kk = 1:(length(gbap.optA(q).B)-(gbap.optA(q).N + 0))
    %         qq = kk + gbap.optA(q).N + 0;
    %         extinctionIndex = find(gbap.angleVec == gbap.grainExtinction{q,kk},1);
    %         exinctionImage = gbap.imageStack{extinctionIndex,2};
    %         imageExtinctionComposite(repmat(gbap.optA(q).L == qq,[1 1 3])) = ...
    %             exinctionImage(repmat(gbap.optA(q).L == qq,[1 1 3]));
    %     end
    %     %     imshow(imageExtinctionComposite)
    %     mask = boundarymask(gbap.optA(q).L);
    % %     imageExtinctionComposite = imoverlay(imageExtinctionComposite,mask,gbap.CM(q,:));
    %     imageExtinctionComposite = imoverlay(imageExtinctionComposite,mask,'w');
    %     imshow(imageExtinctionComposite)
    imshow(gbap.imageExtinctionComposite{q})
    
    title(['threshold=',num2str(gbap.threshfacVec(q),2)],...
        'Interpreter','Latex','FontName','Times New Roman','FontSize',gbap.fontSize+1)
end
% set(gca,'FontName','Times New Roman') set(gca,'FontSize',gbap.fontSize)
dim = [0.50 0.98 0.00 0.00]; %[x y w h]
str = 'Extinction composite';
annotation('textbox',dim,'String',str,'FitBoxToText','on',...
    'EdgeColor','none','HorizontalAlignment','center',...
    'VerticalAlignment','middle',...
    'FontName','Times New Roman','FontWeight','bold','FontSize',gbap.fontSize+1);
set(gcf,'color','w')
toc

%% Maximum composite image

% figure
scrsz = get(groot,'ScreenSize');
figure('Position',...
    [scrsz(3)*0.02 scrsz(4)*0.15 scrsz(3)*0.95 scrsz(4)*0.70]);
%'Position' definition: [left bottom width height]

tic
tiledlayout(1+(size(gbap.threshfacVec,1)>1),ceil(size(gbap.threshfacVec,1)/2),'TileSpacing','tight','Padding','compact');
for q = 1:size(gbap.threshfacVec,1)
    nexttile
    %     %     baseImage = (gbap.imageStack{1,2});
    %     imageMaximumComposite = gbap.imageStack{1,2};
    %     for kk = 1:(length(gbap.optA(q).B)-(gbap.optA(q).N + 0))
    %         qq = kk + gbap.optA(q).N + 0;
    %         maximumIndex = find(gbap.angleVec == gbap.grainMaximum{q,kk},1);
    %         maximumImage = gbap.imageStack{maximumIndex,2};
    %         imageMaximumComposite(repmat(gbap.optA(q).L == qq,[1 1 3])) = ...
    %             maximumImage(repmat(gbap.optA(q).L == qq,[1 1 3]));
    %     end
    %     %     imshow(imageMaximumComposite)
    %
    %     mask = boundarymask(gbap.optA(q).L);
    % %     imageMaximumComposite = imoverlay(imageMaximumComposite,mask,gbap.CM(q,:));
    %     imageMaximumComposite = imoverlay(imageMaximumComposite,mask,'w');
    %     imshow(imageMaximumComposite)
    imshow(gbap.imageMaximumComposite{q})
    
    title(['threshold=',num2str(gbap.threshfacVec(q),2)],...
        'Interpreter','Latex','FontName','Times New Roman','FontSize',gbap.fontSize+1)
end
% set(gca,'FontName','Times New Roman') set(gca,'FontSize',gbap.fontSize)
dim = [0.50 0.98 0.00 0.00]; %[x y w h]
str = 'Brightest composite';
annotation('textbox',dim,'String',str,'FitBoxToText','on',...
    'EdgeColor','none','HorizontalAlignment','center',...
    'VerticalAlignment','middle',...
    'FontName','Times New Roman','FontWeight','bold','FontSize',gbap.fontSize+1);
set(gcf,'color','w')
toc


%% Maximum composite image with mode values

% figure
scrsz = get(groot,'ScreenSize');
figure('Position',...
    [scrsz(3)*0.02 scrsz(4)*0.15 scrsz(3)*0.95 scrsz(4)*0.70]);
%'Position' definition: [left bottom width height]

tic
tiledlayout(1+(size(gbap.threshfacVec,1)>1),ceil(size(gbap.threshfacVec,1)/2),'TileSpacing','tight','Padding','compact');
for q = 1:size(gbap.threshfacVec,1)
    nexttile
    %     %     baseImage = (gbap.imageStack{1,2});
    %     imageMaximumModeComposite = gbap.imageStack{1,2};
    %     for kk = 1:(length(gbap.optA(q).B)-(gbap.optA(q).N + 0))
    %         qq = kk + gbap.optA(q).N + 0;
    %
    %         maximumIndex = find(gbap.angleVec == gbap.grainMaximum{q,kk},1);
    %         maximumImage = gbap.imageStack{maximumIndex,2};
    %         maximumImage = imoverlay(maximumImage,gbap.optA(q).L == qq,((squeeze(im2double(gbap.grainModeArray{q}(maximumIndex,kk,:))))./1'));
    %
    %         imageMaximumModeComposite(repmat(gbap.optA(q).L == qq,[1 1 3])) = maximumImage(repmat(gbap.optA(q).L == qq,[1 1 3]));
    %     end
    %
    %     mask = boundarymask(gbap.optA(q).L);
    % %     imageMaximumModeComposite = imoverlay(imageMaximumModeComposite,mask,gbap.CM(q,:));
    %     imageMaximumModeComposite = imoverlay(imageMaximumModeComposite,mask,'w');
    %     imshow(imageMaximumModeComposite)
    imshow(gbap.imageMaximumModeComposite{q})
    
    title(['threshold=',num2str(gbap.threshfacVec(q),2)],...
        'Interpreter','Latex','FontName','Times New Roman','FontSize',gbap.fontSize+1)
end
% set(gca,'FontName','Times New Roman') set(gca,'FontSize',gbap.fontSize)
dim = [0.50 0.98 0.00 0.00]; %[x y w h]
str = 'Mode composite';
annotation('textbox',dim,'String',str,'FitBoxToText','on',...
    'EdgeColor','none','HorizontalAlignment','center',...
    'VerticalAlignment','middle',...
    'FontName','Times New Roman','FontWeight','bold','FontSize',gbap.fontSize+1);
set(gcf,'color','w')
toc



%% Maximum composite image with mean values

% figure
scrsz = get(groot,'ScreenSize');
figure('Position',...
    [scrsz(3)*0.02 scrsz(4)*0.15 scrsz(3)*0.95 scrsz(4)*0.70]);
%'Position' definition: [left bottom width height]

tic
tiledlayout(1+(size(gbap.threshfacVec,1)>1),ceil(size(gbap.threshfacVec,1)/2),'TileSpacing','tight','Padding','compact');
for q = 1:size(gbap.threshfacVec,1)
    nexttile
    %     %     baseImage = (gbap.imageStack{1,2});
    %     imageMaximumMeanComposite = gbap.imageStack{1,2};
    %     for kk = 1:(length(gbap.optA(q).B)-(gbap.optA(q).N + 0))
    %         qq = kk + gbap.optA(q).N + 0;
    %         maximumIndex = find(gbap.angleVec == gbap.grainMaximum{q,kk},1);
    %         maximumImage = gbap.imageStack{maximumIndex,2};
    %         % maximumImage = imoverlay(maximumImage,gbap.optA(q).L ==
    %         % qq,rescale(squeeze(im2double(gbap.grainMeanArray{q}(maximumIndex,kk,:))),0,1)');
    %         maximumImage = imoverlay(maximumImage,gbap.optA(q).L == qq,((squeeze(im2double(gbap.grainMeanArray{q}(maximumIndex,kk,:))))./255)');
    %         imageMaximumMeanComposite(repmat(gbap.optA(q).L == qq,[1 1 3])) = maximumImage(repmat(gbap.optA(q).L == qq,[1 1 3]));
    %     end
    %     mask = boundarymask(gbap.optA(q).L);
    % %     imageMaximumMeanComposite = imoverlay(imageMaximumMeanComposite,mask,gbap.CM(q,:));
    %     imageMaximumMeanComposite = imoverlay(imageMaximumMeanComposite,mask,'w');
    %     imshow(imageMaximumMeanComposite)
    imshow(gbap.imageMaximumMeanComposite{q})
    
    title(['threshold=',num2str(gbap.threshfacVec(q),2)],...
        'Interpreter','Latex','FontName','Times New Roman','FontSize',gbap.fontSize+1)
end
% set(gca,'FontName','Times New Roman') set(gca,'FontSize',gbap.fontSize)
dim = [0.50 0.98 0.00 0.00]; %[x y w h]
str = 'Mean composite';
annotation('textbox',dim,'String',str,'FitBoxToText','on',...
    'EdgeColor','none','HorizontalAlignment','center',...
    'VerticalAlignment','middle',...
    'FontName','Times New Roman','FontWeight','bold','FontSize',gbap.fontSize+1);
set(gcf,'color','w')
toc


%% Maximum composite image with median values

% figure
scrsz = get(groot,'ScreenSize');
figure('Position',...
    [scrsz(3)*0.02 scrsz(4)*0.15 scrsz(3)*0.95 scrsz(4)*0.70]);
%'Position' definition: [left bottom width height]

tic
tiledlayout(1+(size(gbap.threshfacVec,1)>1),ceil(size(gbap.threshfacVec,1)/2),'TileSpacing','tight','Padding','compact');
for q = 1:size(gbap.threshfacVec,1)
    nexttile
    %     %     baseImage = (gbap.imageStack{1,2});
    %     imageMaximumMedianComposite = gbap.imageStack{1,2};
    %     for kk = 1:(length(gbap.optA(q).B)-(gbap.optA(q).N + 0))
    %         qq = kk + gbap.optA(q).N + 0;
    %         maximumIndex = find(gbap.angleVec == gbap.grainMaximum{q,kk},1);
    %         maximumImage = gbap.imageStack{maximumIndex,2};
    %         % maximumImage = imoverlay(maximumImage,gbap.optA(q).L ==
    %         % qq,rescale(squeeze(im2double(gbap.grainMeanArray{q}(maximumIndex,kk,:))),0,1)');
    %         maximumImage = imoverlay(maximumImage,gbap.optA(q).L == qq,((squeeze(im2double(gbap.grainMedianArray{q}(maximumIndex,kk,:))))./1)');
    %         imageMaximumMedianComposite(repmat(gbap.optA(q).L == qq,[1 1 3])) = maximumImage(repmat(gbap.optA(q).L == qq,[1 1 3]));
    %     end
    %     mask = boundarymask(gbap.optA(q).L);
    % %     imageMaximumMedianComposite = imoverlay(imageMaximumMedianComposite,mask,gbap.CM(q,:));
    %     imageMaximumMedianComposite = imoverlay(imageMaximumMedianComposite,mask,'w');
    %     imshow(imageMaximumMedianComposite)
    imshow(gbap.imageMaximumMedianComposite{q})
    
    title(['threshold=',num2str(gbap.threshfacVec(q),2)],...
        'Interpreter','Latex','FontName','Times New Roman','FontSize',gbap.fontSize+1)
end
% set(gca,'FontName','Times New Roman') set(gca,'FontSize',gbap.fontSize)
dim = [0.50 0.98 0.00 0.00]; %[x y w h]
str = 'Median composite';
annotation('textbox',dim,'String',str,'FitBoxToText','on',...
    'EdgeColor','none','HorizontalAlignment','center',...
    'VerticalAlignment','middle',...
    'FontName','Times New Roman','FontWeight','bold','FontSize',gbap.fontSize+1);
set(gcf,'color','w')
toc


%% Schmidt plot (extinction)

% figure
scrsz = get(groot,'ScreenSize');
figure('Position',...
    [scrsz(3)*0.02 scrsz(4)*0.15 scrsz(3)*0.95 scrsz(4)*0.70]);
% 'Position' definition: [left bottom width height]

tiledlayout(1+(size(gbap.threshfacVec,1)>1),ceil(size(gbap.threshfacVec,1)/2),'TileSpacing','loose','Padding','loose');
% tiledlayout(1+(size(gbap.threshfacVec,1)>1),ceil(size(gbap.threshfacVec,1)/2));
for q = 1:size(gbap.threshfacVec,1)
    nexttile
    
%     ps = polarscatter(deg2rad([gbap.grainExtinction{q,:}]),[gbap.grainTheta{q,:}],[],[gbap.grainExtinctionModeColormap{q,:}]','filled');
if gbap.statOption == 1
    ps = polarscatter(deg2rad([gbap.grainExtinction{q,:}]),[gbap.grainTheta{q}(:)],[],[gbap.grainExtinctionModeColormap{q,:}]','filled');
elseif gbap.statOption == 2
    ps = polarscatter(deg2rad([gbap.grainExtinction{q,:}]),[gbap.grainTheta{q}(:)],[],[gbap.grainExtinctionMeanColormap{q,:}]','filled');
elseif gbap.statOption == 3
    ps = polarscatter(deg2rad([gbap.grainExtinction{q,:}]),[gbap.grainTheta{q}(:)],[],[gbap.grainExtinctionMedianColormap{q,:}]','filled');
end

%         polarscatter(deg2rad([gbap.grainMaximum{q,:}]),[gbap.grainTheta{q,:}],[],[gbap.grainMaximumModeColormap{q,:}]','filled')
    ps.SizeData = 10; %marker size (default=36)
    
    title(['threshold=',num2str(gbap.threshfacVec(q),2)],...
        'Interpreter','Latex','FontName','Times New Roman','FontSize',gbap.fontSize+1)
    %     hold off
    ax = gca;
    ax.RLim = [0 90];
    set(gca,'FontName','Times New Roman')
    set(gca,'FontSize',gbap.fontSize)
    
end
dim = [0.50 0.98 0.00 0.00]; %[x y w h]
str = 'Extinction c-axes';
annotation('textbox',dim,'String',str,'FitBoxToText','on',...
    'EdgeColor','none','HorizontalAlignment','center',...
    'VerticalAlignment','middle',...
    'FontName','Times New Roman','FontWeight','bold','FontSize',gbap.fontSize+1);
set(gcf,'color','w')


%% Schmidt plot (brightest)

% figure
scrsz = get(groot,'ScreenSize');
figure('Position',...
    [scrsz(3)*0.02 scrsz(4)*0.15 scrsz(3)*0.95 scrsz(4)*0.70]);
% 'Position' definition: [left bottom width height]

% tiledlayout(1+(size(gbap.threshfacVec,1)>1),ceil(size(gbap.threshfacVec,1)/2),'TileSpacing','loose','Padding','loose');
tiledlayout(1+(size(gbap.threshfacVec,1)>1),ceil(size(gbap.threshfacVec,1)/2));
for q = 1:size(gbap.threshfacVec,1)
    nexttile
    
    %             polarscatter(deg2rad([gbap.grainExtinction{q,:}]),[gbap.grainTheta{q,:}],[],[gbap.grainExtinctionModeColormap{q,:}]','filled')
%     ps = polarscatter(deg2rad([gbap.grainMaximum{q,:}]),[gbap.grainTheta{q,:}],[],[gbap.grainMaximumModeColormap{q,:}]','filled');
if gbap.statOption == 1
    ps = polarscatter(deg2rad([gbap.grainMaximum{q,:}]),[gbap.grainTheta{q}(:)],[],[gbap.grainMaximumModeColormap{q,:}]','filled');
elseif gbap.statOption == 2
    ps = polarscatter(deg2rad([gbap.grainMaximum{q,:}]),[gbap.grainTheta{q}(:)],[],[gbap.grainMaximumMeanColormap{q,:}]','filled');
elseif gbap.statOption == 3
    ps = polarscatter(deg2rad([gbap.grainMaximum{q,:}]),[gbap.grainTheta{q}(:)],[],[gbap.grainMaximumMedianColormap{q,:}]','filled');
end
ps.SizeData = 10; %marker size (default=36)
    
    title(['threshold=',num2str(gbap.threshfacVec(q),2)],...
        'Interpreter','Latex','FontName','Times New Roman','FontSize',gbap.fontSize+1)
    %     hold off
    ax = gca;
    ax.RLim = [0 90];
    set(gca,'FontName','Times New Roman')
    set(gca,'FontSize',gbap.fontSize)
end
dim = [0.50 0.98 0.00 0.00]; %[x y w h]
str = 'Brightest c-axes';
annotation('textbox',dim,'String',str,'FitBoxToText','on',...
    'EdgeColor','none','HorizontalAlignment','center',...
    'VerticalAlignment','middle',...
    'FontName','Times New Roman','FontWeight','bold','FontSize',gbap.fontSize+1);
set(gcf,'color','w')


%%

% figure
scrsz = get(groot,'ScreenSize');
figure('Position',...
    [scrsz(3)*0.20 scrsz(4)*0.15 scrsz(3)*0.50 scrsz(4)*0.70]);
% 'Position' definition: [left bottom width height]

% tiledlayout(1+(size(gbap.threshfacVec,1)>1),ceil(size(gbap.threshfacVec,1)/2),'TileSpacing','tight','Padding','compact');
tLayout = tiledlayout(2,2,'TileSpacing','tight','Padding','compact');
nexttile
% imshow(gbap.imageExtinctionComposite{qIndex})
if gbap.statOption == 1
    imshow(gbap.imageExtinctionComposite{qIndex})
elseif gbap.statOption == 2
    imshow(gbap.imageExtinctionComposite{qIndex})
elseif gbap.statOption == 3
    imshow(gbap.imageExtinctionComposite{qIndex})
end
title('Extinction',...
    'Interpreter','Latex','FontName','Times New Roman','FontSize',gbap.fontSize+1)
nexttile
%     ps = polarscatter(deg2rad([gbap.grainExtinction{qIndex,:}]),[gbap.grainTheta{qIndex,:}],[],[gbap.grainExtinctionModeColormap{qIndex,:}]','filled');
if gbap.statOption == 1
    ps = polarscatter(deg2rad([gbap.grainExtinction{qIndex,:}]),[gbap.grainTheta{qIndex}(:)],[],[gbap.grainExtinctionModeColormap{qIndex,:}]','filled');
elseif gbap.statOption == 2
    ps = polarscatter(deg2rad([gbap.grainExtinction{qIndex,:}]),[gbap.grainTheta{qIndex}(:)],[],[gbap.grainExtinctionMeanColormap{qIndex,:}]','filled');
elseif gbap.statOption == 3
    ps = polarscatter(deg2rad([gbap.grainExtinction{qIndex,:}]),[gbap.grainTheta{qIndex}(:)],[],[gbap.grainExtinctionMedianColormap{qIndex,:}]','filled');
end
ps.SizeData = 10; %marker size (default=36)

% title(['threshold=',num2str(gbap.threshfacVec(q),2)],...
%     'Interpreter','Latex','FontName','Times New Roman','FontSize',gbap.fontSize+1)
%     hold off
ax = gca;
ax.RLim = [0 90];
set(gca,'FontName','Times New Roman')
set(gca,'FontSize',gbap.fontSize)
% set(gcf,'color','w')

nexttile
% imshow(gbap.imageMaximumModeComposite{qIndex})
if gbap.statOption == 1
    imshow(gbap.imageMaximumModeComposite{qIndex})
elseif gbap.statOption == 2
    imshow(gbap.imageMaximumMeanComposite{qIndex})
elseif gbap.statOption == 3
    imshow(gbap.imageMaximumMedianComposite{qIndex})
end
title('Median brightest',...
    'Interpreter','Latex','FontName','Times New Roman','FontSize',gbap.fontSize+1)
nexttile

%     ps = polarscatter(deg2rad([gbap.grainMaximum{qIndex,:}]),[gbap.grainTheta{qIndex,:}],[],[gbap.grainMaximumModeColormap{qIndex,:}]','filled');
if gbap.statOption == 1
    ps = polarscatter(deg2rad([gbap.grainMaximum{qIndex,:}]),[gbap.grainTheta{qIndex}(:)],[],[gbap.grainMaximumModeColormap{qIndex,:}]','filled');
elseif gbap.statOption == 2
    ps = polarscatter(deg2rad([gbap.grainMaximum{qIndex,:}]),[gbap.grainTheta{qIndex}(:)],[],[gbap.grainMaximumMeanColormap{qIndex,:}]','filled');
elseif gbap.statOption == 3
    ps = polarscatter(deg2rad([gbap.grainMaximum{qIndex,:}]),[gbap.grainTheta{qIndex}(:)],[],[gbap.grainMaximumMedianColormap{qIndex,:}]','filled');
end
ps.SizeData = 10; %marker size (default=36)

% title(['threshold=',num2str(gbap.threshfacVec(q),2)],...
%     'Interpreter','Latex','FontName','Times New Roman','FontSize',gbap.fontSize+1)
%     hold off
ax = gca;
ax.RLim = [0 90];
set(gca,'FontName','Times New Roman')
set(gca,'FontSize',gbap.fontSize)
set(gcf,'color','w')


%% Summary figure

if gbap.stats(qIndex).meanEquivDia < 5 %less than 5mm grain diameter
    %set variables for 1cm scale line for images
    scaleLengthReal = 10; scaleString = '$\textbf{1 cm}$'; %mm real scale length of line
    scaleWidthReal = 1.05; %mm real scale width of line
else
    %set variables for 2cm scale line for images
    scaleLengthReal = 20; scaleString = '$\textbf{2 cm}$'; %mm real scale length of line
    scaleWidthReal = 1.25; %mm real scale width of line
end
scaleLength = floor(scaleLengthReal/gbap.mmPerPixel);
scaleWidth = floor(scaleWidthReal/gbap.mmPerPixel);

if size(gbap.imageStack{1,2},1) >= 1e3
    scaleBoxHeight = floor(10*scaleWidthReal/gbap.mmPerPixel); %number of pixels
else
    scaleBoxHeight = floor(4*scaleWidthReal/gbap.mmPerPixel); %number of pixels
end

%add pixels to bottom of image and add 1cm scale line
imageScaledExtinct = padarray(gbap.imageExtinctionComposite{qIndex},floor(scaleBoxHeight),000,'post');
% imageScaledExtinct(...
%     size(imageScaledExtinct,1)-floor(scaleBoxHeight/2):size(imageScaledExtinct,1)-floor(scaleBoxHeight/2)+scaleWidth,...
%     floor(size(imageScaledExtinct,2)/2)-floor(scaleLength/2):ceil(size(imageScaledExtinct,2)/2)+ceil(scaleLength/2),...
%     1:size(imageScaledExtinct,3)) = uint8(255);
imageScaledExtinct(...
    size(imageScaledExtinct,1)-floor(scaleBoxHeight/2):size(imageScaledExtinct,1)-floor(scaleBoxHeight/2)+scaleWidth,...
    floor(scaleLength/2):floor(scaleLength/2)+scaleLength,...
    1:size(imageScaledExtinct,3)) = uint8(255);
% imageScaledExtinct = insertText(imageScaledExtinct,...
%     [floor(scaleLength/2)+floor(scaleLength/2) (size(imageScaledExtinct,1)-floor(scaleBoxHeight/2))+1],...
%     '1 cm','BoxColor','black',...
%     'FontSize',64,'TextColor','white','AnchorPoint','CenterBottom');




if gbap.statOption == 1
    imageScaledMaximum = padarray(gbap.imageMaximumModeComposite{qIndex},floor(scaleBoxHeight),000,'post');
elseif gbap.statOption == 2
    imageScaledMaximum = padarray(gbap.imageMaximumMeanComposite{qIndex},floor(scaleBoxHeight),000,'post');
elseif gbap.statOption == 3
    imageScaledMaximum = padarray(gbap.imageMaximumMedianComposite{qIndex},floor(scaleBoxHeight),000,'post');
end
% imageScaledMaximum(...
%     size(imageScaledMaximum,1)-floor(scaleBoxHeight/2):size(imageScaledMaximum,1)-floor(scaleBoxHeight/2)+scaleWidth,...
%     floor(size(imageScaledMaximum,2)/2)-floor(scaleLength/2):ceil(size(imageScaledMaximum,2)/2)+ceil(scaleLength/2),...
%     1:size(imageScaledMaximum,3)) = uint8(255);
imageScaledMaximum(...
    size(imageScaledMaximum,1)-floor(scaleBoxHeight/2):size(imageScaledMaximum,1)-floor(scaleBoxHeight/2)+scaleWidth,...
    floor(scaleLength/2):floor(scaleLength/2)+scaleLength,...
    1:size(imageScaledMaximum,3)) = uint8(255);
% imageScaledMaximum = insertText(imageScaledMaximum,...
%     [floor(scaleLength/2)+floor(scaleLength/2) (size(imageScaledMaximum,1)-floor(scaleBoxHeight/2))+1],...
%     '1 cm','BoxColor','black',...
%     'FontSize',64,'TextColor','white','AnchorPoint','CenterBottom');

% figure
scrsz = get(groot,'ScreenSize');
figure('Position',...
    [scrsz(3)*0.20 scrsz(4)*0.05 scrsz(3)*0.60 scrsz(4)*0.85]);
% 'Position' definition: [left bottom width height]

% tiledlayout(1+(size(gbap.threshfacVec,1)>1),ceil(size(gbap.threshfacVec,1)/2),'TileSpacing','tight','Padding','compact');
tLayout = tiledlayout(2,3,'TileSpacing','tight','Padding','compact');
nexttile
% imshow(gbap.imageExtinctionComposite{qIndex})
imshow(imageScaledExtinct)
text(...
    floor(scaleLength/2)+floor(scaleLength/2),...
    (size(imageScaledExtinct,1)-floor(scaleBoxHeight/2)-25),...
    scaleString,'Color','white','HorizontalAlignment','center',...
    'Interpreter','Latex','FontName','Times New Roman','FontSize',gbap.fontSize+0,'FontWeight','bold')
title('Extinction',...
    'Interpreter','Latex','FontName','Times New Roman','FontSize',gbap.fontSize+1)

nexttile
%     ps = polarscatter(deg2rad([gbap.grainExtinction{qIndex,:}]),[gbap.grainTheta{qIndex,:}],[],[gbap.grainExtinctionModeColormap{qIndex,:}]','filled');
%     ps = polarscatter(deg2rad([gbap.grainExtinction{qIndex,:}]),[gbap.grainTheta{qIndex}(:)],[],[gbap.grainExtinctionModeColormap{qIndex,:}]','filled');
if gbap.statOption == 1
    ps = polarscatter(deg2rad([gbap.grainExtinction{qIndex,:}]),[gbap.grainTheta{qIndex}(:)],[],[gbap.grainExtinctionModeColormap{qIndex,:}]','filled');
elseif gbap.statOption == 2
    ps = polarscatter(deg2rad([gbap.grainExtinction{qIndex,:}]),[gbap.grainTheta{qIndex}(:)],[],[gbap.grainExtinctionMeanColormap{qIndex,:}]','filled');
elseif gbap.statOption == 3
    ps = polarscatter(deg2rad([gbap.grainExtinction{qIndex,:}]),[gbap.grainTheta{qIndex}(:)],[],[gbap.grainExtinctionMedianColormap{qIndex,:}]','filled');
end
ps.SizeData = 10; %marker size (default=36)
minSz = 07;
maxSz = minSz*5;
% SizeDataScaling = (maxSz-minSz)/(max(gbap.stats(qIndex).equivDia)-min(gbap.stats(qIndex).equivDia))*gbap.stats(qIndex).equivDia+minSz;
% ps.SizeData = SizeDataScaling; %marker size (default=36)
ax = gca;
ax.RLim = [0 90];
% Recreate rho ticks
rTickTheta = (92)*pi/180 * ones(size(ax.RTick)); 
th = text(rTickTheta, ax.RTick, ax.RTickLabel, 'Color','k', ...
    'HorizontalAlignment','right','VerticalAlignment','Middle',...
    'Interpreter','Latex','FontName','Times New Roman','FontSize',gbap.fontSize); 
ax.RTickLabel = {};
set(ax,'FontName','Times New Roman')
set(ax,'FontSize',gbap.fontSize)

nexttile
histogram(gbap.stats(qIndex).Area,'FaceColor',gbap.CM(qIndex,:),'FaceAlpha',1)
xlabel('grain area [mm$\textsuperscript{2}$]',...
    'Interpreter','LaTeX','FontName','Times New Roman','FontSize',gbap.fontSize)
ylabel('frequency [-]',...
    'Interpreter','LaTeX','FontName','Times New Roman','FontSize',gbap.fontSize)
    title([num2str(gbap.threshfacVec(qIndex),'%.1f'),': ','$\bar{A}$=',...
        num2str(gbap.stats(qIndex).meanArea,'%.1f'),'mm$\textsuperscript{2}$',...
        ', $A_{f}$=',num2str(gbap.stats(qIndex).areaFraction,'%.2f')],...
        'Interpreter','LaTeX','FontName','Times New Roman','FontSize',gbap.fontSize-1)
        axis tight
    grid on
    set(gca,'FontName','Times New Roman')
    set(gca,'FontSize',gbap.fontSize)
    
nexttile
    imshow(imageScaledMaximum)
    text(...
    floor(scaleLength/2)+floor(scaleLength/2),...
    (size(imageScaledMaximum,1)-floor(scaleBoxHeight/2)-25),...
    scaleString,'Color','white','HorizontalAlignment','center',...
    'Interpreter','Latex','FontName','Times New Roman','FontSize',gbap.fontSize+0,'FontWeight','bold')
text(1,1,[num2str(gbap.thicknessArrayInterpCrop(1,1),'%.2f'),' mm'],'HorizontalAlignment','right',...
    'Interpreter','Latex','FontName','Times New Roman','FontSize',gbap.fontSize+0)
text(size(imageScaledMaximum,2),1,[num2str(gbap.thicknessArrayInterpCrop(1,end),'%.2f'),' mm'],'HorizontalAlignment','left',...
    'Interpreter','Latex','FontName','Times New Roman','FontSize',gbap.fontSize+0)
text(1,size(imageScaledMaximum,2),[num2str(gbap.thicknessArrayInterpCrop(end,1),'%.2f'),' mm'],'HorizontalAlignment','right',...
    'Interpreter','Latex','FontName','Times New Roman','FontSize',gbap.fontSize+0)
text(size(imageScaledMaximum,2),size(imageScaledMaximum,2),[num2str(gbap.thicknessArrayInterpCrop(end,end),'%.2f'),'$\pm$',num2str(gbap.thicknessVecTolerance,'%.2f'),' mm'],'HorizontalAlignment','left',...
    'Interpreter','Latex','FontName','Times New Roman','FontSize',gbap.fontSize+0)
% gbap.thicknessArrayInterpCrop


% imshow(gbap.imageMaximumModeComposite{qIndex})
title('Median brightest',...
    'Interpreter','Latex','FontName','Times New Roman','FontSize',gbap.fontSize+1)
    
nexttile
% fig = nexttile;
% h = polaraxes(fig);
% h = polaraxes();

%     ps = polarscatter(deg2rad([gbap.grainMaximum{qIndex,:}]),[gbap.grainTheta{qIndex,:}],[],[gbap.grainMaximumModeColormap{qIndex,:}]','filled');
%     ps = polarscatter(deg2rad([gbap.grainMaximum{qIndex,:}]),[gbap.grainTheta{qIndex}(:)],[],[gbap.grainMaximumModeColormap{qIndex,:}]','filled');
if gbap.statOption == 1
    ps = polarscatter(deg2rad([gbap.grainMaximum{qIndex,:}]),[gbap.grainTheta{qIndex}(:)],[],[gbap.grainMaximumModeColormap{qIndex,:}]','filled');
elseif gbap.statOption == 2
    ps = polarscatter(deg2rad([gbap.grainMaximum{qIndex,:}]),[gbap.grainTheta{qIndex}(:)],[],[gbap.grainMaximumMeanColormap{qIndex,:}]','filled');
elseif gbap.statOption == 3
    ps = polarscatter(deg2rad([gbap.grainMaximum{qIndex,:}]),[gbap.grainTheta{qIndex}(:)],[],[gbap.grainMaximumMedianColormap{qIndex,:}]','filled');
end
    
    
    % ps.SizeData = 10; %marker size (default=36)

% minSz = 07;
% maxSz = minSz*5;
SizeDataScaling = (maxSz-minSz)/(max(gbap.stats(qIndex).equivDia)-min(gbap.stats(qIndex).equivDia))*gbap.stats(qIndex).equivDia+minSz;
% ps.SizeData = 10*ones(size([gbap.grainMaximum{qIndex,:}])); %marker size (default=36)
ps.SizeData = SizeDataScaling; %marker size (default=36)


% title(['threshold=',num2str(gbap.threshfacVec(q),2)],...
%     'Interpreter','Latex','FontName','Times New Roman','FontSize',gbap.fontSize+1)
%     hold off
% ax = gca;

ax = gca;
ax.RLim = [0 90];
% Recreate rho ticks
rTickTheta = (92)*pi/180 * ones(size(ax.RTick)); 
th = text(rTickTheta, ax.RTick, ax.RTickLabel, 'Color','k', ...
    'HorizontalAlignment','right','VerticalAlignment','Middle',...
    'Interpreter','Latex','FontName','Times New Roman','FontSize',gbap.fontSize); 
ax.RTickLabel = {};
set(ax,'FontName','Times New Roman')
set(ax,'FontSize',gbap.fontSize)
set(gcf,'color','w')

nexttile
histogram(gbap.stats(qIndex).equivDia,'FaceColor',gbap.CM(qIndex,:),'FaceAlpha',1)
xlabel('equiv. grain dia. [mm]',...
    'Interpreter','LaTeX','FontName','Times New Roman','FontSize',gbap.fontSize)
ylabel('frequency [-]',...
    'Interpreter','LaTeX','FontName','Times New Roman','FontSize',gbap.fontSize)
title([num2str(gbap.threshfacVec(qIndex),'%.1f'),': ','$\bar{d}$=',...
    num2str(gbap.stats(qIndex).meanEquivDia,'%.1f'),'mm',...
    ', N=',num2str(gbap.stats(qIndex).numGrains)],...
    'Interpreter','LaTeX','FontName','Times New Roman','FontSize',gbap.fontSize-1)
    axis tight
    grid on
    set(gca,'FontName','Times New Roman')
    set(gca,'FontSize',gbap.fontSize)
    
    
    
    
    if savePlotFlag == 1
%         savefig(gcf,fullfile([pwd,'/Figures/',str,'.fig']),'compact')
        savefig(gcf,fullfile(gbap.fileNamePrefix,'results',[strrep(directoryNameList{fileInput},'.mat',''),'_summary','.fig']),'compact')
        exportgraphics(gcf,fullfile(gbap.fileNamePrefix,'results',[strrep(directoryNameList{fileInput},'.mat',''),'_summary','.png']))
    end


%% Check cropped interpolated thickness field

% %crop ice thickness array by the image cropping
% gbap.thicknessArrayInterpCrop = gbap.thicknessArrayInterp(...
%     gbap.winCrop.XLimits(1):gbap.winCrop.XLimits(end),...
%     gbap.winCrop.YLimits(1):gbap.winCrop.YLimits(end));
% 
% gbap.thicknessColumnsCrop = gbap.thicknessColumns(...
%     gbap.winCrop.XLimits(1):gbap.winCrop.XLimits(end),...
%     gbap.winCrop.YLimits(1):gbap.winCrop.YLimits(end));
% 
% gbap.thicknessRowsCrop = gbap.thicknessRows(...
%     gbap.winCrop.XLimits(1):gbap.winCrop.XLimits(end),...
%     gbap.winCrop.YLimits(1):gbap.winCrop.YLimits(end));

figure
surf(gbap.thicknessArrayInterpCrop,'EdgeColor','none','FaceColor','interp');
% surf(gbap.thicknessColumnsCrop,gbap.thicknessRowsCrop,gbap.thicknessArrayInterpCrop,'EdgeColor','none','FaceColor','interp');
title('Thickness field of cropped image');
zlim([0 max(gbap.thicknessArrayInterpCrop(:))])
xlabel('column')
ylabel('row')
set(gca, 'YDir','reverse')
colorbar
view(-38,20)


%% Summary figure with every 30 grains

% plotOpt = 1;
% plotOpt = 0;

% if plotOpt == 1

if (length(gbap.optA(qIndex).B)-(gbap.optA(qIndex).N + 0)) <= 130
    
    % grainMaxNum = 15;
    grainMaxNum = 20;
    
    grainCount = 0;
    for k=1:ceil( (length(gbap.optA(qIndex).B)-(gbap.optA(qIndex).N + 0)) / grainMaxNum )
        
        kkLowerLim = 1+(grainMaxNum*grainCount);
        if k == ceil( (length(gbap.optA(qIndex).B)-(gbap.optA(qIndex).N + 0)) / grainMaxNum )
            kkUpperLim = (length(gbap.optA(qIndex).B)-(gbap.optA(qIndex).N + 0));
        else
            kkUpperLim = (grainMaxNum+grainMaxNum*grainCount);
        end
        
        
        % figure
        scrsz = get(groot,'ScreenSize');
        figure('Position',...
            [scrsz(3)*0.20 scrsz(4)*0.15 scrsz(3)*0.50 scrsz(4)*0.70]);
        % 'Position' definition: [left bottom width height]
        
        % tiledlayout(1+(size(gbap.threshfacVec,1)>1),ceil(size(gbap.threshfacVec,1)/2),'TileSpacing','tight','Padding','compact');
        tLayout = tiledlayout(2,2,'TileSpacing','tight','Padding','compact');
        nexttile
        imshow(gbap.imageExtinctionComposite{qIndex})
        title('Extinction',...
            'Interpreter','Latex','FontName','Times New Roman','FontSize',gbap.fontSize+1)
        
        % for kk = 1:(length(gbap.optA(qIndex).B)-(gbap.optA(qIndex).N + 0))
        for kk = kkLowerLim:kkUpperLim
            hold on
            % stats(q).regionProps.Centroid
            qq = kk + gbap.optA(qIndex).N;
            text(gbap.stats(qIndex).regionProps.Centroid(qq,1),gbap.stats(qIndex).regionProps.Centroid(qq,2),[num2str(kk)],'Color','w','HorizontalAlignment','center','FontSize',gbap.fontSize)
        end
        hold off
        
        
        nexttile
        %     ps = polarscatter(deg2rad([gbap.grainExtinction{qIndex,:}]),[gbap.grainTheta{qIndex,:}],[],[gbap.grainExtinctionModeColormap{qIndex,:}]','filled');
        if gbap.statOption == 1
            ps = polarscatter(deg2rad([gbap.grainExtinction{qIndex,:}]),[gbap.grainTheta{qIndex}(:)],[],[gbap.grainExtinctionModeColormap{qIndex,:}]','filled');
        elseif gbap.statOption == 2
            ps = polarscatter(deg2rad([gbap.grainExtinction{qIndex,:}]),[gbap.grainTheta{qIndex}(:)],[],[gbap.grainExtinctionMeanColormap{qIndex,:}]','filled');
        elseif gbap.statOption == 3
            ps = polarscatter(deg2rad([gbap.grainExtinction{qIndex,:}]),[gbap.grainTheta{qIndex}(:)],[],[gbap.grainExtinctionMedianColormap{qIndex,:}]','filled');
        end
        ps.SizeData = 10; %marker size (default=36)
        % text(deg2rad([gbap.grainExtinction{qIndex,:}]),[gbap.grainTheta{qIndex}(:)]-5, num2str( (1:(length(gbap.optA(qIndex).B)-(gbap.optA(qIndex).N + 0)))' ),'HorizontalAlignment','center')
        text(deg2rad([gbap.grainExtinction{qIndex,kkLowerLim:kkUpperLim}]),[gbap.grainTheta{qIndex}(kkLowerLim:kkUpperLim)]-5, num2str( (kkLowerLim:kkUpperLim)' ),'HorizontalAlignment','center','FontSize',gbap.fontSize)
        
        
        % title(['threshold=',num2str(gbap.threshfacVec(q),2)],...
        %     'Interpreter','Latex','FontName','Times New Roman','FontSize',gbap.fontSize+1)
        %     hold off
        ax = gca;
        ax.RLim = [0 90];
        set(gca,'FontName','Times New Roman')
        set(gca,'FontSize',gbap.fontSize)
        % set(gcf,'color','w')
        
        nexttile
        if gbap.statOption == 1
            imshow(gbap.imageMaximumModeComposite{qIndex})
        elseif gbap.statOption == 2
            imshow(gbap.imageMaximumMeanComposite{qIndex})
        elseif gbap.statOption == 3
            imshow(gbap.imageMaximumMedianComposite{qIndex})
        end
        title('Median brightest',...
            'Interpreter','Latex','FontName','Times New Roman','FontSize',gbap.fontSize+1)
        for kk = kkLowerLim:kkUpperLim
            hold on
            % stats(q).regionProps.Centroid
            qq = kk + gbap.optA(qIndex).N;
            text(gbap.stats(qIndex).regionProps.Centroid(qq,1),gbap.stats(qIndex).regionProps.Centroid(qq,2),[num2str(kk)],'Color','k','HorizontalAlignment','center','FontSize',gbap.fontSize)
        end
        hold off
        nexttile
        
        %     ps = polarscatter(deg2rad([gbap.grainMaximum{qIndex,:}]),[gbap.grainTheta{qIndex,:}],[],[gbap.grainMaximumModeColormap{qIndex,:}]','filled');
        if gbap.statOption == 1
            ps = polarscatter(deg2rad([gbap.grainMaximum{qIndex,:}]),[gbap.grainTheta{qIndex}(:)],[],[gbap.grainMaximumModeColormap{qIndex,:}]','filled');
        elseif gbap.statOption == 2
            ps = polarscatter(deg2rad([gbap.grainMaximum{qIndex,:}]),[gbap.grainTheta{qIndex}(:)],[],[gbap.grainMaximumMeanColormap{qIndex,:}]','filled');
        elseif gbap.statOption == 3
            ps = polarscatter(deg2rad([gbap.grainMaximum{qIndex,:}]),[gbap.grainTheta{qIndex}(:)],[],[gbap.grainMaximumMedianColormap{qIndex,:}]','filled');
        end
        ps.SizeData = 10; %marker size (default=36)
        text(deg2rad([gbap.grainMaximum{qIndex,kkLowerLim:kkUpperLim}]),[gbap.grainTheta{qIndex}(kkLowerLim:kkUpperLim)]-5, num2str( (kkLowerLim:kkUpperLim)' ),'HorizontalAlignment','center','FontSize',gbap.fontSize)
        
        % title(['threshold=',num2str(gbap.threshfacVec(q),2)],...
        %     'Interpreter','Latex','FontName','Times New Roman','FontSize',gbap.fontSize+1)
        %     hold off
        ax = gca;
        ax.RLim = [0 90];
        set(gca,'FontName','Times New Roman')
        set(gca,'FontSize',gbap.fontSize)
        set(gcf,'color','w')
        
        grainCount = grainCount + 1;
        
    end
    [ [1:(length(gbap.optA(qIndex).B)-(gbap.optA(qIndex).N + 0))]' [gbap.grainTheta{qIndex}(:)] [gbap.grainExtinction{qIndex,:}]' ]
end



%%

% figure
% bar3(gbap.thicknessArrayInterpCrop);

% ['CAT = ',num2str(gbap.colorTempChrom), ' K']


%%

% lowInclinationIndex = [gbap.grainTheta{qIndex}(:)] < 40;
% lowInclinationColor = [gbap.grainMaximumModeColormap{qIndex,lowInclinationIndex}]'*255;
% [gbap.grainTheta{qIndex}(lowInclinationIndex)' lowInclinationColor ]


%%

% angleDiff = ([gbap.grainExtinction{qIndex,:}] - [gbap.grainMaximum{qIndex,:}])';
% figure;plot(angleDiff(:),'o')


%%

figure
subplot(2,1,1)
plot([gbap.grainExtinction{qIndex,:}],'o')
subplot(2,1,2)
plot([gbap.stats(qIndex).Orientation],'o')
% 
figure
plot([gbap.grainExtinction{qIndex,:}]-(gbap.stats(qIndex).Orientation)','o')

