clear all 
close all

%% ***** Finite difference method Cdc42 & Rac1 & Rho & Par & Scribble ***** 
%% Parameters
Dm = 0.1;           % Membrane diffusion coefficient, [micrometer^2/s]
Dc = 10;            % Cytosolic diffusion coefficient, [micrometer^2/s]
n = 4;              % Hill coefficient of inhibition responses, [-]

%% Parameters Cdc42
Ic = 0.5;           % Cdc42 activation input rate, [microM/s]
delta_C = 1;        % Decay rate of activated Cdc42, [1/s]
Ctot = 4.4;         % Total Cdc42 concentration, [microM]
Dca = Dm;           % Membrane diffusion coefficient of Cdc42, [micrometer^2/s]
Dci = Dc;           % Cytosolic diffusion coefficient of Cdc42, [micrometer^2/s]
alpha_C = 2.5;      % Cdc42 dependent activation rate, [1/s] 

%% Parameters Rac
Ir = 0.5;           % Rac activation input rate, [microM/s]
delta_R = 1;        % Decay rate of activated Rac, [1/s]
Rtot = 2.4;         % Total Rac concentration, [microM]
Dra = Dm;           % Membrane diffusion coefficient of Rac, [micrometer^2/s]
Dri = Dc;           % Cytosolic diffusion coefficient of RAC, [micrometer^2/s]
beta_R = 1.5;       % Active Rac dependent inhibition rate, [microM]  

%% Parameters Rho
Irho = 3.3;         % Rho activation rate, [microM/s]
delta_Rho = 1;      % Decay rate of activated Rho, [1/s]
Rhotot = 4;         % Total Rho concentration, [microM]
Drhoa = Dm;         % Membrane diffusion coefficient of Rho, [micrometer^2/s]
Drhoi = Dc;         % Cytosolic diffusion coefficient of Cdc42, [micrometer^2/s]
beta_rho = 0.7;     % Active Rho dependent inhibition rate, [microM]
alpha_rho=2.5;      % Rho dependent activation rate, [1/s]

%% Parameters PAR
k_on_P = 0.5;       % Par association rate or on-rate, [microM/s]
k_off_P = 2;        % Par dissociation rate or off-rate, [1/s]
Ptot = 1.6;         % Effective total of Par complex concentration, [microM]
Dpc = 0.2;          % Membrane diffusion coefficient of Par complex, [micrometer^2/s]
Dpun = 2;           % Cytosolic diffusion coefficient of unbound Par proteins, [micrometer^2/s]
beta_PR = 2;        % Active Par complex dependent inhibition rate, [microM]
beta_PS = 0.2;      % Par-complex dependent inhibition rate, [microM]

%% Parameters Scrib
k_on_S = 4;         % Scribble association rate or on-rate, [microM/s]
k_off_S = 2;        % Scribble dissociation rate or off-rate, [1/s]
Stot = 1.6;         % Effective total of Scribble complex concentration, [microM]
Dsc = 0.2;          % Membrane diffusion coefficient of Scribble complex, [micrometer^2/s]
Dsun = 2;           % Cytosolic diffusion coefficient of unbound Scribble proteins, [micrometer^2/s]
beta_SP = 0.2;      % Scribble-complex dependent inhibition rate, [microM]

%% Parameters Integrin
I_I = 10;           % Integrin activation input rate, [microM/s]
delta_I = 1;        % Decay rate of activated integrin, [1/s]
Itot = 30;          % Effective total of integrins, [microM]
Itot1 = 5;          % Auxiliary variable, [microM]
D_Ii = 10;          % Cytosolic diffusion coefficient of inactive integrins, [micrometer^2/s]
alpha_I = 1.5;      % Integrin-dependent activation rate, [1/s]
l=1;     % If activation by integrin is on left side of the cell (l=1)
r=0;     % If activation by integrin is on right side of the cell (r=1)

%% Common parameters
x_max=10;               % Length, L, [micrometer]
Nx=30;                  % Number of divisions along length, [-]
delta_x=x_max/Nx;       % Length increment
x=0:delta_x:x_max;      % Position, x, [micrometer]

t_max=100;              % Simulation time, t, [s]
delta_t=0.005;          % Time increment, [s]
Nt=t_max/delta_t;       % Number of time steps, [-]
t=0:delta_t:t_max;      % Time, t, [s]

%% Allocation
Ca=zeros(length(x),1); Ci=zeros(length(x),1);
Rhoa=zeros(length(x),1); Rhoi=zeros(length(x),1);
Ra=zeros(length(x),1); Ri=zeros(length(x),1);
Pc=zeros(length(x),1); Pun=zeros(length(x),1);
Sc=zeros(length(x),1); Sun=zeros(length(x),1);
Ia=zeros(length(x),1); Ii=zeros(length(x),1);
A=zeros(12*length(x),12*length(x)); Vector=zeros(12*length(x),length(t));
coupl=zeros(12*length(x),length(t)); RHS=zeros(12*length(x),length(t));

Diff_Pc=zeros(11,1);Diff_Sc=zeros(11,1);
Diff_Rac=zeros(11,1);Diff_Rho=zeros(11,1);Diff_Cdc=zeros(11,1);

%% Definition of matrix A
%% *************** Cdc42 ***************
b_Ca=1/delta_t-delta_C-2*Dca/(delta_x^2);           % Cdc42 active
a_Ca=Dca/(delta_x^2);                             
c_Ca=Dca/(delta_x^2);                           
b_Ci=1/delta_t-2*Dci/(delta_x^2);                   % Cdc42 inactive
a_Ci=Dci/(delta_x^2);                           
c_Ci=Dci/(delta_x^2);                           
% Coupling of Ci to Ca
d_Ci_Ca=delta_C;

%% *************** Rac1 ***************
b_Ra=1/delta_t-delta_R-2*Dra/(delta_x^2);           % Rac active
a_Ra=Dra/(delta_x^2);                             
c_Ra=Dra/(delta_x^2);                             
b_Ri=1/delta_t-2*Dri/(delta_x^2);                   % Rac inactive
a_Ri=Dri/(delta_x^2);                           
c_Ri=Dri/(delta_x^2);                           
% Coupling of Ri to Ra
d_Ri_Ra=delta_R;

%% *************** Rho ***************
b_rhoa=1/delta_t-delta_Rho-2*Drhoa/(delta_x^2);     % Rho active
a_rhoa=Drhoa/(delta_x^2);                           
c_rhoa=Drhoa/(delta_x^2);                           
b_rhoi=1/delta_t-2*Drhoi/(delta_x^2);               % Rho inactive
a_rhoi=Drhoi/(delta_x^2);                         
c_rhoi=Drhoi/(delta_x^2);                         
% Coupling of Rhoi to Rhoa
d_Rhoi_Rhoa=delta_Rho;

%% *************** Par ***************
b_Pc=1/delta_t-k_off_P-2*Dpc/(delta_x^2);           % Par complex
a_Pc=Dpc/(delta_x^2);                             
c_Pc=Dpc/(delta_x^2);                             
b_Pun=1/delta_t-2*Dpun/(delta_x^2);                 % Par unbounded
a_Pun=Dpun/(delta_x^2);                           
c_Pun=Dpun/(delta_x^2);                           
% Coupling of Pun to Pc
d_Pun_Pc = k_off_P;

%% *************** Scribble ***************
b_Sc=1/delta_t-k_off_S-2*Dsc/(delta_x^2);           % Scribble complex
a_Sc=Dsc/(delta_x^2);                             
c_Sc=Dsc/(delta_x^2);                             
b_Sun=1/delta_t-2*Dsun/(delta_x^2);                 % Scribble unbounded
a_Sun=Dsun/(delta_x^2);                           
c_Sun=Dsun/(delta_x^2);                           
% Coupling of Sun to Sc
d_Sun_Sc=k_off_S;

for test = 1:11
    Itot = (test-1)*Itot1;

%% *************** Integrin ***************
% along the cell
b_Ia=0;                                             % I active
a_Ia=0;                            
c_Ia=0;                           
b_Ii=1/delta_t-2*D_Ii/(delta_x^2);                  % I inactive
a_Ii=D_Ii/(delta_x^2);                           
c_Ii=D_Ii/(delta_x^2);                          
% at the membrane
d_I_Ii=I_I/Itot;
b_Ia_lr=1/delta_t-delta_I;
b_Ii_lr=b_Ii-I_I/Itot;
d_Ii_I=delta_I;

%% Creating matrix A
for i=2:length(x)-1
    A(i,i)=b_Ca;                                        % 1) Ca
    A(i,i-1)=a_Ca;
    A(i,i+1)=c_Ca;
    A(length(x)+i,length(x)+i)=b_Ra;                    % 2) Ra
    A(length(x)+i,length(x)+i-1)=a_Ra;
    A(length(x)+i,length(x)+i+1)=c_Ra;
    A(2*length(x)+i,2*length(x)+i)=b_rhoa;              % 3) Rhoa
    A(2*length(x)+i,2*length(x)+i-1)=a_rhoa;
    A(2*length(x)+i,2*length(x)+i+1)=c_rhoa;
    A(3*length(x)+i,3*length(x)+i)=b_Ci;                % 4) Ci
    A(3*length(x)+i,3*length(x)+i-1)=a_Ci;
    A(3*length(x)+i,3*length(x)+i+1)=c_Ci;
    A(4*length(x)+i,4*length(x)+i)=b_Ri;                % 5) Ri
    A(4*length(x)+i,4*length(x)+i-1)=a_Ri;
    A(4*length(x)+i,4*length(x)+i+1)=c_Ri;
    A(5*length(x)+i,5*length(x)+i)=b_rhoi;              % 6) Rhoi
    A(5*length(x)+i,5*length(x)+i-1)=a_rhoi;
    A(5*length(x)+i,5*length(x)+i+1)=c_rhoi;
    A(6*length(x)+i,6*length(x)+i)=b_Pc;                % 7) PAR, Pc
    A(6*length(x)+i,6*length(x)+i-1)=a_Pc;
    A(6*length(x)+i,6*length(x)+i+1)=c_Pc;
    A(7*length(x)+i,7*length(x)+i)=b_Pun;               % 8) Pun
    A(7*length(x)+i,7*length(x)+i-1)=a_Pun;
    A(7*length(x)+i,7*length(x)+i+1)=c_Pun;
    A(8*length(x)+i,8*length(x)+i)=b_Sc;                % 9) Sc
    A(8*length(x)+i,8*length(x)+i-1)=a_Sc;
    A(8*length(x)+i,8*length(x)+i+1)=c_Sc;
    A(9*length(x)+i,9*length(x)+i)=b_Sun;               % 10) Sun
    A(9*length(x)+i,9*length(x)+i-1)=a_Sun;
    A(9*length(x)+i,9*length(x)+i+1)=c_Sun;   
    A(10*length(x)+i,10*length(x)+i)=b_Ia;              % 11) Ia
    A(10*length(x)+i,10*length(x)+i-1)=a_Ia;
    A(10*length(x)+i,10*length(x)+i+1)=c_Ia;
    A(11*length(x)+i,11*length(x)+i)=b_Ii;              % 12) Ii
    A(11*length(x)+i,11*length(x)+i-1)=a_Ii;
    A(11*length(x)+i,11*length(x)+i+1)=c_Ii;
    
    % Coupling members (linear) between variables  
    A(3*length(x)+i,i)=d_Ci_Ca;
    A(4*length(x)+i,length(x)+i)=d_Ri_Ra;
    A(5*length(x)+i,2*length(x)+i)=d_Rhoi_Rhoa;
    A(7*length(x)+i,6*length(x)+i)=d_Pun_Pc;
    A(9*length(x)+i,8*length(x)+i)=d_Sun_Sc;
end

%% Boundary conditions, no flux exchange at the borders, no protein passage through cell membrane
    A(1,1)=a_Ca+b_Ca;                               % Ca 
    A(1,2)=c_Ca;
    A(length(x),length(x))=b_Ca+c_Ca; 
    A(length(x),length(x)-1)=a_Ca;
    A(length(x)+1,length(x)+1)=a_Ra+b_Ra;           % Ra 
    A(length(x)+1,length(x)+2)=c_Ra;                                                                                                                                                                                                                                                                                                                                                                                                                                      
    A(2*length(x),2*length(x))=b_Ra+c_Ra; 
    A(2*length(x),2*length(x)-1)=a_Ra;
    A(2*length(x)+1,2*length(x)+1)=a_rhoa+b_rhoa;   % Rhoa 
    A(2*length(x)+1,2*length(x)+2)=c_rhoa;                                                                                                                                                                                                                                                                                                                                                                                                                                        
    A(3*length(x),3*length(x))=b_rhoa+c_rhoa; 
    A(3*length(x),3*length(x)-1)=a_rhoa;
    A(3*length(x)+1,3*length(x)+1)=a_Ci+b_Ci;       % Ci 
    A(3*length(x)+1,3*length(x)+2)=c_Ci;                                                                                                                                                                                                                                                                                                                                                                                                                                            
    A(4*length(x),4*length(x))=b_Ci+c_Ci; 
    A(4*length(x),4*length(x)-1)=a_Ci;
    A(4*length(x)+1,4*length(x)+1)=a_Ri+b_Ri;       % Ri 
    A(4*length(x)+1,4*length(x)+2)=c_Ri;                                                                                                                                                                                                                                                                                                                                                                                                                                            
    A(5*length(x),5*length(x))=b_Ri+c_Ri; 
    A(5*length(x),5*length(x)-1)=a_Ri;    
    A(5*length(x)+1,5*length(x)+1)=a_rhoi+b_rhoi;   % Rhoi 
    A(5*length(x)+1,5*length(x)+2)=c_rhoi;                                                                                                                                                                                                                                                                                                                                                                                                                                            
    A(6*length(x),6*length(x))=b_rhoi+c_rhoi; 
    A(6*length(x),6*length(x)-1)=a_rhoi;
    A(6*length(x)+1,6*length(x)+1)=a_Pc+b_Pc;       % Pc 
    A(6*length(x)+1,6*length(x)+2)=c_Pc;                                                                                                                                                                                                                                                                                                                                                                                                                                            
    A(7*length(x),7*length(x))=b_Pc+c_Pc; 
    A(7*length(x),7*length(x)-1)=a_Pc; 
    A(7*length(x)+1,7*length(x)+1)=a_Pun+b_Pun;     % Pun 
    A(7*length(x)+1,7*length(x)+2)=c_Pun;                                                                                                                                                                                                                                                                                                                                                                                                                                            
    A(8*length(x),8*length(x))=b_Pun+c_Pun; 
    A(8*length(x),8*length(x)-1)=a_Pun;    
    A(8*length(x)+1,8*length(x)+1)=a_Sc+b_Sc;       % Sc 
    A(8*length(x)+1,8*length(x)+2)=c_Sc;                                                                                                                                                                                                                                                                                                                                                                                                                                            
    A(9*length(x),9*length(x))=b_Sc+c_Sc; 
    A(9*length(x),9*length(x)-1)=a_Sc;     
    A(9*length(x)+1,9*length(x)+1)=a_Sun+b_Sun;     % Sun
    A(9*length(x)+1,9*length(x)+2)=c_Sun;                                                                                                                                                                                                                                                                                                                                                                                                                                            
    A(10*length(x),10*length(x))=b_Sun+c_Sun;
    A(10*length(x),10*length(x)-1)=a_Sun;  
    
    % Activated integrin on left or right side of the cell
    if l==1
        for m=1:5
    % left border - coupling
    A(10*length(x)+m,10*length(x)+m)=b_Ia_lr;           % Ia
    A(10*length(x)+m,10*length(x)+m+1)=c_Ia;
        if m==1
        A(11*length(x)+m,11*length(x)+m)=b_Ii_lr+a_Ii;  % Ii
        A(11*length(x)+m,11*length(x)+m+1)=c_Ii;
        else
        A(11*length(x)+m,11*length(x)+m)=b_Ii_lr;       % Ii
        end
    A(10*length(x)+m,11*length(x)+m)=d_I_Ii;
    A(11*length(x)+m,10*length(x)+m)=d_Ii_I;
        end
    % right border - no coupling
    A(11*length(x),11*length(x))=b_Ia+c_Ia;             % Ia
    A(11*length(x),11*length(x)-1)=a_Ia;
    A(12*length(x),12*length(x))=b_Ii+c_Ii;             % Ii
    A(12*length(x),12*length(x)-1)=a_Ii;
    end
	
    if r==1      
    % left border - no coupling
    A(10*length(x)+1,10*length(x)+1)=b_Ia+a_Ia;         % Ia
    A(10*length(x)+1,10*length(x)+2)=a_Ia;
    A(11*length(x)+1,11*length(x)+1)=b_Ii+a_Ii;         % Ii
    A(11*length(x)+1,11*length(x)+2)=c_Ii;
      for m=1:5
    % right border - coupling      
    A(11*length(x)+1-m,11*length(x)+1-m)=b_Ia_lr;               % Ia
    A(11*length(x)+1-m,11*length(x)-1+1-m)=a_Ia;
        if m==1
        A(12*length(x)+1-m,12*length(x)+1-m)=b_Ii_lr+c_Ii;      % Ii
        A(12*length(x)+1-m,12*length(x)-1+1-m)=a_Ii;
        else
        A(12*length(x)+1-m,12*length(x)+1-m)=b_Ii_lr;           % Ii
        end
    A(11*length(x)+1-m,12*length(x)+1-m)=d_I_Ii;
    A(12*length(x)+1-m,11*length(x)+1-m)=d_Ii_I;
      end
    end
    
    % Linear coupling members on boarders (left and right side)
    A(3*length(x)+1,1)=d_Ci_Ca;
    A(4*length(x),length(x))=d_Ci_Ca;
    A(4*length(x)+1,length(x)+1)=d_Ri_Ra;
    A(5*length(x),2*length(x))=d_Ri_Ra;
    A(5*length(x)+1,2*length(x)+1)=d_Rhoi_Rhoa;
    A(6*length(x),3*length(x))=d_Rhoi_Rhoa;    
    A(7*length(x)+1,6*length(x)+1)=d_Pun_Pc;
    A(8*length(x),7*length(x))=d_Pun_Pc;
    A(9*length(x)+1,8*length(x)+1)=d_Sun_Sc;
    A(10*length(x),9*length(x))=d_Sun_Sc;
    
%% Set initial values
Ca(:,1)=1.75;         % [microM]
Ci(:,1)=2.65;         % [microM]
Ra(:,1)=1;            % [microM]
Ri(:,1)=1.4;          % [microM]
Rhoa(:,1)=0.8;        % [microM]
Rhoi(:,1)=3.2;        % [microM]
Pc(:,1)=0.6;          % [microM]
Pun(:,1)=1;           % [microM]
Sc(:,1)=0.6;          % [microM]
Sun(:,1)=1;           % [microM]

 for i=1:length(x)
     % Activation at left side
     if l==1
        if i<=5 
            Ia(i,1)=(Itot/i^2)*exp(1/i^2-1);
        else
            Ia(i,1)=0;
        end
     end
     % Activation at right side
     if r==1
     if i<=length(x)-5
        Ia(i,1)=0;     
     else
        Ia(i,1)=(Itot/(length(x)+1-i)^2)*exp(1/((length(x)+1-i))^2-1); 
     end
     end
 end
 Ii(:,1)=Itot-Ia(:,1);

Vector(1:length(x),1)=Ca(:,1);
Vector(length(x)+1:2*length(x),1)=Ra(:,1);
Vector(2*length(x)+1:3*length(x),1)=Rhoa(:,1);
Vector(3*length(x)+1:4*length(x),1)=Ci(:,1);
Vector(4*length(x)+1:5*length(x),1)=Ri(:,1);
Vector(5*length(x)+1:6*length(x),1)=Rhoi(:,1);
Vector(6*length(x)+1:7*length(x),1)=Pc(:,1);
Vector(7*length(x)+1:8*length(x),1)=Pun(:,1);
Vector(8*length(x)+1:9*length(x),1)=Sc(:,1);
Vector(9*length(x)+1:10*length(x),1)=Sun(:,1);
Vector(10*length(x)+1:11*length(x),1)=Ia(:,1);
Vector(11*length(x)+1:12*length(x),1)=Ii(:,1);

%% Nonlinear coupling members from equations and solving equations
k1=0;
for j=1:length(t)-1      
        for i=1:length(x)
            coupl(i,j)=(Ic+alpha_rho*Vector(2*length(x)+i,j))*(Vector(3*length(x)+i,j)/Ctot); 
            coupl(length(x)+i,j)=((Ir+alpha_I*Vector(10*length(x)+i,j))/(1+(Vector(2*length(x)+i,j)^n)/(beta_rho^n)+(Vector(6*length(x)+i,j)^n)/(beta_PR^n)))*(Vector(4*length(x)+i,j)/Rtot);
            coupl(2*length(x)+i,j)=(Irho/(1+((Vector(length(x)+i,j)^n)/(beta_R^n))))*(Vector(5*length(x)+i,j)/Rhotot); 
            coupl(3*length(x)+i,j)=-coupl(i,j);
            coupl(4*length(x)+i,j)=-coupl(length(x)+i,j);
            coupl(5*length(x)+i,j)=-coupl(2*length(x)+i,j);
            coupl(6*length(x)+i,j)=((k_on_P+alpha_C*Vector(i,j))/(1+((Vector(8*length(x)+i,j)^n)/(beta_SP^n))))*(Vector(7*length(x)+i,j)/Ptot);
            coupl(7*length(x)+i,j)=-coupl(6*length(x)+i,j);
            coupl(8*length(x)+i,j)=(k_on_S/(1+((Vector(6*length(x)+i,j)^n)/(beta_PS^n))))*(Vector(9*length(x)+i,j)/Stot);
            coupl(9*length(x)+i,j)=-coupl(8*length(x)+i,j);         
        end
        
    RHS(:,j)=A(:,:)*Vector(:,j)+coupl(:,j);
    Vector(:,j+1)=delta_t*RHS(:,j);

    % Check if values go below zero
    for i=1:12*length(x)
        if Vector(i,j+1)<0
            Vector(i,j+1)=0;
            k1=k1+1;
        end           
    end
end

%% For plotting critical value of Ia
Diff_Pc(test)=abs(Vector(7*length(x),length(t))-Vector(6*length(x)+1,length(t)));
Diff_Sc(test)=abs(Vector(9*length(x),length(t))-Vector(8*length(x)+1,length(t)));
Diff_Rac(test)=abs(Vector(2*length(x),length(t))-Vector(length(x)+1,length(t)));
Diff_Rho(test)=abs(Vector(3*length(x),length(t))-Vector(2*length(x)+1,length(t)));
Diff_Cdc(test)=abs(Vector(length(x),length(t))-Vector(1,length(t)));

%% Plotting the results at the end of simulation
y1=Vector(1:length(x),length(t));                   % Ca
y2=Vector(length(x)+1:2*length(x),length(t));       % Ra
y3=Vector(2*length(x)+1:3*length(x),length(t));     % Rhoa
y4=Vector(3*length(x)+1:4*length(x),length(t));     % Ci
y5=Vector(4*length(x)+1:5*length(x),length(t));     % Ri
y6=Vector(5*length(x)+1:6*length(x),length(t));     % Rhoi
y7=Vector(6*length(x)+1:7*length(x),length(t));     % Pc
y8=Vector(7*length(x)+1:8*length(x),length(t));     % Pun
y9=Vector(8*length(x)+1:9*length(x),length(t));     % Sc
y10=Vector(9*length(x)+1:10*length(x),length(t));   % Sun
y11=Vector(10*length(x)+1:11*length(x),length(t));  % Ia
y12=Vector(11*length(x)+1:12*length(x),length(t));  % Ii

% Depict colors for plots
green1=[0.4660,0.6740,0.1880];red1=[0.6350,0.0780,0.1840];purple1=[0.4940,0.1840,0.5560];
yellow1=[0.9290,0.6940,0.1250];blue1=[0,0.4470,0.7410];orange1=[0.8500,0.3250,0.0980];

%% Plot results
if test==7      % outputs for Ia=30
%% Figure 4a additional - Initial Ia
figure(1)
plot(x,Vector(10*length(x)+1:11*length(x),1),'LineWidth',2.0)
title('Initial values of integrin')
xlabel('x [µm]')
ylabel('Concentration [µM]')
ax = gca;
ax.FontSize = 12;
exportgraphics(ax,'Fig_4A_additional.tif','Resolution',300)

%% Figure 4a - Cdc42-Rac1-Rhoa
figure(2)
plot(x,y1,'Color',green1,'LineWidth',2.0);
hold on
plot(x,y2,'Color',purple1,'LineWidth',2.0)
plot(x,y3,'Color',red1,'LineWidth',2.0)
txt = 'Rac';txt1='Cdc42';txt2='Rhoa';
text(1.8,4,txt,'Color', purple1, 'FontWeight','bold', 'FontSize',12)
text(6,3,txt1,'Color', green1, 'FontWeight','bold','FontSize',12)
text(5,1.5,txt2,'Color', red1, 'FontWeight','bold','FontSize',12)
xlabel('x [µm]')
ylabel('Concentration [µM]')
ax = gca;
ax.FontSize = 12;
exportgraphics(ax,'Fig_4A.tif','Resolution',300)

%% Figure 4b - Pc-Sc
figure(3)
plot(x,y7,'Color',blue1,'LineWidth',2.0);
hold on
plot(x,y9,'Color',orange1,'LineWidth',2.0);
txt = 'PAR';txt1='Scribble';
text(6.0,1.25,txt,'Color', blue1, 'FontWeight','bold', 'FontSize',12)
text(7,0.1,txt1,'Color', orange1, 'FontWeight','bold','FontSize',12)
xlabel('x [µm]')
ylabel('Concentration [µM]')
ax = gca;
ax.FontSize = 12;
exportgraphics(ax,'Fig_4B.tif','Resolution',300)

%% Figure 4c additional - Testing equilibrium at basal and apical side
figure(4)
subplot(2,2,1)
plot((1:100:length(t))*delta_t,Vector(length(x)+1,1:100:length(t)),'Color',purple1,'LineWidth',1.0)
hold on
plot((1:100:length(t))*delta_t,Vector(2*length(x)+1,1:100:length(t)),'Color',red1,'LineWidth',1.0)
plot((1:100:length(t))*delta_t,Vector(1,1:100:length(t)),'Color',green1,'LineWidth',1.0)
plot((1:1000:length(t))*delta_t,Vector(10*length(x)+1,1:1000:length(t)),'Color',yellow1,'LineWidth',1.0)
txt = 'Rac1';txt1='Cdc42';txt2='Rhoa';txt3='Ia';
text(22,5.5,txt,'Color', purple1, 'FontWeight','bold', 'FontSize',11)
text(40,2.0,txt1,'Color', green1, 'FontWeight','bold','FontSize',11)
text(70,1.5,txt2,'Color', red1, 'FontWeight','bold','FontSize',11)
text(10,11,txt3,'Color', yellow1, 'FontWeight','bold','FontSize',11)
xlabel('t [s]')
ylabel('Concentration [µM]')
title('Testing equilibrium at basal side')
ax = gca;
ax.FontSize = 10;

subplot(2,2,2)
plot((1:100:length(t))*delta_t,Vector(6*length(x)+1,1:100:length(t)),'Color',blue1,'LineWidth',1.0)
hold on
plot((1:100:length(t))*delta_t,Vector(8*length(x)+1,1:100:length(t)),'Color',orange1,'LineWidth',1.0)
txt = 'PAR';txt1='Scribble';
text(50,0.1,txt,'Color', blue1, 'FontWeight','bold', 'FontSize',11)
text(70,1.3,txt1,'Color', orange1, 'FontWeight','bold','FontSize',11)
xlabel('t [s]')
ylabel('Concentration [µM]')
title('Testing equilibrium at basal side')
ax = gca;
ax.FontSize = 10;

subplot(2,2,3)
plot((1:100:length(t))*delta_t,Vector(2*length(x),1:100:length(t)),'Color',purple1,'LineWidth',1.0)
hold on
plot((1:100:length(t))*delta_t,Vector(3*length(x),1:100:length(t)),'Color',red1,'LineWidth',1.0)
plot((1:100:length(t))*delta_t,Vector(length(x),1:100:length(t)),'Color',green1,'LineWidth',1.0)
plot((1:100:length(t))*delta_t,Vector(11*length(x),1:100:length(t)),'Color',yellow1,'LineWidth',1.0)
txt = 'Rac1';txt1='Cdc42';txt2='Rhoa';txt3='Ia';
text(6,0.5,txt,'Color', purple1, 'FontWeight','bold', 'FontSize',11)
text(70,2.5,txt1,'Color', green1, 'FontWeight','bold','FontSize',11)
text(60,1.8,txt2,'Color', red1, 'FontWeight','bold','FontSize',11)
text(40,0.2,txt3,'Color', yellow1, 'FontWeight','bold','FontSize',11)
xlabel('t [s]')
ylabel('Concentration [µM]')
title('Testing equilibrium at apical side')
ax = gca;
ax.FontSize = 10;

subplot(2,2,4)
plot((1:100:length(t))*delta_t,Vector(7*length(x),1:100:length(t)),'Color',blue1,'LineWidth',1.0)
hold on
plot((1:1000:length(t))*delta_t,Vector(9*length(x),1:1000:length(t)),'Color',orange1,'LineWidth',1.0)
txt = 'PAR';txt1='Scribble';
text(85,1.3,txt,'Color', blue1, 'FontWeight','bold', 'FontSize',11)
text(80,0.1,txt1,'Color', orange1, 'FontWeight','bold','FontSize',11)
xlabel('t [s]')
ylabel('Concentration [µM]')
title('Testing equilibrium at apical side')
ax = gca;
ax.FontSize = 10;
end

%% Figure 7a - Cdc42-Rac1-Rhoa response in dependence of Ia
if test==4
figure(5)    
plot(x,y1,'Color',green1,'LineWidth',2.0);
hold on
plot(x,y2,'Color',purple1,'LineWidth',2.0)
plot(x,y3,'Color',red1,'LineWidth',2.0)
txt = 'Ia=15,'; txt1='_____';
text(6.5,7.0,txt,'Color', 'k', 'FontSize',12)
text(7.7,7.2,txt1,'Color', 'k', 'FontSize',12, 'Interpreter',"none")
end
if test==7
figure(5)    
plot(x,y1,'Color', green1,'LineStyle','--','LineWidth',2.0);
hold on
plot(x,y2,'Color',purple1,'LineStyle','--','LineWidth',2.0)
plot(x,y3,'Color',red1,'LineStyle','--','LineWidth',2.0)
txt1 = 'Ia=30,  - - - - -';
text(6.5,6.5,txt1,'Color', 'k', 'FontSize',12)
txt2 = 'Rac1';txt3='Cdc42';txt4='Rhoa';
text(1.8,4,txt2,'Color', purple1, 'FontWeight','bold', 'FontSize',12)
text(6,3,txt3,'Color', green1, 'FontWeight','bold','FontSize',12)
text(5,1.5,txt4,'Color', red1, 'FontWeight','bold','FontSize',12)
xlabel('x [µm]')
ylabel('Concentration [µM]')
ax = gca;
ax.FontSize = 12;
exportgraphics(ax,'Fig_7A.tif','Resolution',300)
end

%% Figure 7b - Pc-Sc response in dependence of Ia
if test==4
figure(6)    
plot(x,y7,'Color',blue1,'LineWidth',2.0);
hold on
plot(x,y9,'Color',orange1,'LineWidth',2.0);
txt = 'Ia=15,'; txt1='  _____';
text(6.5,0.8,txt,'Color', 'k', 'FontSize',12)
text(7.5,0.825,txt1,'Color', 'k', 'FontSize',12, 'Interpreter',"none")
end
if test==7
figure(6)    
plot(x,y7,'Color',blue1,'LineStyle','--','LineWidth',2.0);
hold on
plot(x,y9,'Color',orange1,'LineStyle','--','LineWidth',2.0);
txt1 = 'Ia=30,  - - - - -';
text(6.5,0.7,txt1,'Color', 'k', 'FontSize',12)
txt2 = 'PAR';txt3='Scribble';
text(6.,1.25,txt2,'Color', blue1, 'FontWeight','bold', 'FontSize',12)
text(7,0.1,txt3,'Color', orange1, 'FontWeight','bold','FontSize',12)
xlabel('x [µm]')
ylabel('Concentration [µM]')
ax = gca;
ax.FontSize = 12;
exportgraphics(ax,'Fig_7B.tif','Resolution',300)
end

%% Figure 7C - Determine critical amount of Ia to achieve cell polarization (Rac1, Rho, Cdc42)
if test==11
figure(7)
x_int=(0:1:test-1)*Itot1;   %[0,5,10,15,20,25,30,35,40,45,50];
plot(x_int, Diff_Rac(:),'-o','Color',purple1,'LineWidth',2.0)
hold on
plot(x_int, Diff_Rho(:),'-o','Color',red1,'LineWidth',2.0)
plot(x_int, Diff_Cdc(:),'-o','Color',green1,'LineWidth',2.0)
txt = 'Rac_{diff}';txt1='Rho_{diff}';txt2='Cdc_{diff}';
text(40,6.2,txt,'Color', purple1, 'FontWeight','bold', 'FontSize',12)
text(40,1.6,txt1,'Color', red1, 'FontWeight','bold','FontSize',12)
text(40,3.0,txt2,'Color', green1, 'FontWeight','bold','FontSize',12)
xlabel('Integrin level [µM]')
ylabel('Concentration [µM]')
title('Apical-basal difference vs. Integrin level')
ax = gca;
ax.FontSize = 12;
exportgraphics(ax,'Fig_7C.tif','Resolution',300)
end
%% Figure 7D - Determine critical amount of Ia to achieve cell polarization (Pc and Sc)
if test==11
figure(8)
x_int=(0:1:test-1)*Itot1;   %[0,5,10,15,20,25,30,35,40,45,50];
plot(x_int, Diff_Pc(:),'-o','Color',blue1,'LineWidth',2.0)
hold on
plot(x_int, Diff_Sc(:),'-o','Color',orange1,'LineWidth',2.0)
txt = 'PAR_{diff}';txt1='Scribble_{diff}';
text(40,1.2,txt,'Color', blue1, 'FontWeight','bold', 'FontSize',12)
text(40,1.1,txt1,'Color', orange1, 'FontWeight','bold','FontSize',12)
xlabel('Integrin level [µM]')
ylabel('Concentration [µM]')
title('Apical-basal difference vs. Integrin level')
ax = gca;
ax.FontSize = 12;
exportgraphics(ax,'Fig_7D.tif','Resolution',300)
end

end