clear all 
close all

%% ***** Finite difference method EMT *****
%% Parameters
Dm = 0.1;           % Membrane diffusion coefficient, [micrometer^2/s]
Dc = 10;            % Cytosolic diffusion coefficient, [micrometer^2/s]
n = 4;              % Hill coefficient of inhibition responses, [-]

%% Parameters Cdc42
Ic = 0.5;           % Cdc42 activation input rate, [microM/s]
delta_C = 1;        % Decay rate of activated Cdc42, [1/s]
Ctot = 4.4;         % Total Cdc42 concentration, [microM]
Dca = Dm;           % Membrane diffusion coefficient of Cdc42, [micrometer^2/s]
Dci = Dc;           % Cytosolic diffusion coefficient of Cdc42, [micrometer^2/s]
alpha_C = 2.5;      % Cdc42 dependent activation rate, [1/s] 

%% Parameters Rac
Ir = 0.5;           % Rac activation input rate, [microM/s]
delta_R = 1;        % Decay rate of activated Rac, [1/s]
Rtot = 2.4;         % Total Rac concentration, [microM]
Dra = Dm;           % Membrane diffusion coefficient of Rac, [micrometer^2/s]
Dri = Dc;           % Cytosolic diffusion coefficient of RAC, [micrometer^2/s]
beta_R = 1.5;       % Active Rac dependent inhibition rate, [microM]   

%% Parameters Rho
Irho = 3.3;         % Rho activation rate, [microM/s]
delta_Rho = 1;      % Decay rate of activated Rho, [1/s]
Rhotot = 4;         % Total Rho concentration, [microM]
Drhoa = Dm;         % Membrane diffusion coefficient of Rho, [micrometer^2/s]
Drhoi = Dc;         % Cytosolic diffusion coefficient of Cdc42, [micrometer^2/s]
beta_rho = 0.7;     % Active Rho dependent inhibition rate, [microM]
alpha_rho=2.5;      % Rho dependent activation rate, [1/s]

%% Parameters PAR
k_on_P = 0.5;       % Par association rate or on-rate, [microM/s]
k_off_P = 2;        % Par dissociation rate or off-rate, [1/s]
Ptot = 1.6;         % Effective total of Par complex concentration, [microM]
Dpc = 0.2;          % Membrane diffusion coefficient of Par complex, [micrometer^2/s]
Dpun = 2;           % Cytosolic diffusion coefficient of unbound Par proteins, [micrometer^2/s]
beta_PR = 2;        % Active Par complex dependent inhibition rate, [microM]
beta_PS = 0.2;      % Par-complex dependent inhibition rate, [microM]

%% Parameters Scrib
k_on_S = 4;         % Scribble association rate or on-rate, [microM/s]
k_off_S = 2;        % Scribble dissociation rate or off-rate, [1/s]
Stot = 1.6;         % Effective total of Scribble complex concentration, [microM]
Dsc = 0.2;          % Membrane diffusion coefficient of Scribble complex, [micrometer^2/s]
Dsun = 2;           % Cytosolic diffusion coefficient of unbound Scribble proteins, [micrometer^2/s]
beta_SP = 0.2;      % Scribble-complex dependent inhibition rate, [microM]

%% Parameters Integrin
I_I = 10;           % Integrin activation input rate, [microM/s]
delta_I = 1;        % Decay rate of activated integrin, [1/s]
Itot = 30;          % Effective total of integrins, [microM]
D_Ii = 10;          % Cytosolic diffusion coefficient of inactive integrins, [micrometer^2/s]
alpha_I = 1.5;      % Integrin-dependent activation rate, [1/s]
l=1;     % If activation by integrin is on left side of the cell (l=1)
r=0;     % If activation by integrin is on right side of the cell (r=1)

%% Common parameters
x_max=10;               % Length, L, [micrometer]
Nx=30;                  % Number of divisions along length, [-]
delta_x=x_max/Nx;       % Length increment
x=0:delta_x:x_max;      % Position, x, [micrometer]

t_max=300;              % Simulation time, t, [s]
delta_t=0.005;          % Time increment, [s]
Nt=t_max/delta_t;       % Number of time steps, [-]
t=0:delta_t:t_max;      % Time, t, [s]

%% Allocation
Ca=zeros(length(x),1); Ci=zeros(length(x),1);
Rhoa=zeros(length(x),1); Rhoi=zeros(length(x),1);
Ra=zeros(length(x),1); Ri=zeros(length(x),1);
Pc=zeros(length(x),1); Pun=zeros(length(x),1);
Sc=zeros(length(x),1); Sun=zeros(length(x),1);
Ia=zeros(length(x),1); Ii=zeros(length(x),1);
A=zeros(12*length(x),12*length(x)); Vector=zeros(12*length(x),length(t));
coupl=zeros(12*length(x),length(t)); RHS=zeros(12*length(x),length(t));

%% Definition of matrix A
%% *************** Cdc42 ***************
b_Ca=1/delta_t-delta_C-2*Dca/(delta_x^2);           % Cdc42 active
a_Ca=Dca/(delta_x^2);                             
c_Ca=Dca/(delta_x^2);                           
b_Ci=1/delta_t-2*Dci/(delta_x^2);                   % Cdc42 inactive
a_Ci=Dci/(delta_x^2);                           
c_Ci=Dci/(delta_x^2);                           
% Coupling of Ci to Ca
d_Ci_Ca=delta_C;

%% *************** Rac1 ***************
b_Ra=1/delta_t-delta_R-2*Dra/(delta_x^2);           % Rac active
a_Ra=Dra/(delta_x^2);                             
c_Ra=Dra/(delta_x^2);                             
b_Ri=1/delta_t-2*Dri/(delta_x^2);                   % Rac inactive
a_Ri=Dri/(delta_x^2);                           
c_Ri=Dri/(delta_x^2);                           
% Coupling of Ri to Ra
d_Ri_Ra=delta_R;

%% *************** Rho ***************
b_rhoa=1/delta_t-delta_Rho-2*Drhoa/(delta_x^2);     % Rho active
a_rhoa=Drhoa/(delta_x^2);                           
c_rhoa=Drhoa/(delta_x^2);                           
b_rhoi=1/delta_t-2*Drhoi/(delta_x^2);               % Rho inactive
a_rhoi=Drhoi/(delta_x^2);                         
c_rhoi=Drhoi/(delta_x^2);                         
% Coupling of Rhoi to Rhoa
d_Rhoi_Rhoa=delta_Rho;

%% *************** Par ***************
b_Pc=1/delta_t-k_off_P-2*Dpc/(delta_x^2);           % Par complex
a_Pc=Dpc/(delta_x^2);                             
c_Pc=Dpc/(delta_x^2);                             
b_Pun=1/delta_t-2*Dpun/(delta_x^2);                 % Par unbounded
a_Pun=Dpun/(delta_x^2);                           
c_Pun=Dpun/(delta_x^2);                           
% Coupling of Pun to Pc
d_Pun_Pc = k_off_P;

%% *************** Scribble ***************
b_Sc=1/delta_t-k_off_S-2*Dsc/(delta_x^2);           % Scribble complex
a_Sc=Dsc/(delta_x^2);                             
c_Sc=Dsc/(delta_x^2);                             
b_Sun=1/delta_t-2*Dsun/(delta_x^2);                 % Scribble unbounded
a_Sun=Dsun/(delta_x^2);                           
c_Sun=Dsun/(delta_x^2);                           
% Coupling of Sun to Sc
d_Sun_Sc=k_off_S;

%% *************** Integrin ***************
% along the cell
b_Ia=0;                                             % I active
a_Ia=0;                            
c_Ia=0;                           
b_Ii=1/delta_t-2*D_Ii/(delta_x^2);                  % I inactive
a_Ii=D_Ii/(delta_x^2);                           
c_Ii=D_Ii/(delta_x^2);                          
% at the membrane
d_I_Ii=I_I/Itot;
b_Ia_lr=1/delta_t-delta_I;
b_Ii_lr=b_Ii-I_I/Itot;
d_Ii_I=delta_I;

%% Creating matrix A
for i=2:length(x)-1
    A(i,i)=b_Ca;                                        % 1) Ca
    A(i,i-1)=a_Ca;
    A(i,i+1)=c_Ca;
    A(length(x)+i,length(x)+i)=b_Ra;                    % 2) Ra
    A(length(x)+i,length(x)+i-1)=a_Ra;
    A(length(x)+i,length(x)+i+1)=c_Ra;
    A(2*length(x)+i,2*length(x)+i)=b_rhoa;              % 3) Rhoa
    A(2*length(x)+i,2*length(x)+i-1)=a_rhoa;
    A(2*length(x)+i,2*length(x)+i+1)=c_rhoa;
    A(3*length(x)+i,3*length(x)+i)=b_Ci;                % 4) Ci
    A(3*length(x)+i,3*length(x)+i-1)=a_Ci;
    A(3*length(x)+i,3*length(x)+i+1)=c_Ci;
    A(4*length(x)+i,4*length(x)+i)=b_Ri;                % 5) Ri
    A(4*length(x)+i,4*length(x)+i-1)=a_Ri;
    A(4*length(x)+i,4*length(x)+i+1)=c_Ri;
    A(5*length(x)+i,5*length(x)+i)=b_rhoi;              % 6) Rhoi
    A(5*length(x)+i,5*length(x)+i-1)=a_rhoi;
    A(5*length(x)+i,5*length(x)+i+1)=c_rhoi;
    A(6*length(x)+i,6*length(x)+i)=b_Pc;                % 7) PAR, Pc
    A(6*length(x)+i,6*length(x)+i-1)=a_Pc;
    A(6*length(x)+i,6*length(x)+i+1)=c_Pc;
    A(7*length(x)+i,7*length(x)+i)=b_Pun;               % 8) Pun
    A(7*length(x)+i,7*length(x)+i-1)=a_Pun;
    A(7*length(x)+i,7*length(x)+i+1)=c_Pun;
    A(8*length(x)+i,8*length(x)+i)=b_Sc;                % 9) Sc
    A(8*length(x)+i,8*length(x)+i-1)=a_Sc;
    A(8*length(x)+i,8*length(x)+i+1)=c_Sc;
    A(9*length(x)+i,9*length(x)+i)=b_Sun;               % 10) Sun
    A(9*length(x)+i,9*length(x)+i-1)=a_Sun;
    A(9*length(x)+i,9*length(x)+i+1)=c_Sun;   
    A(10*length(x)+i,10*length(x)+i)=b_Ia;              % 11) Ia
    A(10*length(x)+i,10*length(x)+i-1)=a_Ia;
    A(10*length(x)+i,10*length(x)+i+1)=c_Ia;
    A(11*length(x)+i,11*length(x)+i)=b_Ii;              % 12) Ii
    A(11*length(x)+i,11*length(x)+i-1)=a_Ii;
    A(11*length(x)+i,11*length(x)+i+1)=c_Ii;

    % Coupling members (linear) between variables
    A(3*length(x)+i,i)=d_Ci_Ca;
    A(4*length(x)+i,length(x)+i)=d_Ri_Ra;
    A(5*length(x)+i,2*length(x)+i)=d_Rhoi_Rhoa;
    A(7*length(x)+i,6*length(x)+i)=d_Pun_Pc;
    A(9*length(x)+i,8*length(x)+i)=d_Sun_Sc;
end

%% Boundary conditions, no flux exchange at the borders, no protein passage through cell membrane
    A(1,1)=a_Ca+b_Ca;                               % Ca 
    A(1,2)=c_Ca;
    A(length(x),length(x))=b_Ca+c_Ca; 
    A(length(x),length(x)-1)=a_Ca;
    A(length(x)+1,length(x)+1)=a_Ra+b_Ra;           % Ra 
    A(length(x)+1,length(x)+2)=c_Ra;                                                                                                                                                                                                                                                                                                                                                                                                                                      
    A(2*length(x),2*length(x))=b_Ra+c_Ra; 
    A(2*length(x),2*length(x)-1)=a_Ra;
    A(2*length(x)+1,2*length(x)+1)=a_rhoa+b_rhoa;   % Rhoa 
    A(2*length(x)+1,2*length(x)+2)=c_rhoa;                                                                                                                                                                                                                                                                                                                                                                                                                                        
    A(3*length(x),3*length(x))=b_rhoa+c_rhoa; 
    A(3*length(x),3*length(x)-1)=a_rhoa;
    A(3*length(x)+1,3*length(x)+1)=a_Ci+b_Ci;       % Ci 
    A(3*length(x)+1,3*length(x)+2)=c_Ci;                                                                                                                                                                                                                                                                                                                                                                                                                                            
    A(4*length(x),4*length(x))=b_Ci+c_Ci; 
    A(4*length(x),4*length(x)-1)=a_Ci;
    A(4*length(x)+1,4*length(x)+1)=a_Ri+b_Ri;       % Ri 
    A(4*length(x)+1,4*length(x)+2)=c_Ri;                                                                                                                                                                                                                                                                                                                                                                                                                                            
    A(5*length(x),5*length(x))=b_Ri+c_Ri; 
    A(5*length(x),5*length(x)-1)=a_Ri;    
    A(5*length(x)+1,5*length(x)+1)=a_rhoi+b_rhoi;   % Rhoi 
    A(5*length(x)+1,5*length(x)+2)=c_rhoi;                                                                                                                                                                                                                                                                                                                                                                                                                                            
    A(6*length(x),6*length(x))=b_rhoi+c_rhoi; 
    A(6*length(x),6*length(x)-1)=a_rhoi;
    A(6*length(x)+1,6*length(x)+1)=a_Pc+b_Pc;       % Pc 
    A(6*length(x)+1,6*length(x)+2)=c_Pc;                                                                                                                                                                                                                                                                                                                                                                                                                                            
    A(7*length(x),7*length(x))=b_Pc+c_Pc; 
    A(7*length(x),7*length(x)-1)=a_Pc; 
    A(7*length(x)+1,7*length(x)+1)=a_Pun+b_Pun;     % Pun 
    A(7*length(x)+1,7*length(x)+2)=c_Pun;                                                                                                                                                                                                                                                                                                                                                                                                                                            
    A(8*length(x),8*length(x))=b_Pun+c_Pun; 
    A(8*length(x),8*length(x)-1)=a_Pun;    
    A(8*length(x)+1,8*length(x)+1)=a_Sc+b_Sc;       % Sc 
    A(8*length(x)+1,8*length(x)+2)=c_Sc;                                                                                                                                                                                                                                                                                                                                                                                                                                            
    A(9*length(x),9*length(x))=b_Sc+c_Sc; 
    A(9*length(x),9*length(x)-1)=a_Sc;     
    A(9*length(x)+1,9*length(x)+1)=a_Sun+b_Sun;     % Sun
    A(9*length(x)+1,9*length(x)+2)=c_Sun;                                                                                                                                                                                                                                                                                                                                                                                                                                            
    A(10*length(x),10*length(x))=b_Sun+c_Sun;
    A(10*length(x),10*length(x)-1)=a_Sun;  
    
    % Activated integrin on left or right side of the cell
    if l==1
        for m=1:5
    % left border - coupling
    A(10*length(x)+m,10*length(x)+m)=b_Ia_lr;           % Ia
    A(10*length(x)+m,10*length(x)+m+1)=c_Ia;
        if m==1
        A(11*length(x)+m,11*length(x)+m)=b_Ii_lr+a_Ii;  % Ii
        A(11*length(x)+m,11*length(x)+m+1)=c_Ii;
        else
        A(11*length(x)+m,11*length(x)+m)=b_Ii_lr;       % Ii
        end
    A(10*length(x)+m,11*length(x)+m)=d_I_Ii;
    A(11*length(x)+m,10*length(x)+m)=d_Ii_I;
        end
    % right border - no coupling
    A(11*length(x),11*length(x))=b_Ia+c_Ia;             % Ia
    A(11*length(x),11*length(x)-1)=a_Ia;
    A(12*length(x),12*length(x))=b_Ii+c_Ii;             % Ii
    A(12*length(x),12*length(x)-1)=a_Ii;
    end
	
    if r==1      
    % left border - no coupling
    A(10*length(x)+1,10*length(x)+1)=b_Ia+a_Ia;         % Ia
    A(10*length(x)+1,10*length(x)+2)=a_Ia;
    A(11*length(x)+1,11*length(x)+1)=b_Ii+a_Ii;         % Ii
    A(11*length(x)+1,11*length(x)+2)=c_Ii;
      for m=1:5
    % right border - coupling      
    A(11*length(x)+1-m,11*length(x)+1-m)=b_Ia_lr;               % Ia
    A(11*length(x)+1-m,11*length(x)-1+1-m)=a_Ia;
        if m==1
        A(12*length(x)+1-m,12*length(x)+1-m)=b_Ii_lr+c_Ii;      % Ii
        A(12*length(x)+1-m,12*length(x)-1+1-m)=a_Ii;
        else
        A(12*length(x)+1-m,12*length(x)+1-m)=b_Ii_lr;           % Ii
        end
    A(11*length(x)+1-m,12*length(x)+1-m)=d_I_Ii;
    A(12*length(x)+1-m,11*length(x)+1-m)=d_Ii_I;
      end
    end
    
    % Linear coupling members on boarders (left and right side)
    A(3*length(x)+1,1)=d_Ci_Ca;
    A(4*length(x),length(x))=d_Ci_Ca;
    A(4*length(x)+1,length(x)+1)=d_Ri_Ra;
    A(5*length(x),2*length(x))=d_Ri_Ra;
    A(5*length(x)+1,2*length(x)+1)=d_Rhoi_Rhoa;
    A(6*length(x),3*length(x))=d_Rhoi_Rhoa;    
    A(7*length(x)+1,6*length(x)+1)=d_Pun_Pc;
    A(8*length(x),7*length(x))=d_Pun_Pc;
    A(9*length(x)+1,8*length(x)+1)=d_Sun_Sc;
    A(10*length(x),9*length(x))=d_Sun_Sc;

%% Set initial values
Ca(:,1)=1.75;         % [microM]
Ci(:,1)=2.65;         % [microM]
Ra(:,1)=1;            % [microM]
Ri(:,1)=1.4;          % [microM]
Rhoa(:,1)=0.8;        % [microM]
Rhoi(:,1)=3.2;        % [microM]
Pc(:,1)=0.6;          % [microM]
Pun(:,1)=1;           % [microM]
Sc(:,1)=0.6;          % [microM]
Sun(:,1)=1;           % [microM]

 for i=1:length(x)
     % Activation at left side
     if l==1
        if i<=5 
            Ia(i,1)=(Itot/i^2)*exp(1/i^2-1);
        else
            Ia(i,1)=0;
        end
     end
     % Activation at right side
     if r==1
     if i<=length(x)-5
        Ia(i,1)=0;     
     else
        Ia(i,1)=(Itot/(length(x)+1-i)^2)*exp(1/((length(x)+1-i))^2-1); 
     end
     end
 end
 Ii(:,1)=Itot-Ia(:,1);

Vector(1:length(x),1)=Ca(:,1);
Vector(length(x)+1:2*length(x),1)=Ra(:,1);
Vector(2*length(x)+1:3*length(x),1)=Rhoa(:,1);
Vector(3*length(x)+1:4*length(x),1)=Ci(:,1);
Vector(4*length(x)+1:5*length(x),1)=Ri(:,1);
Vector(5*length(x)+1:6*length(x),1)=Rhoi(:,1);
Vector(6*length(x)+1:7*length(x),1)=Pc(:,1);
Vector(7*length(x)+1:8*length(x),1)=Pun(:,1);
Vector(8*length(x)+1:9*length(x),1)=Sc(:,1);
Vector(9*length(x)+1:10*length(x),1)=Sun(:,1);
Vector(10*length(x)+1:11*length(x),1)=Ia(:,1);
Vector(11*length(x)+1:12*length(x),1)=Ii(:,1);

%% *************** EMT ***************
k1=0;
Rhotot1=Rhotot;Rtot1=Rtot;Ctot1=Ctot;amount=zeros(length(t),1);
a_rho=zeros(length(x));a_rac=zeros(length(x));a_cdc=zeros(length(x));
a1=100;b1=0.043;
%% Set the option for GTP that degrades
option=3;   %[option: 1 for Rac degradation; 2 for Rho degradation; 3 for Cdc degradation
% Set option for downregulation or upregulation
downregulation = 1;
upregulation = 0;

%% Nonlinear coupling members from equations and solving equations
for test = 1:6  % 0, 20, 40, 60, 80, 100
    if test > 1    % Reset to initial tot concentrations
        Rhotot=Rhotot1;Ctot=Ctot1;Rtot=Rtot1;
    end

for j=1:length(t)-1      
        for i=1:length(x)
            coupl(i,j)=(Ic+alpha_rho*Vector(2*length(x)+i,j))*(Vector(3*length(x)+i,j)/Ctot); 
            coupl(length(x)+i,j)=((Ir+alpha_I*Vector(10*length(x)+i,j))/(1+(Vector(2*length(x)+i,j)^n)/(beta_rho^n)+(Vector(6*length(x)+i,j)^n)/(beta_PR^n)))*(Vector(4*length(x)+i,j)/Rtot);
            coupl(2*length(x)+i,j)=(Irho/(1+((Vector(length(x)+i,j)^n)/(beta_R^n))))*(Vector(5*length(x)+i,j)/Rhotot); 
            coupl(3*length(x)+i,j)=-coupl(i,j);
            coupl(4*length(x)+i,j)=-coupl(length(x)+i,j);
            coupl(5*length(x)+i,j)=-coupl(2*length(x)+i,j);
            coupl(6*length(x)+i,j)=((k_on_P+alpha_C*Vector(i,j))/(1+((Vector(8*length(x)+i,j)^n)/(beta_SP^n))))*(Vector(7*length(x)+i,j)/Ptot);
            coupl(7*length(x)+i,j)=-coupl(6*length(x)+i,j);
            coupl(8*length(x)+i,j)=(k_on_S/(1+((Vector(6*length(x)+i,j)^n)/(beta_PS^n))))*(Vector(9*length(x)+i,j)/Stot);
            coupl(9*length(x)+i,j)=-coupl(8*length(x)+i,j);       
        end
        
    RHS(:,j)=A(:,:)*Vector(:,j)+coupl(:,j);
    Vector(:,j+1)=delta_t*RHS(:,j);
    
    if j<=20000
    amount(j)=100;
    end
    if j>40000
        amount(j+1)=rate;
    end

%% Degradation / upregulation of RAC1
%% Constant percentage along cell length, increaes in time
if j>=20000 && j<=40000 && option==1    % [100-200s]
        if j==20000
            steady_state_rac = Vector(length(x)+1:2*length(x),j);
            steady_state_inactive_rac = Vector(4*length(x)+1:5*length(x),j);
        end

        if test == 1    % There is no degradation / upregulation (rate = 0)
        a=100;
        green2=[214/255,238/255,183/255];red2=[232/255,138/255,153/255];purple2=[229/255,204/255,255/255];
        yellow2=[0.9290,0.6940,0.1250];blue2=[156/255,204/255,236/255];orange2=[247/255,208/255,191/255];
        end
        if test == 2    % 20% of active component is degraded / upregulated
        a=80;
        green2=[193/255,226/255,149/255];red2=[212/255,104/255,122/255];purple2=[195/255,148/255,205/255];
        yellow2=[0.9290,0.6940,0.1250];blue2=[126/255,190/255,232/255];orange2=[239/255,185/255,162/255];
        end
        if test == 3    % 40% of active component is degraded / upregulated
        a=60;
        green2=[178/255,216/255,127/255];red2=[192/255,49/255,73/255];purple2=[182/255,113/255,196/255];
        yellow2=[0.9290,0.6940,0.1250];blue2=[92/255,172/255,225/255];orange2=[232/255,159/255,128/255];
        end
        if test == 4    % 60% of active component is degraded / upregulated  
        a=40;
        green2=[153/255,198/255,94/255];red2=[196/255,72/255,93/255];purple2=[168/255,92/255,183/255];
        yellow2=[0.9290,0.6940,0.1250];blue2=[58/255,153/255,216/255];orange2=[230/255,136/255,96/255];
        end
        if test == 5    % 80% of active component is degraded / upregulated
        a=20;
        green2=[134/255,183/255,69/255];red2=[179/255,50/255,71/255];purple2=[149/255,69/255,165/255];
        yellow2=[0.9290,0.6940,0.1250];blue2=[31/255,136/255,206/255];orange2=[223/255,110/255,62/255];
        end
        if test == 6    % 100% of active component is degraded / upregulated
        a=0;
        green2=[0.4660,0.6740,0.1880];red2=[0.6350,0.0780,0.1840];purple2=[0.4940,0.1840,0.5560];
        yellow2=[0.9290,0.6940,0.1250];blue2=[0,0.4470,0.7410];orange2=[0.8500,0.3250,0.0980];
        end
        
        if downregulation == 1 
        rate = (a1-a)*exp(-b1*(j-20000)*delta_t)+a;
        end
        if upregulation ==1
        rate = 100+100-((a1-a)*exp(-b1*(j-20000)*delta_t)+a);
        end

        % Decrease / increase the amount of active Rac 
        for kk = 1:length(x)
         Vector(length(x)+kk,j+1)=0.01*rate*steady_state_rac(kk);
         a_rac(kk)= (1-0.01*rate)*steady_state_rac(kk);    
        end
        
        % Update total amount of Rac by decrease / increase of Ra
        Rtot = Rtot1 - sum(a_rac(:))/31;
        
        amount(j+1) = rate;
end

    %% Downregulation / upregulation of Rhoa
    %% Constant percentage along cell length, increaes in time
    if j>=20000 && j<=40000 && option==2    % [100-200s]    
        if j==20000
            steady_state = Vector(2*length(x)+1:3*length(x),j);
            steady_state_inactive = Vector(5*length(x)+1:6*length(x),j);
        end
        
        if test == 1    % There is no degradation / upregulation (rate = 0)
        a=100;
        green2=[214/255,238/255,183/255];red2=[232/255,138/255,153/255];purple2=[229/255,204/255,255/255];
        yellow2=[0.9290,0.6940,0.1250];blue2=[156/255,204/255,236/255];orange2=[247/255,208/255,191/255];
        end
        if test == 2    % 20% of active component is degraded / upregulated
        a=80;
        green2=[193/255,226/255,149/255];red2=[212/255,104/255,122/255];purple2=[195/255,148/255,205/255];
        yellow2=[0.9290,0.6940,0.1250];blue2=[126/255,190/255,232/255];orange2=[239/255,185/255,162/255];
        end
        if test == 3    % 40% of active component is degraded / upregulated
        a=60;
        green2=[178/255,216/255,127/255];red2=[192/255,49/255,73/255];purple2=[182/255,113/255,196/255];
        yellow2=[0.9290,0.6940,0.1250];blue2=[92/255,172/255,225/255];orange2=[232/255,159/255,128/255];
        end
        if test == 4    % 60% of active component is degraded / upregulated  
        a=40;
        green2=[153/255,198/255,94/255];red2=[196/255,72/255,93/255];purple2=[168/255,92/255,183/255];
        yellow2=[0.9290,0.6940,0.1250];blue2=[58/255,153/255,216/255];orange2=[230/255,136/255,96/255];
        end
        if test == 5    % 80% of active component is degraded / upregulated
        a=20;
        green2=[134/255,183/255,69/255];red2=[179/255,50/255,71/255];purple2=[149/255,69/255,165/255];
        yellow2=[0.9290,0.6940,0.1250];blue2=[31/255,136/255,206/255];orange2=[223/255,110/255,62/255];
        end
        if test == 6    % 100% of active component is degraded / upregulated
        a=0;
        green2=[0.4660,0.6740,0.1880];red2=[0.6350,0.0780,0.1840];purple2=[0.4940,0.1840,0.5560];
        yellow2=[0.9290,0.6940,0.1250];blue2=[0,0.4470,0.7410];orange2=[0.8500,0.3250,0.0980];
        end

        if downregulation == 1 
        rate = (a1-a)*exp(-b1*(j-20000)*delta_t)+a;
        end
        if upregulation ==1
        rate = 100+100-((a1-a)*exp(-b1*(j-20000)*delta_t)+a);
        end

        % Decrease / increase the amount of active Rho 
        for kk = 1:length(x)
            Vector(2*length(x)+kk,j+1)=0.01*rate*steady_state(kk);
            a_rho(kk)= (1-0.01*rate)*steady_state(kk); 
        end
        
        % Update total amount of Rho by decrease / increase of Rhoa
        Rhotot = Rhotot1 - sum(a_rho(:))/31;
              
        amount(j+1) = rate;
       
    end

%% Degradation / upregulation of Cdc42
%% Constant percentage along cell length, increaes in time
    if j>=20000 && j<=40000 && option==3    % [100-200s]
        if j==20000
            steady_state_cdc = Vector(1:length(x),j);
            steady_state_inactive_cdc = Vector(3*length(x)+1:4*length(x),j);
        end
        
        if test == 1    % There is no degradation / upregulation (rate = 0)
        a=100;
        green2=[214/255,238/255,183/255];red2=[232/255,138/255,153/255];purple2=[229/255,204/255,255/255];
        yellow2=[0.9290,0.6940,0.1250];blue2=[156/255,204/255,236/255];orange2=[247/255,208/255,191/255];
        end
        if test == 2    % 20% of active component is degraded / upregulated
        a=80;
        green2=[193/255,226/255,149/255];red2=[212/255,104/255,122/255];purple2=[195/255,148/255,205/255];
        yellow2=[0.9290,0.6940,0.1250];blue2=[126/255,190/255,232/255];orange2=[239/255,185/255,162/255];
        end
        if test == 3    % 40% of active component is degraded / upregulated
        a=60;
        green2=[178/255,216/255,127/255];red2=[192/255,49/255,73/255];purple2=[182/255,113/255,196/255];
        yellow2=[0.9290,0.6940,0.1250];blue2=[92/255,172/255,225/255];orange2=[232/255,159/255,128/255];
        end
        if test == 4    % 60% of active component is degraded / upregulated   
        a=40;
        green2=[153/255,198/255,94/255];red2=[196/255,72/255,93/255];purple2=[168/255,92/255,183/255];
        yellow2=[0.9290,0.6940,0.1250];blue2=[58/255,153/255,216/255];orange2=[230/255,136/255,96/255];
        end
        if test == 5    % 80% of active component is degraded / upregulated
        a=20;
        green2=[134/255,183/255,69/255];red2=[179/255,50/255,71/255];purple2=[149/255,69/255,165/255];
        yellow2=[0.9290,0.6940,0.1250];blue2=[31/255,136/255,206/255];orange2=[223/255,110/255,62/255];
        end
        if test == 6    % 100% of active component is degraded / upregulated
        a=0;
        green2=[0.4660,0.6740,0.1880];red2=[0.6350,0.0780,0.1840];purple2=[0.4940,0.1840,0.5560];
        yellow2=[0.9290,0.6940,0.1250];blue2=[0,0.4470,0.7410];orange2=[0.8500,0.3250,0.0980];
        end

        if downregulation == 1 
        rate = (a1-a)*exp(-b1*(j-20000)*delta_t)+a;
        end
        if upregulation ==1
        rate = 100+100-((a1-a)*exp(-b1*(j-20000)*delta_t)+a);
        end
        
        % Decrease / increase the amount of active Cdc42 
        for kk = 1:length(x)
         Vector(kk,j+1)=0.01*rate*steady_state_cdc(kk);
         a_cdc(kk)= (1-0.01*rate)*steady_state_cdc(kk);    
        end
        
        % Update total amount of Cdc42 by decrease of Ca
        Ctot = Ctot1 - sum(a_cdc(:))/31;
        
        amount(j+1) = rate;    
        
    end    

    %% Check if values go below zero
    for i=1:12*length(x)
        if Vector(i,j+1)<0
            Vector(i,j+1)=0;
            k1=k1+1;
        end           
    end
end

%% Plotting the results at the end of simulation
y1=Vector(1:length(x),length(t));                   % Ca
y2=Vector(length(x)+1:2*length(x),length(t));       % Ra
y3=Vector(2*length(x)+1:3*length(x),length(t));     % Rhoa
y4=Vector(3*length(x)+1:4*length(x),length(t));     % Ci
y5=Vector(4*length(x)+1:5*length(x),length(t));     % Ri
y6=Vector(5*length(x)+1:6*length(x),length(t));     % Rhoi
y7=Vector(6*length(x)+1:7*length(x),length(t));     % Pc
y8=Vector(7*length(x)+1:8*length(x),length(t));     % Pun
y9=Vector(8*length(x)+1:9*length(x),length(t));     % Sc
y10=Vector(9*length(x)+1:10*length(x),length(t));   % Sun
y11=Vector(10*length(x)+1:11*length(x),length(t));  % Ia
y12=Vector(11*length(x)+1:12*length(x),length(t));  % Ii

% Depict colors for plots
green1=[0.4660,0.6740,0.1880];red1=[0.6350,0.0780,0.1840];purple1=[0.4940,0.1840,0.5560];
yellow1=[0.9290,0.6940,0.1250];blue1=[0,0.4470,0.7410];orange1=[0.8500,0.3250,0.0980];

%% Figure 8 EMT - Rac1
figure(1)
plot(x, y2, 'Color', purple2, 'LineWidth',1)
my_map_rac=ColorMapRac;
colormap(my_map_rac)
hold on
title('Rac1','FontSize',12)
xlabel('x [µm]','FontSize',12)
ylabel('Concentration [µM]','FontSize',12)
if option==1, exportgraphics(gca,'Fig_8A_Rac1.tif','Resolution',300), end
if option==3, exportgraphics(gca,'Fig_8B_Rac1.tif','Resolution',300), end
if option==2, exportgraphics(gca,'Fig_8C_Rac1.tif','Resolution',300), end
    
%% Figure 8 EMT - Rho
figure(2)
plot(x, y3, 'Color', red2, 'LineWidth',1)
hold on
title('Rho','FontSize',12)
xlabel('x [µm]','FontSize',12)
ylabel('Concentration [µM]','FontSize',12)
if option==1, exportgraphics(gca,'Fig_8A_Rho.tif','Resolution',300), end
if option==3, exportgraphics(gca,'Fig_8B_Rho.tif','Resolution',300), end
if option==2, exportgraphics(gca,'Fig_8C_Rho.tif','Resolution',300), end

%% Figure 8 EMT - Cdc42
figure(3)
plot(x, y1, 'Color', green2, 'LineWidth',1)
hold on
title('Cdc42','FontSize',12)
xlabel('x [µm]','FontSize',12)
ylabel('Concentration [µM]','FontSize',12)
if option==1, exportgraphics(gca,'Fig_8A_Cdc42.tif','Resolution',300), end
if option==3, exportgraphics(gca,'Fig_8B_Cdc42.tif','Resolution',300), end
if option==2, exportgraphics(gca,'Fig_8C_Cdc42.tif','Resolution',300), end

%% Figure 8 EMT - PAR
figure(4)
plot(x, y7, 'Color', blue2, 'LineWidth',1)
hold on
title('Par','FontSize',12)
xlabel('x [µm]','FontSize',12)
ylabel('Concentration [µM]','FontSize',12)
if option==1, exportgraphics(gca,'Fig_8A_Par.tif','Resolution',300), end
if option==3, exportgraphics(gca,'Fig_8B_Par.tif','Resolution',300), end
if option==2, exportgraphics(gca,'Fig_8C_Par.tif','Resolution',300), end

%% Figure 8 EMT - Scribble
figure(5)
plot(x, y9, 'Color', orange2, 'LineWidth',1)
hold on
title('Scribble','FontSize',12)
xlabel('x [µm]','FontSize',12)
ylabel('Concentration [µM]','FontSize',12)
if option==1, exportgraphics(gca,'Fig_8A_Sc.tif','Resolution',300), end
if option==3, exportgraphics(gca,'Fig_8B_Sc.tif','Resolution',300), end
if option==2, exportgraphics(gca,'Fig_8C_Sc.tif','Resolution',300), end

%% Figure 3 Degradation / upregulation profile
figure(6)
xlim([0 length(t)*delta_t])
xlabel('time [s]','FontSize',12)
ylabel('Concentration of active component [%]','FontSize',12)
hold on
if test == 1
plot((1:1:length(t))*delta_t,amount(:),'Color',[0.753, 0.753, 0.753],'LineWidth',2.0) 
end
if test == 2
plot((1:1:length(t))*delta_t,amount(:),'Color',[0.627, 0.627, 0.627],'LineWidth',2.0)
end
if test == 3
plot((1:1:length(t))*delta_t,amount(:),'Color',[0.502, 0.502, 0.502],'LineWidth',2.0)
end
if test == 4
plot((1:1:length(t))*delta_t,amount(:),'Color',[0.376, 0.376, 0.376],'LineWidth',2.0)
end
if test == 5
plot((1:1:length(t))*delta_t,amount(:),'Color',[0.251, 0.251, 0.251],'LineWidth',2.0)
end
if test == 6
plot((1:1:length(t))*delta_t,amount(:),'Color',[0, 0, 0],'LineWidth',2.0)
end
if downregulation==1, exportgraphics(gca,'Fig_3A.tif','Resolution',300), end
if upregulation==1, exportgraphics(gca,'Fig_3B.tif','Resolution',300), end

%% Figure 8 additional - Plotting end state of no and one EMT - Rho, Cdc42, Rac1
figure(7)
if test == 1
plot(x,y1,'Color',green1,'LineStyle','--','LineWidth',2.0);
hold on
plot(x,y2,'Color',purple1,'LineStyle','--','LineWidth',2.0)
plot(x,y3,'Color',red1,'LineStyle','--','LineWidth',2.0)
txt6='No EMT';txt7= '- - - - -';
text(6.5,6.6,txt6,'Color', 'k', 'FontWeight','bold','FontSize',12)
text(8.2,6.6,txt7,'Color', 'k', 'FontWeight','bold', 'FontSize',12, 'Interpreter',"none")
end
if test == 4
plot(x,y1,'Color',green1,'LineWidth',2.0);
hold on
plot(x,y2,'Color',purple1,'LineWidth',2.0)
plot(x,y3,'Color',red1,'LineWidth',2.0)
txt = 'Rac1';txt1='Cdc42';txt2='Rhoa';
txt3 = 'EMT';txt4= '_____';
text(1.8,4,txt,'Color', purple1, 'FontWeight','bold', 'FontSize',12)
text(7,1.3,txt1,'Color', green1, 'FontWeight','bold','FontSize',12)
text(5,1.3,txt2,'Color', red1, 'FontWeight','bold','FontSize',12)
text(6.5,6,txt3,'Color', 'k', 'FontWeight','bold', 'FontSize',12)
text(8.2,6.2,txt4,'Color', 'k', 'FontWeight','bold', 'FontSize',12, 'Interpreter',"none")
end
ylim([0 8])
xlabel('x [µm]','FontSize',12)
ylabel('Concentration [µM]','FontSize',12)

%% Figure 8 additional - Plotting end state of no and one EMT - PAR, Scribble
figure(8)
if test == 1
plot(x,y7,'Color',blue1,'LineStyle','--','LineWidth',2.0);
hold on
plot(x,y9,'Color',orange1,'LineStyle','--','LineWidth',2.0)
txt6='No EMT';txt7= '- - - - -';
text(7.5,0.25,txt6,'Color', 'k', 'FontWeight','bold','FontSize',12)
text(8.8,0.25,txt7,'Color', 'k', 'FontWeight','bold', 'FontSize',12, 'Interpreter',"none")
end
if test == 4
plot(x,y7,'Color',blue1,'LineWidth',2.0);
hold on
plot(x,y9,'Color',orange1,'LineWidth',2.0)
txt = 'Par';txt1='Scribble';
txt3 = 'EMT';txt4= '_____';
text(9,1.1,txt,'Color', blue1, 'FontWeight','bold', 'FontSize',12)
text(0.2,1.15,txt1,'Color', orange1, 'FontWeight','bold','FontSize',12)
text(7.5,0.15,txt3,'Color', 'k', 'FontWeight','bold', 'FontSize',12)
text(8.8,0.19,txt4,'Color', 'k', 'FontWeight','bold', 'FontSize',12, 'Interpreter',"none")
end
ylim([0 1.5])
xlabel('x [µm]','FontSize',12)
ylabel('Concentration [µM]','FontSize',12)

end