import numpy as np
import matplotlib.pyplot as plt
#
# Stokes solution
#
def stokesSolution(z,Ub,omega,nphases,kvisc=1e-6,nondimres=False):
    '''
        This function computes the Stokes (1851) solution for static wall and oscillatory far field velocity Ub
    INPUT
        z:          [Nz x 1 array] vertical coordinate array
        Ub:         [float] Maximum wave orbital velocity in m/s
        omega:      [float] Wave frequency in rad/s .or. 1/s
        nphases:    [integer] Number of wave phases where solution is required
        kvisc:      [float] Kinematic viscosity of the fluid in m^2/s: Default 1e-6
        nondimres:  [Boolean] Non-Dimensional Result Flag: Default False
    OUTPUT
        u:          [Nz x nphases array] Result with the velocity
    '''
    nphases = int(nphases)
    u = np.zeros((len(z),nphases))
    Tw = 2*np.pi/omega
    t = np.arange(Tw/nphases,Tw+(Tw/nphases),Tw/nphases)
    K = np.sqrt(omega/(2*kvisc))
    for i in range(0,len(t)):
        u[:,i] = Ub*np.sin(omega*t[i]) + Ub*np.exp(-K*z)*np.sin(K*z - omega*t[i])

    if(nondimres):
        u =  u/Ub
    else:
        u = u

    return u
#
# Fancy Plot
# 
def fixPlot(thickness=1.5, fontsize=20, markersize=8, labelsize=15, texuse=False, tickSize = 15):
    '''
        This plot sets the default plot parameters
    INPUT
        thickness:      [float] Default thickness of the axes lines
        fontsize:       [integer] Default fontsize of the axes labels
        markersize:     [integer] Default markersize
        labelsize:      [integer] Default label size
    OUTPUT
        None
    '''
    # Set the thickness of plot axes
    plt.rcParams['axes.linewidth'] = thickness    
    # Set the default fontsize
    plt.rcParams['font.size'] = fontsize    
    # Set the default markersize
    plt.rcParams['lines.markersize'] = markersize    
    # Set the axes label size
    plt.rcParams['axes.labelsize'] = labelsize
    # Enable LaTeX rendering
    plt.rcParams['text.usetex'] = texuse
    # Tick size
    plt.rcParams['xtick.major.size'] = tickSize
    plt.rcParams['ytick.major.size'] = tickSize
    plt.rcParams['xtick.direction'] = 'in'
    plt.rcParams['ytick.direction'] = 'in'