function [Polars] = Static(Wing)
    % Function that reads a "Static" folder within wing folder of results 
    % and saves it into structure Polars. 
    % Input: 
    %   - Wing: String with wing folder name
    % Output:
    %   - Polars: Structure with the static results of a wing for both 
    %             locked and free conditions

    

    % Move path to desired "Wing" folder and then move into "Static"
    cd(Wing)
    cd('Static')
    
    % List files in directory
    S = dir;
    L = length(dir);
    N = L-2;
    % Go through each file in the directory
    for i=3:1:L
        file = S(i).name;   % Obtain file name from directory variable
        fid = fopen(file, 'rt');    % Open the file
        tline = fgetl(fid);         % Extract first lin
        headers = strsplit(tline, ',');     % Column titles

        % Scan table
        datacell = textscan(fid, '%f%f%f%f%f%f%f%f%f%f%f%f%f', 'Delimiter',',', 'CollectOutput', 1);
        fclose(fid);                % Close file
        datavalues = datacell{1};   % Save data into array
        datavalues = datavalues(1:end-20,:); % Remove last two rows (saving error)
        Average = mean(datavalues);         % Compute average of each column
        Dev = std(datavalues);              % Compute std of each column
        
        % Get file info from name
        ix=strfind(file,'_');               % Find index where _ appear              
        speed = file(ix(3)+1:1:ix(4)-1);    % Speed [m/s]
        hinge = string(file(ix(4)+1:1:ix(5)-1));    % Hinge condition
        alpha = str2double(file(ix(6)+1:1:ix(7)-1));    % Alpha[º]
        
        Res(i-2).file=file;                 % Save file name
        Res(i-2).speed=str2double(speed);   % Save velocity from file name
        Res(i-2).hinge=hinge;               % Save hinge condition from file name

        % Convert hinge condition into binary variable
        % 0 if "Free" and 1 if "Locked"
        if hinge=='Free'
            Res(i-2).hinge2=0;
        else
            Res(i-2).hinge2=1;
        end
        Res(i-2).alpha=alpha;               % Save angle of attack
        Res(i-2).results = [Average; Dev];  % Save average value and standard deviation
    end
    
    % Find files corresponding to speed = 10m/s
    is = find(arrayfun(@(Res) ismember(10, Res.speed), Res));
    
    % Find files corresponding to free hinge
    ihf = find(arrayfun(@(Res) ismember(0, Res.hinge2), Res));

    % Find files corresponding to locked hinge
    ilf = find(arrayfun(@(Res) ismember(1, Res.hinge2), Res));

    % Initialise results structure
    Polars0.free = [];
    Polars0.locked = [];

    % Check every file at the desired speed and save at the corresponding 
    % data structure (free or locked)
    for i=is
        if (ismember(i,ihf))
            Polars0.free=[Polars0.free; Res(i)];
        elseif (ismember(i,ilf))
            Polars0.locked=[Polars0.locked; Res(i)];
        end
    end

    % Save data from the files
    headers = headers(1:13);        % Save headers
    headers(1) = {'Alpha [º]'};     % Rename 1st header
    headers(13) = {'Trigger [V]'};  % Rename 13th header
    ff = size(Polars0.free,1);      % Number of files "Locked"
    lf = size(Polars0.locked,1);    % Number of files "Free"

    % Initialize tables were data is saved
    Pfree = zeros(2,13,ff);
    Plocked = zeros(2,13,lf);
    
    for i=1:ff
        Pfree(:,:,i) = Polars0.free(i).results;     % Save the values from the table
        Pfree(:,1,i) = Polars0.free(i).alpha;       % Save the angle of attack from the file name
    end
    for i=1:lf
        Plocked(:,:,i) = Polars0.locked(i).results; % Save the values from the table
        Plocked(:,1,i) = Polars0.locked(i).alpha;   % Save the angle of attack from the file name
    end
    
    % Save the data in each structure and put them in ascending order
    [~,Iff] = sort(Pfree(1,1,:));
    Polars.free=Pfree(:,:,Iff);
    [~,Ill] = sort(Plocked(1,1,:));
    Polars.locked=Plocked(:,:,Ill);
    Polars.headers = headers;
    
    % Go back to original folder
    cd ..
    cd ..
end