%% Script for obtaining transects using a contour line based on Vaklodingen
%% Developed by Pim Willemsen
%% 2018 11 15

clear all, close all, clc
cd 'C:\Checkouts\OET\'
addpathfast('C:\Checkouts\OET\');
oetsettings();
cd 'M:\Documents\PhD\Scripts\MATLAB\Chapter1\'

%% Set initial parameters (specified for Paulinapolder)
location = 'Paulinapolder_VegEdgePerp';
disp(['location is ',location])
KB_N = 'KB114_4948'; 	% Set specific Kaartblad North
KB_S = 'KB114_5150'; 	% Set specific Kaartblad South
max_cont = 2.8; 		% Set level at which the diketoe is assumed
Orient = 'south'; 		% Determine whether the marsh is located at the "north" or "south" shore
specific_L = 250; 		% Spacing in between transects
length_transect = 2500; % Length of transect
wave_dir = 344;         % Direction of dominant waves (from North)
marsh_edge = 1.73;      % MHWN, assumed as the vegetation edge
mtl = 0.19;             % MTL (Van der Wal et al.)
mlws = mtl-0.5*4.70;    % MLWS, (minimum spring tide MTL - 1/2*STR; Van der Wal et al.)
Hm0 =  0.5*(2.36+1.93); % Normative wave height
Tm = 0.5*(5.4+4);       % Normative wave period
h = 0.5*(5.98+5.8);     % Normative water level 
Hm0_daily = 0.2;        % Daily wave height (from Callaghan et al.)
Tm_daily = 3;           % Daily wave period (assumed, based on Hu et al.)
h_daily = 0.19+0.5*4.7; % Daily water level (maximum spring tide MTL + 1/2*STR; Van der Wal et al.)
kN_bare = 0.001;        % Nikuradse for tidal flats (Vuik et al. 2019)
kN_saltmarsh = 0.02;    % Nikuradse salt marsh with vegetation (Vuik et al. 2016)
kN_normative = 0.05;    % Nikuradse salt marsh in storm conditions, when stems are broken (Vuik et al. 2019)
Veg_height =  0.24;     % (Vuik et al. 2016)
Veg_nstems =  865;      % (Vuik et al. 2016)
Veg_diamtr =  5.1;      % (Vuik et al. 2016)

Hm0_Tm_h_ref = 'average of 140 and 114';
Hm0Dir_ref = '128';

AI_TB = 3.78709e5;                  % Top boundary of map
AI_BB = 3.73233e5;                  % Bottom boundary of map
AI_LB = 3.6256e4;                   % Left boundary of map
AI_RB = 4.08e4;                     % Right boundary of map
XL1 = 3.8199e4; XL2 = 3.8369e4;     % Coordinates left boundary marsh
YL1 = 3.75373e5; YL2 = 3.75544e5;   % Coordinates left boundary marsh
XR1 = 3.9642e4; XR2 = 3.9674e4; 	% Coordinates right boundary marsh
YR1 = 3.74546e5; YR2 = 3.74788e5;   % Coordinates right boundary marsh

% time series of available years of Vaklodingen
T_start_N = 1;			% Index of first year to use
T_end_N = 41; 			% Index of latest year to use
T_N = 43-1;
time_series_N = [1950 1955 1958:2:1966 1967 1968:2:1988 1989 1990 1992 1994:2007 2010:2014];

T_start_S = 1;			% Index of first year to use
T_end_S = 35; 			% Index of latest year to use
T_S = 36-1;
time_series_S = [1955 1958:2:1966 1968:2:1976 1980:2:1988 1989 1990:2:1994 1996:2005 2008 2011:2014];


%% Load data
%url_grid = ['http://opendap.deltares.nl/thredds/dodsC/opendap/rijkswaterstaat/vaklodingen/vaklodingen' KB '.nc'];
url_grid_N = ['D:\Vaklodingen\vaklodingen' KB_N '.nc'];
url_grid_S = ['D:\Vaklodingen\vaklodingen' KB_S '.nc'];
G_N.x = ncread(url_grid_N,'x'); 	% x lon
G_N.y = ncread(url_grid_N,'y'); 	% y lat
G_N.z = ncread(url_grid_N,'z',[1 1 T_N],[Inf Inf 1]); % z value
nan_matr_N = isnan(G_N.z); 			% data is 0; NaN values are 1

G_S.x = ncread(url_grid_S,'x'); 	% x lon
G_S.y = ncread(url_grid_S,'y'); 	% y lat
G_S.z = ncread(url_grid_S,'z',[1 1 T_S],[Inf Inf 1]); % z value
nan_matr_S = isnan(G_S.z); 			% data is 0; NaN values are 1

Grep_X_N = repmat(G_N.x,1,625);
Grep_Y_N = repmat(G_N.y',500,1);

Grep_X_S = repmat(G_S.x,1,625);
Grep_Y_S = repmat(G_S.y',500,1);

Grep_X = [Grep_X_N Grep_X_S];
Grep_Y = [Grep_Y_N Grep_Y_S];
G.z = [G_N.z G_S.z];

figure; contour(Grep_X,Grep_Y,G.z);
axis equal

%% Create outside boundaries of salt marsh (left and right)

% Left boundary
ML = (YL2-YL1) / (XL2-XL1);
BL = YL1 - ML*XL1;

% Right boundary
MR = (YR2-YR1) / (XR2-XR1);
BR = YR1 - MR*XR1;

% Vectors
X_vec = [min(min(Grep_X)):1:max(max(Grep_X))];
YL_vec = ML.*X_vec + BL;
YR_vec = MR.*X_vec + BR;

%% Find indices of area of interest
AI_log  = Grep_X>AI_LB & Grep_X<AI_RB & Grep_Y>AI_BB & Grep_Y<AI_TB; %logical for area of interest
AI_X = Grep_X;
AI_X(AI_log==0) = NaN;
AI_Y = Grep_Y;
AI_Y(AI_log==0) = NaN;
AI_Z = G.z;
AI_Z(AI_log==0) = NaN;

%% Create contour line
figure;
[C1 h1] = contour(AI_X,AI_Y,AI_Z,[max_cont max_cont],'b'); hold on;
xlim([AI_LB, AI_RB]);
ylim([AI_BB, AI_TB]);

%% Create seperate arrays with contour line segments
C_lines = find(C1(1,:)==max_cont);
C_lines_cum = cumsum(C1(2,C_lines));
for i = 1:length(C_lines)
    C_lines_sep{i} = C1(:,C_lines(i)+1:C_lines_cum(i)+1);
end

%% Connect two parts of contourline when using a contour stretching over 2 KB
% for Paulina the right segments are 1 and 13 (also in this order)
% last contour part is segment 20.
% segment 1 and 13 are merged in new segment 21 and 1 and 13 are emptied
% 
C_lines_sep{21} = [C_lines_sep{1} C_lines_sep{13}];
C_lines_sep{1} = [];
C_lines_sep{13} = [];


%% Intersect left boundary and diketoe contour
for i = 1:length(C_lines_sep)
    if size(C_lines_sep{i},2)>=2
        [x0{i}, y0{i}, iout0{i},jout0{i}] = intersections(C_lines_sep{i}(1,:),...
            C_lines_sep{i}(2,:),X_vec,YL_vec);
        if isempty(x0{i})
            x0{i} = NaN;
            y0{i} = NaN;
        end
        x0_vec(i) = x0{i};
        y0_vec(i) = y0{i};
    end
end

plot(X_vec,YL_vec,'k');
plot(x0_vec,y0_vec,'xr');

%% Intersect right boundary and diketoe contour
for i = 1:length(C_lines_sep)
    if size(C_lines_sep{i},2)>=2
        [x1{i}, y1{i}, iout1{i},jout1{i}] = intersections(C_lines_sep{i}(1,:),...
            C_lines_sep{i}(2,:),X_vec,YR_vec);
        if isempty(x1{i})
            x1{i} = NaN;
            y1{i} = NaN;
        end
        x1_vec(i) = x1{i};
        y1_vec(i) = y1{i};
    end
end

plot(X_vec,YR_vec,'k');
plot(x1_vec,y1_vec,'xr');


%% find intersection closest to salt marsh
y0_vec(y0_vec==0) = NaN;
y1_vec(y1_vec==0) = NaN;
if strcmp(Orient,'north') == 1
    [y0_vec_repr,y0_vec_repr_i] = min(y0_vec);
    [y1_vec_repr,y1_vec_repr_i] = min(y1_vec);
elseif strcmp(Orient,'south') == 1
    [y0_vec_repr,y0_vec_repr_i] = max(y0_vec);
    [y1_vec_repr,y1_vec_repr_i] = max(y1_vec);
end

%% Construct contourline part in between left and right boundary
C_lines_select = C_lines_sep{y0_vec_repr_i}(:,...
    ceil(min([iout0{y0_vec_repr_i},iout1{y1_vec_repr_i}])):...
    floor(max([iout0{y0_vec_repr_i},iout1{y1_vec_repr_i}])));

plot(C_lines_select(1,:),C_lines_select(2,:),'.b','linewidth',3)

windowSize = 20;
b = (1/windowSize)*ones(1,windowSize);
a = 1;
C_lines_select_smooth(1,:) = filter(b,a,C_lines_select(1,:));
C_lines_select_smooth(2,:) = filter(b,a,C_lines_select(2,:));


%% calculate length of line segments in selected contour line part
for i = 1:length(C_lines_select)-1
    dx(i) = C_lines_select(1,i+1)-C_lines_select(1,i);
    dy(i) = C_lines_select(2,i+1)-C_lines_select(2,i);
    d(i) = sqrt(dx(i)^2 + dy(i)^2);
end

d_tot = sum(d);
d_cum = cumsum(d);
disp(['total length of coastline is ',num2str(d_tot),'m']);

%% set number of transects
N_transects = floor((d_tot-0.5*specific_L) / specific_L);
disp(['spacing of transects is set to ',num2str(specific_L),'m']);
disp(['number of transects is ',num2str(N_transects)]);

%% Create lines with the same angles as dominant wind direction
wave_dir_rad = deg2rad(wave_dir);

for i = 1:N_transects
    perp_i(i)       = find((d_cum>(i*specific_L+(0.5*specific_L))),1,'first');

    trans_dir_rad(i) = wave_dir_rad-pi();
    Y_length(i) = cos(trans_dir_rad(i))*length_transect;
    X_length(i) = sin(trans_dir_rad(i))*length_transect;

    KML_dat_xy{i}(1,1)      = C_lines_select_smooth(1,perp_i(i));
    KML_dat_xy{i}(2,1)      = C_lines_select_smooth(2,perp_i(i));
    
    KML_dat_xy{i}(1,2)      = C_lines_select_smooth(1,perp_i(i))-X_length(i);
    KML_dat_xy{i}(2,2)      = C_lines_select_smooth(2,perp_i(i))-Y_length(i);  
        
    slope_trans(i) = (KML_dat_xy{i}(2,2)-KML_dat_xy{i}(2,1))/...
        (KML_dat_xy{i}(1,2)-KML_dat_xy{i}(1,1));
    perp_Y{i} = (slope_trans(i))*(X_vec-C_lines_select_smooth(1,perp_i(i)))+...
        C_lines_select_smooth(2,perp_i(i));
    
    
    [lon,lat]=convertCoordinates(KML_dat_xy{i}(1,:) ,KML_dat_xy{i}(2,:) ,...
        'CS1.code',28992,'CS2.code',4326);
    KML_dat_latlon{i}(1,:)  = lon;
    KML_dat_latlon{i}(2,:)  = lat;
    
    plot(X_vec,perp_Y{i},'--k');
    plot(KML_dat_xy{i}(1,:),KML_dat_xy{i}(2,:),'or','linewidth',2);
end

xlabel('x coordinate [m]')
ylabel('y coordinate [m]')
title({['location is ',location];['spacing between transects is ',num2str(specific_L),'m']});

%% Adapted Vaklodingen2Transects script

% vaklodingen2transects.m
%
% This program creates a KML file for the Vaklodingen. Bathymetric
% profiles are drawn at the Earth surface connecting the defined
% coordinates.
% .kml file and .mat file are produced for all the years.
% The figure on matlab is relative to the specified period.
% Transects of the Western Scheldt have been defined manually on
% GoogleEarth and then saved on "myplaces".
% The function KMLline is called to create the lines in Google Earth
% The function convertCoordinates is called to change coordinate
% system.
%
% --------------------------------------------------------------------
% Copyright (C) 2010 Deltares
%       Giorgio Santinelli
%
%       Giorgio.Santinelli@deltares.nl
%
%       Deltares
%       P.O. Box 177
%       2600 MH Delft
%       The Netherlands
%
% See also: KMLline, convertCoordinates, googlePlot
% --------------------------------------------------------------------

nametrans   = location;
startdate   = '01-Jan-1950';
enddate     = datestr(now, 'dd-mmm-yyyy');

%set coordinates in inputmatrix for getting Vaklodingen data
GE_lonlat = zeros(5,N_transects);
for j = 1:N_transects
    GE_lonlat(1,j) =  KML_dat_latlon{1,j}(1,1);
    GE_lonlat(2,j) =  KML_dat_latlon{1,j}(2,1);
    GE_lonlat(4,j) =  KML_dat_latlon{1,j}(1,2);
    GE_lonlat(5,j) =  KML_dat_latlon{1,j}(2,2);
end

for j = 1:size(GE_lonlat,2)
    j=j
    % Define starting point (lon1, lat1) and ending point (lon2, lat2) in
    % sexagesimal DMS, coordinate system: 'WGS 84'
    lon1 = GE_lonlat(1,j); lat1 = GE_lonlat(2,j); % deg
    lon2 = GE_lonlat(4,j); lat2 = GE_lonlat(5,j); %deg
    % lat1 = 51.2407; lon1 = 3.3047; %
    % lat2 = 51.2801; lon2 = 3.3159; %
    
    % 1D Resolution of transect
    stepsize = 5; % metres
    
    
    % Read the path on pc-drive
    ncpath = ['D:\Vaklodingen\']; % Rijkswaterstaat data
    
    % Read the path from the opendap
    %ncpath = 'http://opendap.deltares.nl/thredds/catalog/opendap/rijkswaterstaat/vaklodingen/catalog.html';
    ncfile = '*.nc';
    fss = opendap_catalog(ncpath);
    fns = dir(fullfile(ncpath,ncfile));
  
    % Convert coordinate system from 'WGS 84' (code:4326) to
    % 'Amersfoort / RD New' (code:28992)
    [x1, y1] = convertCoordinates(lon1, lat1, 'CS1.code', '4326', 'CS2.code', '28992');
    [x2, y2] = convertCoordinates(lon2, lat2, 'CS1.code', '4326', 'CS2.code', '28992');
    
    % Define distances (dx, dy), length, number of steps (based on stepsize),
    % transect points in 'WGS 84' (X_0, Y_0) and 'Amersfoort / RD New' (lon_0, lat_0)
    dx = x2-x1; dy = y2-y1;
    len = sqrt(dx^2+dy^2);
    nstep = len/stepsize;
    stepsize_x = dx/nstep; stepsize_y = dy/nstep;
    X_0 = x1:stepsize_x:x1+((nstep)-1)*stepsize_x;
    Y_0 = y1:stepsize_y:y1+((nstep)-1)*stepsize_y;
    [lon_0, lat_0] = convertCoordinates(X_0, Y_0, 'CS1.code', '28992', 'CS2.code', '4326');
    
    
    % Find related KB (urlcross) and date of measurement (date_KB)
    i_KB = 1;
    date_KB = [];
    for i = 2:length(fns)
        i = i
        url = fullfile(ncpath, fns(i).name); % path on pc-drive
%         url = fns{i}; % path from the opendap
        lon = nc_varget(url, 'lon');
        lat = nc_varget(url, 'lat');
        
        if ismember(true, ((min(lon1,lon2)<lon)&(max(lon1,lon2)>lon))) &&...
                ismember(true, ((min(lat1,lat2)<lat)&(max(lat1,lat2)>lat)));
            urlcross{i_KB} = url;
            date{i_KB} = nc_cf_time(url, 'time');
            date_i = date{i_KB};
            date_KB = union(date_KB, date_i);
            i_KB = i_KB+1;
        end
    end
    
    % remove first year if not sufficient data is present
    date_KB = date_KB(2:end);
    
    disp('Related KaartBladen and dates of measurement found.')
    
    % Create folder in the directory
    mkdir('GE_Transects')
    
    % Read bathymetry for the related KB, for every year
    stepX_0 = 0:stepsize:((nstep)-1)*stepsize;
    for k = 1:length(date_KB) % Cycle of dates of measurement
        z_0{k} = [];
        x_0{k} = [];
        
        for i = 1:(i_KB-1) % Cycle of the related KB
            [tf_date n_year] = ismember(date_KB, date{i});
            
            if tf_date(k) % Make it if, in date_KB, the related KB (urlcross) has data.
                Z{i} = squeeze(nc_varget(urlcross{i}, 'z', [n_year(k)-1,0,0], [1,-1,-1]));
                y{i} = nc_varget(urlcross{i}, 'y');
                x{i} = nc_varget(urlcross{i}, 'x');
                [X{i} Y{i}] = meshgrid(x{i}, y{i}); % Create the grid for interp2
                
                xline{i} = 0:stepsize:((nstep)-1)*stepsize;
                zline{i} = interp2(X{i}, Y{i}, Z{i}, X_0, Y_0);
                
                dumz = (zline{i})';
                z_0{k} = vertcat(z_0{k}, dumz);
                dumx = (xline{i})';
                x_0{k} = vertcat(x_0{k}, dumx);
            end
        end
        
        z_nnan{k} = nan(size(z_0{k}));
        x_nnan{k} = nan(size(z_0{k}));
        
        if ~isempty(z_0{k}(~isnan(z_0{k})))
            z_nnan{j,k} = z_0{k}(~isnan(z_0{k}));
            x_nnan{j,k} = x_0{k}(~isnan(z_0{k}));
            z_kml{j,k} = interp1(x_nnan{j,k}, z_nnan{j,k}, stepX_0); % the cell variable of the bathymetry along the defined transect
            x_kml{j,k} = stepX_0;
        else
            z_kml{j,k} = nan(size(stepX_0));
            x_kml{j,k} = nan(size(stepX_0));
        end
        
    end
    clear dumx dumz
    
    disp('z_kml has been created successfully. ')
    
    % Define timeIn and timeOut for Google Earth
    timeIn = date_KB;
    timeOut = date_KB(2:end);
    timeOut(end+1,1) = date_KB(end)+1;
    
    % Create matrix z_KML for KMLline
    for k = 1:length(date_KB)
        z_KML(k,:) = z_kml{j,k};
    end
      
    % Create matrixes lat_0 and lon_0 for KMLline
    lat_0 = repmat(lat_0, size(z_KML,1), 1);
    lon_0 = repmat(lon_0, size(z_KML,1), 1);
    
    lat_0_matr{j} = lat_0;
    lon_0_matr{j} = lon_0;
    
    % Make KML file
    fname = ['GE_Transects\',nametrans,'_S',num2str(specific_L),...
        '_L',num2str(length_transect),'_',num2str(j),'.kml'];
    KMLline(lat_0', lon_0', z_KML', 'timeIn',timeIn,'timeOut',timeOut,...
        'fileName', fname, 'lineColor',jet(size(z_KML,1)),'lineWidth',2,...
        'fillColor',jet(size(z_KML,1)),'zScaleFun', @(z_KML)(z_KML+60)*10);
    
    % KB within the specified period
    date_KB_Tnan = date_KB(date_KB >= datenum(startdate) & date_KB <= datenum(enddate));
    [dummy, date_KB_index] = ismember(date_KB_Tnan, date_KB);
    
    % Bathymetries ~NaNs within the specified period
    kk = 1;
    for tt = 1:length(date_KB_index)
        if ~isequalwithequalnans(z_kml{j,date_KB_index(tt)},NaN(1,length(stepX_0)))
            date_KB_Time(kk) = date_KB_Tnan(tt);
            kk = kk+1;
        end
    end
    [dummy, date_KB_index] = ismember(date_KB_Time, date_KB);
    
    % Plot bathymetry within the specified period
    figure;
    col = jet(length(date_KB_Time));
    for kk = 1:length(date_KB_index)
        plot(x_kml{j,date_KB_index(kk)},z_kml{j,date_KB_index(kk)},'Color',col(kk,:),'LineWidth', 1.5)
        grid on
        hold on
    end
    legend(datestr(date_KB_Time));
    set(gcf,'Position',[5 35 1040 760])
    set(gca,'FontSize',12)
    title(['Transect from (',num2str(lat1,'%5.2f'),', ',num2str(lon1,'%5.2f'),...
        ') to (',num2str(lat2,'%5.2f'),', ',num2str(lon2,'%5.2f'),')'],...
        'FontSize',12,'FontWeight','bold')
    xlabel('Distance along transect,  x [m]','FontSize',12)
    ylabel('Bed level,  z_b [m]','FontSize',12)

    % Interpolate data near diketoe
    timevec{j} = datetime(date_KB,'convertfrom','datenum');
    if j == 1 
        unique_meas{j} = [1:31 33:42];
        % not: 32
    end
    if  j == 2 || j == 3 || j == 4 || j == 5 || j == 6 || j == 7
        unique_meas{j} = [1:31 33:36 38:43];
        % not: 32 37
    end
    timevec_used{j} = datetime(date_KB(unique_meas{j}),'convertfrom','datenum'); 
    for i = 1:length(lat_0)
        T_kml_matrix{j}(:,i) = year(timevec_used{j})';
    end
    for i = 1:length(unique_meas{j})
        X_kml_matrix{j}(i,:) = x_kml{j,1};
    end
    
    if isnan(z_KML(1,1)) == 1
        z_KML(1,1) = max_cont;
        z_KML(1,:) = interp1(X_kml_matrix{j}(1,~isnan(z_KML(1,:))),...
            z_KML(1,~isnan(z_KML(1,:))),X_kml_matrix{j}(1,:));
    end
    
     z_KML_unique = z_KML(unique_meas{j},:);
    
    for i = 1:length(lat_0)
        z_kml_matrix_interpT{j}(:,i) = interp1(T_kml_matrix{j}(~isnan(z_KML_unique(:,i)),i),...
            z_KML_unique(~isnan(z_KML_unique(:,i)),i),T_kml_matrix{j}(:,i));
    end
    
    z_kml_matrix_interpT{j}(isnan(z_kml_matrix_interpT{j})) = 50;
    
    % plot interpolated data 
    figure;
    SPlot = surf(X_kml_matrix{j},T_kml_matrix{j},z_kml_matrix_interpT{j}-100);
    hold on;
    set(SPlot,'edgecolor','none');
    shading interp
    view(2)
    caxis([-10-100 5-100]);
    
    [cont_interpT,H] = contour(X_kml_matrix{j},T_kml_matrix{j},z_kml_matrix_interpT{j},[-5:1:5],'ShowText','on','linecolor',[0.5, 0.5, 0.5],'linewidth',1);
    [cont_interpT,H] = contour(X_kml_matrix{j},T_kml_matrix{j},z_kml_matrix_interpT{j},[marsh_edge marsh_edge],'ShowText','on','linecolor',[0, 0, 0],'linewidth',2);
    [cont_interpT,H] = contour(X_kml_matrix{j},T_kml_matrix{j},z_kml_matrix_interpT{j},[max_cont max_cont],'ShowText','on','linecolor',[0, 0, 0],'linewidth',2);
    
    for k = 1:length(unique_meas{j})
        MLWS_length(k,j) = find(z_kml_matrix_interpT{1,j}(k,:)<mlws,1,'first');
    end
    MLWS_length_sum(j) = max(MLWS_length(:,j));
    
    % Set values impossible for interpolation to NaN
    z_kml_matrix_interpT{1,j}(z_kml_matrix_interpT{1,j}==50) = NaN;
    
    % Extrapolate values at dike if not present
%     z_kml_matrix_interpT{1,j} = fillmissing(z_kml_matrix_interpT{1,j},'linear',2);
    z_kml_matrix_interpT{1,j} = inpaint_nans(z_kml_matrix_interpT{1,j},3);
    
    for k = 1:length(unique_meas{j})
        
        % Set "to deep" data to NaN
        z_kml_matrix_interpT{1,j}(k,MLWS_length_sum(j):end) = NaN;
        
        %create vegetation field
        vegetation{1,j}(k,:) = z_kml_matrix_interpT{1,j}(k,:);
        vegetation{1,j}(k,(z_kml_matrix_interpT{1,j}(k,:)>marsh_edge)) = 999999;
        vegetation{1,j}(k,(z_kml_matrix_interpT{1,j}(k,:)<marsh_edge)) = 0;
        vegetation{1,j}(k,(vegetation{1,j}(k,:)==999999)) = 1;
        vegetation{1,j}(k,(z_kml_matrix_interpT{1,j}(k,:)==999)) = 999;
        
        %create roughness field for daily
        roughness_daily{1,j}(k,:) = z_kml_matrix_interpT{1,j}(k,:);
        roughness_daily{1,j}(k,(z_kml_matrix_interpT{1,j}(k,:)>marsh_edge)) = 999999;
        roughness_daily{1,j}(k,(z_kml_matrix_interpT{1,j}(k,:)<marsh_edge)) = 0.001;
        roughness_daily{1,j}(k,(roughness_daily{1,j}(k,:)==999999)) = 0.02;
        roughness_daily{1,j}(k,(z_kml_matrix_interpT{1,j}(k,:)==999)) = 999;
        
        %create roughness field for normative
        roughness_norm{1,j}(k,:) = z_kml_matrix_interpT{1,j}(k,:);
        roughness_norm{1,j}(k,(z_kml_matrix_interpT{1,j}(k,:)>marsh_edge)) = 999999;
        roughness_norm{1,j}(k,(z_kml_matrix_interpT{1,j}(k,:)<marsh_edge)) = 0.001;
        roughness_norm{1,j}(k,(roughness_norm{1,j}(k,:)==999999)) = 0.05;
        roughness_norm{1,j}(k,(z_kml_matrix_interpT{1,j}(k,:)==999)) = 999;
    end
    timeIn = date_KB(1:end);
    timeOut = date_KB(2:end);
    timeOut(end+1,1) = date_KB(end)+1;
    KMLline(lat_0(1,1:MLWS_length_sum(j))', lon_0(1,1:MLWS_length_sum(j))', z_KML(1,1:MLWS_length_sum(j))'.*0, 'timeIn',timeIn,'timeOut',timeOut,...
        'fileName', fname, 'lineColor',[0,0,0],'lineWidth',2,...
        'fillColor','none','zScaleFun', @(z_KML)(z_KML));
    
    clear z_KML
    clear trans_length_temp
end

Paulinapolder_S250.location          = location;
Paulinapolder_S250.Z                 = z_kml_matrix_interpT;
Paulinapolder_S250.Veg               = vegetation;
Paulinapolder_S250.roughness_daily   = roughness_daily;
Paulinapolder_S250.roughness_norm    = roughness_norm;
Paulinapolder_S250.grid_size_m       = 5;
Paulinapolder_S250.T                 = timevec_used;
Paulinapolder_S250.spacing           = specific_L;
Paulinapolder_S250.MLWS              = mlws;
Paulinapolder_S250.veg_edge          = marsh_edge;
Paulinapolder_S250.wave_direction    = wave_dir;
Paulinapolder_S250.dike_toe          = max_cont;
Paulinapolder_S250.Hm0_norm          = Hm0;
Paulinapolder_S250.Tm_norm           = Tm;
Paulinapolder_S250.h_norm            = h;
Paulinapolder_S250.Hm0_daily         = Hm0_daily;
Paulinapolder_S250.Tm_daily          = Tm_daily;
Paulinapolder_S250.h_daily           = h_daily;
Paulinapolder_S250.Hm0_Tm_h_ref      = Hm0_Tm_h_ref;
Paulinapolder_S250.Hm0Dir_ref        = Hm0Dir_ref
Paulinapolder_S250.Veg_height        = Veg_height;
Paulinapolder_S250.Veg_nstems        = Veg_nstems;
Paulinapolder_S250.Veg_diamtr        = Veg_diamtr;
Paulinapolder_S250.kN_bare           = kN_bare;
Paulinapolder_S250.kN_marsh_norm     = kN_normative;
Paulinapolder_S250.kN_marsh_daily    = kN_saltmarsh;

fprintf(['The file ',nametrans,'_L',num2str(specific_L),'.kml has been saved on the folder ''GE_Transects''\n']);
fprintf(['please save the matfile manually''\n']);









