function [solpoints] = MembraneModelFunction(F1,L1,L2,rad,P,k,alpha,D1,AC,S,plots,feedback,boundaryreaction)

%% Constants
%---------
%Input constants:
%F1 = Flowrate (g/h)
%L1 = Boundary layer length/thickness (m)
%L2 = Membrane length/thickness (m)
%rad = Membrane radius (m)
%P = Porosity of the membrane
%k = Surface reaction rate (m/s)
%alpha = Concentration ratio at boundary
%D1 = Diffusion coefficient in the liquid (m2/s)
%AC = Absorption coefficient (1/m)
%S = Specific surface area of the catalyst (m2/m3)
%plots = a boolean to indicate whether plots will be outputted or not
%feedback = a boolean to indicate whether stats are outputted from the bvpsolver
%boundaryreaction = a boolean to indicate whether the reaction flux at the boundary will be taken into account in the flux balance

%Conversion of input constants
Area = (rad^2)*pi;                      %Surface area membrane (m^2)
u1 = (F1*(1e-6/3600))/Area;             %Velocity pre-membrane (m/s) Shu:F1 [mL/h]*(1m3/1e6mL*1h/3600s]
u2 = u1/P;                              %Velocity in membrane (m/s) Assuming cylindrical pores
D2 = D1*P;                              %Diffusion coefficient membrane (m2/s)
Pe_1 = u1*L1/D1;                        %Peclet number before membrane
Pe_2 = u2*L2/D2;                        %Peclet number in membrane

%% Solve BVP

%Define the xmesh and its data point density the BVPsolver will use
xmesh = linspace(0,1,101);

%initialize a solution on the xmesh (c = 1 on both the start of the liquid and membrane and dc/dx is 0 on both)
solinit = bvpinit(xmesh, @guess);

%Check for feedback flag and toggle appropriately
if feedback == true
    options = bvpset('RelTol',1e-3,'Stats','on','AbsTol',1e-3);
else
    options = bvpset('RelTol',1e-3,'Stats','off','AbsTol',1e-3);
end

%bvp5c to solve this boundary value problem
sol = bvp5c(@bvpfcn, @bcfcn, solinit, options);  

%These solutions are returned to the script that calls this function
solpoints = [sol.y(1,1),sol.y(1,end)/P,sol.y(3,1)/P,sol.y(3,end)/P];

%Check for plots flag and output plots if 1
if plots == true
    figure('Name' , 'Danckwerts'); hold on; grid on;
    plot(sol.x, sol.y(1,:), 'blue',Linewidth=3)                                                             %plot of the concentration profile fluid
    plot([sol.x(end),sol.x(end)],[sol.y(1,length(sol.y)),sol.y(3,1)],'--g',Linewidth=3)                     %plot of the concentration gap (if present)
    plot((sol.x+1), sol.y(3,:),'red',Linewidth=3)                                                           %plot of the concentration profile membrane
    legend(' C in Boundary layer',' Concentration gap',' C in Membrane','location','SouthWest')
    xlabel({'Dimensionless axial coordinate','X = x/L1 for 0<X<1 and X = x/L2 for 1<X<2'});
    ylabel({'Dimensionless concentration';'C = c/c_i_n'});
    title(['Concentration profile at F1 = ' num2str(F1) ' mL/h'])
    ylim([0 1]); xlim([0 2])
    h = gca; set(h,'fontsize',24,'FontName','GillSansMT');
    hold off
end


%Define the differential equations of this system
function dydx = bvpfcn(x,y)
    dydx = [y(2)                                                %C1'  = dC1/dX
            Pe_1*y(2)                                           %C1'' = Pe_1*dC1/dX
            y(4)                                                %C2'  = dC2/dX    
            Pe_2*y(4)+(k*exp(-x*L2*AC))*S*L2^2/D2*y(3)];        %C2'' = Pe_2*y(4)+((k*exp(-X*L2/LDL)*S*L2^2)/D2)*C2
end

%Define the boundary conditions of this system depending on boundaryreaction flag
function res = bcfcn(ya,yb)
    if boundaryreaction == false
        res =  [ya(1) - 1                                                               %Inlet concentration = 1
                P*(yb(1)-(1/Pe_1)*yb(2)) - (ya(3)-(1/Pe_2)*ya(4))                       %Flux continuity at membrane surface: P*(C1-(1/Pe_1)*dC1/dX) == C2-(1/Pe_2)*dC2/dX
                ya(3)-yb(1)*alpha                                                       %Concentration continuity at membrane surface C1*alpha = C2
                yb(4)];                                                                 %dC2/dX = 0 at the end of the membrane
    else
        res =  [ya(1) - 1                                                               %Inlet concentration = 1
                %P*(yb(1)-(1/Pe_1)*yb(2)) - (ya(3)-(1/Pe_2)*ya(4)+(k/u2)*ya(3)*(1-P) %Flux continuity at membrane surface (including reaction): P*(C1-(1/Pe_1)*dC1/dX) == C2-(1/Pe_2)*dC2/dX+(k/u2)*C2*(1-P)
                P*(yb(1)-(1/Pe_1)*yb(2)) - (ya(3)-(1/Pe_2)*ya(4)+(k/u2)*ya(3)*(1-P)) %shu Change to alpha but it is P
                ya(3)-yb(1)*alpha                                                       %Concentration continuity at membrane surface C1*alpha = C2
                yb(4)];                                                                 %dC2/dX = 0 at the end of the membrane
    end
end

function g = guess(x)
    g =    [1      %c1 = 1 at x = 0 (begin fluid)
            0      %dc1 = 0 at x = 0 (begin fluid)
            1      %c2 = 1 at x = 0 (begin membrane)
            0];    %dc2 = 0 at x = 0 (begin membrane)
    end
end
