%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%                                                           %%%  
%%% fit unique value for K and AC for each of the experiments %%%
%%%                                                           %%%  
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Script takes a long time to run, but calculated values can  
% be inserted into the script so it runs considerably faster
% In order to run script using previously calculated values, set
% justplotfinal to true

clear all; close all; clc; format compact;
set(0,'DefaultLegendAutoUpdate','off')
%set(0,'DefaultFigureWindowStyle','docked')
set(0,'DefaultFigureWindowStyle','normal')

global alpha rad D1 P L1 boundaryreaction Fmeasurements S   % General globals used in all scripts
global L2_fit tofit                                         % Globals specific for this script
global itK Ktest AC                                         % Globals required for the fit of k
global itAC ACtest AnsmindevK mk                            % Globals required for the fit of AC

%% script control
basecaseonly     = false ;                  % Do not fit any datapoints - just generatre profiles or plot graphs for the set variables
comparebasecase  = true  ;                  % Compare set parameters manually to chosen measurement (see below)
plots            = false ;                  % Plot individual membrane graph for each flowrate
feedback         = false ; %                 % Show bvp solver statistics of MembraneModelFunction.m
boundaryreaction = false ;                  % Include boundary reaction flux in flux boundary condition
Kfit             = true  ;                  % Determine the fit for K using LSQ
ACfit            = false ;                  % Determine the fit for AC using LSQ (after fitting K)
justplotfinal    = false ;                 % Can be used if Kfit and ACfit have previously been determined and fittings are turned off

% Choose for which membrane measurement you want to get a fit for K (and AC)
Measurements = {'d21_9' 'd10_6' 'd4_3' 'd3_4' 'd2_74' 'd1_69' 'd1_17' 'd0_65' 'd0_26'};
Measurement = 'd3_4';                      % Chosen measurement - can be changed as desired

% The number of datapoints in the model can be adjusted if desired
Fcomparison = [0.001:.25:12];                    % Values of the flow rate to plot in comparing graphs
%Fcomparison = [1 2 4 6 8 10];               % Default to match measurement points

%% Variables (set values for base case)
Fmeasurements = [1 2 4 6 8 10];             % g/h (assumed to be equal to ml/h)
k = 0.3e-7;                                  % Surface reaction rate (m/s)
alpha = 1;                                  % Concentration ratio at the boundary
L2 = 3.44E-6;                               % Membrane thickness (m)
AC = 1.073E6;                               % Absorption coefficient (1/m)
maxdev_AC = 0.25;                           % Maximum deviation from literary AC when fitting

%% constants
rad = 0.01;%20/2/1000;                 % Radius membrane (m) %Shu: I changed this value
D1 = 4.6e-10;                           % Diffusion coefficient pre-membrane (m2/s)%Shu: I changed from the literature value
P = 0.45;                               % Porosity
L1 = 0.01; %1/100;                    % Thickness of the mass transfer boundary layer (m)%Shu: I changed this value
S = 192802500;                               % Specific surface area of the catalyst (m2/m3)

%% Importing Experimental data
Data = readmatrix('dataV2.xlsx','range','A1:K91');
All_L2 = [21.9 10.6 4.3 3.4 2.74 1.69 1.17 0.65 0.26]*1e-6; %(m)
nmeasurements = length(All_L2);

for n = 1:nmeasurements
    idx1 = 10*n-10;
    nrepeats(n) = find(isnan(Data(5+idx1,:)),1)-1; % Number of (experimental repeats+1) for 1 membrane
    Exp_Data{n}=Data([5:10]+idx1,[2:nrepeats(n)]);
    %Exp_Data{n}
end

%% Base case: Calculation of concentration as function of flow rate for given K AC alpha values
for n = 1:length(Fcomparison)
    Cpoints(n,:) = MembraneModelFunction(Fcomparison(n),L1,L2,rad,P,k,alpha,D1,AC,S,plots,feedback,boundaryreaction);
end

% Plotting results from base case with set K
figure(); hold on; grid on
plot(Fcomparison,Cpoints(:,1),'-*k','Markersize',16,'Linewidth',3);
plot(Fcomparison,Cpoints(:,2),'-ok','Markersize',16,'Linewidth',3);
plot(Fcomparison,Cpoints(:,3),'-xk','Markersize',16,'Linewidth',3);
plot(Fcomparison,Cpoints(:,4),'-dk','Markersize',16,'Linewidth',3);

legend('Cbulk','Cmem-','Cmem+','Cout','fontsize', 20, 'Location', 'southeast');
h = gca; set(h,'fontsize',24,'FontName','GillSansMT');
ylabel({'Dimensionless concentration [C = c/c_i_n]'});
xlabel('Flow rate [mL/h]')
title('Concentrations within system (base case)')

% Comparing base case with experimental data
idx2 = find(contains(Measurements,Measurement));
mstring = '*xods';
legendset = [];

if comparebasecase == true && justplotfinal == false
    figure(); hold on; grid on
    plot(Fcomparison,Cpoints(:,4),'.-b','MarkerSize',35,'LineWidth',3);

    for m = 1:nrepeats(idx2)-1
        plot(Fmeasurements,Exp_Data{idx2}(:,m),'--k','Marker',mstring(m),'MarkerSize',16,'LineWidth',3);
        legendset = [legendset, {[' Experiment ' num2str(m)]}];
    end

    legend([' Model',legendset],'fontsize', 20, 'Location', 'southeast');
    h = gca; set(h,'fontsize',24,'FontName','GillSansMT');
    ylim([0  1]); ylabel({'Dimensionless concentration [C = c/c_i_n]'}); 
    xlim([1 10]); xlabel('Flow rate [mL/h]');
    title('Comparing Cout with experiments (base case)');
    text(1.1,.9925,{['k0      = ' num2str(k) ' m/s'];['alpha = ' num2str(alpha)];['L2      = ' num2str(L2*1e6) ' μm'];['AC     = ' num2str(AC/1e6) '\times10\^6/m'];['P       = ' num2str(P)]},'fontsize', 20,'HorizontalAlignment', 'left','VerticalAlignment', 'top');
    text(10*0.9925,1*0.9925,'Note that k has been set and not fit using fminsearch','fontsize', 20,'HorizontalAlignment', 'right','VerticalAlignment', 'top')

elseif comparebasecase == false
    figure(); hold on; grid on
    plot(Fcomparison,Cpoints(:,4),'.-b','MarkerSize',35,'LineWidth',3);
    legend(' Model','fontsize', 20, 'Location', 'southeast');
    h = gca; set(h,'fontsize',24,'FontName','GillSansMT');
    ylim([0  1]); ylabel({'Dimensionless concentration [C = c/c_i_n]'}); 
    xlim([1 10]); xlabel('Flow rate [mL/h]');
    title('Plotting Cout (base case)');
    text(1.1,.9925,{['k0      = ' num2str(k) ' m/s'];['alpha = ' num2str(alpha)];['L2      = ' num2str(L2*1e6) ' μm'];['AC     = ' num2str(AC/1e6) '\times10\^6/m'];['P       = ' num2str(P)]},'fontsize', 20,'HorizontalAlignment', 'left','VerticalAlignment', 'top');
    text(10*0.9925,1*0.9925,'Note that k has been set and not fit using fminsearch','fontsize', 20,'HorizontalAlignment', 'right','VerticalAlignment', 'top')
end

if basecaseonly == 1
    warning('Basecaseonly = true. No fitting of data')
    % In order to fit data, use Basecaseonly = false at top of script
    return
end

%% Optimization K fit as before
if Kfit == true && justplotfinal == false

Ktest = [];
%options = optimset('Display','iter','TolFun',1e-7,'TolX',1e-7,'MaxFunEvals',150);
% Can be used as alternative to display iteration results
options = optimset('Display','none','TolFun',1e-9,'TolX',1e-9,'MaxFunEvals',150);

for mk = 1:length(All_L2)
    itK = 0;
    L2_fit = All_L2(mk);
    tofit = Exp_Data{mk};
%    AnsmindevK(mk) = fminsearchbnd(@balanceK,k,0,+inf,options); % searches for the K giving the lowest deviation from the datapoints
    Y = reshape(tofit,[],1);
    X = repmat(Fmeasurements', size(tofit,2),1);
    [beta,r,J,cov,mse] = nlinfit(X, Y, @fitK, k);
    ci = nlparci(beta,r,'Jacobian',J);
    
    AnsmindevK(mk) = beta;
    
    disp(['LSQ fit value for k: ' num2str(AnsmindevK(mk)), '(95% confidence interval: ',num2str(ci),')'])    
%    disp(['LSQ fit value for k: ' num2str(f)])   
    
    for n = 1:length(Fcomparison) % Compare the found optimal value of k with experimental data
        Cpoints(n,:) = MembraneModelFunction(Fcomparison(n),L1,L2_fit,rad,P,AnsmindevK(mk),alpha,D1,AC,S,plots,feedback,boundaryreaction);
    end

    % plotting comparison
    figure(); hold on; grid on 
            plot(Fcomparison,Cpoints(:,4),'-b','MarkerSize',35,'LineWidth',3);
            savefilename = "experiment_"+int2str(mk)+"_repetition_"+int2str(m);
            Csave = Cpoints(:,4);
            save(savefilename,'Fcomparison','Csave');
        for m = 1:nrepeats(mk)-1
            %plot(Fmeasurements,Exp_Data{mk}(:,m),'--k','Marker',mstring(m),'MarkerSize',16,'LineWidth',3);
            plot(Fmeasurements,Exp_Data{mk}(:,m),'k','Marker',mstring(m),'MarkerSize',16);
        end
    
        legend([' Model',legendset],'fontsize', 20, 'Location', 'southeast');
        h = gca; set(h,'fontsize',24,'FontName','GillSansMT');
%         ylim([0  1]); ylabel({'Dimensionless concentration [C = c/c_i_n]'}); 
        ylim([0  1]); ylabel({'c_p/c_b'}); 
        xlim([1 10]); xlabel('Flow rate [mL/h]');
        title('Comparing modeled Cp with experiments');
        %text(1.1,.9925,{['k0      = ' num2str(AnsmindevK(mk)) ' m/s'];['alpha = ' num2str(alpha)];['L2      = ' num2str(L2_fit*1e6) ' μm'];['AC     = ' num2str(AC/1e6) '\times10\^6/m'];['P       = ' num2str(P)]},'fontsize', 20,'HorizontalAlignment', 'left','VerticalAlignment', 'top');
        text(1.1,.9925,{['k_o    = ' num2str(AnsmindevK(mk)) ' m.s^{-1}'];['\alpha      = ' num2str(alpha)];['L_m   = ' num2str(L2_fit*1e6) ' μm'];['AC   = ' num2str(AC/1e6) '\times10^6 m^{-1}'];['\epsilon      = ' num2str(P)]},'fontsize', 20,'HorizontalAlignment', 'left','VerticalAlignment', 'top');
end
end
%% Optimization AC fit
if ACfit == true && justplotfinal == false

if Kfit == false
    error('Cannot fit AC without fitting K first')
    % Change Kfit to true at the beginning of the script
end 

ACtest = [];
% options = optimset('Display','iter','TolFun',1e-7,'TolX',1e-7,'MaxFunEvals',150);
% Can be used as alternative to display iteration results
options = optimset('Display','none','TolFun',1e-3,'TolX',1e-3,'MaxFunEvals',150);


for mk = 1:length(All_L2)
    itAC = 0;
    L2_fit = All_L2(mk);
    tofit = Exp_Data{mk};
    
    AnsmindevAC(mk) = fminsearchbnd(@balanceAC,AC,AC*(1-maxdev_AC),AC*(1+maxdev_AC),options); % searches for the K giving the lowest deviation from the datapoints
    disp(['LSQ fit value for AC: ' num2str(AnsmindevAC(mk))])

    for n = 1:length(Fcomparison) % Compare the found optimal value of k with experimental data
        Cpoints(n,:) = MembraneModelFunction(Fcomparison(n),L1,L2_fit,rad,P,AnsmindevK(mk),alpha,D1,AnsmindevAC(mk),S,plots,feedback,boundaryreaction);
    end

    % plotting comparison
    figure(); hold on; grid on 
        plot(Fcomparison,Cpoints(:,4),'.-b','MarkerSize',35,'LineWidth',3);

        for m = 1:nrepeats(mk)-1
            plot(Fmeasurements,Exp_Data{mk}(:,m),'--k','Marker',mstring(m),'MarkerSize',16,'LineWidth',3);
        end
    
        legend([' Model',legendset],'fontsize', 20, 'Location', 'southeast');
        h = gca; set(h,'fontsize',24,'FontName','GillSansMT');
        ylim([0  1]); ylabel({'Dimensionless concentration [C = c/c_i_n]'}); 
        xlim([1 10]); xlabel('Flow rate [mL/h]');
        title('Comparing modeled Cout with experiments');
        text(1.1,.9925,{['k0      = ' num2str(AnsmindevK(mk)) ' m/s'];['alpha = ' num2str(alpha)];['L2      = ' num2str(L2_fit*1e6) ' μm'];['AC     = ' num2str(AnsmindevAC(mk)/1e6) '\times10\^6/m'];['P       = ' num2str(P)]},'fontsize', 20,'HorizontalAlignment', 'left','VerticalAlignment', 'top');
end
end


%% Additional processing unique to script 2 and 3

if justplotfinal == true
if boundaryreaction == false 
    AnsmindevK = [0.778313065052032	0.565624094754457	0.199108661340550	0.215661995284725	0.206416003648657	0.193760528072744	0.198107525475323	0.280571960926056	0.484884468197823];
    AnsmindevAC= [1083125.07896356	1080813.99632243	1076260.33776512	1076551.49192128	1075731.30034790	1073082.29806798	1072958.85096553	1072979.42548270	1073000];
elseif boundaryreaction == true 
    AnsmindevK = [0.00138716856960673	0.000918151445686814	0.000267416842281820	0.000285503133789463	0.000257988309051141	0.000197643997853446	0.000174858455022785	0.000154381010215844	0.000108817930129591]
    AnsmindevAC= [1341249.58260702	1341229.80126835	1341237.11020416	1341043.18897157	1340498.74668141	1341246.81846887	1230798.20633938	1094046.65242899	1073000]
end

for mk = 1:length(All_L2)
    L2_fit = All_L2(mk);
    tofit = Exp_Data{mk};
    
    for n = 1:length(Fcomparison) % Compare the found optimal value of k with experimental data
        Cpoints(n,:) = MembraneModelFunction(Fcomparison(n),L1,L2_fit,rad,P,AnsmindevK(mk),alpha,D1,AnsmindevAC(mk),S,plots,feedback,boundaryreaction);
    end

    % plotting comparison
    figure(); hold on; grid on 
        plot(Fcomparison,Cpoints(:,4),'.-b','MarkerSize',35,'LineWidth',3);

        for m = 1:nrepeats(mk)-1
            plot(Fmeasurements,Exp_Data{mk}(:,m),'--k','Marker',mstring(m),'MarkerSize',16,'LineWidth',3);
        end
    
        legend([' Model',legendset],'fontsize', 20, 'Location', 'southeast');
        h = gca; set(h,'fontsize',24,'FontName','GillSansMT');
        ylim([0  1]); ylabel({'Dimensionless concentration [C = c/c_i_n]'}); 
        xlim([1 10]); xlabel('Flow rate [mL/h]');
        title('Comparing modeled Cout with experiments');
        text(1.1,.9925,{['k_o      = ' num2str(AnsmindevK(mk)) ' m/s'];['alpha = ' num2str(alpha)];['L2      = ' num2str(L2_fit*1e6) ' μm'];['AC     = ' num2str(AnsmindevAC(mk)/1e6) '\times10\^6/m'];['P       = ' num2str(P)]},'fontsize', 20,'HorizontalAlignment', 'left','VerticalAlignment', 'top');

end
end

%% Relation between k and F
%Shu:both figures are the same
figure(); hold on; grid on 
plot(All_L2([3:end]),AnsmindevK([3:end]),'*k','MarkerSize',16,'LineWidth',3)
legend([' Data'],'fontsize', 20, 'Location', 'southeast');
h = gca; set(h,'fontsize',24,'FontName','GillSansMT');
ylabel({'Fitted reaction constant (k_o)'}); 
xlabel('Membrane thickness (L_2)');
title('Relation between membrane thickness and reaction constant 0.26 to 4.3 um');

figure(); hold on; grid on 
plot(All_L2([1:end]),AnsmindevK([1:end]),'*k','MarkerSize',16,'LineWidth',3)
legend([' Data'],'fontsize', 20, 'Location', 'southeast');
h = gca; set(h,'fontsize',24,'FontName','GillSansMT');
ylabel({'Fitted reaction constant, k_o [m.s^{-1}]'}); 
xlabel('Membrane thickness, L_2 [m]');
title('Relation between membrane thickness and reaction constant');

%% Plotting model an exp data
Pe=[9.8 19.6 39.2 58.8 78.4 98];%MB
F=[1 2 4 6 8 10]; %Flow (mL/h)

%AGLT6_4 Experiment data with 1% 6L
C161=[0.247 0.123 0.218 0.260 0.288 0.288];
C162= [0.124 0.131 0.210 0.317 0.370 0.412];

%AGLT3_3 Experiment data with 1% 3L
C131=[0.123 0.135 0.203 0.288 0.315 0.324];
C132= [0.162 0.216 0.351 0.405 0.456 0.487];
C133= [0.151 0.219 0.335 0.409 0.521 0.548];

%AGPT150_2 Experiment data with 1.5%
C1501=[0.301 0.439 0.630 0.712 0.795 0.756];
C1502= [0.274 0.356 0.532 0.603 0.644 0.659];


%AGPT100_4 Experiment data with 1%
C1001=[0.264 0.403 0.570 0.637 0.681 0.723];
C1002=[0.296 0.429 0.605 0.681 0.690 0.718];
C1003= [0.274 0.389 0.556 0.625 0.671 0.666];

%AGPT070_2 Experiment data 0.70%
C0701=[0.321 0.486 0.601 0.658 0.740 0.702];
C0702=[0.297 0.388 0.578 0.650 0.673 0.746];

%AGPT050_2 Experiment data with 0.5%
C0501=[0.384 0.487 0.674 0.726 0.750 0.764];
C0502= [0.361 0.466 0.642 0.708 0.750 0.754];

%AGPT030_2 Experiment data 0.3%
C0301=[0.372 0.548 0.672 0.724 0.779 0.816];
C0302=[0.361 0.515 0.654 0.736 0.774 0.820];


%AGPT015_4 Experiment data 0.15%
C0151=[0.431 0.537 0.712 0.768 0.758 0.813];
C0152=[0.406 0.542 0.681 0.756 0.811 0.820];


%AGPT005_1 Experiment data with 0.05%
C0051=[0.419 0.575 0.744 0.794 0.849 0.879];
C0052= [0.522 0.647 0.794 0.847 0.876 0.887];


figure(); hold on; grid on 
file_list = dir(fullfile('./','experiment*.mat')); % creates a list of all the files ending with .mat
color_list = {'#bae4bc', '#7bccc4' ,'#9ecae1' ,'#6baed6', '#4292c6', '#009ec5', '#0074d9' ,'#08519c','#08306b'};
legend_list = [21.9 10.6 4.3 3.4 2.74 1.69 1.17 0.65 0.26]; %(µm)
for i = length(file_list):-1:1
    data = load(fullfile(file_list(i).folder,file_list(i).name));
    plot(data.Fcomparison,data.Csave,'-','Linewidth',3,'Color',color_list{i},'DisplayName',[sprintf('%.1f',legend_list(i)) ' µm'])
    hold on    
end
legend show
%lgd = legend('Location','northwest','FontSize',16);
lgd = legend('Location','southeast','FontSize',16,'NumColumns',3);
title(lgd,'TiO_2 layer thickness')
hold on

%add experimental data and format
ax(1)=gca;
set(ax(1),'Position',[0.12 0.12 0.75 0.70])
set(ax(1),'XColor','k','YColor','k');
set(gca, 'FontSize', 20);
set(gca, 'YTick', 0:0.2:1,'XTick', 0:2:10);
width=650;
height=700;
set(gcf,'position',[0,0,width,height])

plot (F, C161,'o','DisplayName',char({'21.9  \mum'}),'MarkerSize', 10,'MarkerEdgeColor',[0.672 0.884 0.724],'MarkerFaceColor',[0.672 0.884 0.724]);
plot (F, C162,'o','HandleVisibility','off','Color','k', 'MarkerSize', 10,'MarkerEdgeColor',[0.672 0.884 0.724],'MarkerFaceColor',[0.672 0.884 0.724]);

plot (F, C131,'o','DisplayName',char({'10.6  \mum'}),'MarkerSize',10,'MarkerEdgeColor',[0.24 0.812 0.72],'MarkerFaceColor',[0.24 0.812 0.72]);
plot (F, C132,'o','HandleVisibility','off','Color','k', 'MarkerSize', 10,'MarkerEdgeColor',[0.24 0.812 0.72],'MarkerFaceColor',[0.24 0.812 0.72]);
plot (F, C133,'o','HandleVisibility','off','Color','k', 'MarkerSize', 10,'MarkerEdgeColor',[0.24 0.812 0.72],'MarkerFaceColor',[0.24 0.812 0.72]);

xlim([0 10.46])
ylim([0 1])
grid on
xlabel('Flow [mL.h^{-1}]');
ylabel('\it{c_p/c_b}');

ax(2)=axes('Position',get(ax(1),'Position'),'XAxisLocation','top','YAxisLocation','right','Color','none','XColor','k','YColor','k','FontSize', 20);
set(ax,'box','on')
ylim([0 1])
xlim([0 102.58])
set(gca, 'YTick', [],'XTick',0:20:400)

hold on

plot (Pe,C1501,'o','DisplayName',char({' 4.3   \mum'}),'MarkerSize', 10,'MarkerEdgeColor',[0.776 0.859 0.937],'MarkerFaceColor',[0.776 0.859 0.937]);
plot (Pe, C1502,'o','HandleVisibility','off','Color','k', 'MarkerSize', 10,'MarkerEdgeColor',[0.776 0.859 0.937],'MarkerFaceColor',[0.776 0.859 0.937]);

plot (Pe,C1001,'o','DisplayName',char({' 3.4   \mum'}),'MarkerSize', 10,'MarkerEdgeColor',[0.620 0.792 0.882],'MarkerFaceColor',[0.620 0.792 0.882]);
plot (Pe, C1002,'o','HandleVisibility','off','Color','k', 'MarkerSize', 10,'MarkerEdgeColor',[0.620 0.792 0.882],'MarkerFaceColor',[0.620 0.792 0.882]);
plot (Pe, C1003,'o','HandleVisibility','off','Color','k', 'MarkerSize', 10,'MarkerEdgeColor',[0.620 0.792 0.882],'MarkerFaceColor',[0.620 0.792 0.882]);

plot (Pe,C0701,'o','DisplayName',char({' 2.74 \mum'}),'MarkerSize', 10,'MarkerEdgeColor','#009ec5','MarkerFaceColor','#009ec5');%#3ccbb4
plot (Pe, C0702,'o','HandleVisibility','off','Color','k', 'MarkerSize', 10,'MarkerEdgeColor','#009ec5','MarkerFaceColor','#009ec5');

plot (Pe,C0501,'o','DisplayName',char({' 1.69 \mum'}),'MarkerSize', 10,'MarkerEdgeColor',[0 0.632 0.788],'MarkerFaceColor',[0 0.632 0.788]); 
plot (Pe, C0502,'o','HandleVisibility','off','Color','k', 'MarkerSize', 10,'MarkerEdgeColor',[0 0.632 0.788],'MarkerFaceColor',[0 0.632 0.788]); 

plot (Pe,C0301,'o','HandleVisibility','off','MarkerSize',10,'MarkerEdgeColor',[0.129 0.443 0.710],'MarkerFaceColor',[0.129 0.443 0.710]); %#bcf185
plot (Pe, C0302,'o','HandleVisibility','off','Color','k', 'MarkerSize',10,'MarkerEdgeColor',[0.129 0.443 0.710],'MarkerFaceColor',[0.129 0.443 0.710]); 

plot (Pe,C0151,'o','HandleVisibility','off','MarkerSize',10,'MarkerEdgeColor',[0 0.464 0.868],'MarkerFaceColor',[0 0.464 0.868]);
plot (Pe, C0152,'o','HandleVisibility','off','Color','k', 'MarkerSize',10,'MarkerEdgeColor',[0 0.464 0.868],'MarkerFaceColor',[0 0.464 0.868]);

plot (Pe,C0051,'o','HandleVisibility','off','MarkerSize',10,'MarkerEdgeColor',[0 0.256 0.668],'MarkerFaceColor',[0 0.256 0.668]);
plot (Pe, C0052,'o','HandleVisibility','off','Color','k', 'MarkerSize',10,'MarkerEdgeColor',[0 0.256 0.668],'MarkerFaceColor',[0 0.256 0.668]);

hold off
xlabel('Pe');
text(-0.1,1.1,'a)','Color','k','FontSize',20);

ax(2).GridAlpha = 0.05
ax(1).YGrid = 'on'
ax(1).XGrid = 'on'

%% Functions
function f = fitK(k2, x)
global alpha rad D1 P L1 boundaryreaction S
global L2_fit 
global AC
    for n = 1:length(x) 
        Ktestpoints(n,:) = MembraneModelFunction(x(n),L1,L2_fit,rad,P,k2,alpha,D1,AC,S,false,false,boundaryreaction);
    end   
    f = Ktestpoints(:,4);
end

function f = balanceK(k2)
global alpha rad D1 P L1 boundaryreaction Fmeasurements S
global L2_fit tofit
global itK Ktest AC

    itK = [itK+ 1];
    Ktest = [Ktest, k2];
    Ktestpoints = zeros(length(Fmeasurements),4);
    disp(['Iteration number ' num2str(itK) ', Value for K: ' num2str(k2)])
    for n = 1:length(Fmeasurements)
        
        Ktestpoints(n,:) = MembraneModelFunction(Fmeasurements(n),L1,L2_fit,rad,P,k2,alpha,D1,AC,S,false,false,boundaryreaction);
    end   
    Ymodel = Ktestpoints(:,4);
    SumOfSquares =  sum((Ymodel-tofit).^2,'all');
    f = abs(SumOfSquares);
end

function f = balanceAC(AC2)
global alpha rad D1 P L1 boundaryreaction Fmeasurements S
global L2_fit tofit
global itAC ACtest AnsmindevK mk 

    itAC = [itAC+ 1];
    ACtest = [ACtest, AC2];
    ACtestpoints = zeros(length(Fmeasurements),4);
    disp(['Iteration number ' num2str(itAC) ', Value for AC: ' num2str(AC2)])
    for n = 1:length(Fmeasurements)
        ACtestpoints(n,:) = MembraneModelFunction(Fmeasurements(n),L1,L2_fit,rad,P,AnsmindevK(mk),alpha,D1,AC2,S,false,false,boundaryreaction);
    end        
    Ymodel = ACtestpoints(:,4);
    SumOfSquares =  sum((Ymodel-tofit).^2,'all');   
    f = abs(SumOfSquares);
end




