#Section 1: Create clean R workspace ----
rm(list=ls())
graphics.off()
gc(FALSE)
cat("\014")
Sys.setenv(LANG = "en")
options(max.print = 20000)

#Section 2: Open libraries and datasets ----
#Open packages:
Pkgs2Load <- c("readxl","tidyr", "ggplot2", "ggh4x", "emmeans", "multcomp", "glmmTMB", "DHARMa", "vegan")
invisible(lapply(Pkgs2Load, library, character.only = TRUE))

#Set the working directory.
setwd("[ADD FILE LOCATION]")

#Import data files
Lel <- read.csv("GB_4loc_Data_Lelystad_2019-2021.csv", row.names = 1)

Wag <- read.csv("GB_4loc_Data_Wageningen_2019-2022.csv", row.names = 1)
Wag22 <- read.csv("GB_4loc_Data_Wageningen_early_2022.csv", row.names = 1)

Alm20 <- read.csv("GB_4loc_Data_Almere_2020.csv", row.names = 1)
Alm21 <- read.csv("GB_4loc_Data_Almere_2021-2022.csv", row.names = 1)

Val <- read.csv("GB_4loc_Data_Valthermond_2020-2021.csv", row.names = 1)


#Section 3: Prepare the data ----
#Subsection 3.1: Lelystad ----
#Subsection 3.1.1: Add diversity indices
Lel$Abundance <- rowSums(Lel[,11:65])
Lel$Richness  <- rowSums(Lel[,11:64] / Lel[,11:64], na.rm = TRUE)
Lel$Shannon   <- diversity(Lel[,11:64],index = "shannon")
Lel$Simpson   <- diversity(Lel[,11:64],index = "simpson")
Lel$Invsimpson<- diversity(Lel[,11:64],index = "invsimpson")
Lel$Evenness <- Lel$Invsimpson / Lel$Richness

#Subsection 3.2: Wageningen ----
#Subsection 3.2.1: Add diversity indices
Wag$Abundance <- rowSums(Wag[,10:71])
Wag$Richness  <- rowSums(Wag[,10:70] / Wag[,10:70], na.rm = TRUE)
Wag$Shannon   <- diversity(Wag[,10:70],index = "shannon")
Wag$Simpson   <- diversity(Wag[,10:70],index = "simpson")
Wag$Invsimpson<- diversity(Wag[,10:70],index = "invsimpson")
Wag$Evenness <- Wag$Invsimpson / Wag$Richness

Wag22$Abundance <- rowSums(Wag22[,8:51])
Wag22$Richness  <- rowSums(Wag22[,8:50] / Wag22[,8:50], na.rm = TRUE)
Wag22$Shannon   <- diversity(Wag22[,8:50],index = "shannon")
Wag22$Simpson   <- diversity(Wag22[,8:50],index = "simpson")
Wag22$Invsimpson<- diversity(Wag22[,8:50],index = "invsimpson")
Wag22$Evenness <- Wag22$Invsimpson / Wag22$Richness

#Subsection 3.3: Almere ----
#Subsection 3.2.1: Add diversity indices
Alm20$Abundance <- rowSums(Alm20[,8:20])
Alm20$Richness  <- rowSums(Alm20[,8:20] / Alm20[,8:20], na.rm = TRUE)
Alm20$Shannon   <- diversity(Alm20[,8:20],index = "shannon")
Alm20$Simpson   <- diversity(Alm20[,8:20],index = "simpson")
Alm20$Invsimpson<- diversity(Alm20[,8:20],index = "invsimpson")
Alm20$Evenness <- Alm20$Invsimpson / Alm20$Richness

Alm21$Abundance <- rowSums(Alm21[,9:52])
Alm21$Richness  <- rowSums(Alm21[,9:51] / Alm21[,9:51], na.rm = TRUE)
Alm21$Shannon   <- diversity(Alm21[,9:51],index = "shannon")
Alm21$Simpson   <- diversity(Alm21[,9:51],index = "simpson")
Alm21$Invsimpson<- diversity(Alm21[,9:51],index = "invsimpson")
Alm21$Evenness <- Alm21$Invsimpson / Alm21$Richness

#Subsection 3.4: Valthermond ----
#Subsection 3.4.1: Add diversity indices
Val$Abundance <- rowSums(Val[,7:31])
Val$Richness  <- rowSums(Val[,7:31] / Val[,7:31], na.rm = TRUE)
Val$Shannon   <- diversity(Val[,7:31],index = "shannon")
Val$Simpson   <- diversity(Val[,7:31],index = "simpson")
Val$Invsimpson<- diversity(Val[,7:31],index = "invsimpson")
Val$Evenness <- Val$Invsimpson / Val$Richness

#Subsection 3.5: Merge datasets ----
#Subsection 3.5.1: Change column names

colnames(Wag22)[1] <- "ID"
Wag22$Location <- "Case-study"

colnames(Alm20)[4] <- "Treatment"
Alm20$Location <- "Almere"

colnames(Alm21)[4] <- "Treatment"
Alm21$Location <- "Almere"

colnames(Val)[2] <- "Year"
colnames(Val)[3] <- "ID"
colnames(Val)[5] <- "Crop"
colnames(Val)[6] <- "Treatment"
Val$Location <- "Valthermond"


#Subsection 3.5.2: Bind datasets
Total <- rbind(Lel[,c(2,3,8:10,66:71)],
               Wag[,c(1,2,7:9,72:77)],
               Wag22[,c(1:3,7,52:58)],
               Alm20[,c(1:4,21:27)],
               Alm21[,c(2:4,8,53:59)],
               Val[,c(2,3,5,6,32:38)])

#Subsection 3.5.3: Make treatment names consistent
Total$Treatment <- factor(Total$Treatment,
                       levels = c("mono", "REF_SPACE", "strip", "STRIP", "STRIP_3"),
                       labels = c("Monoculture", "Monoculture", "Strip", "Strip", "Strip"))
Total$Crop <- ifelse(Total$Crop == "barley", "Barley", Total$Crop)
Total$Crop <- ifelse(Total$Crop == "barley/bean", "Barley/Beans", Total$Crop)
Total$Crop <- ifelse(Total$Crop == "grass/clover", "Grass/Clover", Total$Crop)
Total$Crop <- ifelse(Total$Crop == "potato", "Potato", Total$Crop)

Total$CropGroup <- factor(Total$Crop,
                          levels = c("Barley", "Barley/Beans", "Beans", "Oat", "Wheat","Broccoli","Cabbage","Celeriac","Parsnip","Onion","Potato","Pumpkin","Grass","Grass/Clover"),
                          labels = c("Cereal", "Cereal", "Cereal", "Cereal", "Cereal","Brassica","Brassica","Root","Root","Onion","Potato","Pumpkin","Grass","Grass"))

Total$Crop <- factor(Total$Crop,
                        levels = c("Barley", "Barley/Beans" , "Beans","Broccoli","Cabbage","Celeriac","Grass","Grass/Clover" ,"Oat","Onion","Parsnip","Potato","Pumpkin","Wheat"),
                        labels = c("Barley", "Barley\nBeans", "Beans","Broccoli","Cabbage","Celeriac","Grass","Grass\nClover","Oat","Onion","Parsnip","Potato","Pumpkin","Wheat"))


#Subsection 3.5.4: Make location-year interaction variable
Total$LocYearCrop <- paste(Total$LocYear, Total$Crop, sep = "_")

#Section 4: Abundance / Activity density ----
TMA1 <- glmmTMB(log10(Abundance+1) ~ 1 + Treatment + 
                  Treatment*LocYearCrop, 
                family = gaussian(),
                dispformula = ~ 1,
                ziformula =~ 0,
                data = Total[Total$Location != "Case-study",], na.action = na.fail)
HSD.TMA1 <- emmeans(TMA1, pairwise ~ Treatment | LocYearCrop, type = "response")
CLD.TMA1 <- cld(HSD.TMA1$emmeans, Letters = letters)
CLD.TMA1$.group <- gsub(" ", "", CLD.TMA1$.group)
CLD.TMA1 <- separate(CLD.TMA1, LocYearCrop, into = c("Location","Year","Crop"), sep = "_")


TMA2 <- glmmTMB(log10(Abundance+1) ~ 1 + Treatment +
                  (1|Location/Year/Crop), 
                family = gaussian(),
                dispformula = ~ 1,
                ziformula =~ 0,
                data = Total[Total$Location != "Case-study",], na.action = na.fail)
HSD.TMA2 <- emmeans(TMA2, pairwise ~ Treatment, type = "response")
CLD.TMA2 <- cld(HSD.TMA2$emmeans, Letters = letters)
CLD.TMA2$.group <- gsub(" ", "", CLD.TMA2$.group)


#Section 5: Taxonomic Richness ----
TMR1 <- glmmTMB(Richness ~ 1 + Treatment + 
                  Treatment * LocYearCrop, 
                family = gaussian(),
                dispformula = ~ 1,
                ziformula =~ 0,
                data = Total[Total$Location != "Case-study",], na.action = na.fail)
HSD.TMR1 <- emmeans(TMR1, pairwise ~ Treatment | LocYearCrop)
CLD.TMR1 <- cld(HSD.TMR1$emmeans, Letters = letters)
CLD.TMR1$.group <- gsub(" ", "", CLD.TMR1$.group)
CLD.TMR1 <- separate(CLD.TMR1, LocYearCrop, into = c("Location","Year","Crop"), sep = "_")


TMR2 <- glmmTMB(Richness ~ 1 + Treatment + 
                  (1|Location/Year/Crop), 
                family = gaussian(),
                dispformula = ~ 1,
                ziformula =~ 0,
                data = Total[Total$Location != "Case-study",], na.action = na.fail)
HSD.TMR2 <- emmeans(TMR2, pairwise ~ Treatment)
CLD.TMR2 <- cld(HSD.TMR2$emmeans, Letters = letters)
CLD.TMR2$.group <- gsub(" ", "", CLD.TMR2$.group)


#Section 6: Evenness ----
TME1 <- glmmTMB(Evenness ~ 1 + Treatment + 
                  LocYearCrop +
                  Treatment*LocYearCrop, 
                family = gaussian(),
                dispformula = ~ 1,
                ziformula =~ 0,
                data = Total[Total$Location != "Case-study" & Total$Simpson > 0 & Total$Simpson < 1,], na.action = na.fail)
HSD.TME1 <- emmeans(TME1, pairwise ~ Treatment | LocYearCrop)
CLD.TME1 <- cld(HSD.TME1$emmeans, Letters = letters)
CLD.TME1$.group <- gsub(" ", "", CLD.TME1$.group)
CLD.TME1 <- separate(CLD.TME1, LocYearCrop, into = c("Location","Year","Crop"), sep = "_")

TME2 <- glmmTMB(Evenness ~ 1 + Treatment +  
                  (1|Location/Year/Crop), 
                family = gaussian(),
                dispformula = ~ 1,
                ziformula =~ 0,
                data = Total[Total$Location != "Case-study" & Total$Simpson > 0 & Total$Simpson < 1,], na.action = na.fail)
HSD.TME2 <- emmeans(TME2, pairwise ~ Treatment)
CLD.TME2 <- cld(HSD.TME2$emmeans, Letters = letters)
CLD.TME2$.group <- gsub(" ", "", CLD.TME2$.group)


#Section 7a: Shannon index ----
TMSh1 <- glmmTMB(Shannon ~ 1 + Treatment + 
                   LocYearCrop +
                   Treatment*LocYearCrop, 
                 family = gaussian,
                 dispformula = ~ 1,
                 ziformula =~ 0,
                 data = Total[Total$Location != "Case-study" & Total$Shannon > 0,], na.action = na.fail)
HSD.TMSh1 <- emmeans(TMSh1, pairwise ~ Treatment | LocYearCrop)
CLD.TMSh1 <- cld(HSD.TMSh1$emmeans, Letters = letters)
CLD.TMSh1$.group <- gsub(" ", "", CLD.TMSh1$.group)
CLD.TMSh1 <- separate(CLD.TMSh1, LocYearCrop, into = c("Location","Year","Crop"), sep = "_")

TMSh2 <- glmmTMB(Shannon ~ 1 + Treatment + 
                   (1|Location/Year/Crop), 
                 family = gaussian,
                 dispformula = ~ 1,
                 ziformula =~ 0,
                 data = Total[Total$Location != "Case-study" & Total$Shannon > 0,], na.action = na.fail)
HSD.TMSh2 <- emmeans(TMSh2, pairwise ~ Treatment)
CLD.TMSh2 <- cld(HSD.TMSh2$emmeans, Letters = letters)
CLD.TMSh2$.group <- gsub(" ", "", CLD.TMSh2$.group)


#Section 7b: Inverse simpson index ----
TMIS1 <- glmmTMB(Invsimpson ~ 1 + Treatment + 
                   LocYearCrop +
                   Treatment*LocYearCrop , 
                 family = Gamma(),
                 dispformula = ~ 1,
                 ziformula =~ 0,
                 data = Total[Total$Location != "Case-study" & Total$Simpson > 0 & Total$Simpson < 1,], na.action = na.fail)
HSD.TMIS1 <- emmeans(ref_grid(TMIS1, regrid = "response"), pairwise ~ Treatment | LocYearCrop)
CLD.TMIS1 <- cld(HSD.TMIS1$emmeans, Letters = letters)
CLD.TMIS1$.group <- gsub(" ", "", CLD.TMIS1$.group)
CLD.TMIS1 <- separate(CLD.TMIS1, LocYearCrop, into = c("Location","Year","Crop"), sep = "_")

TMIS2 <- glmmTMB(Invsimpson ~ 1 + Treatment +  
                   (1|Location/Year/Crop), 
                 family = Gamma(),
                 dispformula = ~ 1,
                 ziformula =~ 0,
                 data = Total[Total$Location != "Case-study" & Total$Simpson > 0 & Total$Simpson < 1,], na.action = na.fail)
HSD.TMIS2 <- emmeans(ref_grid(TMIS2, regrid = "response"), pairwise ~ Treatment)
CLD.TMIS2 <- cld(HSD.TMIS2$emmeans, Letters = letters)
CLD.TMIS2$.group <- gsub(" ", "", CLD.TMIS2$.group)




#Section 8: Make figures for appendix ----
setwd("[SET LOCATION]")

Treatment.colors <- c("#895F39", "#7FA24D")


#Subsection 8.1: Abundance / Activity density ----
png(filename = "Ground beetle abundance - Without case-study.png", width = 8000, height = 4000, res = 400)
ggplot(data = Total[Total$Location != "Case-study",], aes(x = Crop, y = Abundance, fill = Treatment)) +
  geom_jitter(data = Total[Total$Location != "Case-study",], 
              aes(x = Crop, y = Abundance, group = Treatment, color = Treatment), 
              shape = 1, size = 2, stroke = 1.5, alpha = 0.75, position = position_jitterdodge(dodge.width = 0.75)) +
  geom_pointrange(data = CLD.TMA1[CLD.TMA1$Location != "Case-study",], 
                  aes(x = Crop, y = response, group = Treatment, ymin = lower.CL, ymax = upper.CL, color = Treatment), 
                  shape = 15, size = 0.75, linewidth = 1, position = position_dodge(width = 0.6)) +
  facet_grid2(Year ~ Location, scales = "free", independent = "y", space = "free_x") +
  #geom_text(data = CLD.TMA1[CLD.TMA1$Location != "Case-study",], aes(label = .group, x = Crop, group = Treatment, y = 1.3*upper.CL), 
  #          position = position_dodge(width = 0.75), size = 6) +
  scale_color_manual(values = Treatment.colors) +
  labs(x = "Crop", y = "Ground beetle activity density") +
  theme_bw() + 
  theme(legend.position = "none",
        title = element_text(size = 25), strip.text = element_text(size = 20),
        axis.text.x = element_text(size = 15, color = "black"),axis.text.y = element_text(size = 16, color = "black"),
        panel.grid.major.x = element_blank())
dev.off()


#Subsection 8.2: Taxonomic Richness ----
png(filename = "Ground beetle taxonomic richness - Without case-study.png", width = 8000, height = 4000, res = 400)
ggplot(data = Total[Total$Location != "Case-study",], aes(x = Crop, y = Richness, fill = Treatment)) +
  geom_jitter(data = Total[Total$Location != "Case-study",], 
              aes(x = Crop, y = Richness, group = Treatment, color = Treatment), 
              shape = 1, size = 2, stroke = 1.5, alpha = 0.75, position = position_jitterdodge(dodge.width = 0.75)) +
  geom_pointrange(data = CLD.TMR1[CLD.TMR1$Location != "Case-study",], 
                  aes(x = Crop, y = emmean, group = Treatment, ymin = lower.CL, ymax = upper.CL, color = Treatment), 
                  shape = 15, size = 0.75, linewidth = 1, position = position_dodge(width = 0.6)) +
  facet_grid2(Year ~ Location, scales = "free_x", independent = "none", space = "free_x") +
  #geom_text(data = CLD.TMR1[CLD.TMR1$Location != "Case-study",], aes(label = .group, x = Crop, group = Treatment, y = 1.2*upper.CL), 
  #          position = position_dodge(width = 0.75), size = 6) +
  scale_color_manual(values = Treatment.colors) +
  labs(x = "Crop", y = "Ground beetle taxonomic richness") +
  theme_bw() + 
  theme(legend.position = "none",
        title = element_text(size = 25), strip.text = element_text(size = 20),
        axis.text.x = element_text(size = 15, color = "black"),axis.text.y = element_text(size = 16, color = "black"),
        panel.grid.major.x = element_blank())
dev.off()


#Subsection 8.3: Evenness (Inverse Simpson / Richness) ----
png(filename = "Ground beetle evenness - Without case-study.png", width = 8000, height = 4000, res = 400)
ggplot(data = Total[(Total$Simpson > 0 & Total$Simpson < 1) & Total$Location != "Case-study",], aes(x = Crop, y = Evenness, fill = Treatment)) +
  geom_jitter(data = Total[(Total$Simpson > 0 & Total$Simpson < 1) & Total$Location != "Case-study",], 
              aes(x = Crop, y = Evenness, group = Treatment, color = Treatment), 
              shape = 1, size = 2, stroke = 1.5, alpha = 0.75, position = position_jitterdodge(dodge.width = 0.75)) +
  geom_pointrange(data = CLD.TME1[CLD.TME1$Location != "Case-study",], 
                  aes(x = Crop, y = emmean, group = Treatment, ymin = lower.CL, ymax = upper.CL, color = Treatment), 
                  shape = 15, size = 0.75, linewidth = 1, position = position_dodge(width = 0.6)) +
  facet_grid2(Year ~ Location, scales = "free", independent = "y", space = "free_x") +
  #geom_text(data = CLD.TME1[CLD.TME1$Location != "Case-study",], aes(label = .group, x = Crop, group = Treatment, y = 1.3*upper.CL), 
  #          position = position_dodge(width = 0.75), size = 6) +
  scale_color_manual(values = Treatment.colors) +
  labs(x = "Crop", y = "Ground beetle evenness") +
  theme_bw() +
  theme(legend.position = "none",
        title = element_text(size = 25), strip.text = element_text(size = 20),
        axis.text.x = element_text(size = 15, color = "black"),axis.text.y = element_text(size = 16, color = "black"),
        panel.grid.major.x = element_blank())
dev.off()


#Subsection 8.4: Shannon-Weaver index ----
png(filename = "Ground beetle Shannon index - Without case-study.png", width = 8000, height = 4000, res = 400)
ggplot(data = Total[Total$Location != "Case-study",], aes(x = Crop, y = Shannon, fill = Treatment)) +
  geom_jitter(data = Total[Total$Location != "Case-study",], 
              aes(x = Crop, y = Shannon, group = Treatment, color = Treatment), 
              shape = 1, size = 2, stroke = 1.5, alpha = 0.75, position = position_jitterdodge(dodge.width = 0.75)) +
  geom_pointrange(data = CLD.TMSh1[CLD.TMSh1$Location != "Case-study",], aes(x = Crop, y = emmean, group = Treatment, ymin = lower.CL, ymax = upper.CL, color = Treatment), 
                  shape = 15, size = 0.75, linewidth = 1, position = position_dodge(width = 0.6)) +
  facet_grid2(Year ~ Location, scales = "free_x", independent = "none", space = "free_x") +
  #geom_text(data = CLD.TMSh1[CLD.TMSh1$Location != "Case-study",], aes(label = .group, x = Crop, group = Treatment, y = 1.3*upper.CL), 
  #          position = position_dodge(width = 0.75), size = 6) +
  scale_color_manual(values = Treatment.colors) +
  labs(x = "Crop", y = "Ground beetle Shannon entropy") +
  theme_bw() +
  theme(legend.position = "none",
        title = element_text(size = 25), strip.text = element_text(size = 20),
        axis.text.x = element_text(size = 15, color = "black"),axis.text.y = element_text(size = 16, color = "black"),
        panel.grid.major.x = element_blank())
dev.off()


#Subsection 8.5: Inverse Simpson index ----
png(filename = "Ground beetle inverse simpson index - Without case-study.png", width = 8000, height = 4000, res = 400)
ggplot(data = Total[(Total$Simpson > 0 & Total$Simpson < 1) & Total$Location != "Case-study",], aes(x = Crop, y = Invsimpson, fill = Treatment)) +
  geom_jitter(data = Total[(Total$Simpson > 0 & Total$Simpson < 1) & Total$Location != "Case-study",], 
              aes(x = Crop, y = Invsimpson, group = Treatment, color = Treatment), 
              shape = 1, size = 2, stroke = 1.5, alpha = 0.75, position = position_jitterdodge(dodge.width = 0.75)) +
  geom_pointrange(data = CLD.TMIS1[CLD.TMIS1$Location != "Case-study",], 
                  aes(x = Crop, y = response, group = Treatment, ymin = lower.CL, ymax = upper.CL, color = Treatment), 
                  shape = 15, size = 0.75, linewidth = 1, position = position_dodge(width = 0.6)) +
  facet_grid2(Year ~ Location, scales = "free", independent = "y", space = "free_x") +
  #geom_text(data = CLD.TMIS1[CLD.TMIS1$Location != "Case-study",], aes(label = .group, x = Crop, group = Treatment, y = 1.3*upper.CL), 
  #          position = position_dodge(width = 0.75), size = 6) +
  scale_color_manual(values = Treatment.colors) +
  labs(x = "Crop", y = "Ground beetle inverse Simpson index") +
  theme_bw() +
  theme(legend.position = "none",
        title = element_text(size = 25), strip.text = element_text(size = 20),
        axis.text.x = element_text(size = 15, color = "black"),axis.text.y = element_text(size = 16, color = "black"),
        panel.grid.major.x = element_blank())
dev.off()


#Section 9: Case-study figure
CaseStudy <- Total[Total$Location == "Case-study",]
CaseStudy$Crop <- factor(CaseStudy$Crop,
                          levels = c("Cabbage", "Oat", "Pumpkin", "Barley", "Potato", "Grass"))

#Subsection 9.1: Abundance / Activity density ----
png(filename = "Ground beetle abundance - Case-study.png", width = 2250, height = 1000, res = 200)
ggplot(data = CaseStudy, aes(x = Crop, y = Abundance, fill = Treatment)) +
  geom_jitter(data = CaseStudy, 
              aes(x = Crop, y = Abundance, group = Treatment, color = Treatment), 
              shape = 1, size = 2, stroke = 1.5, alpha = 0.75, position = position_jitterdodge(dodge.width = 0.75)) +
  geom_pointrange(data = CLD.TMA1[CLD.TMA1$Location == "Case-study",], 
                  aes(x = Crop, y = response, group = Treatment, ymin = lower.CL, ymax = upper.CL, color = Treatment), 
                  shape = 15, size = 0.75, linewidth = 1, position = position_dodge(width = 0.6)) +
  facet_grid2(Year ~ Location, scales = "free", independent = "y", space = "free_x") +
  #geom_text(data = CLD.TMA1[CLD.TMA1$Location == "Case-study",], aes(label = .group, x = Crop, group = Treatment, y = 1.3*upper.CL), 
  #          position = position_dodge(width = 0.75), size = 10) +
  scale_color_manual(values = Treatment.colors) +
  labs(x = "Crop", y = "Ground beetle activity density") +
  theme_bw() + 
  theme(legend.position = "none",
        title = element_text(size = 20), strip.text = element_text(size = 20),
        axis.text.x = element_text(size = 15, color = "black"),axis.text.y = element_text(size = 16, color = "black"),
        panel.grid.major.x = element_blank())
dev.off()


#Subsection 9.2: Taxonomic Richness ----
png(filename = "Ground beetle taxonomic richness - Case-study.png", width = 2250, height = 1000, res = 200)
ggplot(data = CaseStudy, aes(x = Crop, y = Richness, fill = Treatment)) +
  geom_jitter(data = CaseStudy, 
              aes(x = Crop, y = Richness, group = Treatment, color = Treatment),
              shape = 1, size = 2, stroke = 1.5, alpha = 0.75, position = position_jitterdodge(dodge.width = 0.75)) +
  geom_pointrange(data = CLD.TMR1[CLD.TMR1$Location == "Case-study",], 
                  aes(x = Crop, y = emmean, group = Treatment, ymin = lower.CL, ymax = upper.CL, color = Treatment), 
                  shape = 15, size = 0.75, linewidth = 1, position = position_dodge(width = 0.6)) +
  facet_grid2(Year ~ Location, scales = "free_x", independent = "none", space = "free_x") +
  #geom_text(data = CLD.TMR1[CLD.TMR1$Location == "Case-study",], aes(label = .group, x = Crop, group = Treatment, y = 1.2*upper.CL), 
  #          position = position_dodge(width = 0.75), size = 6) +
  scale_color_manual(values = Treatment.colors) +
  labs(x = "Crop", y = "Ground beetle taxonomic richness") +
  theme_bw() +
  theme(legend.position = "none",
        title = element_text(size = 20), strip.text = element_text(size = 20),
        axis.text.x = element_text(size = 15, color = "black"),axis.text.y = element_text(size = 16, color = "black"),
        panel.grid.major.x = element_blank())
dev.off()


#Subsection 9.3: Evenness (Inverse Simpson / Richness) ----
png(filename = "Ground beetle evenness - Case-study.png", width = 2250, height = 1000, res = 200)
ggplot(data = CaseStudy[(CaseStudy$Simpson > 0 & CaseStudy$Simpson < 1),], aes(x = Crop, y = Evenness, fill = Treatment)) +
  geom_jitter(data = CaseStudy[(CaseStudy$Simpson > 0 & CaseStudy$Simpson < 1),], 
              aes(x = Crop, y = Evenness, group = Treatment, color = Treatment), 
              shape = 1, size = 2, stroke = 1.5, alpha = 0.75, position = position_jitterdodge(dodge.width = 0.75)) +
  geom_pointrange(data = CLD.TME1[CLD.TME1$Location == "Case-study",], 
                  aes(x = Crop, y = emmean, group = Treatment, ymin = lower.CL, ymax = upper.CL, color = Treatment), 
                  shape = 15, size = 0.75, linewidth = 1, position = position_dodge(width = 0.6)) +
  facet_grid2(Year ~ Location, scales = "free", independent = "y", space = "free_x") +
  #geom_text(data = CLD.TME1[CLD.TME1$Location == "Case-study",], aes(label = .group, x = Crop, group = Treatment, y = 1.3*upper.CL), 
  #          position = position_dodge(width = 0.75), size = 6) +
  scale_color_manual(values = Treatment.colors) +
  labs(x = "Crop", y = "Ground beetle evenness") +
  theme_bw() +
  theme(legend.position = "none",
        title = element_text(size = 20), strip.text = element_text(size = 20),
        axis.text.x = element_text(size = 15, color = "black"),axis.text.y = element_text(size = 16, color = "black"),
        panel.grid.major.x = element_blank())
dev.off()


#Subsection 9.4: Shannon-Weaver index ----
png(filename = "Ground beetle Shannon index - Case-study.png", width = 2250, height = 1000, res = 200)
ggplot(data = CaseStudy, aes(x = Crop, y = Shannon, fill = Treatment)) +
  geom_jitter(data = CaseStudy, 
              aes(x = Crop, y = Shannon, group = Treatment, color = Treatment), 
              shape = 1, size = 2, stroke = 1.5, alpha = 0.75, position = position_jitterdodge(dodge.width = 0.75)) +
  geom_pointrange(data = CLD.TMSh1[CLD.TMSh1$Location == "Case-study",], 
                  aes(x = Crop, y = emmean, group = Treatment, ymin = lower.CL, ymax = upper.CL, color = Treatment), 
                  shape = 15, size = 0.75, linewidth = 1, position = position_dodge(width = 0.6)) +
  facet_grid2(Year ~ Location, scales = "free_x", independent = "none", space = "free_x") +
  #geom_text(data = CLD.TMSh1[CLD.TMSh1$Location == "Case-study",], aes(label = .group, x = Crop, group = Treatment, y = 1.3*upper.CL), 
  #          position = position_dodge(width = 0.75), size = 6) +
  scale_color_manual(values = Treatment.colors) +
  labs(x = "Crop", y = "Ground beetle Shannon entropy") +
  theme_bw() +
  theme(legend.position = "none",
        title = element_text(size = 20), strip.text = element_text(size = 20),
        axis.text.x = element_text(size = 15, color = "black"),axis.text.y = element_text(size = 16, color = "black"),
        panel.grid.major.x = element_blank())
dev.off()


#Subsection 9.5: Inverse Simpson index ----
png(filename = "Ground beetle inverse simpson index - Case-study.png", width = 2250, height = 1000, res = 200)
ggplot(data = CaseStudy[(CaseStudy$Simpson > 0 & CaseStudy$Simpson < 1),], aes(x = Crop, y = Invsimpson, fill = Treatment)) +
  geom_jitter(data = CaseStudy[(CaseStudy$Simpson > 0 & CaseStudy$Simpson < 1),], 
              aes(x = Crop, y = Invsimpson, group = Treatment, color = Treatment), 
              shape = 1, size = 2, stroke = 1.5, alpha = 0.75, position = position_jitterdodge(dodge.width = 0.75)) +
  geom_pointrange(data = CLD.TMIS1[CLD.TMIS1$Location == "Case-study",], 
                  aes(x = Crop, y = response, group = Treatment, ymin = lower.CL, ymax = upper.CL, color = Treatment), 
                  shape = 15, size = 0.75, linewidth = 1, position = position_dodge(width = 0.6)) +
  facet_grid2(Year ~ Location, scales = "free", independent = "y", space = "free_x") +
  #geom_text(data = CLD.TMIS1[CLD.TMIS1$Location == "Case-study",], aes(label = .group, x = Crop, group = Treatment, y = 1.3*upper.CL), 
  #          position = position_dodge(width = 0.75), size = 6) +
  scale_color_manual(values = Treatment.colors) +
  labs(x = "Crop", y = "Ground beetle inverse Simpson index") +
  theme_bw() +
  theme(legend.position = "none",
        title = element_text(size = 20), strip.text = element_text(size = 20),
        axis.text.x = element_text(size = 15, color = "black"),axis.text.y = element_text(size = 16, color = "black"),
        panel.grid.major.x = element_blank())
dev.off()





#section 10: Make figures for overall paterns  ----
#Subsection 10.1: Abundance / Activity density ----
png(filename = "Ground beetle abundance - Overall effect.png", width = 4000, height = 5000, res = 1000)
ggplot(data = Total, aes(x = Treatment, y = Abundance)) +
  geom_pointrange(data = CLD.TMA2, 
                  aes(x = Treatment, y = response, ymin = lower.CL, ymax = upper.CL, color = Treatment), 
                  shape = 15, size = 2.5, linewidth = 5) +
  scale_color_manual(values = Treatment.colors) +
  labs(x = "Cropping design", y = "Ground beetle activity density") +
  theme_bw() +
  theme(legend.position = "none", axis.title.x = element_blank() ,
        title = element_text(size = 20), 
        axis.text.x = element_text(size = 18, color = "black"),axis.text.y = element_text(size = 16, color = "black"),
        panel.grid.major.x = element_blank(), panel.grid.minor.y = element_blank())
dev.off()


#Subsection 10.2: Taxonomic Richness ----
png(filename = "Ground beetle taxonomic richness - Overall effect.png", width = 4000, height = 5000, res = 1000)
ggplot(data = Total, aes(x = Treatment, y = Richness)) +
  geom_pointrange(data = CLD.TMR2, 
                  aes(x = Treatment, y = emmean, ymin = lower.CL, ymax = upper.CL, color = Treatment), 
                  shape = 15, size = 2.5, linewidth = 5) +
  #geom_text(data = CLD.TMR2, aes(label = .group, x = Treatment, y = upper.CL + 0.05), size = 8) +
  scale_color_manual(values = Treatment.colors) +
  labs(x = "Cropping design", y = "Ground beetle taxonomic richness") +
  theme_bw() +
  theme(legend.position = "none", axis.title.x = element_blank() ,
        title = element_text(size = 20), 
        axis.text.x = element_text(size = 18, color = "black"),axis.text.y = element_text(size = 16, color = "black"),
        panel.grid.major.x = element_blank(), panel.grid.minor.y = element_blank())
dev.off()

#Subsection 10.3: Evenness index ----
png(filename = "Ground beetle evenness - Overall effect.png", width = 4000, height = 5000, res = 1000)
ggplot(data = Total, aes(x = Treatment, y = Evenness)) +
  geom_pointrange(data = CLD.TME2, 
                  aes(x = Treatment, y = emmean, ymin = lower.CL, ymax = upper.CL, color = Treatment), 
                  shape = 15, size = 2.5, linewidth = 5) +
  scale_color_manual(values = Treatment.colors) +
  labs(x = "Cropping design", y = "Ground beetle evenness") +
  theme_bw() +
  theme(legend.position = "none", axis.title.x = element_blank() ,
        title = element_text(size = 20), 
        axis.text.x = element_text(size = 18, color = "black"),axis.text.y = element_text(size = 16, color = "black"),
        panel.grid.major.x = element_blank(), panel.grid.minor.y = element_blank())
dev.off()

#Subsection 10.4: Shannon index ----
png(filename = "Ground beetle shannon index - Overall effect.png", width = 4000, height = 5000, res = 1000)
ggplot(data = Total, aes(x = Treatment, y = Shannon)) +
  geom_pointrange(data = CLD.TMSh2, 
                  aes(x = Treatment, y = emmean, ymin = lower.CL, ymax = upper.CL, color = Treatment), 
                  shape = 15, size = 2.5, linewidth = 5) +
  #geom_text(data = CLD.TMSh2, aes(label = .group, x = Treatment, y = upper.CL + 0.01), size = 8) +
  scale_color_manual(values = Treatment.colors) +
  labs(x = "Cropping design", y = "Ground beetle Shannon entropy") +
  theme_bw() +
  theme(legend.position = "none", axis.title.x = element_blank() ,
        title = element_text(size = 20), 
        axis.text.x = element_text(size = 18, color = "black"),axis.text.y = element_text(size = 16, color = "black"),
        panel.grid.major.x = element_blank(), panel.grid.minor.y = element_blank())
dev.off()

#Subsection 10.5: Inverse Simpson index ----
png(filename = "Ground beetle inverse Simpson index - Overall effect.png", width = 4000, height = 5000, res = 1000)
ggplot(data = Total, aes(x = Treatment, y = Invsimpson)) +
  geom_pointrange(data = CLD.TMIS2, 
                  aes(x = Treatment, y = response, ymin = lower.CL, ymax = upper.CL, color = Treatment), 
                  shape = 15, size = 2.5, linewidth = 5) +
  #geom_text(data = CLD.TMIS2, aes(label = .group, x = Treatment, y = upper.CL + 0.01), size = 8) +
  scale_color_manual(values = Treatment.colors) +
  labs(x = "Cropping design", y = "Ground beetle inverse Simpson index") +
  theme_bw() +
  theme(legend.position = "none", axis.title.x = element_blank() ,
        title = element_text(size = 20), 
        axis.text.x = element_text(size = 18, color = "black"),axis.text.y = element_text(size = 16, color = "black"),
        panel.grid.major.x = element_blank(), panel.grid.minor.y = element_blank())
dev.off()