# Import necessary libraries
import numpy as np  # For numerical computations
from scipy.interpolate import interp1d  # For interpolation operations
from PythonFiles.createFit2 import createFit2  # Import the createFit2 function
from PythonFiles.disper import disper  # Import the disper function

# Define the HU method for calculating KLm
def HU_method_Hs(KLn, Urn, Hs, X, T, d):
    """
    HU_METHOD_HS(KLn, Urn, Hs, X, T, d)
    Calculate KLm using the HU method.

    Parameters:
        KLn : array-like, shape (m, n)
            KLn values for different wavelengths and Ursell numbers.
        Urn : array-like
            Ursell numbers corresponding to the data in KLn.
        Hs : float
            Significant wave height.
        X : float
            Distance from the wave source.
        T : float
            Wave period.
        d : float
            Water depth.

    Returns:
        KLm : float
            KLm value at the given position X.
    """

    # Get the number of rows in the input matrix KLn
    # m is the number of wavelengths
    m = len(KLn[:, 0])

    # Calculate the wavelength L and the current Ursell number Ur
    # L = 2*pi/disper(2*pi/T,d)  % obtain wavelength
    L = 2 * np.pi / disper(2 * np.pi / T, d)  # Wavelength calculated using the disper function
    # Ur = Hs*L^2/d^3            % obtain Ur number
    Ur = Hs * L ** 2 / d ** 3  # Ursell number formula

    # Determine the range of Ur, taking the minimum and maximum values of Urn and the current Ur
    # Range1 = min([round(min(Urn)) round(Ur)])
    # Range2 = max([round(max(Urn)) round(Ur)])
    # Ur_range = [Range1:1:Range2]
    Range1 = min(round(np.min(Urn)), round(Ur))  # Take the smaller value as the lower bound
    Range2 = max(round(np.max(Urn)), round(Ur))  # Take the larger value as the upper bound
    Ur_range = np.arange(Range1, Range2 + 1)  # Construct the range array for Ur

    # Initialize the KL array with shape (m, len(Ur_range))
    KL = np.zeros((m, len(Ur_range)))

    # Iterate through each row to fit Ur and KL relationships for each position in KLn
    # for i = 1:m
    # [fitresult, gof] = createFit2(Urn, KLn(i,:))
    for i in range(m):
        # Use createFit2 to perform the fitting and obtain parameters a and b
        # Temp = coeffvalues(fitresult)
        # a = Temp(1)
        # b = Temp(2)
        fitresult = createFit2(Urn, KLn[i, :])  # Obtain the fitting parameters
        a = fitresult[0]  # Fitted parameter a
        b = fitresult[1]  # Fitted parameter b

         # Avoid Ur_range being 0 (to prevent division by zero)
        Ur_safe = np.where(Ur_range == 0, 1e-10, Ur_range)
        
        # KL(i,:) = 1./(1+a.*Ur_range.^b)
        KL[i, :] = 1 / (1 + a * Ur_safe ** b)  # Calculate KL values using the fitted formula

    # Find the index position corresponding to the current Ur in the Ur_range
    # ind = find(Ur_range-round(Ur)==0)
    try:
        ind = np.where(Ur_range == round(Ur))[0][0]  # Locate the closest Ur value
    except IndexError:
        # If no match is found, return NaN
        return np.nan

    # Check if X/L meets the condition and perform interpolation
    # if X/L < 1
    # KLm = NaN
    if X / L < 1:
        KLm = np.nan  # If X/L is less than 1, return NaN
    else:
        # KLm = interp1 (1:m,KL(:,ind),X/L)
        # Use an interpolation function to calculate KLm, returning NaN for out-of-bound cases
        interp_func = interp1d(np.arange(1, m + 1), KL[:, ind], bounds_error=False, fill_value=np.nan)
        KLm = interp_func(X / L)  # Interpolate to compute KLm

    # Return KLm
    return KLm

