% This function plots the rates with fits of all runs of one sweep type
% (interactive selection of which sweep).
% You need the useful_colormaps function [1] that was included in the code
% documentation earlier as well, just for the colors in the plots.
%
% Date: 18-01-2024
%
% Bibliography
% [1] Smith, N. J., Walt, S. van der, & Firing, E. (2015). Magma, inferno,
%     plasma and viridis colormaps,
%     https://github.com/BIDS/colormap/blob/master/colormaps.py.

close all
clc
clearvars

plot_font_name      = 'Arial';  % Name of font in plot
plot_font_size      = 8;        % Font size in plot
resol_for_print     = 300;      % Resolution for .tiff outputs
pos_plot            = [0.2 0.5 5.1 3.4]; % Position vector of the figure in inches
% [x-coordinate bottom left corner y-coordinate width height] in inches
% Modify if it does not fit your screen
pos_ax              = [0.5 0.4 pos_plot(3 : 4) - 0.6]; % Position vector of the axes in inches

%% Interactive selection of assays to be plotted
[assays_file_name, assays_path_name]    = uigetfile('*.mat', 'Select the output of the assay processing (Data_assays.mat)');
load([assays_path_name assays_file_name], 'Assays_processed')

% dialog box for selecting which assays to plot
panel1              = dialog('Name', 'Select assay', 'Position', [50 50 200 200]);
all_assays          = fieldnames(Assays_processed);
assay_menu          = uicontrol('Style', 'popupmenu', 'units', 'normalized', ...
                        'Position', [0.1 0.3 0.3 0.3], ...
                        'String', ['empty'; all_assays], 'Parent', panel1);
% button wfen done with selecting assays
uicontrol('Style', 'pushbutton', 'units', 'normalized', ...
                        'Position', [0.65 0.4 0.3 0.2], ...
                        'String', 'Start', 'Parent', panel1, 'Callback', 'uiresume(panel1)');
% Wait until selection is made
uiwait(panel1)
% Indices of selected assays
assay_list          = flipud(nonzeros([assay_menu.Value] - 1));
% choice is completed, close the dialog box again
close(panel1)
Assay_name          = all_assays{assay_list};

%% Extract data from the large structure
Proteins            = Assays_processed.(Assay_name).Proteins;       % Protein that is varied in this assay
GTPase              = Proteins.GTPase{1};                           % GTPase in this assay
Proteins_var        = cat(1, Proteins.QuaEff, Proteins.LinEff);     % Find the varied proteins
if isempty(Proteins_var)                                            % If no effectors, then the varied protein is the GTPase
    Proteins_var    = Proteins.GTPase;
end
varied_protein      = Proteins_var{1};                              % The first varied protein will be the one on the x-axis
uni_run_names       = Assays_processed.(Assay_name).k_runs.name;    % Unique run names
run_name_list       = Assays_processed.(Assay_name).Data.Run;       % List of run names corresponding to each data point
t                   = Assays_processed.(Assay_name).Data.Time;      % Time points
y                   = Assays_processed.(Assay_name).Data.GTP_remaining; % GTP remaining data
err                 = Assays_processed.(Assay_name).Data.Error;     % Errors on GTP remaining
c                   = Assays_processed.(Assay_name).Data.([varied_protein '_conc']); % Concentration that was varied
c_GTPase            = Assays_processed.(Assay_name).Data.([GTPase '_conc']); % GTPase background concentration

% Discard data points with too low GTP reminaing (<0.05%)
run_name_list       = run_name_list(y > 0.05);
t                   = t(y > 0.05);
err                 = err(y > 0.05);
c                   = c(y > 0.05);
c_GTPase            = c_GTPase(y > 0.05);
y                   = y(y > 0.05);

% Correction factors per run
c_corr              = Assays_processed.(Assay_name).conc_corr(:, 1);
% Concentration values for the fitted rate lines
c_probe             = linspace(0, ceil(max(c(:, 1) + 0.1) * 10) / 10, 1e2)';

if numel(Proteins_var) == 2
    disp('Plotting multiple variable proteins of multiple runs not supported')
    return
elseif ~isempty(cat(1, Proteins.QuaEff, Proteins.LinEff)) % If an effector is present
    GTPase_assay    = Assays_processed.(Assay_name).GTPase_assay;
    % Rate parameters
    k1              = Assays_processed.(GTPase_assay).k_tot.est(1);
    k2              = Assays_processed.(GTPase_assay).k_tot.est(2);
    k3              = Assays_processed.(Assay_name).k_runs.est(:, 2);
    
    % Find Cdc42 concentration per run
    GTPase_uni      = unique(cell2table([run_name_list cellfun(@(x) num2str(x), num2cell(c_GTPase), 'UniformOutput', false)]), 'stable');
    GTPase_uni      = cellfun(@(x) str2double(x), table2cell(GTPase_uni(:, 2)));

    % Fit values of rates (only quadratic dependency on quadratic effectors)
    y_fit           = bsxfun(@plus, k1 * c_corr' .* GTPase_uni' + k2 * (c_corr' .* GTPase_uni') .^ 2, ...
                        bsxfun(@times, k3' .* c_corr' .* GTPase_uni', c_probe .^ (1 + ~isempty(Proteins.QuaEff))));
else % No effectors present, only GTPase is varied
    % Rate parameters
    k1              = Assays_processed.(Assay_name).k_runs.est(:, 1);
    k2              = Assays_processed.(Assay_name).k_runs.est(:, 2);
    
    % Fit values of rates (only quadratic dependency on Cdc24)
    y_fit           = bsxfun(@times, k1', c_probe) + bsxfun(@times, k2', c_probe .^ 2);
    
end

%% Rate plot as function of concentration

% Define figure
fig_rate_conc     = figure;
set(fig_rate_conc, 'Renderer', 'painters', 'Units', 'inches', 'Position', pos_plot);

% Define axis
ax_fig_rate_conc  = axes('Parent', fig_rate_conc);
set(ax_fig_rate_conc, 'Units', 'inches', 'Position', pos_ax, ...
    'FontName', 'Arial', 'FontSize', plot_font_size, ...
    'LabelFontSizeMultiplier', 1, 'TitleFontSizeMultiplier', 1, ...
    'NextPlot', 'add', 'LineWidth', 1, 'Box', 'on', 'color', 'none');
xlabel(ax_fig_rate_conc, [varied_protein ' [\muM]']);
ylabel(ax_fig_rate_conc, 'overall GTP hydr. rate [h^{-1}]');

% colors of the lines
plot_colors         = useful_colormaps('viridis', max(10, 3 + 2 * numel(uni_run_names))); % [1]

%% Plot data points and fit

y_max_plot          = 0; % Maximum y-value, gets updated and relevant for plot scaling
for j = 1 : numel(uni_run_names) % iterate over all runs
    % Indices of points to select corrsponding to the run in this iteration
    ind_p           = strcmp(run_name_list, uni_run_names{j});
    % Error bars for rates
    err_bar         = errorbar(c(ind_p), -log(y(ind_p)) ./ t(ind_p), ...
                        -log(y(ind_p)) ./ t(ind_p) + log(y(ind_p) + err(ind_p)) ./ t(ind_p), ...
                        -log(y(ind_p) - err(ind_p)) ./ t(ind_p) + log(y(ind_p)) ./ t(ind_p), ...
                        'LineStyle', 'none', 'Color', plot_colors(1 + 2 * j, :), ...
                        'Marker', 'd', 'MarkerSize', 8, 'LineWidth', 1, ...
                        'Parent', ax_fig_rate_conc, 'DisplayName', uni_run_names{j});
    % Maximum y-value, relevant for plot scaling   
    y_max_plot      = max(max(y_max_plot, -log(y(ind_p)) ./ t(ind_p) ...
                        -log(y(ind_p) - err(ind_p)) ./ t(ind_p) ...
                        + log(y(ind_p)) ./ t(ind_p)));
end

% Make rate fit lines
fit_lines               = cell(numel(uni_run_names), 1);
for j = 1 : numel(uni_run_names) % iterate over all runs
    fit_lines{j}        = line(c_probe, y_fit(:, j), ...
                            'LineStyle', '--', 'Color', plot_colors(1 + 2 * j, :), ...
                            'LineWidth', 1, 'Parent', ax_fig_rate_conc, ...
                            'DisplayName', 'Fits');
    % Remove fit entries in legend, except for one
    if j > 1
        fit_lines{j}.Annotation.LegendInformation.IconDisplayStyle = 'off';
    end
end

%% Adjust axis properties
x_limits                = [min(c_probe(:, 1)) ceil(max(c(:, 1) + 0.1) * 10) / 10];
x_tick_spacing          = round(diff(x_limits) / 5 * 10) / 10;
y_limits1               = [0 max(0.5, ceil(max(y_max_plot) * 10) / 10)];
y_tick_spacing1         = round(diff(y_limits1) / 5 * 10) / 10;
set(ax_fig_rate_conc, 'XLim', x_limits, 'XTick', 0 : x_tick_spacing : x_limits(2), ...
    'YLim', y_limits1, 'YTick', 0 : y_tick_spacing1 : y_limits1(2));

le_fits                 = legend(ax_fig_rate_conc, 'show');
set(le_fits, 'FontName', plot_font_name, 'FontSize', plot_font_size - 1, ...
    'Units', 'inches', 'Location', 'EastOutside', 'Box', 'off');

%% Export figure to .tiff and .pdf
print(fig_rate_conc, '-dtiffn', strcat('-r', num2str(resol_for_print)), ...
    ['Rate as function of ' varied_protein ' concentration'])
fig_rate_conc.PaperOrientation  = 'landscape';
fig_rate_conc.PaperType         = 'A3';
print(fig_rate_conc, '-dpdf', ['Rate as function of ' varied_protein ' concentration'])