%% Plot of time dependent runs
% Colormaps from [1].
%
% Date: 23-11-2023
%
% Bibliography
% [1] Smith, N. J., Walt, S. van der, & Firing, E. (2015). Magma, inferno,
%     plasma and viridis colormaps,
%     https://github.com/BIDS/colormap/blob/master/colormaps.py.

clc
clearvars
close all
addpath('./Additional functions/') % Folder with the model functions

%% Plot properties that can be changed
x_tick_spacing      = 0 : 10;
y_limits            = [5 105];
y_tick_spacing      = 10 : 30 : 100;
plot_font_name      = 'Arial';
plot_font_size      = 10;
print_res           = 150;

%% Extract data

%%%%%%%%%%%%%%%%%%%%%%%%%
% Determine which assay %
%%%%%%%%%%%%%%%%%%%%%%%%%

[file_name1, file_name2] = uigetfile('.mat', 'Select Data_assays.mat file');
load([file_name2 file_name1], 'Assays_processed')

% dialog box for selecting which assays to plot
panel1              = dialog('Name', 'Select assay', 'Position', [50 50 350 200]);
all_assays          = fieldnames(Assays_processed);
% popupmenus for order in which to plot which assays
box_assay           = uicontrol('Style', 'popupmenu', 'units', 'normalized', ...
                        'Position', [0.1 0.2 0.35 0.2], ...
                        'String', ['empty'; all_assays], 'Parent', panel1);
% button when done with selecting assays
uicontrol('Style', 'pushbutton', 'units', 'normalized', ...
                        'Position', [0.6 0.2 0.35 0.2], ...
                        'String', 'Start', 'Parent', panel1, 'Callback', 'uiresume(panel1)');
% Wait until selection is made
uiwait(panel1)
% Indices of selected assays
assay_no            = box_assay.Value - 1;
% choice is completed, close the dialog box again
close(panel1)
assay_choice        = all_assays{assay_no};

%%%%%%%%%%%%%%%%%%%%%%%
% Determine which run %
%%%%%%%%%%%%%%%%%%%%%%%
% dialog box for selecting which runs to plot
panel2              = dialog('Name', 'Select run', 'Position', [70 50 350 200]);
all_runs            = unique(Assays_processed.(assay_choice).Data.Run, 'stable');
% popupmenus for order in which to plot which runs
box_run             = uicontrol('Style', 'popupmenu', 'units', 'normalized', ...
                        'Position', [0.1 0.2 0.35 0.2], ...
                        'String', ['empty'; all_runs], 'Parent', panel2);
% button when done with selecting assays
uicontrol('Style', 'pushbutton', 'units', 'normalized', ...
                        'Position', [0.6 0.2 0.35 0.2], ...
                        'String', 'Start', 'Parent', panel2, 'Callback', 'uiresume(panel2)');
% Wait until selection is made
uiwait(panel2)
% Indices of selected assays
run_no              = box_run.Value - 1;
% choice is completed, close the dialog box again
close(panel2)
run_choice          = all_runs{run_no};

%% Extract data (protein part)
% GTPase in this assay
GTPase              = Assays_processed.(assay_choice).Proteins.GTPase{1};
% Find the varied proteins
Proteins_var        = cat(2, Assays_processed.(assay_choice).Proteins.QuaEff, ...
                        Assays_processed.(assay_choice).Proteins.LinEff);
% If no effectors, then the varied protein is the GTPase
if isempty(Proteins_var)
    Proteins_var    = Assays_processed.(assay_choice).Proteins.GTPase;
end
% The first varied protein will be color coded as indicated in the legend
varied_protein      = Proteins_var{1};

%% Extract data further
Data_tot            = Assays_processed.(assay_choice).Data(strcmp(Assays_processed.(assay_choice).Data.Run, run_choice), :);
pars                = Assays_processed.(assay_choice).k_runs.est(strcmp(Assays_processed.(assay_choice).k_runs.name, run_choice), :);
t                   = Data_tot.Time;
y                   = 100 * Data_tot.GTP_remaining;
err                 = 100 * Data_tot.Error;
x                   = Data_tot.([varied_protein '_conc']);
t_probe             = linspace(0, max(t) * 1.5, 1e3)';

t_uni               = unique(t);
x_uni               = unique(x);
plot_colors         = useful_colormaps('viridis', max(10, 3 + 2 * numel(x_uni))); % colormaps from [1]

if isempty(Assays_processed.(assay_choice).Proteins.LinEff) && isempty(Assays_processed.(assay_choice).Proteins.QuaEff)
    rate_fit    = pars(1) * x_uni' + pars(2) * x_uni' .^ 2;
elseif isempty(Assays_processed.(assay_choice).Proteins.LinEff) && ~isempty(Assays_processed.(assay_choice).Proteins.QuaEff) && ...
        numel(Assays_processed.(assay_choice).Proteins.QuaEff) == 1
    GTPase_conc = mode(Data_tot.([GTPase '_conc']));
    conc_corr   = Assays_processed.(assay_choice).conc_corr(run_no);
    rate_fit    = GTPase_conc * conc_corr * (pars(1) + pars(2) * x_uni' .^ 2);
elseif ~isempty(Assays_processed.(assay_choice).Proteins.LinEff) && isempty(Assays_processed.(assay_choice).Proteins.QuaEff) && ...
        numel(Assays_processed.(assay_choice).Proteins.LinEff) == 1
    GTPase_conc = mode(Data_tot.([GTPase '_conc']));
    conc_corr   = Assays_processed.(assay_choice).conc_corr(run_no);
    rate_fit    = GTPase_conc * conc_corr * (pars(1) + pars(2) * x_uni');
else
    error('Plotting assays with multiple effectors currently not supported')
end
    
y_fit               = 100 * exp(-bsxfun(@times, rate_fit,  t_probe));

%% Define figure and axis
x_limits            = [0 max(t) * 1.1];
fig_rate_fit        = gcf;
set(fig_rate_fit, 'Renderer', 'painters', 'PaperPositionMode', 'auto', 'Units', 'inches', 'Position', [1.5 1.5 4.5 3.1]);
ax_fig_rate_fit     = axes('Parent', fig_rate_fit);
set(ax_fig_rate_fit, 'Units', 'inches', 'Position', [0.6 0.5 3.3 2.3], 'FontName', plot_font_name, ...
    'FontSize', plot_font_size, 'LabelFontSizeMultiplier', 1, 'TitleFontSizeMultiplier', 1, 'NextPlot', 'add', ...
    'LineWidth', 1, 'Box', 'on', 'color', 'none', 'XLim', x_limits, 'XTick', 0 : 10, ...
    'YLim', y_limits, 'YTick', y_tick_spacing, 'YScale', 'log');
ax_fig_rate_fit.YLabel.String  = 'fraction: remaining GTP';
ax_fig_rate_fit.XLabel.String  = 'time [h]';

%% Plot data points and fit
y_max_plot          = 0;
fit_lines           = cell(numel(x_uni), 1);
for j = 1 : numel(x_uni)
    ind_p           = x == x_uni(j);
    err_bar         = errorbar(t(ind_p), y(ind_p), err(ind_p), 'LineStyle', 'none', 'Color', plot_colors(1 + 2 * j, :), ...
                        'Marker', 'd', 'MarkerSize', 8, 'LineWidth', 1, 'Parent', ax_fig_rate_fit, ...
                        'DisplayName', sprintf('%s \\muM', num2str(x_uni(j))));
    fit_lines{j}    = line(t_probe, y_fit(:, j), 'LineStyle', '--', 'Color', plot_colors(1 + 2 * j, :), ...
                        'LineWidth', 1, 'Parent', ax_fig_rate_fit);
    fit_lines{j}.Annotation.LegendInformation.IconDisplayStyle = 'off';
end

%% Legend
le_fits             = legend(ax_fig_rate_fit, 'show');
set(le_fits, 'FontName', plot_font_name, 'FontSize', plot_font_size - 1, ...
    'Units', 'inches', 'Location', 'SouthWest', 'Box', 'off');

% Position protein legend text
lp                  = le_fits.Position;
ap                  = ax_fig_rate_fit.Position;
y_le_end            = exp(log(y_limits(1)) + (sum(lp([2 4])) - ap(2)) / ap(4) * diff(log(y_limits)));
y_line              = exp(log(y_limits(1)) + ((sum(lp([2 4])) - ap(2)) / ap(4) + 0.05) * diff(log(y_limits)));
y_text              = exp(log(y_limits(1)) + ((sum(lp([2 4])) - ap(2)) / ap(4) + 0.10)* diff(log(y_limits)));
x_line              = [x_limits(1) + ((lp(1) - ap(1)) / ap(3) + 0.00) * diff(x_limits) ...
                        x_limits(1) + ((sum(lp([1 3])) - ap(1)) / ap(3) + 0.00) * diff(x_limits)];

Varied_protein_text = text(x_line(1) + diff(x_line) / 2, y_text, varied_protein, 'FontName', plot_font_name, 'FontSize', plot_font_size - 1);
Varied_protein_line = line(x_line, y_line * [1 1], 'LineStyle', '-', 'Color', [0 0 0], 'Parent', ax_fig_rate_fit);
Varied_protein_line.Annotation.LegendInformation.IconDisplayStyle = 'off';

% Export figure to .tiff and .pdf
print(fig_rate_fit, '-dtiffn', ['Semilog_GTP_time_plot_' varied_protein])
print(fig_rate_fit, '-dpdf',   ['Semilog_GTP_time_plot_' varied_protein])