from pathlib import Path

import h5py
import lmfit
import matplotlib.gridspec as gridspec
import matplotlib.pyplot as plt
import numpy as np
import scipy.stats as st
import uncertainties as un
from mpl_toolkits.axes_grid1.inset_locator import inset_axes
from quantify_core.data.handling import (
    load_dataset,
    set_datadir,
)
from scipy.special import wofz
from uncertainties.umath import sqrt

set_datadir(Path("../data"))


def lorentz(x, offset, gamma, center, amp):
    lorentzfunc = (
        amp / np.pi * (gamma / 2) / ((x - center) ** 2 + (gamma / 2) ** 2) + offset
    )
    return lorentzfunc


def gaussian(x, offset, amp, center, sigma):
    exponent = ((x - center) / sigma) ** 2
    gaussian = amp * np.exp(-exponent / 2)
    return offset + gaussian


def double_gaussian(x, offset, amp_1, amp_2, center_1, center_2, sigma_1, sigma_2):
    exponent_1 = ((x - center_1) / sigma_1) ** 2
    exponent_2 = ((x - center_2) / sigma_2) ** 2
    gaussian_1 = np.abs(amp_1) * np.exp(-exponent_1 / 2)
    gaussian_2 = np.abs(amp_2) * np.exp(-exponent_2 / 2)
    return offset + gaussian_1 + gaussian_2


def Voigt(x, offset, center, amp, sigma, gamma):
    return offset + amp * np.real(
        wofz((x - center + 1j * gamma) / sigma / np.sqrt(2))
    ) / sigma / np.sqrt(2 * np.pi)


lifetime_limit = 0.013
fit_error = 0.1


def calc_voigt_width(sigma, gamma):
    if sigma.stderr is None:
        sig = un.ufloat(sigma.value, fit_error * sigma.value)
    else:
        sig = un.ufloat(sigma.value, sigma.stderr)
    if gamma.stderr is None:
        gam = 2 * un.ufloat(gamma.value, fit_error * gamma.value)
    else:
        gam = 2 * un.ufloat(gamma.value, gamma.stderr)
    return 0.5346 * gam + sqrt(
        0.2166 * gam**2 + (2 * sig * np.sqrt(2 * np.log(2))) ** 2
    )


# Load and anlyse NV Data

with h5py.File("../data/20220804/2022-08-04_confocal_scan.h5", "r") as h5f:
    counts_large_map = h5f["cyclops_2d"][:]

dataset_nv = load_dataset("20220805-161039-403-0e202c-line_scan")

start_1, end_1 = 270, 370
start_2, end_2 = 550, 750
start_3, end_3 = 1040, 1140

fit_model_voigt = lmfit.Model(Voigt)
fit_model_voigt.set_param_hint("offset", value=10)
fit_model_voigt.set_param_hint("gamma", value=0.02, min=lifetime_limit / 2)
fit_model_voigt.set_param_hint("sigma", value=0.04, min=0.001)
fit_model_voigt.set_param_hint("center", value=67.55)
fit_model_voigt.set_param_hint("amp", value=16)
fit_result_voigt = fit_model_voigt.fit(
    dataset_nv.y0[start_1:end_1],
    x=dataset_nv.x0[start_1:end_1],
    params=fit_model_voigt.make_params(),
)

fit_model_voigt_2 = lmfit.Model(Voigt)
fit_model_voigt_2.set_param_hint("offset", value=10)
fit_model_voigt_2.set_param_hint("gamma", value=0.02, min=lifetime_limit / 2)
fit_model_voigt_2.set_param_hint("sigma", value=0.04, min=0.001)
fit_model_voigt_2.set_param_hint("center", value=70.25)
fit_model_voigt_2.set_param_hint("amp", value=290)
fit_result_voigt_2 = fit_model_voigt_2.fit(
    dataset_nv.y0[start_2:end_2],
    x=dataset_nv.x0[start_2:end_2],
    params=fit_model_voigt_2.make_params(),
)

fit_model_voigt_3 = lmfit.Model(Voigt)
fit_model_voigt_3.set_param_hint("offset", value=10)
fit_model_voigt_3.set_param_hint("gamma", value=0.02, min=lifetime_limit / 2)
fit_model_voigt_3.set_param_hint("sigma", value=0.04, min=0.001)
fit_model_voigt_3.set_param_hint("center", value=73.7)
fit_model_voigt_3.set_param_hint("amp", value=290)
fit_result_voigt_3 = fit_model_voigt_3.fit(
    dataset_nv.y0[start_3:end_3],
    x=dataset_nv.x0[start_3:end_3],
    params=fit_model_voigt_3.make_params(),
)

# Load and anlyse SnV data

with h5py.File(
    "../data/20230213/135318_Measurement_autoNVsearch/Z_depth_52.4/141449_scan2d_.hdf5",
    "r",
) as h5f:
    x = h5f["x"][:]
    y = h5f["x"][:]
    counts = h5f["countrate"][:]

dataset_snv_1 = load_dataset("20230316-205912-136-74ef68-fast_line_scan")

lorentz_centers_1, peaks_fwhm_1 = list(), list()
exclude_scans_1 = [16, 33, 35, 60, 62, 77, 89]

for rep in range(len(dataset_snv_1.x0)):
    if rep not in exclude_scans_1:
        fit_model_lorentz = lmfit.Model(lorentz)
        fit_model_lorentz.set_param_hint("offset", value=200)
        fit_model_lorentz.set_param_hint("gamma", value=0.05)
        fit_model_lorentz.set_param_hint("center", value=123)
        fit_model_lorentz.set_param_hint("amp", value=4000)
        fit_result_lorentz = fit_model_lorentz.fit(
            dataset_snv_1.y0[rep],
            x=dataset_snv_1.x1,
            params=fit_model_lorentz.make_params(),
        )
        peaks_fwhm_1.append(fit_result_lorentz.params["gamma"].value)
        lorentz_centers_1.append(fit_result_lorentz.params["center"].value)

traces_with_snv_1 = list()
for i in range(len(dataset_snv_1.x0)):
    if dataset_snv_1.x0[i] not in exclude_scans_1:
        traces_with_snv_1.append(dataset_snv_1.y0[i])

summed_gaussian_1 = dataset_snv_1.y0.sum(axis=0)

bin_freq_1 = np.arange(-0.75, 0.75, 0.005)
bin_sum_1 = np.zeros(len(bin_freq_1) - 1)
for i in range(len(traces_with_snv_1)):
    cent = lorentz_centers_1[i]
    act_binned, _, _ = st.binned_statistic(
        dataset_snv_1.x1 - cent, traces_with_snv_1[i], statistic="sum", bins=bin_freq_1
    )
    bin_sum_1 += act_binned

fit_model_gauss_1 = lmfit.Model(gaussian)
fit_model_gauss_1.set_param_hint("offset", value=100)
fit_model_gauss_1.set_param_hint("amp", value=10000)
fit_model_gauss_1.set_param_hint("center", value=123)
fit_model_gauss_1.set_param_hint("sigma", value=0.2)
fit_result_gauss_1 = fit_model_gauss_1.fit(
    summed_gaussian_1,
    x=dataset_snv_1.x1,
    params=fit_model_gauss_1.make_params(),
)

fit_model_lorentz_1 = lmfit.Model(lorentz)
fit_model_lorentz_1.set_param_hint("offset", value=100)
fit_model_lorentz_1.set_param_hint("gamma", value=0.1)
fit_model_lorentz_1.set_param_hint("center", value=0)
fit_model_lorentz_1.set_param_hint("amp", value=100000)
fit_result_lorentz_1 = fit_model_lorentz_1.fit(
    bin_sum_1,
    x=bin_freq_1[:-1],
    params=fit_model_lorentz_1.make_params(),
)

norm_gauss_1 = max(
    gaussian(
        x=np.linspace(dataset_snv_1.x1[0].values, dataset_snv_1.x1[-1].values, 1000),
        **fit_result_gauss_1.best_values,
    )
)
norm_lorentz_1 = max(
    lorentz(
        x=np.linspace(bin_freq_1[0], bin_freq_1[-1], 1000),
        **fit_result_lorentz_1.best_values,
    )
)

dataset_snv_2 = load_dataset("20230320-205050-772-1af654-fast_line_scan")
freqiencies_corrected_2 = dataset_snv_2.x1 - 13600
lorentz_centers_2, peaks_fwhm_snv_2 = list(), list()
left_scans_2 = [
    2,
    3,
    10,
    16,
    20,
    30,
    32,
    33,
    43,
    46,
    49,
    53,
    54,
    56,
    58,
    66,
    69,
    70,
    71,
    72,
    73,
    74,
    79,
    80,
    83,
    92,
    93,
    98,
    99,
]
right_scans_2 = [
    0,
    7,
    9,
    12,
    15,
    17,
    19,
    21,
    23,
    24,
    25,
    26,
    27,
    28,
    36,
    38,
    39,
    44,
    45,
    48,
    52,
    55,
    57,
    59,
    61,
    63,
    65,
    67,
    68,
    75,
    76,
    77,
    78,
    81,
    82,
    86,
    87,
    90,
    91,
    96,
]
include_scans_2 = left_scans_2 + right_scans_2
include_scans_2.sort()

for rep in range(len(dataset_snv_2.x0)):
    if rep in include_scans_2:
        fit_model_lorentz = lmfit.Model(lorentz)
        fit_model_lorentz.set_param_hint("offset", value=50)
        fit_model_lorentz.set_param_hint("gamma", value=0.05, min=0, max=0.1)
        fit_model_lorentz.set_param_hint("center", value=123.3, min=123.1, max=123.5)
        fit_model_lorentz.set_param_hint("amp", value=4000)
        fit_result_lorentz = fit_model_lorentz.fit(
            dataset_snv_2.y0[rep],
            x=freqiencies_corrected_2,
            params=fit_model_lorentz.make_params(),
        )
        lorentz_centers_2.append(fit_result_lorentz.params["center"].value)
        peaks_fwhm_snv_2.append(fit_result_lorentz.params["gamma"].value)

traces_with_snv_2 = list()
for i in range(len(dataset_snv_2.x0)):
    if dataset_snv_2.x0[i] in include_scans_2:
        traces_with_snv_2.append(dataset_snv_2.y0[i])

summed_gaussian_2 = dataset_snv_2.y0.sum(axis=0)

bin_freq_2 = np.arange(-0.75, 0.75, 0.005)
bin_sum_2 = np.zeros(len(bin_freq_2) - 1)
for i in range(len(traces_with_snv_2)):
    cent = lorentz_centers_2[i]
    act_binned, _, _ = st.binned_statistic(
        freqiencies_corrected_2 - cent,
        traces_with_snv_2[i],
        statistic="sum",
        bins=bin_freq_2,
    )
    bin_sum_2 += act_binned

fit_model_gauss_2 = lmfit.Model(double_gaussian)
fit_model_gauss_2.set_param_hint("offset", value=5000)
fit_model_gauss_2.set_param_hint("amp_1", value=100000, min=0)
fit_model_gauss_2.set_param_hint("center_1", value=123.2)
fit_model_gauss_2.set_param_hint("sigma_1", value=0.1)
fit_model_gauss_2.set_param_hint("amp_2", value=500000, min=0)
fit_model_gauss_2.set_param_hint("center_2", value=123.35)
fit_model_gauss_2.set_param_hint("sigma_2", value=0.1)
fit_result_gauss_2 = fit_model_gauss_2.fit(
    summed_gaussian_2,
    x=freqiencies_corrected_2,
    params=fit_model_gauss_2.make_params(),
)

fit_model_lorentz_2 = lmfit.Model(lorentz)
fit_model_lorentz_2.set_param_hint("offset", value=100)
fit_model_lorentz_2.set_param_hint("gamma", value=0.1)
fit_model_lorentz_2.set_param_hint("center", value=0)
fit_model_lorentz_2.set_param_hint("amp", value=100000)
fit_result_lorentz_2 = fit_model_lorentz_2.fit(
    bin_sum_2,
    x=bin_freq_2[:-1],
    params=fit_model_lorentz_2.make_params(),
)

norm_gauss_2 = max(
    double_gaussian(
        x=np.linspace(
            freqiencies_corrected_2[0].values, freqiencies_corrected_2[-1].values, 1000
        ),
        **fit_result_gauss_2.best_values,
    )
)
norm_lorentz_2 = max(
    lorentz(
        x=np.linspace(bin_freq_2[0], bin_freq_2[-1], 1000),
        **fit_result_lorentz_2.best_values,
    )
)
dataset_snv_3 = load_dataset("20230320-230754-584-58d6c6-fast_line_scan")
freqiencies_corrected_3 = dataset_snv_3.x1 - 13600

lorentz_centers_3, peaks_fwhm_snv_3 = list(), list()
exclude_scans_3 = [
    6,
    7,
    9,
    14,
    17,
    23,
    24,
    25,
    28,
    29,
    36,
    41,
    45,
    46,
    47,
    48,
    49,
    53,
    58,
    59,
    61,
    62,
    63,
    64,
    65,
    66,
    67,
    75,
    76,
    77,
    81,
    82,
    83,
    85,
    86,
    87,
    89,
    90,
    94,
    95,
    97,
    99,
]

for rep in range(len(dataset_snv_3.x0)):
    if rep not in exclude_scans_3:
        fit_model_lorentz = lmfit.Model(lorentz)
        fit_model_lorentz.set_param_hint("offset", value=200)
        fit_model_lorentz.set_param_hint("gamma", value=0.05, min=0)
        fit_model_lorentz.set_param_hint("center", value=123.4)
        fit_model_lorentz.set_param_hint("amp", value=4000, min=0)
        fit_result_lorentz = fit_model_lorentz.fit(
            dataset_snv_3.y0[rep],
            x=freqiencies_corrected_3,
            params=fit_model_lorentz.make_params(),
        )
        peaks_fwhm_snv_3.append(fit_result_lorentz.params["gamma"].value)
        lorentz_centers_3.append(fit_result_lorentz.params["center"].value)

traces_with_snv_3 = list()
for i in range(len(dataset_snv_3.x0)):
    if dataset_snv_3.x0[i] not in exclude_scans_3:
        traces_with_snv_3.append(dataset_snv_3.y0[i])

summed_gaussian_3 = dataset_snv_3.y0.sum(axis=0)

bin_freq_3 = np.arange(-0.75, 0.75, 0.005)
bin_sum_3 = np.zeros(len(bin_freq_3) - 1)
for i in range(len(traces_with_snv_3)):
    cent = lorentz_centers_3[i]
    act_binned, _, _ = st.binned_statistic(
        freqiencies_corrected_3 - cent,
        traces_with_snv_3[i],
        statistic="sum",
        bins=bin_freq_3,
    )
    bin_sum_3 += act_binned

fit_model_gauss_3 = lmfit.Model(double_gaussian)
fit_model_gauss_3.set_param_hint("offset", value=10000)
fit_model_gauss_3.set_param_hint("amp_1", value=100000)
fit_model_gauss_3.set_param_hint("center_1", value=123.3)
fit_model_gauss_3.set_param_hint("sigma_1", value=0.21)
fit_model_gauss_3.set_param_hint("amp_2", value=70000)
fit_model_gauss_3.set_param_hint("center_2", value=123.5)
fit_model_gauss_3.set_param_hint("sigma_2", value=0.1)
fit_result_gauss_3 = fit_model_gauss_3.fit(
    summed_gaussian_3,
    x=freqiencies_corrected_3,
    params=fit_model_gauss_3.make_params(),
)

fit_model_lorentz_3 = lmfit.Model(lorentz)
fit_model_lorentz_3.set_param_hint("offset", value=100)
fit_model_lorentz_3.set_param_hint("gamma", value=0.1, min=0)
fit_model_lorentz_3.set_param_hint("center", value=0)
fit_model_lorentz_3.set_param_hint("amp", value=500, min=0)
fit_result_lorentz_3 = fit_model_lorentz_3.fit(
    bin_sum_3,
    x=bin_freq_3[:-1],
    params=fit_model_lorentz_3.make_params(),
)

norm_gauss_3 = max(
    double_gaussian(
        x=np.linspace(
            freqiencies_corrected_3[0].values, freqiencies_corrected_3[-1].values, 1000
        ),
        **fit_result_gauss_3.best_values,
    )
)
norm_lorentz_3 = max(
    lorentz(
        x=np.linspace(bin_freq_3[0], bin_freq_3[-1], 1000),
        **fit_result_lorentz_3.best_values,
    )
)

nv_color_fit = "lightcoral"
nv_color_data = "firebrick"

snv_color_fit = "orange"
snv_color_data = "darkorange"
snv_gaussian_color_fit = "tan"
snv_gaussian_color_data = "peru"

fig = plt.figure(figsize=(12, 8))
grid = gridspec.GridSpec(2, 1, hspace=0.25, figure=fig)

nv_panels = gridspec.GridSpecFromSubplotSpec(
    1,
    2,
    grid[0, :],
    width_ratios=[
        1.5,
        3.5,
    ],
    wspace=0.2,
)

nv_ple_panels = gridspec.GridSpecFromSubplotSpec(
    1,
    3,
    nv_panels[0, 1],
    wspace=0.22,
)

snv_panels = gridspec.GridSpecFromSubplotSpec(
    1,
    2,
    grid[1, :],
    width_ratios=[
        1.5,
        3.5,
    ],
    wspace=0.2,
)

snv_ple_panels = gridspec.GridSpecFromSubplotSpec(
    1,
    3,
    snv_panels[0, 1],
    wspace=0.22,
)

ax_1 = plt.subplot(nv_panels[0, 0])
ax_2 = plt.subplot(nv_ple_panels[0, 0])
ax_3 = plt.subplot(nv_ple_panels[0, 1])
ax_4 = plt.subplot(nv_ple_panels[0, 2])

ax_5 = plt.subplot(snv_panels[0, 0])
ax_6 = plt.subplot(snv_ple_panels[0, 0])
ax_7 = plt.subplot(snv_ple_panels[0, 1])
ax_8 = plt.subplot(snv_ple_panels[0, 2])

ax_1.text(-0.2, 1.08, "(a)", transform=ax_1.transAxes, fontsize=12, va="top")
img = ax_1.pcolormesh(
    -1 * x,
    -1 * y,
    counts.transpose()[::-1] / 1000,
    vmax=8,
    cmap="Oranges",
)
ax_1.set_xlabel("X Position (µm)", fontsize=12)
ax_1.set_ylabel("Y Position (µm)", fontsize=12)
ax_1.set_xticks([0, 10, 20, 30, 40, 50])
axinset = inset_axes(
    ax_1,
    width="10%",
    height="30%",
    bbox_to_anchor=(-0.1, 0.04, 0.95, 0.94),
    bbox_transform=ax_1.transAxes,
    loc="lower right",
)
cbar = fig.colorbar(img, cax=axinset, orientation="vertical")
cbar.set_label("Counts (kCts/s)", color="black", fontsize=10)
cbar.ax.tick_params(color="black", labelsize=10, labelcolor="black")
cbar.outline.set_edgecolor("black")

ax_2.text(-0.34, 1.08, "(b)", transform=ax_2.transAxes, fontsize=12, va="top")
ax_2.scatter(
    dataset_snv_1.x1,
    summed_gaussian_1 / norm_gauss_1,
    s=15,
    color=snv_gaussian_color_data,
    label="Summed Counts",
)
ax_2.plot(
    np.linspace(dataset_snv_1.x1[0].values, dataset_snv_1.x1[-1].values, 1000),
    gaussian(
        x=np.linspace(dataset_snv_1.x1[0].values, dataset_snv_1.x1[-1].values, 1000),
        **fit_result_gauss_1.best_values,
    )
    / norm_gauss_1,
    color=snv_gaussian_color_fit,
    linestyle="solid",
    label="(Double) Gaussian Fit",
)
ax_2.scatter(
    bin_freq_1[:-1] + fit_result_gauss_1.params["center"].value,
    bin_sum_1 / norm_lorentz_1,
    s=15,
    color=snv_color_data,
    label="Centered Summed Counts",
)
ax_2.plot(
    np.linspace(bin_freq_1[0], bin_freq_1[-2], 1000)
    + fit_result_gauss_1.params["center"].value,
    lorentz(
        x=np.linspace(bin_freq_1[0], bin_freq_1[-2], 1000),
        **fit_result_lorentz_1.best_values,
    )
    / norm_lorentz_1,
    color=snv_color_fit,
    linestyle="solid",
    label="Lorentzian Fit",
)
ax_2.set_xlim(122.7, 123.3)
ax_2.annotate(
    "",
    xy=(123.09, 0.5),
    xycoords="data",
    xytext=(123.17, 0.5),
    textcoords="data",
    arrowprops=dict(arrowstyle="-|>", color=snv_gaussian_color_fit, lw=2),
)
ax_2.annotate(
    "",
    xy=(122.88, 0.5),
    xycoords="data",
    xytext=(122.96, 0.5),
    textcoords="data",
    arrowprops=dict(arrowstyle="<|-", color=snv_gaussian_color_fit, lw=2),
)
ax_2.annotate(
    "{:.1f}\nMHz".format(
        1000 * 2 * np.sqrt(2 * np.log(2)) * fit_result_gauss_1.params["sigma"].value
    ),
    xy=(123.18, 0.45),
    xycoords="data",
    xytext=(123.18, 0.45),
    color=snv_gaussian_color_fit,
    textcoords="data",
    fontsize=8,
)
ax_2.annotate(
    "{:.1f} MHz\n+/- {:.1f} MHz".format(
        1000 * fit_result_lorentz_1.params["gamma"].value,
        1000 * fit_result_lorentz_1.params["gamma"].stderr,
    ),
    xy=(122.72, 0.94),
    xycoords="data",
    xytext=(122.71, 0.94),
    textcoords="data",
    color=snv_color_fit,
    fontsize=8,
)
ax_2.set_ylabel("Normalized Counts", fontsize=12)
ax_2.set_ylim(-0.05, 1.08)

ax_3.scatter(
    freqiencies_corrected_2,
    summed_gaussian_2 / norm_gauss_2,
    s=15,
    color=snv_gaussian_color_data,
    label="Summed Counts",
)
ax_3.plot(
    np.linspace(
        freqiencies_corrected_2[0].values, freqiencies_corrected_2[-1].values, 1000
    ),
    double_gaussian(
        x=np.linspace(
            freqiencies_corrected_2[0].values, freqiencies_corrected_2[-1].values, 1000
        ),
        **fit_result_gauss_2.best_values,
    )
    / norm_gauss_2,
    color=snv_gaussian_color_fit,
    linestyle="solid",
    label="(Double) Gaussian Fit",
)
ax_3.scatter(
    bin_freq_2[:-1] + fit_result_gauss_2.params["center_2"].value,
    bin_sum_2 / norm_lorentz_2,
    s=15,
    color=snv_color_data,
    label="Centered Summed Counts",
)
ax_3.plot(
    np.linspace(bin_freq_2[0], bin_freq_2[-2], 1000)
    + fit_result_gauss_2.params["center_2"].value,
    lorentz(
        x=np.linspace(bin_freq_2[0], bin_freq_2[-2], 1000),
        **fit_result_lorentz_2.best_values,
    )
    / norm_lorentz_2,
    color=snv_color_fit,
    linestyle="solid",
    label="Lorentzian Fit",
)
ax_3.set_xlim(123, 123.6)
ax_3.annotate(
    "",
    xy=(123.13, 0.5),
    xycoords="data",
    xytext=(123.21, 0.5),
    textcoords="data",
    arrowprops=dict(arrowstyle="<|-", color=snv_gaussian_color_fit, lw=2),
)
ax_3.annotate(
    "",
    xy=(123.37, 0.5),
    xycoords="data",
    xytext=(123.45, 0.5),
    textcoords="data",
    arrowprops=dict(arrowstyle="-|>", color=snv_gaussian_color_fit, lw=2),
)
ax_3.annotate(
    "176.0\nMHz",
    xy=(123.5, 0.45),
    xycoords="data",
    xytext=(123.45, 0.45),
    color=snv_gaussian_color_fit,
    textcoords="data",
    fontsize=8,
)
ax_3.annotate(
    "{:.1f} MHz\n+/- {:.1f} MHz".format(
        1000 * fit_result_lorentz_2.params["gamma"].value,
        1000 * fit_result_lorentz_2.params["gamma"].stderr,
    ),
    xy=(123.04, 0.94),
    xycoords="data",
    xytext=(123.01, 0.94),
    textcoords="data",
    color=snv_color_fit,
    fontsize=8,
)
ax_3.set_ylim(-0.05, 1.08)
ax_3.set_xlabel("Laser Excitation Frequency w.r.t. 484.4 THz (GHz)", fontsize=11)

ax_4.scatter(
    freqiencies_corrected_3,
    summed_gaussian_3 / norm_gauss_3,
    s=15,
    color=snv_gaussian_color_data,
    label="Summed Counts",
)
ax_4.plot(
    np.linspace(
        freqiencies_corrected_3[0].values, freqiencies_corrected_3[-1].values, 1000
    ),
    double_gaussian(
        x=np.linspace(
            freqiencies_corrected_3[0].values, freqiencies_corrected_3[-1].values, 1000
        ),
        **fit_result_gauss_3.best_values,
    )
    / norm_gauss_3,
    color=snv_gaussian_color_fit,
    linestyle="solid",
    label="(Double) Gaussian Fit",
)
ax_4.scatter(
    bin_freq_3[:-1] + fit_result_gauss_3.params["center_1"].value,
    bin_sum_3 / norm_lorentz_3,
    s=15,
    color=snv_color_data,
    label="Centered Summed Counts",
)
ax_4.plot(
    np.linspace(bin_freq_3[0], bin_freq_3[-2], 1000)
    + fit_result_gauss_3.params["center_1"].value,
    lorentz(
        x=np.linspace(bin_freq_3[0], bin_freq_3[-2], 1000),
        **fit_result_lorentz_3.best_values,
    )
    / norm_lorentz_3,
    color=snv_color_fit,
    linestyle="solid",
    label="Lorentzian Fit",
)
ax_4.annotate(
    "",
    xy=(123.205, 0.5),
    xycoords="data",
    xytext=(123.285, 0.5),
    textcoords="data",
    arrowprops=dict(arrowstyle="<|-", color=snv_gaussian_color_fit, lw=2),
)
ax_4.annotate(
    "",
    xy=(123.484, 0.5),
    xycoords="data",
    xytext=(123.564, 0.5),
    textcoords="data",
    arrowprops=dict(arrowstyle="-|>", color=snv_gaussian_color_fit, lw=2),
)
ax_4.annotate(
    "211.0\nMHz",
    xy=(123.58, 0.45),
    xycoords="data",
    xytext=(123.58, 0.45),
    color=snv_gaussian_color_fit,
    textcoords="data",
    fontsize=8,
)
ax_4.annotate(
    "{:.1f} MHz\n+/- {:.1f}\nMHz".format(
        1000 * fit_result_lorentz_3.params["gamma"].value,
        1000 * fit_result_lorentz_3.params["gamma"].stderr,
    ),
    xy=(123.12, 0.9),
    xycoords="data",
    xytext=(123.12, 0.9),
    textcoords="data",
    color=snv_color_fit,
    fontsize=8,
)
ax_4.set_xlim(123.1, 123.7)
ax_4.set_ylim(-0.05, 1.08)

ax_5.text(-0.2, 1.08, "(c)", transform=ax_5.transAxes, fontsize=12, va="top")
img = ax_5.pcolormesh(
    np.linspace(0, 60, 600),
    np.linspace(0, 60, 600) - 10,
    counts_large_map.transpose(),
    vmax=42,
    cmap="Reds",
)
ax_5.set_xlim(0, 45)
ax_5.set_ylim(0, 45)
ax_5.set_xlabel("X Position (µm)", fontsize=12)
ax_5.set_ylabel("Y Position (µm)", fontsize=12)
ax_5.set_yticks([0, 10, 20, 30, 40])
axinset = inset_axes(
    ax_5,
    width="10%",
    height="30%",
    bbox_to_anchor=(-0.12, 0.02, 0.94, 0.94),
    bbox_transform=ax_5.transAxes,
    loc="upper right",
)
cbar = fig.colorbar(img, cax=axinset, orientation="vertical")
cbar.set_label("Counts (kCts/s)", color="black", fontsize=10)
cbar.ax.tick_params(color="black", labelsize=10, labelcolor="black")
cbar.outline.set_edgecolor("black")

ax_6.text(-0.34, 1.08, "(d)", transform=ax_6.transAxes, fontsize=12, va="top")
max_1 = max(
    Voigt(
        x=np.linspace(dataset_nv.x0[start_1].values, dataset_nv.x0[end_1].values, 1000),
        **fit_result_voigt.best_values,
    )
)
ax_6.scatter(dataset_nv.x0, dataset_nv.y0 / max_1, s=14, color=nv_color_data)
ax_6.plot(
    np.linspace(dataset_nv.x0[start_1].values, dataset_nv.x0[end_1].values, 1000),
    Voigt(
        x=np.linspace(dataset_nv.x0[start_1].values, dataset_nv.x0[end_1].values, 1000),
        **fit_result_voigt.best_values,
    )
    / max_1,
    color=nv_color_fit,
    linestyle="solid",
    label="Voigt Fit",
)
ax_6.set_xlim(67.35, 67.75)
ax_6.set_ylim(0, 1.1)
width = calc_voigt_width(
    fit_result_voigt.params["sigma"], fit_result_voigt.params["gamma"]
)
ax_6.annotate(
    "",
    xy=(67.46, 0.5),
    xycoords="data",
    xytext=(67.54, 0.5),
    textcoords="data",
    arrowprops=dict(arrowstyle="<|-", color=nv_color_fit, lw=2),
)
ax_6.annotate(
    "",
    xy=(67.565, 0.5),
    xycoords="data",
    xytext=(67.655, 0.5),
    textcoords="data",
    arrowprops=dict(arrowstyle="-|>", color=nv_color_fit, lw=2),
)
ax_6.annotate(
    "{:.1f} MHz\n+/- {:.1f} MHz".format(1000 * width.n, 1000 * width.s),
    xy=(67.59, 0.7),
    xycoords="data",
    xytext=(67.59, 0.6),
    textcoords="data",
    color=nv_color_fit,
    fontsize=8,
)
ax_6.set_ylabel("Normalized Counts", fontsize=11)

max_2 = max(
    Voigt(
        x=np.linspace(dataset_nv.x0[start_2].values, dataset_nv.x0[end_2].values, 1000),
        **fit_result_voigt_2.best_values,
    )
)
ax_7.scatter(dataset_nv.x0, dataset_nv.y0 / max_2, s=14, color=nv_color_data)
ax_7.plot(
    np.linspace(dataset_nv.x0[start_2].values, dataset_nv.x0[end_2].values, 1000),
    Voigt(
        x=np.linspace(dataset_nv.x0[start_2].values, dataset_nv.x0[end_2].values, 1000),
        **fit_result_voigt_2.best_values,
    )
    / max_2,
    color=nv_color_fit,
    linestyle="solid",
    label="Voigt Fit",
)
ax_7.set_xlim(70.025, 70.425)
ax_7.set_ylim(0, 1.1)
width_2 = calc_voigt_width(
    fit_result_voigt_2.params["sigma"], fit_result_voigt_2.params["gamma"]
)
ax_7.annotate(
    "",
    xy=(70.11, 0.5),
    xycoords="data",
    xytext=(70.19, 0.5),
    textcoords="data",
    arrowprops=dict(arrowstyle="<|-", color=nv_color_fit, lw=2),
)
ax_7.annotate(
    "",
    xy=(70.242, 0.5),
    xycoords="data",
    xytext=(70.322, 0.5),
    textcoords="data",
    arrowprops=dict(arrowstyle="-|>", color=nv_color_fit, lw=2),
)
ax_7.annotate(
    "{:.1f} MHz\n+/- {:.1f} MHz".format(1000 * width_2.n, 1000 * width_2.s),
    xy=(70.255, 0.6),
    xycoords="data",
    xytext=(70.255, 0.6),
    textcoords="data",
    color=nv_color_fit,
    fontsize=8,
)

max_3 = max(
    Voigt(
        x=np.linspace(dataset_nv.x0[start_3].values, dataset_nv.x0[end_3].values, 1000),
        **fit_result_voigt_3.best_values,
    )
)
ax_8.scatter(dataset_nv.x0, dataset_nv.y0 / max_3, s=14, color=nv_color_data)
ax_8.plot(
    np.linspace(dataset_nv.x0[start_3].values, dataset_nv.x0[end_3].values, 1000),
    Voigt(
        x=np.linspace(dataset_nv.x0[start_3].values, dataset_nv.x0[end_3].values, 1000),
        **fit_result_voigt_3.best_values,
    )
    / max_3,
    color=nv_color_fit,
    linestyle="solid",
    label="Voigt\nFit",
)
ax_8.set_xlim(73.55, 73.95)
ax_8.set_ylim(0, 1.1)
width_3 = calc_voigt_width(
    fit_result_voigt_3.params["sigma"], fit_result_voigt_2.params["gamma"]
)
ax_8.annotate(
    "",
    xy=(73.64, 0.5),
    xycoords="data",
    xytext=(73.72, 0.5),
    textcoords="data",
    arrowprops=dict(arrowstyle="<|-", color=nv_color_fit, lw=2),
)
ax_8.annotate(
    "",
    xy=(73.766, 0.5),
    xycoords="data",
    xytext=(73.846, 0.5),
    textcoords="data",
    arrowprops=dict(arrowstyle="-|>", color=nv_color_fit, lw=2),
)
ax_8.annotate(
    "{:.1f} MHz\n+/- {:.1f} MHz".format(1000 * width_3.n, 1000 * width_3.s),
    xy=(73.78, 0.6),
    xycoords="data",
    xytext=(73.78, 0.6),
    textcoords="data",
    color=nv_color_fit,
    fontsize=8,
)

fig.text(
    0.64,
    0.06,
    "Laser Excitation Frequency w.r.t. 470.4 THz (GHz)",
    ha="center",
    fontsize=12,
)

fig.savefig(Path("../Fig_6.png"), dpi=600, bbox_inches="tight")
