from pathlib import Path

import lmfit
import matplotlib.pyplot as plt
import numpy as np
import uncertainties as un
from quantify_core.data.handling import (
    load_dataset,
    set_datadir,
)
from scipy.special import wofz
from uncertainties.umath import sqrt


def lorentz(x, offset, gamma, center, amp):
    lorentzfunc = (
        amp / np.pi * (gamma / 2) / ((x - center) ** 2 + (gamma / 2) ** 2) + offset
    )
    return lorentzfunc


def gaussian(x, offset, amp, center, sigma):
    exponent = ((x - center) / sigma) ** 2
    gaussian = amp * np.exp(-exponent / 2)
    return offset + gaussian


def Voigt(x, offset, center, amp, sigma, gamma):
    return offset + amp * np.real(
        wofz((x - center + 1j * gamma) / sigma / np.sqrt(2))
    ) / sigma / np.sqrt(2 * np.pi)


def calc_voigt_width(sigma, gamma):
    if sigma.stderr is None:
        sig = un.ufloat(sigma.value, fit_error_fwhm * sigma.value)
    else:
        sig = un.ufloat(sigma.value, sigma.stderr)
    if gamma.stderr is None:
        gam = 2 * un.ufloat(gamma.value, fit_error_fwhm * gamma.value)
    else:
        gam = 2 * un.ufloat(gamma.value, gamma.stderr)
    return 0.5346 * gam + sqrt(
        0.2166 * gam**2 + (2 * sig * np.sqrt(2 * np.log(2))) ** 2
    )


set_datadir(Path("../data"))

# analysis parameters
lifetime_limit = 0.013  # NV center lifetime-limited linewidth, see e.g. Hermans et al. 2023 New J.Phys. 25 013011
fit_error_fwhm = 0.1  # in case fit does not give standard error
fit_error_center = 0.01  # in case fit does not give standard error

# load and analyze data

peaks_center, peaks_fwhm = list(), list()

# Lines 1 and 2

dataset_nv = load_dataset("20220803-191253-212-12f43c-line_scan")
print(dataset_nv.attrs["freq_step"])
start_1, end_1 = 250, 350
start_2, end_2 = 380, 480

fit_model_voigt = lmfit.Model(Voigt)
fit_model_voigt.set_param_hint("offset", value=10)
fit_model_voigt.set_param_hint("gamma", value=0.02, min=lifetime_limit)
fit_model_voigt.set_param_hint("sigma", value=0.04, min=0.001)
fit_model_voigt.set_param_hint("center", value=65)
fit_model_voigt.set_param_hint("amp", value=290)
fit_result_voigt = fit_model_voigt.fit(
    dataset_nv.y0[start_1:end_1],
    x=dataset_nv.x0[start_1:end_1],
    params=fit_model_voigt.make_params(),
)

fit_model_voigt_2 = lmfit.Model(Voigt)
fit_model_voigt_2.set_param_hint("offset", value=10)
fit_model_voigt_2.set_param_hint("gamma", value=0.02, min=lifetime_limit)
fit_model_voigt_2.set_param_hint("sigma", value=0.04, min=0.001)
fit_model_voigt_2.set_param_hint("center", value=66.35)
fit_model_voigt_2.set_param_hint("amp", value=290)
fit_result_voigt_2 = fit_model_voigt_2.fit(
    dataset_nv.y0[start_2:end_2],
    x=dataset_nv.x0[start_2:end_2],
    params=fit_model_voigt_2.make_params(),
)

if fit_result_voigt.params["center"].stderr is None:
    peaks_center.append(
        un.ufloat(
            fit_result_voigt.params["center"].value,
            fit_error_center * fit_result_voigt.params["center"].value,
        )
    )
else:
    peaks_center.append(
        un.ufloat(
            fit_result_voigt.params["center"].value,
            fit_result_voigt.params["center"].stderr,
        )
    )

if fit_result_voigt_2.params["center"].stderr is None:
    peaks_center.append(
        un.ufloat(
            fit_result_voigt_2.params["center"].value,
            fit_error_center * fit_result_voigt_2.params["center"].value,
        )
    )
else:
    peaks_center.append(
        un.ufloat(
            fit_result_voigt_2.params["center"].value,
            fit_result_voigt_2.params["center"].stderr,
        )
    )

peaks_fwhm.append(
    calc_voigt_width(fit_result_voigt.params["sigma"], fit_result_voigt.params["gamma"])
)
peaks_fwhm.append(
    calc_voigt_width(
        fit_result_voigt_2.params["sigma"], fit_result_voigt_2.params["gamma"]
    )
)

# Lines 3 and 4

dataset_nv = load_dataset("20220804-172230-755-ac5b7f-line_scan")
print(dataset_nv.attrs["freq_step"])
start_1, end_1 = 30, 130
start_2, end_2 = 340, 440

fit_model_voigt = lmfit.Model(Voigt)
fit_model_voigt.set_param_hint("offset", value=10)
fit_model_voigt.set_param_hint("gamma", value=0.02, min=lifetime_limit)
fit_model_voigt.set_param_hint("sigma", value=0.04, min=0.001)
fit_model_voigt.set_param_hint("center", value=71.6)
fit_model_voigt.set_param_hint("amp", value=290)
fit_result_voigt = fit_model_voigt.fit(
    dataset_nv.y0[start_1:end_1],
    x=dataset_nv.x0[start_1:end_1],
    params=fit_model_voigt.make_params(),
)

fit_model_voigt_2 = lmfit.Model(Voigt)
fit_model_voigt_2.set_param_hint("offset", value=10)
fit_model_voigt_2.set_param_hint("gamma", value=0.02, min=lifetime_limit)
fit_model_voigt_2.set_param_hint("sigma", value=0.04, min=0.001)
fit_model_voigt_2.set_param_hint("center", value=77.8)
fit_model_voigt_2.set_param_hint("amp", value=290)
fit_result_voigt_2 = fit_model_voigt_2.fit(
    dataset_nv.y0[start_2:end_2],
    x=dataset_nv.x0[start_2:end_2],
    params=fit_model_voigt_2.make_params(),
)

if fit_result_voigt.params["center"].stderr is None:
    peaks_center.append(
        un.ufloat(
            fit_result_voigt.params["center"].value,
            fit_error_center * fit_result_voigt.params["center"].value,
        )
    )
else:
    peaks_center.append(
        un.ufloat(
            fit_result_voigt.params["center"].value,
            fit_result_voigt.params["center"].stderr,
        )
    )

if fit_result_voigt_2.params["center"].stderr is None:
    peaks_center.append(
        un.ufloat(
            fit_result_voigt_2.params["center"].value,
            fit_error_center * fit_result_voigt_2.params["center"].value,
        )
    )
else:
    peaks_center.append(
        un.ufloat(
            fit_result_voigt_2.params["center"].value,
            fit_result_voigt_2.params["center"].stderr,
        )
    )

peaks_fwhm.append(
    calc_voigt_width(fit_result_voigt.params["sigma"], fit_result_voigt.params["gamma"])
)
peaks_fwhm.append(
    calc_voigt_width(
        fit_result_voigt_2.params["sigma"], fit_result_voigt_2.params["gamma"]
    )
)

# Lines 5 to 7

dataset_nv = load_dataset("20220805-120837-702-2da9c4-line_scan")
print(dataset_nv.attrs["freq_step"])
start_1, end_1 = 40, 140
start_2, end_2 = 170, 270
start_3, end_3 = 660, 760

fit_model_voigt = lmfit.Model(Voigt)
fit_model_voigt.set_param_hint("offset", value=10)
fit_model_voigt.set_param_hint("gamma", value=0.02, min=lifetime_limit)
fit_model_voigt.set_param_hint("sigma", value=0.04, min=0.001)
fit_model_voigt.set_param_hint("center", value=66.7)
fit_model_voigt.set_param_hint("amp", value=290)
fit_result_voigt = fit_model_voigt.fit(
    dataset_nv.y0[start_1:end_1],
    x=dataset_nv.x0[start_1:end_1],
    params=fit_model_voigt.make_params(),
)

fit_model_voigt_2 = lmfit.Model(Voigt)
fit_model_voigt_2.set_param_hint("offset", value=10)
fit_model_voigt_2.set_param_hint("gamma", value=0.02, min=lifetime_limit)
fit_model_voigt_2.set_param_hint("sigma", value=0.04, min=0.001)
fit_model_voigt_2.set_param_hint("center", value=69.5)
fit_model_voigt_2.set_param_hint("amp", value=290)
fit_result_voigt_2 = fit_model_voigt_2.fit(
    dataset_nv.y0[start_2:end_2],
    x=dataset_nv.x0[start_2:end_2],
    params=fit_model_voigt_2.make_params(),
)

fit_model_voigt_3 = lmfit.Model(Voigt)
fit_model_voigt_3.set_param_hint("offset", value=10)
fit_model_voigt_3.set_param_hint("gamma", value=0.02, min=lifetime_limit)
fit_model_voigt_3.set_param_hint("sigma", value=0.04, min=0.001)
fit_model_voigt_3.set_param_hint("center", value=79.2)
fit_model_voigt_3.set_param_hint("amp", value=290)
fit_result_voigt_3 = fit_model_voigt_3.fit(
    dataset_nv.y0[start_3:end_3],
    x=dataset_nv.x0[start_3:end_3],
    params=fit_model_voigt_3.make_params(),
)

if fit_result_voigt.params["center"].stderr is None:
    peaks_center.append(
        un.ufloat(
            fit_result_voigt.params["center"].value,
            fit_error_center * fit_result_voigt.params["center"].value,
        )
    )
else:
    peaks_center.append(
        un.ufloat(
            fit_result_voigt.params["center"].value,
            fit_result_voigt.params["center"].stderr,
        )
    )
if fit_result_voigt_2.params["center"].stderr is None:
    peaks_center.append(
        un.ufloat(
            fit_result_voigt_2.params["center"].value,
            fit_error_center * fit_result_voigt_2.params["center"].value,
        )
    )
else:
    peaks_center.append(
        un.ufloat(
            fit_result_voigt_2.params["center"].value,
            fit_result_voigt_2.params["center"].stderr,
        )
    )
if fit_result_voigt_3.params["center"].stderr is None:
    peaks_center.append(
        un.ufloat(
            fit_result_voigt_3.params["center"].value,
            fit_error_center * fit_result_voigt_3.params["center"].value,
        )
    )
else:
    peaks_center.append(
        un.ufloat(
            fit_result_voigt_3.params["center"].value,
            fit_result_voigt_3.params["center"].stderr,
        )
    )

peaks_fwhm.append(
    calc_voigt_width(fit_result_voigt.params["sigma"], fit_result_voigt.params["gamma"])
)
peaks_fwhm.append(
    calc_voigt_width(
        fit_result_voigt_2.params["sigma"], fit_result_voigt_2.params["gamma"]
    )
)
peaks_fwhm.append(
    calc_voigt_width(
        fit_result_voigt_3.params["sigma"], fit_result_voigt_3.params["gamma"]
    )
)

# Lines 8 to 10

dataset_nv = load_dataset("20220805-142007-638-353985-line_scan")
print(dataset_nv.attrs["freq_step"])
start_1, end_1 = 1830, 1930
start_2, end_2 = 1960, 2060
start_3, end_3 = 2130, 2230
start_4, end_4 = 2660, 2760
start_5, end_5 = 2790, 2890
start_6, end_6 = 2930, 3030

fit_model_voigt_4 = lmfit.Model(Voigt)
fit_model_voigt_4.set_param_hint("offset", value=10)
fit_model_voigt_4.set_param_hint("gamma", value=0.02, min=lifetime_limit)
fit_model_voigt_4.set_param_hint("sigma", value=0.04, min=0.001)
fit_model_voigt_4.set_param_hint("center", value=84.2)
fit_model_voigt_4.set_param_hint("amp", value=290)
fit_result_voigt_4 = fit_model_voigt_4.fit(
    dataset_nv.y0[start_4:end_4],
    x=dataset_nv.x0[start_4:end_4],
    params=fit_model_voigt_4.make_params(),
)

fit_model_voigt_5 = lmfit.Model(Voigt)
fit_model_voigt_5.set_param_hint("offset", value=10)
fit_model_voigt_5.set_param_hint("gamma", value=0.02, min=lifetime_limit)
fit_model_voigt_5.set_param_hint("sigma", value=0.04, min=0.001)
fit_model_voigt_5.set_param_hint("center", value=86.85)
fit_model_voigt_5.set_param_hint("amp", value=290)
fit_result_voigt_5 = fit_model_voigt_5.fit(
    dataset_nv.y0[start_5:end_5],
    x=dataset_nv.x0[start_5:end_5],
    params=fit_model_voigt_5.make_params(),
)

fit_model_voigt_6 = lmfit.Model(Voigt)
fit_model_voigt_6.set_param_hint("offset", value=10)
fit_model_voigt_6.set_param_hint("gamma", value=0.02, min=lifetime_limit)
fit_model_voigt_6.set_param_hint("sigma", value=0.04, min=0.001)
fit_model_voigt_6.set_param_hint("center", value=89.6)
fit_model_voigt_6.set_param_hint("amp", value=290)
fit_result_voigt_6 = fit_model_voigt_6.fit(
    dataset_nv.y0[start_6:end_6],
    x=dataset_nv.x0[start_6:end_6],
    params=fit_model_voigt_6.make_params(),
)

if fit_result_voigt_4.params["center"].stderr is None:
    peaks_center.append(
        un.ufloat(
            fit_result_voigt_4.params["center"].value,
            fit_error_center * fit_result_voigt_4.params["center"].value,
        )
    )
else:
    peaks_center.append(
        un.ufloat(
            fit_result_voigt_4.params["center"].value,
            fit_result_voigt_4.params["center"].stderr,
        )
    )
if fit_result_voigt_5.params["center"].stderr is None:
    peaks_center.append(
        un.ufloat(
            fit_result_voigt_5.params["center"].value,
            fit_error_center * fit_result_voigt_5.params["center"].value,
        )
    )
else:
    peaks_center.append(
        un.ufloat(
            fit_result_voigt_5.params["center"].value,
            fit_result_voigt_5.params["center"].stderr,
        )
    )
if fit_result_voigt_6.params["center"].stderr is None:
    peaks_center.append(
        un.ufloat(
            fit_result_voigt_6.params["center"].value,
            fit_error_center * fit_result_voigt_6.params["center"].value,
        )
    )
else:
    peaks_center.append(
        un.ufloat(
            fit_result_voigt_6.params["center"].value,
            fit_result_voigt_6.params["center"].stderr,
        )
    )

peaks_fwhm.append(
    calc_voigt_width(
        fit_result_voigt_4.params["sigma"], fit_result_voigt_4.params["gamma"]
    )
)
peaks_fwhm.append(
    calc_voigt_width(
        fit_result_voigt_5.params["sigma"], fit_result_voigt_5.params["gamma"]
    )
)
peaks_fwhm.append(
    calc_voigt_width(
        fit_result_voigt_6.params["sigma"], fit_result_voigt_6.params["gamma"]
    )
)

# Lines 11 to 13

dataset_nv = load_dataset("20220805-161039-403-0e202c-line_scan")
print(dataset_nv.attrs["freq_step"])
start_1, end_1 = 270, 370
start_2, end_2 = 550, 750
start_3, end_3 = 1040, 1140

fit_model_voigt = lmfit.Model(Voigt)
fit_model_voigt.set_param_hint("offset", value=10)
fit_model_voigt.set_param_hint("gamma", value=0.02, min=lifetime_limit)
fit_model_voigt.set_param_hint("sigma", value=0.04, min=0.001)
fit_model_voigt.set_param_hint("center", value=67.55)
fit_model_voigt.set_param_hint("amp", value=16)
fit_result_voigt = fit_model_voigt.fit(
    dataset_nv.y0[start_1:end_1],
    x=dataset_nv.x0[start_1:end_1],
    params=fit_model_voigt.make_params(),
)

fit_model_voigt_2 = lmfit.Model(Voigt)
fit_model_voigt_2.set_param_hint("offset", value=10)
fit_model_voigt_2.set_param_hint("gamma", value=0.02, min=lifetime_limit)
fit_model_voigt_2.set_param_hint("sigma", value=0.04, min=0.001)
fit_model_voigt_2.set_param_hint("center", value=70.25)
fit_model_voigt_2.set_param_hint("amp", value=290)
fit_result_voigt_2 = fit_model_voigt_5.fit(
    dataset_nv.y0[start_2:end_2],
    x=dataset_nv.x0[start_2:end_2],
    params=fit_model_voigt_2.make_params(),
)

fit_model_voigt_3 = lmfit.Model(Voigt)
fit_model_voigt_3.set_param_hint("offset", value=10)
fit_model_voigt_3.set_param_hint("gamma", value=0.02, min=lifetime_limit)
fit_model_voigt_3.set_param_hint("sigma", value=0.04, min=0.001)
fit_model_voigt_3.set_param_hint("center", value=73.7)
fit_model_voigt_3.set_param_hint("amp", value=290)
fit_result_voigt_3 = fit_model_voigt_3.fit(
    dataset_nv.y0[start_3:end_3],
    x=dataset_nv.x0[start_3:end_3],
    params=fit_model_voigt_3.make_params(),
)

if fit_result_voigt.params["center"].stderr is None:
    peaks_center.append(
        un.ufloat(
            fit_result_voigt.params["center"].value,
            fit_error_center * fit_result_voigt.params["center"].value,
        )
    )
else:
    peaks_center.append(
        un.ufloat(
            fit_result_voigt.params["center"].value,
            fit_result_voigt.params["center"].stderr,
        )
    )
if fit_result_voigt_2.params["center"].stderr is None:
    peaks_center.append(
        un.ufloat(
            fit_result_voigt_2.params["center"].value,
            fit_error_center * fit_result_voigt_2.params["center"].value,
        )
    )
else:
    peaks_center.append(
        un.ufloat(
            fit_result_voigt_2.params["center"].value,
            fit_result_voigt_2.params["center"].stderr,
        )
    )
if fit_result_voigt_3.params["center"].stderr is None:
    peaks_center.append(
        un.ufloat(
            fit_result_voigt_3.params["center"].value,
            fit_error_center * fit_result_voigt_3.params["center"].value,
        )
    )
else:
    peaks_center.append(
        un.ufloat(
            fit_result_voigt_3.params["center"].value,
            fit_result_voigt_3.params["center"].stderr,
        )
    )

peaks_fwhm.append(
    calc_voigt_width(fit_result_voigt.params["sigma"], fit_result_voigt.params["gamma"])
)
peaks_fwhm.append(
    calc_voigt_width(
        fit_result_voigt_2.params["sigma"], fit_result_voigt_2.params["gamma"]
    )
)
peaks_fwhm.append(
    calc_voigt_width(
        fit_result_voigt_3.params["sigma"], fit_result_voigt_3.params["gamma"]
    )
)

# Lines 14 and 15

dataset_nv = load_dataset("20220805-165726-695-28cca1-line_scan")
print(dataset_nv.attrs["freq_step"])
start_1, end_1 = 600, 700
start_2, end_2 = 1070, 1170

fit_model_voigt = lmfit.Model(Voigt)
fit_model_voigt.set_param_hint("offset", value=10)
fit_model_voigt.set_param_hint("gamma", value=0.02, min=lifetime_limit)
fit_model_voigt.set_param_hint("sigma", value=0.04, min=0.001)
fit_model_voigt.set_param_hint("center", value=68)
fit_model_voigt.set_param_hint("amp", value=16)
fit_result_voigt = fit_model_voigt.fit(
    dataset_nv.y0[start_1:end_1],
    x=dataset_nv.x0[start_1:end_1],
    params=fit_model_voigt.make_params(),
)

fit_model_voigt_2 = lmfit.Model(Voigt)
fit_model_voigt_2.set_param_hint("offset", value=10)
fit_model_voigt_2.set_param_hint("gamma", value=0.02, min=lifetime_limit)
fit_model_voigt_2.set_param_hint("sigma", value=0.04, min=0.001)
fit_model_voigt_2.set_param_hint("center", value=77.3)
fit_model_voigt_2.set_param_hint("amp", value=290)
fit_result_voigt_2 = fit_model_voigt_5.fit(
    dataset_nv.y0[start_2:end_2],
    x=dataset_nv.x0[start_2:end_2],
    params=fit_model_voigt_2.make_params(),
)

if fit_result_voigt.params["center"].stderr is None:
    peaks_center.append(
        un.ufloat(
            fit_result_voigt.params["center"].value,
            fit_error_center * fit_result_voigt.params["center"].value,
        )
    )
else:
    peaks_center.append(
        un.ufloat(
            fit_result_voigt.params["center"].value,
            fit_result_voigt.params["center"].stderr,
        )
    )
if fit_result_voigt_2.params["center"].stderr is None:
    peaks_center.append(
        un.ufloat(
            fit_result_voigt_2.params["center"].value,
            fit_error_center * fit_result_voigt_2.params["center"].value,
        )
    )
else:
    peaks_center.append(
        un.ufloat(
            fit_result_voigt_2.params["center"].value,
            fit_result_voigt_2.params["center"].stderr,
        )
    )

peaks_fwhm.append(
    calc_voigt_width(fit_result_voigt.params["sigma"], fit_result_voigt.params["gamma"])
)
peaks_fwhm.append(
    calc_voigt_width(
        fit_result_voigt_2.params["sigma"], fit_result_voigt_2.params["gamma"]
    )
)

# Lines 16

dataset_nv = load_dataset("20220808-111126-486-b14fa5-line_scan")
print(dataset_nv.attrs["freq_step"])
start_1, end_1 = 740, 940

fit_model_voigt = lmfit.Model(Voigt)
fit_model_voigt.set_param_hint("offset", value=10)
fit_model_voigt.set_param_hint("gamma", value=0.02, min=lifetime_limit)
fit_model_voigt.set_param_hint("sigma", value=0.04, min=0.001)
fit_model_voigt.set_param_hint("center", value=71.9)
fit_model_voigt.set_param_hint("amp", value=16)
fit_result_voigt = fit_model_voigt.fit(
    dataset_nv.y0[start_1:end_1],
    x=dataset_nv.x0[start_1:end_1],
    params=fit_model_voigt.make_params(),
)

if fit_result_voigt.params["center"].stderr is None:
    peaks_center.append(
        un.ufloat(
            fit_result_voigt.params["center"].value,
            fit_error_center * fit_result_voigt.params["center"].value,
        )
    )
else:
    peaks_center.append(
        un.ufloat(
            fit_result_voigt.params["center"].value,
            fit_result_voigt.params["center"].stderr,
        )
    )

peaks_fwhm.append(
    calc_voigt_width(fit_result_voigt.params["sigma"], fit_result_voigt.params["gamma"])
)

# Lines 17 and 18

dataset_nv = load_dataset("20220808-133514-347-fab84b-line_scan")
print(dataset_nv.attrs["freq_step"])
start_1, end_1 = 390, 490
start_2, end_2 = 755, 955

fit_model_voigt = lmfit.Model(Voigt)
fit_model_voigt.set_param_hint("offset", value=7)
fit_model_voigt.set_param_hint("gamma", value=0.02, min=lifetime_limit)
fit_model_voigt.set_param_hint("sigma", value=0.04, min=0.001)
fit_model_voigt.set_param_hint("center", value=84.4)
fit_model_voigt.set_param_hint("amp", value=5)
fit_result_voigt = fit_model_voigt.fit(
    dataset_nv.y0[start_1:end_1],
    x=dataset_nv.x0[start_1:end_1],
    params=fit_model_voigt.make_params(),
)

fit_model_voigt_2 = lmfit.Model(Voigt)
fit_model_voigt_2.set_param_hint("offset", value=10)
fit_model_voigt_2.set_param_hint("gamma", value=0.02, min=lifetime_limit)
fit_model_voigt_2.set_param_hint("sigma", value=0.04, min=0.001)
fit_model_voigt_2.set_param_hint("center", value=88.5)
fit_model_voigt_2.set_param_hint("amp", value=290)
fit_result_voigt_2 = fit_model_voigt_5.fit(
    dataset_nv.y0[start_2:end_2],
    x=dataset_nv.x0[start_2:end_2],
    params=fit_model_voigt_2.make_params(),
)

if fit_result_voigt.params["center"].stderr is None:
    peaks_center.append(
        un.ufloat(
            fit_result_voigt.params["center"].value,
            fit_error_center * fit_result_voigt.params["center"].value,
        )
    )
else:
    peaks_center.append(
        un.ufloat(
            fit_result_voigt.params["center"].value,
            fit_result_voigt.params["center"].stderr,
        )
    )
if fit_result_voigt_2.params["center"].stderr is None:
    peaks_center.append(
        un.ufloat(
            fit_result_voigt_2.params["center"].value,
            fit_error_center * fit_result_voigt_2.params["center"].value,
        )
    )
else:
    peaks_center.append(
        un.ufloat(
            fit_result_voigt_2.params["center"].value,
            fit_result_voigt_2.params["center"].stderr,
        )
    )

peaks_fwhm.append(
    calc_voigt_width(fit_result_voigt.params["sigma"], fit_result_voigt.params["gamma"])
)
peaks_fwhm.append(
    calc_voigt_width(
        fit_result_voigt_2.params["sigma"], fit_result_voigt_2.params["gamma"]
    )
)

# Lines 19 and 20

dataset_nv = load_dataset("20220808-154620-154-42e77e-line_scan")
print(dataset_nv.attrs["freq_step"])
start_1, end_1 = 3600, 3800
start_2, end_2 = 4460, 4660

fit_model_voigt = lmfit.Model(Voigt)
fit_model_voigt.set_param_hint("offset", value=7)
fit_model_voigt.set_param_hint("gamma", value=0.02, min=lifetime_limit)
fit_model_voigt.set_param_hint("sigma", value=0.04, min=0.001)
fit_model_voigt.set_param_hint("center", value=72.05)
fit_model_voigt.set_param_hint("amp", value=5)
fit_result_voigt = fit_model_voigt.fit(
    dataset_nv.y0[start_1:end_1],
    x=dataset_nv.x0[start_1:end_1],
    params=fit_model_voigt.make_params(),
)

fit_model_voigt_2 = lmfit.Model(Voigt)
fit_model_voigt_2.set_param_hint("offset", value=10)
fit_model_voigt_2.set_param_hint("gamma", value=0.02, min=lifetime_limit)
fit_model_voigt_2.set_param_hint("sigma", value=0.04, min=0.001)
fit_model_voigt_2.set_param_hint("center", value=80.55)
fit_model_voigt_2.set_param_hint("amp", value=290)
fit_result_voigt_2 = fit_model_voigt_5.fit(
    dataset_nv.y0[start_2:end_2],
    x=dataset_nv.x0[start_2:end_2],
    params=fit_model_voigt_2.make_params(),
)

if fit_result_voigt.params["center"].stderr is None:
    peaks_center.append(
        un.ufloat(
            fit_result_voigt.params["center"].value,
            fit_error_center * fit_result_voigt.params["center"].value,
        )
    )
else:
    peaks_center.append(
        un.ufloat(
            fit_result_voigt.params["center"].value,
            fit_result_voigt.params["center"].stderr,
        )
    )
if fit_result_voigt_2.params["center"].stderr is None:
    peaks_center.append(
        un.ufloat(
            fit_result_voigt_2.params["center"].value,
            fit_error_center * fit_result_voigt_2.params["center"].value,
        )
    )
else:
    peaks_center.append(
        un.ufloat(
            fit_result_voigt_2.params["center"].value,
            fit_result_voigt_2.params["center"].stderr,
        )
    )

peaks_fwhm.append(
    calc_voigt_width(fit_result_voigt.params["sigma"], fit_result_voigt.params["gamma"])
)
peaks_fwhm.append(
    calc_voigt_width(
        fit_result_voigt_2.params["sigma"], fit_result_voigt_2.params["gamma"]
    )
)

print("Max. linewdith")
print(np.max(peaks_fwhm))

print("Min. linewdith")
print(np.min(peaks_fwhm))

fwhm_freq_bin = np.linspace(35, 135, 11)
hist_fwhm = np.histogram([1000 * i.n for i in peaks_fwhm], bins=fwhm_freq_bin)[0]

# Compute CDF
x = np.sort(np.array([i.n for i in peaks_fwhm]))
y = np.arange(1, len(np.array([i.n for i in peaks_fwhm])) + 1) / len(
    np.array([i.n for i in peaks_fwhm])
)

fig, [ax_1, ax_2] = plt.subplots(1, 2, figsize=(8, 3), gridspec_kw={"wspace": 0.24})
ax_1.text(-0.21, 1.1, "(a)", transform=ax_1.transAxes, fontsize=12, va="top")
ax_1.errorbar(
    x=[i.n for i in peaks_center],
    y=[1000 * i.n for i in peaks_fwhm],
    xerr=[i.s for i in peaks_center],
    yerr=[1000 * i.s for i in peaks_fwhm],
    fmt="o",
    capsize=3,
    markersize="2",
)
ax_1.axhline(
    np.median(np.array([1e3 * i.n for i in peaks_fwhm])),
    linestyle="dashed",
    label="Median of {:.1f} MHz".format(
        1e3 * np.median(np.array([i.n for i in peaks_fwhm]))
    ),
)
ax_1.set_xlabel("Center Frequency w.r.t. 470.4 THz (GHz)")
ax_1.set_ylabel("Linewidth (MHz)")
ax_1.legend()

ax_2.text(-0.19, 1.1, "(b)", transform=ax_2.transAxes, fontsize=12, va="top")
ax_2.step(1000 * x, y, where="pre")
ax_2.set_xlabel("Linewidth (MHz)")
ax_2.set_ylabel("Cumulative Distribution Function")

fig.savefig(Path("../Fig_12_sup.png"), dpi=600, bbox_inches="tight")
