%--------------------------------------------
%
% Characteristics of slurry transport regimes: 
% Insights from experiments and interface-resolved Direct 
% Numerical Simulations
% - Tariq Shajahan, Thijs Schouten,Shravan Raaghav,
%    Cees van Rhee,Geert Keetels & Wim-Paul Breugem 
% *(W.P.Breugem@tudelft.nl) 
% International Journal of Multiphase Flow
%
% Laboratory for Aero & Hydrodynamics,
% Process & Energy Department
% Delft University of Technology
% Mekelweg 2, 2628 CD, Delft
% Netherlands
%
% Dredging Engineering, 
% Delft University of Technology,
% 2628 CD Delft, 
% The Netherlands
%
%--------------------------------------------
clc; clear all; close all;

%--------------------------------------------
% Figure 5
if(1)
 % Pipe parameters
 rho_f = 998.19;
 nu_f  = 9.195e-7;
 Dpipe = 0.04;
 Dp    = 2.02e-3;

 % Vbulk % Re     pgrad[SI]  phi_b  vbs/vbl  Ga    ReT
 DNS = [[ 3272      21.83     0.325  0.13     44.0  38.0649]   % D1
        [ 8513      47.10     0.284  0.71     43.6  37.5730]   % D2
        [ 13830     76.34     0.268  1.09     43.2  37.0825]   % D3
        [ 2500       8.17     0.25   0.06     37.9  30.7199]   % S1
        [ 5200      20.53     0.25   0.38     37.9  30.7199]   % S2
        [ 7100      30.09     0.25   0.63     37.9  30.7199]   % S3
        [ 12000     55.80     0.25   1.02     37.9  30.7199]   % S4
        [ 12000     50.44     0.25   1.20     18.9  10.7104]]; % S5

 % pgrad[SI]
 EXP = [[ 34.1     ]   % E1
        [ 60.1     ]   % E2
        [ 81.8     ]]; % E3

 Shields_EXP = [[0.835 ]   % E1
 	        [1.495 ]   % E2
	        [2.083 ]]; % E3

 % Convert Pgrad from SI to non-dimensional
 V_bulk = (DNS(:,1) .* nu_f / Dpipe);
 pgrad = DNS(:,2)./(rho_f*V_bulk(:).^2. / Dp );

 % vtau
 vtau = sqrt(0.25 * 20.0 .* pgrad );
 dx_vtau_by_nuf = (1/16.)*vtau.*DNS(:,1)*(1/20.);

% effective viscosity ratio
 nu_eff_by_nu_f = (1 + 1.25 * (DNS(:,3)./(1. - DNS(:,3)/0.64))).^2.;

% friction factor
 f = pgrad * 40 .* ( 1.0./(DNS(:,3).*DNS(:,4) + (1-DNS(:,3))).^2.0 );
 
 % Re_mix
 Re_mix  = DNS(:,1).*( DNS(:,3).*DNS(:,4) + (1-DNS(:,3)));
 Re_mixE = Re_mix .* ( DNS(:,3)*1.02 + (1-DNS(:,3))) ...
	         ./ (1 + 1.25 * (DNS(:,3)./(1 - DNS(:,3)/0.64))).^2.0;
 Re_tau = DNS(:,1).*vtau;
 Re_tau_e = Re_tau./nu_eff_by_nu_f;

 % Other friction factors
 fpure  = 1 ./ ( ( -1.8 * log10(6.9./Re_mix) ).^2.0 );
 fH = 1 ./ ( ( -1.8 * log10(6.9./Re_mixE) ).^2.0 );

 % Delivered concentration
 phi_d = DNS(:,3) ./ (1.0 + (1-DNS(:,3)).*(1.0./(DNS(:,4))-1.0));

 % Froude number
 Fr =sqrt( ...
	(DNS(:,3) .* DNS(:,4) + (1.0 - DNS(:,3))).^2.0 ...
	          .* (DNS(:,1) ./ DNS(:,5)).^2.0 ...
	           * 0.05^3.0 ...
	);
 % Shields number = (Re/Ga)^2 * (vtau/vbl)^2 * (Dp/Dpipe)^2
 Shields = (DNS(:,1)./DNS(:,5)).^2 .* vtau.^2 * (1/20.)^2;

 % uT/sqrt(gDp)
 uT_by_sq_gDp = DNS(:,6) ./ DNS(:,5) * sqrt(1.02 -1.0);

 % Durand and Condolios
 f_DC_slurry = fpure .* ( 1 + 83.0 * phi_d .* (((Fr.^2) ./ uT_by_sq_gDp)).^-1.5);

 % f_DW_slurry
 f_DNS_slurry = DNS(:,2).*(Dpipe./(0.5 * rho_f *V_bulk.^2)) .* (1.0 ./ (DNS(:,3) .* DNS(:,4) + (1 - DNS(:,3)))).^2.0;

 % f_Exp_slurry
 f_EXP_slurry = EXP(:,1).*(Dpipe./(0.5 * rho_f *V_bulk(1:3).^2)) .* (1.0 ./ (DNS(1:3,3) .* DNS(1:3,4) + (1 - DNS(1:3,3)))).^2.0;

 % f_DC_new [correcting DC52 with density ratio effect]
 f_DC_new = fpure .* (1 + 121*phi_d.*( (Fr.^2.0) .* (DNS(:,5)./DNS(:,6))  ).^(-1.5));

 % Limit deposit velocity based model
 Frldv = sqrt( (phi_d*121/2).^(2.0/3.0) .* DNS(:,6)./DNS(:,5) );

 % Newitt et al 1955
 fN55 = fpure .* (1 + 1100 .* phi_d .* (1./Fr).^2.0 .* (DNS(:,6)./DNS(:,5))./(Fr*sqrt(Dpipe/Dp)) );

 % Newitt sliding bed
 fn55sliding = fpure .* (1 + 66*phi_d .* (1./Fr).^2);

 % Modified equivalent liquid model
 f_elm = fH .* ((1.02 * DNS(:,3)) + (1.0 - DNS(:,3)));

 % Correlation for friction factor in fixed bed regime
 Dh_by_Dpipe = (pi/2.0)/(pi/2.0 + 1);
 ul_by_ubl = 2.0*(1-DNS(:,3));
 Re_f = DNS(:,1) .* ul_by_ubl * Dh_by_Dpipe;

 % e - Half Dp
 eps_half_Dp = 0.5 * (Dp/Dpipe) * 1/Dh_by_Dpipe;
 fHnew_halfDp = 1./(1.8 * log10(6.9./Re_f + (eps_half_Dp/3.7)^1.11    )).^2.0;
 f_FB1 = fHnew_halfDp * (1/Dh_by_Dpipe) .* (ul_by_ubl ./ (DNS(:,4).*DNS(:,3) + (1 - DNS(:,3)))).^2;

 % e - Dp
 eps_full_Dp = 1.0 * (Dp/Dpipe) * 1/Dh_by_Dpipe;
 fHnew_fullDp = 1./(1.8 * log10(6.9./Re_f + (eps_full_Dp/3.7)^1.11    )).^2.0;
 f_FB2 = fHnew_fullDp * (1/Dh_by_Dpipe) .* (ul_by_ubl ./ (DNS(:,4).*DNS(:,3) + (1 - DNS(:,3)))).^2;

 % Single - phase
 Re_mix2 = linspace(1,15000,100);
 fpure2  = 1 ./ ( ( -1.8 * log10(6.9./Re_mix2) ).^2.0 );

 figure(501)
 hold on
 plot(Re_mix2,log10(fpure2),'--k','LineWidth',1.5)
 plot(Re_mix(1:3),log10(f(1:3)),'Marker','s','MarkerSize',12,'Markerfacecolor','red','color','r','LineWidth',1.8)
 plot(Re_mix(1:3),log10(f_EXP_slurry(1:3)),'Marker','s','MarkerSize',12,'Markerfacecolor','black','Color','k','LineWidth',1.8)
 plot(Re_mix(1:3),log10(f_FB2(1:3)),'Marker','*','MarkerSize',12,'LineWidth',1.8)
 plot(Re_mix(1:3),log10(f_DC_new(1:3)),'Marker','o','MarkerSize',12,'LineWidth',1.8)
 plot(Re_mix(1:3),log10(fn55sliding(1:3)),'Marker','^','MarkerSize',12,'LineWidth',1.8)
 plot(Re_mix(1:3),log10(fN55(1:3)),'Marker','^','MarkerSize',12,'LineWidth',1.8)
 plot(Re_mix(1:3),log10(f_elm(1:3)),'Marker','s','MarkerSize',12,'LineWidth',1.5)
 yticks([-2 -1.5 -1 -0.5 0.0 0.5])
 ylim([-1.7 0.5])
 yticklabels({'$10^{-2.0}$','$10^{-1.5}$','$10^{-1.0}$','$10^{-0.5}$','$10^{0.0}$','$10^{0.5}$'})
 legend(  '$f_{SPR}$',...
          '$f_{DNS}$',...
          '$f_{EXP}$',...
          '$f_{FB }$',...
          '$f_{MDC}$',...
          '$f_{NSB}$',...
          '$f_{NHS}$',...
          '$f_{ELM}$',...
          'Location','NorthEast')
 xlabel('$Re_{mix}$')
 ylabel('$f$')

 figure(502)
 hold on
 plot(log10(Shields(1:3)),log10(f(1:3)),'Marker','s','MarkerSize',12,'Markerfacecolor','red','color','r','LineWidth',1.8)
 plot(log10(Shields_EXP(1:3)),log10(f_EXP_slurry(1:3)),'Marker','s','MarkerSize',12,'Markerfacecolor','black','color','k','LineWidth',1.8)
 plot(log10(Shields(4:8)),log10(f(4:8)),'Marker','s','MarkerSize',12,'Markerfacecolor','blue','color','b','LineWidth',1.8)
 yticks([-2 -1.5 -1 -0.5 0.0 0.5])
 ylim([-1.7 0.5])
 yticklabels({'$10^{-2.0}$','$10^{-1.5}$','$10^{-1.0}$','$10^{-0.5}$','$10^{0.0}$','$10^{0.5}$'})
 xlim([-0.75 1])
 xticks([-0.5 0.0 0.5 1.0])
 xticklabels({'$10^{-0.5}$','$10^{0.0}$','$10^{0.5}$','$10^{1.0}$'})
 legend(  'D1-D3',...
          'E1-E3',...
          'S1-S5',...
          'Location','NorthEast')
 xlabel('$\mathcal{\theta}$')
 ylabel('$f$')
end

%--------------------------------------------
% Figure 7
if(1)
 fig7_D1 = load('Fig7_D1.txt');
 fig7_D2 = load('Fig7_D2.txt');
 fig7_D3 = load('Fig7_D3.txt');
 fig7_E1 = load('Fig7_E1.txt');
 fig7_E2 = load('Fig7_E2.txt');
 fig7_E3 = load('Fig7_E3.txt');

 figure(7)
 hold on
 plot(fig7_D1(:,1),fig7_D1(:,3),'-b','LineWidth',2.)
 plot(fig7_D2(:,1),fig7_D2(:,3),'-r','LineWidth',2.)
 plot(fig7_D3(:,1),fig7_D3(:,3),'-k','LineWidth',2.)
 plot(fig7_D1(:,2),fig7_D1(:,3),'--b','LineWidth',2.)
 plot(fig7_D2(:,2),fig7_D2(:,3),'--r','LineWidth',2.)
 plot(fig7_D3(:,2),fig7_D3(:,3),'--k','LineWidth',2.)
 plot(fig7_E1(:,1),fig7_E1(:,2),'-b','Marker','Square','LineWidth',2.)
 plot(fig7_E2(:,1),fig7_E2(:,2),'-r','Marker','Square','LineWidth',2.)
 plot(fig7_E3(:,1),fig7_E3(:,2),'-k','Marker','Square','LineWidth',2.)
 xlabel('$\overline{\langle v_{p/f} \rangle}/v_{bl}$')
 ylabel('$z/D_p $')
 ylim([-10 10])
 legend('D1/E1','D2/E2','D3/E3','Location','NorthWest')
 % inset
 axes('Position',[.75 .27 .15 .36])
 box on
 hold on
 plot(fig7_D1(:,1),fig7_D1(:,3),'-b','LineWidth',2.)
 plot(fig7_D1(:,2),fig7_D1(:,3),'--b','LineWidth',2.)
 xlim([0 0.02])
 ylim([-10 0])
end

%--------------------------------------------
% Figure 9
if(1)
 fig9_D1 = load('Fig9_D1.txt');
 fig9_D2 = load('Fig9_D2.txt');
 fig9_D3 = load('Fig9_D3.txt');
 fig9_E1 = load('Fig9_E1.txt');
 fig9_E2 = load('Fig9_E2.txt');
 fig9_E3 = load('Fig9_E3.txt');

 figure(9)
 hold on
 plot(fig9_E1(:,1),fig9_E1(:,2),'-b','LineWidth',2.,'Marker','s','MarkerSize',8.,'MarkerFaceColor','b')
 plot(fig9_E2(:,1),fig9_E2(:,2),'-r','LineWidth',2.,'Marker','s','MarkerSize',8.,'MarkerFaceColor','r')
 plot(fig9_E3(:,1),fig9_E3(:,2),'-k','LineWidth',2.,'Marker','s','MarkerSize',8.,'MarkerFaceColor','k')
 plot(movmean(fig9_D1(:,1),16),fig9_D1(:,2),'-b','LineWidth',2.)
 plot(movmean(fig9_D2(:,1),16),fig9_D2(:,2),'-r','LineWidth',2.)
 plot(movmean(fig9_D3(:,1),16),fig9_D3(:,2),'-k','LineWidth',2.)
 legend('D1/E1','D2/E2','D3/E3','Location','SouthWest')
 xlabel('$\overline{\phi}(z)$')
 ylabel('$z/D_p$')
 ylim([-10 10])
 % inset
 axes('Position',[.65 .65 .27 .27])
 box on
 hold on
 plot(fig9_D1(:,1),fig9_D1(:,2),'-b','LineWidth',2.)
 plot(fig9_D2(:,1),fig9_D2(:,2),'-r','LineWidth',2.)
 plot(fig9_D3(:,1),fig9_D3(:,2),'-k','LineWidth',2.)
 ylim([-10 10])
end

%--------------------------------------------
% Figure 10
if(1)
 fig10_D1 = load('Fig10_D1.txt');
 fig10_D2 = load('Fig10_D2.txt');
 fig10_D3 = load('Fig10_D3.txt');
 fig10_D1_Turb = load('Fig10_D1_Turb.txt');
 fig10_D2_Turb = load('Fig10_D2_Turb.txt');
 fig10_D3_Turb = load('Fig10_D3_Turb.txt');

 % 10a
 figure(101)
 hold on
 plot( fig10_D1(:,2),fig10_D1(:,1),'LineWidth',2.)
 plot( fig10_D1(:,3),fig10_D1(:,1),'LineWidth',2.)
 plot( fig10_D1(:,4),fig10_D1(:,1),'LineWidth',2.)
 plot( fig10_D1(:,5),fig10_D1(:,1),'LineWidth',2.)
 plot(-fig10_D1(:,2),fig10_D1(:,1),'--k','LineWidth',2.)
 xlim([-0.02 0.02])
 ylim([-10.0 10.0])
 plot(xlim,[10 10],'-k','Linewidth',2.)
 ylabel('$z/D_p$')
 legend('$-dp_e/dy$','$D$','$T$','$Re$','$dp_e/dy$','Location','NorthEast')

 % 10b
 figure(104)
 hold on
 plot(fig10_D1_Turb(:,1),fig10_D1_Turb(:,4),'-','LineWidth',2.0,'Color',[1 0.7 0.1])
 plot(fig10_D1_Turb(:,2),fig10_D1_Turb(:,4),'-r','LineWidth',2.0)
 plot(fig10_D1_Turb(:,3),fig10_D1_Turb(:,4),'-b','LineWidth',2.0)
 plot([0 0],ylim,'--k','LineWidth',1.5)
 ylabel('$z/D_p$')
 xlim([-0.02 0.02])
 plot([-0.02 0.02],[10 10],'-k','LineWidth',1.5)
 ylim([-10 10])
 legend('$T$','$T_S$','$T_F$','Location','SouthWest')

 % 10c
 figure(102)
 hold on
 plot( fig10_D2(:,2),fig10_D2(:,1),'LineWidth',2.)
 plot( fig10_D2(:,3),fig10_D2(:,1),'LineWidth',2.)
 plot( fig10_D2(:,4),fig10_D2(:,1),'LineWidth',2.)
 plot( fig10_D2(:,5),fig10_D2(:,1),'LineWidth',2.)
 plot(-fig10_D2(:,2),fig10_D2(:,1),'--k','LineWidth',2.)
 xlim([-0.02 0.02])
 ylim([-10.0 10.0])
 plot(xlim,[10 10],'-k','Linewidth',2.)
 ylabel('$z/D_p$')
 legend('$-dp_e/dy$','$D$','$T$','$Re$','$dp_e/dy$','Location','NorthEast')

 % 10d
 figure(105)
 hold on
 plot(fig10_D2_Turb(:,1),fig10_D2_Turb(:,4),'-','LineWidth',2.0,'Color',[1 0.7 0.1])
 plot(fig10_D2_Turb(:,2),fig10_D2_Turb(:,4),'-r','LineWidth',2.0)
 plot(fig10_D2_Turb(:,3),fig10_D2_Turb(:,4),'-b','LineWidth',2.0)
 plot([0 0],ylim,'--k','LineWidth',1.5)
 ylabel('$z/D_p$')
 xlim([-0.02 0.02])
 plot([-0.02 0.02],[10 10],'-k','LineWidth',1.5)
 ylim([-10 10])
 legend('$T$','$T_S$','$T_F$','Location','SouthWest')

 % 10e
 figure(103)
 hold on
 plot( fig10_D3(:,2),fig10_D3(:,1),'LineWidth',2.)
 plot( fig10_D3(:,3),fig10_D3(:,1),'LineWidth',2.)
 plot( fig10_D3(:,4),fig10_D3(:,1),'LineWidth',2.)
 plot( fig10_D3(:,5),fig10_D3(:,1),'LineWidth',2.)
 plot(-fig10_D3(:,2),fig10_D3(:,1),'--k','LineWidth',2.)
 xlim([-0.02 0.02])
 ylim([-10.0 10.0])
 plot(xlim,[10 10],'-k','Linewidth',2.)
 ylabel('$z/D_p$')
 legend('$-dp_e/dy$','$D$','$T$','$Re$','$dp_e/dy$','Location','NorthEast')

 % 10f
 figure(106)
 hold on
 plot(fig10_D3_Turb(:,1),fig10_D3_Turb(:,4),'-','LineWidth',2.0,'Color',[1 0.7 0.1])
 plot(fig10_D3_Turb(:,2),fig10_D3_Turb(:,4),'-r','LineWidth',2.0)
 plot(fig10_D3_Turb(:,3),fig10_D3_Turb(:,4),'-b','LineWidth',2.0)
 plot([0 0],ylim,'--k','LineWidth',1.5)
 ylabel('$z/D_p$')
 xlim([-0.02 0.02])
 plot([-0.02 0.02],[10 10],'-k','LineWidth',1.5)
 ylim([-10 10])
 legend('$T$','$T_S$','$T_F$','Location','SouthWest')
end

%--------------------------------------------
% Figure 14
if(1)
 fig14_S1  = load('Fig14_S1.txt');
 fig14_S2  = load('Fig14_S2.txt');
 fig14_S3  = load('Fig14_S3.txt');
 fig14_S4  = load('Fig14_S4.txt');
 fig14_S5  = load('Fig14_S5.txt');
 fig14_S1p = load('Fig14_S1p.txt');
 fig14_S2p = load('Fig14_S2p.txt');
 fig14_S3p = load('Fig14_S3p.txt');
 fig14_S4p = load('Fig14_S4p.txt');
 fig14_S5p = load('Fig14_S5p.txt');

 figure(131)
 hold on
 plot(fig14_S1(:,1),fig14_S1(:,3),'-m','LineWidth',2.)
 plot(fig14_S2(:,1),fig14_S2(:,3),'-k','LineWidth',2.)
 plot(fig14_S3(:,1),fig14_S3(:,3),'-r','LineWidth',2.)
 plot(fig14_S4(:,1),fig14_S4(:,3),'-b','LineWidth',2.)
 plot(fig14_S5(:,1),fig14_S5(:,3),'-g','LineWidth',2.)
 legend('S1','S2','S3','S4','S5','Location','NorthEast')
 xlabel('$\overline{\phi}(z)$')
 ylabel('$z/D_p$')
 ylim([-10 10])

 figure(132)
 hold on
 plot(fig14_S1(:,2),fig14_S1(:,3)  ,'-m' ,'LineWidth',2.)
 plot(fig14_S2(:,2),fig14_S2(:,3)  ,'-k' ,'LineWidth',2.)
 plot(fig14_S3(:,2),fig14_S3(:,3)  ,'-r' ,'LineWidth',2.)
 plot(fig14_S4(:,2),fig14_S4(:,3)  ,'-b' ,'LineWidth',2.)
 plot(fig14_S5(:,2),fig14_S5(:,3)  ,'-g' ,'LineWidth',2.)
 plot(fig14_S1p(:,1),fig14_S1p(:,2),'--m','LineWidth',2.)
 plot(fig14_S2p(:,1),fig14_S2p(:,2),'--k','LineWidth',2.)
 plot(fig14_S3p(:,1),fig14_S3p(:,2),'--r','LineWidth',2.)
 plot(fig14_S4p(:,1),fig14_S4p(:,2),'--b','LineWidth',2.)
 plot(fig14_S5p(:,1),fig14_S5p(:,2),'--g','LineWidth',2.)
 xlabel('$\overline{\langle v_{p,f} \rangle}/v_{bl}$')
 ylabel('$z/D_p$')
 ylim([-10 10])
 legend('S1','S2','S3','S4','S5','Location','SouthEast')
end

%--------------------------------------------
% Figure 15
if(1)
 fig15  = load('Fig15.txt');

 % 15a
 figure(14)
 hold on
 grid on
 yyaxis right
 plot(fig15(:,1),fig15(:,3),'k^','MarkerSize',10,'MarkerEdgeColor','black','Markerfacecolor','k');
 ax = gca;
 ax.YColor = 'k';
 ylabel('$\overline{z}_p/D_p$')
 xlabel('$Re$')
 plot(fig15(5,1),fig15(5,3),'ok','MarkerSize',20,'MarkerEdgeColor','black')
 yyaxis left
 plot(fig15(:,1),fig15(:,2),'rs','MarkerSize',10,'MarkerEdgeColor','red','Markerfacecolor','r');
 ax = gca;
 ax.YColor = 'r';
 plot(fig15(5,1),fig15(5,2),'or','MarkerSize',20,'MarkerEdgeColor','red')
 xlim([2000 13000])
 ylabel('$v_{bs}/v_{bl}$')

 figure(2)
 hold on
 grid on
 yyaxis right
 plot(fig15(:,4),fig15(:,3),'k^','MarkerSize',10,'MarkerEdgeColor','black','Markerfacecolor','k');
 ax = gca;
 ax.YColor = 'k';
 ylabel('$\overline{z}_p/D_p$')
 xlabel('$\theta$')
 plot(fig15(5,4),fig15(5,3),'ok','MarkerSize',20,'MarkerEdgeColor','black')
 yyaxis left
 plot(fig15(:,4),fig15(:,2),'rs','MarkerSize',10,'MarkerEdgeColor','red','Markerfacecolor','r');
 ax = gca;
 ax.YColor = 'r';
 plot(fig15(5,4),fig15(5,2),'or','MarkerSize',20,'MarkerEdgeColor','red')
 ylabel('$v_{bs}/v_{bl}$')

 % Linear fit through velocity S1-S3 03 Jan 2024
 A1 = xlim;
 x1 = linspace(A1(1),1.3,3);
 y1 = 0.7875*x1 + -0.1568;
 plot(x1,y1,'--r','LineWidth',1.5)
end

%--------------------------------------------
% Figure 16
if(1)
 fig16_S1  = load('Fig16_S1.txt');
 fig16_S2  = load('Fig16_S2.txt');
 fig16_S3  = load('Fig16_S3.txt');
 fig16_S4  = load('Fig16_S4.txt');
 fig16_S5  = load('Fig16_S5.txt');
 fig10_S5_Turb = load('Fig10_S5_Turb.txt');

 figure(151)
 hold on
 plot( fig16_S1(:,2),fig16_S1(:,1),'LineWidth',2.)
 plot( fig16_S1(:,3),fig16_S1(:,1),'LineWidth',2.)
 plot( fig16_S1(:,4),fig16_S1(:,1),'LineWidth',2.)
 plot( fig16_S1(:,5),fig16_S1(:,1),'LineWidth',2.)
 plot(-fig16_S1(:,2),fig16_S1(:,1),'--k','LineWidth',2.)
 xlim([-0.02 0.02])
 ylim([-10. 10.])
 ylabel('$z/D_p$')
 plot(xlim,[10 10],'-k','Linewidth',2.)
 legend('$-dp_e/dy$','$D$','$T$','$\mathcal{R}$','$+dp_e/dy$','Location','NorthEast')

 figure(152)
 hold on
 plot( fig16_S2(:,2),fig16_S2(:,1),'LineWidth',2.)
 plot( fig16_S2(:,3),fig16_S2(:,1),'LineWidth',2.)
 plot( fig16_S2(:,4),fig16_S2(:,1),'LineWidth',2.)
 plot( fig16_S2(:,5),fig16_S2(:,1),'LineWidth',2.)
 plot(-fig16_S2(:,2),fig16_S2(:,1),'--k','LineWidth',2.)
 xlim([-0.02 0.02])
 ylim([-10. 10.])
 ylabel('$z/D_p$')
 plot(xlim,[10 10],'-k','Linewidth',2.)
 legend('$-dp_e/dy$','$D$','$T$','$\mathcal{R}$','$+dp_e/dy$','Location','NorthEast')

 figure(153)
 hold on
 plot( fig16_S3(:,2),fig16_S3(:,1),'LineWidth',2.)
 plot( fig16_S3(:,3),fig16_S3(:,1),'LineWidth',2.)
 plot( fig16_S3(:,4),fig16_S3(:,1),'LineWidth',2.)
 plot( fig16_S3(:,5),fig16_S3(:,1),'LineWidth',2.)
 plot(-fig16_S3(:,2),fig16_S3(:,1),'--k','LineWidth',2.)
 xlim([-0.02 0.02])
 ylim([-10. 10.])
 ylabel('$z/D_p$')
 plot(xlim,[10 10],'-k','Linewidth',2.)
 legend('$-dp_e/dy$','$D$','$T$','$\mathcal{R}$','$+dp_e/dy$','Location','NorthEast')

 figure(154)
 hold on
 plot( fig16_S4(:,2),fig16_S4(:,1),'LineWidth',2.)
 plot( fig16_S4(:,3),fig16_S4(:,1),'LineWidth',2.)
 plot( fig16_S4(:,4),fig16_S4(:,1),'LineWidth',2.)
 plot( fig16_S4(:,5),fig16_S4(:,1),'LineWidth',2.)
 plot(-fig16_S4(:,2),fig16_S4(:,1),'--k','LineWidth',2.)
 xlim([-0.02 0.02])
 ylim([-10. 10.])
 ylabel('$z/D_p$')
 plot(xlim,[10 10],'-k','Linewidth',2.)
 legend('$-dp_e/dy$','$D$','$T$','$\mathcal{R}$','$+dp_e/dy$','Location','NorthEast')

 figure(155)
 hold on
 plot( fig16_S5(:,2),fig16_S5(:,1),'LineWidth',2.)
 plot( fig16_S5(:,3),fig16_S5(:,1),'LineWidth',2.)
 plot( fig16_S5(:,4),fig16_S5(:,1),'LineWidth',2.)
 plot( fig16_S5(:,5),fig16_S5(:,1),'LineWidth',2.)
 plot(-fig16_S5(:,2),fig16_S5(:,1),'--k','LineWidth',2.)
 xlim([-0.02 0.02])
 ylim([-10. 10.])
 ylabel('$z/D_p$')
 plot(xlim,[10 10],'-k','Linewidth',2.)
 legend('$-dp_e/dy$','$D$','$T$','$\mathcal{R}$','$+dp_e/dy$','Location','NorthEast')

 figure(156)
 hold on
 plot(fig10_S5_Turb(:,1),fig10_S5_Turb(:,4),'-','LineWidth',2.0,'Color',[1 0.7 0.1])
 plot(fig10_S5_Turb(:,2),fig10_S5_Turb(:,4),'-r','LineWidth',2.0)
 plot(fig10_S5_Turb(:,3),fig10_S5_Turb(:,4),'-b','LineWidth',2.0)
 plot([0 0],ylim,'--k','LineWidth',1.5)
 ylabel('$z/D_p$')
 xlim([-0.01 0.01])
 plot([-0.02 0.02],[10 10],'-k','LineWidth',1.5)
 ylim([-10 10])
 legend('$T$','$T_S$','$T_F$','Location','SouthWest')
end

%--------------------------------------------
% Figure A3
if(1)
 data1 = load('Measurements_Jul07.txt');
 data2 = load('Measurements_Jul14.txt');

 data3 = load('Measurements_Aug04_1.txt');
 data4 = load('Measurements_Aug04_2.txt');
 data5 = load('Measurements_Aug04_3.txt');
 data6 = load('Measurements_Aug04_4.txt');

 % mu = tan(theta)
 data1(:,4) = (data1(:,2) - 2.2)/36.;
 data1(:,5) = (data1(:,3) - 2.2)/36.;
 data2(:,4) = (data2(:,2) - 2.2)/36.;
 data2(:,5) = (data2(:,3) - 2.2)/36.;
 data3(:,4) = (data3(:,2) - 2.2)/36.;
 data3(:,5) = (data3(:,3) - 2.2)/36.;
 data4(:,4) = (data4(:,2) - 2.2)/36.;
 data4(:,5) = (data4(:,3) - 2.2)/36.;
 data5(:,4) = (data5(:,2) - 2.2)/36.;
 data5(:,5) = (data5(:,3) - 2.2)/36.;
 data6(:,4) = (data6(:,2) - 2.2)/36.;
 data6(:,5) = (data6(:,3) - 2.2)/36.;

 All = [data1(:,5);data2(:,5)];
 mean_mu = mean(All); % mean (mu) = 0.3889
 std_mu = sqrt(sum((All - mean_mu).^2)/size(All,1)); % std = 0.0234

 % A3a
 figure(661)
 hold on
 plot(data1(:,1),data1(:,5),'.',...
     'MarkerEdgeColor','red',...
     'MarkerFaceColor','red',...
     'MarkerSize',20)
 plot(data2(:,1),data2(:,5),'.',...
     'MarkerEdgeColor','blue',...
     'MarkerFaceColor','blue',...
     'MarkerSize',20)
 plot(data1(:,1), mean_mu * ones(size(data1,1)),'--k','LineWidth',2.)
 plot(data2(:,1), mean_mu * ones(size(data2,1)),'--k','LineWidth',2.)
 errorbar(data1(:,1), mean_mu * ones(size(data1,1),1), std_mu * ones(size(data1,1),1),'k','LineWidth',0.01)
 errorbar(data2(:,1), mean_mu * ones(size(data2,1),1), std_mu * ones(size(data2,1),1),'k','LineWidth',0.01)
 text(30,0.32,'$\bar{\mu}_c=0.39$')
 pbaspect([2 1 1])
 ylabel('$\mu_c$')
 legend('data set 1','data set 2','Location','SouthEast')

 % A3b
 figure(662)
 hold on
 plot(data3(:,1),data3(:,5),'.',...
     'MarkerEdgeColor','red',...
     'MarkerFaceColor','red',...
     'MarkerSize',20)
 plot(data4(:,1),data4(:,5),'.',...
     'MarkerEdgeColor','blue',...
     'MarkerFaceColor','blue',...
     'MarkerSize',20)
 plot(data5(:,1),data5(:,5),'.',...
     'MarkerEdgeColor','magenta',...
     'MarkerFaceColor','magenta',...
     'MarkerSize',20)
 plot(data6(:,1),data6(:,5),'.',...
     'MarkerEdgeColor','green',...
     'MarkerFaceColor','green',...
     'MarkerSize',20)
 mean_mu_3 = mean(data3(:,5)); % 
 std_mu_3 = sqrt(sum((data3(:,5) - mean_mu_3).^2)/size(data3,1)); % 
 mean_mu_4 = mean(data4(:,5)); % 
 std_mu_4 = sqrt(sum((data4(:,5) - mean_mu_4).^2)/size(data4,1)); % 
 mean_mu_5 = mean(data5(:,5)); % 
 std_mu_5 = sqrt(sum((data5(:,5) - mean_mu_5).^2)/size(data5,1)); % 
 mean_mu_6 = mean(data6(:,5)); % 
 std_mu_6 = sqrt(sum((data6(:,5) - mean_mu_6).^2)/size(data6,1)); % 
 plot(data3(:,1), mean_mu_3 * ones(size(data3,1)),'--k','LineWidth',2.)
 plot(data4(:,1), mean_mu_4 * ones(size(data4,1)),'--k','LineWidth',2.)
 plot(data5(:,1), mean_mu_5 * ones(size(data5,1)),'--k','LineWidth',2.)
 plot(data6(:,1), mean_mu_6 * ones(size(data6,1)),'--k','LineWidth',2.)
 errorbar(data3(:,1), mean_mu_3 * ones(size(data3,1),1), std_mu_3 * ones(size(data3,1),1),'k','LineWidth',0.01)
 errorbar(data4(:,1), mean_mu_4 * ones(size(data4,1),1), std_mu_4 * ones(size(data4,1),1),'k','LineWidth',0.01)
 errorbar(data5(:,1), mean_mu_5 * ones(size(data5,1),1), std_mu_5 * ones(size(data5,1),1),'k','LineWidth',0.01)
 errorbar(data6(:,1), mean_mu_6 * ones(size(data6,1),1), std_mu_6 * ones(size(data6,1),1),'k','LineWidth',0.01)
 text(5,0.47,'$\bar{\mu}_c=0.37$')
 text(22.5,0.47,'$\bar{\mu}_c=0.37$')
 text(33,0.38,'$\bar{\mu}_c=0.43$')
 text(42.5,0.35,'$\bar{\mu}_c=0.44$')
 plot([20.5 20.5],[0.3 0.5],'--k','LineWidth',2.)
 plot([31.5 31.5],[0.3 0.5],'--k','LineWidth',2.)
 plot([40.5 40.5],[0.3 0.5],'--k','LineWidth',2.)
 ylabel('$\mu_c$')
 pbaspect([2 1 1])
end

%--------------------------------------------
% Figure A4
if(1)
 p1 = load('part_data1.txt');
 p2 = load('part_data2.txt');
 nu = load('nu.txt');

 % A4a
 figure(663)
 hold on
 plot(p1(:,1),p1(:,2)/trapz(p1(:,1),p1(:,2)),'LineWidth',2.)
 plot(p2(:,1),p2(:,2)/trapz(p2(:,1),p2(:,2)),'LineWidth',2.)
 xlabel('Diameter [$mm$]')
 ylabel('$pdf [mm^{-1}]$')
 plot([1.973 1.973],ylim,'--k','LineWidth',1.5)
 plot([1.995 1.995],ylim,'--k','LineWidth',1.5)
 xlim([1.8 2.1])
 legend('Before','After','Location','NorthWest')
 grid on

 % A4b
 figure(664)
 hold on
 plot(nu(:,1),nu(:,2),'o','MarkerEdgeColor','k','MarkerSize',7.5,'MarkerFaceColor','k')
 xlabel('$T [^\circ C]$')
 ylabel('$\nu[m^2/s]$')
 xlim([16 30])
 ylim([7.5e-7 11.5e-7])
 grid on
end

%--------------------------------------------
% Figure B2
if(1)
 FigB2_AC1 = load('FigB2_Left_Acorr.txt');
 FigB2_YT1 = load('FigB2_Left_YT.txt');
 FigB2_AC2 = load('FigB2_Right_Acorr.txt');
 FigB2_YT2 = load('FigB2_Right_YT.txt');

 % B2a
 figure(771)
 hold on
 hold on
 plot(FigB2_YT1(:,1),FigB2_YT1(:,2),'-s')
 plot(FigB2_AC1(:,1),FigB2_AC1(:,2),'--')
 plot(xlim,[280 280],'--k')
 plot(xlim,[620 620],'-k')
 plot(xlim,[20 20],'-k')
 xlabel('$\overline{v}_p$ [pixels/frames]')
 ylabel('z [pixels]')
 legend('YT scan','Auto-corr','Location','SouthEast')
 text(0.5, 650,'Top wall')
 text(0.5, 40,'Bottom wall')

 % B2b
 figure(772)
 hold on
 plot(FigB2_YT2(:,1),FigB2_YT2(:,2),'-s')
 plot(FigB2_AC2(:,1),FigB2_AC2(:,2),'--')
 plot(xlim,[40 40],'--k')
 plot(xlim,[0 0],'-k')
 xlabel('$\overline{v}_p [mm/s]$')
 ylabel('z [mm]')
 legend('YT scan','Auto-corr','Location','SouthEast')
end

%--------------------------------------------
% Figure C1
if(1)
 % Load mean velocity profile Um v R
 DNS  = load('Graph0_my.dat');
 ref1 = load('Graph0_ref1.dat');
 ref2 = load('Graph0_ref2.dat');

 % Load mean velocity profile Um v R
 rms_DNS  = load('Graph2_my.dat');
 rms_ref1 = load('Graph2_ref1.dat');
 rms_ref2 = load('Graph2_ref2.dat');

 % C1a
 figure(881)
 hold on
 plot(DNS(:,1),DNS(:,2),'k','LineWidth',1.4)
 plot(ref1(:,1),ref1(:,2),'s','MarkerEdgeColor','k','MarkerSize',7.,'LineWidth',1.5)
 plot(ref2(:,1),ref2(:,2),'d','MarkerEdgeColor','k','MarkerSize',7.,'LineWidth',1.5)
 xlabel('$(1-r/R_{pipe})$')
 ylabel('$\overline{v}/v_{bl}$')
 legend('DNS','Wu and Moin (2008)','Eggels et al. (1994)','Location','SouthEast')

 % C1b
 figure(882)
 hold on
 plot(rms_DNS (:,1),rms_DNS (:,2),'k','LineWidth',1.4)
 plot(rms_ref1(:,1),rms_ref1(:,2),'s','MarkerEdgeColor','k','MarkerSize',7.,'LineWidth',1.5)
 plot(rms_ref2(:,1),rms_ref2(:,2),'d','MarkerEdgeColor','k','MarkerSize',7.,'LineWidth',1.5)
 xlabel('$(1-r/R_{pipe})^+$')
 ylabel('$v_{rms}/v_\tau$')
 legend('DNS','Wu and Moin (2008)','Eggels et al. (1994)','Location','NorthEast')
end
