%-------------------------------------------------------------------------%
% Script to compute output displacement through a multi-layered system
% using the transfer matrix method.
% Flexible for any number and type of layers (homogeneous or enriched).

% Created by X.Kuci, TU/e, 2024
%-------------------------------------------------------------------------%
clc
clear all
% close all
%---------------------------------------------------------------
% Setup problem: load material data (HM structs), define layers

% Load or define layers. Example of 5 layered problem:
load('HM_matrix.mat')
load('HM_UC1')
load('HM_HardRubber.mat')
load('HM_RotatedRubberplus.mat')
load('HM_layer3.mat')

loadMats;

% Homogeneous layer 1 properties (matrix)
K_m = matrix{1}.E/3/(1-2*matrix{1}.nu);
G_m = matrix{1}.E/2/(1+matrix{1}.nu);
rho_m = matrix{1}.rho;
cL_m = sqrt((K_m+(4/3)*G_m)/rho_m);
cS_m = sqrt(G_m/rho_m);

% Homogeneous layer 3 properties (matrix with E/3)
K_3 = layer3{1}.E/3/(1-2*layer3{1}.nu);
G_3 = layer3{1}.E/2/(1+layer3{1}.nu);
rho_3 = layer3{1}.rho;
cL_3 = sqrt((K_3+(4/3)*G_3)/rho_3);
cS_3 = sqrt(G_3/rho_3);

% Define geometry and layer sequence 
layers = struct([]);
% layers().type - string: 'hom' for homogeneous layer, 'enriched' for metamaterial
% layers().W - unit cell size (in meters) for this layer
W = 20e-3; % unit cell size
% layers().N - length of layer (number of unit cells, integer)


% Example for 5 layers:
layers(1).type = 'hom';      % semi-infinite (input)
layers(1).HM   = HM_matrix;
layers(1).W    = W; % arbitrary, not used for 1 layer (semi-infinite)
layers(1).N    = 1;

layers(2).type = 'enriched';
layers(2).HM   = HM_UC1;
layers(2).W    = W;
layers(2).N    = 5;

layers(3).type = 'enriched';      % could also be 'enriched'
layers(3).HM   = HM_HardRubber;
layers(3).W    = W;
layers(3).N    = 5;

layers(4).type = 'enriched';
layers(4).HM   = HM_RotatedRubberplus;
layers(4).W    = W;
layers(4).N    = 5;

layers(5).type = 'hom';      % semi-infinite (output)
layers(5).HM   = HM_layer3;
layers(5).W    = W;
layers(5).N    = 1;
dx = 0.2;%05419248304; % distance from the last interface x_{N-1} to the measurement point x_{dx},
% as in Eq. (47) of the paper. (Here, it was calculated from COSMOL)

% Direction vector 
e_x = [1; 0];

%  Frequency, incident angle, input wavenumber 
f = 100:10:1200;                    % Hz
theta0 = 0*pi/180;               % radians (or angle(degrees)*180/pi)
Nf = numel(f);


% Preallocate outputs:
u_out = zeros(2, Nf);
u_mod = zeros(1, Nf);


for i = 1:Nf
    fi = f(i);
    omegai = 2*pi*fi;

    % Define incident field (for completeness)
    x0 = 0;
    y0 = W/2;
    
%------ Case: first layer = solid -----------%
    % Input: solid plane wave (see Eq.(45) in paper)
    k0i = 2*pi*fi / cL_m;  % wavenumber in first layer 
    Au0 = 1*cos(theta0);
    Av0 = 1*sin(theta0);

   u_in = [Au0 * exp(-1i * (k0i * cos(theta0) * x0 + k0i * sin(theta0) * y0));
        Av0 * exp(-1i * (k0i * cos(theta0) * x0 + k0i * sin(theta0) * y0))]; 

%---------------------------------------------------%
%------ Case: first layer = acoustic -----------%
    % Input: displacement from acoustic pressure (see Eq.(48) in paper)
      % rho_acoust = 1000; 
      % c_acoust = 1500;
      % 
      % k0i = 2*pi*fi / c_acoust; % wavenumber in first layer 
      % k0i_x = k0i*cos(theta0);
      % k0i_y = k0i*sin(theta0);
      % Pin = 1*exp(-1i*(k0i_x*x0 + k0i_y*y0));  % Plane wave pressure
      % u0 = (1i*k0i_x/(rho_acoust*omegai^2))*Pin;
      % v0 = (1i*k0i_y/(rho_acoust*omegai^2))*Pin;
      % u_in = [u0; v0];
%---------------------------------------------------%
%---------------------------------------------------%
    
    % Compute 2x2 transmission tensor at this frequency
    T = get_transm_tensor(layers, fi, k0i, theta0, e_x, dx);

     % Output displacement 
    u_out(:,i) = T * conj(u_in); 

    % Magnitude 
    u_mod(i) = sqrt(abs(u_out(1,i))^2 + abs(u_out(2,i))^2);
    
end

% Plot
% figure;
hold on
plot(f, u_mod,'k', 'LineWidth', 2);
set(gca, 'YScale', 'log')
xlabel('Frequency (Hz)');
ylabel('Output displacement magnitude');

%grid on;
