%-------------------------------------------------------------------------%
% Background
% Compute the elementwise integral (2D only) of the discretised geometry
% Able to distinct between different parts of the geometry (e.g. core,
% coating or matrix)
% 2D
% Created by S.A. Verhoeven, TU/e, 2022
%-------------------------------------------------------------------------%

function integ = elemIntav(coords,elems,Q,parts,sParts)
% Function to determine the discrete integral over the mesh elements. 

% if nargin==4 || nargin>5
%     error('wrong number of input arguments')
% end

% if nargin>3
%     parts = varargin{1};
%     sParts = varargin{2};
% else
%     parts = ones(length(elems));
%     sParts = [1,2,3];
% end    

global coordsElem tol

s = ismember(parts,sParts); % selector
Ae = zeros(length(elems),1);

tol=1e-6;

for i = 1:length(elems)
    coordsElem = coords(elems(i,:),:);
    if length(coordsElem) == 3 % triangular
        fprintf('triangular\n')
        [~,Lidx] = min(coordsElem(:,1)); % Left
        [~,Ridx] = max(coordsElem(:,1)); % Right
        [~,Bidx] = min(coordsElem(:,2)); % Bottom
        [~,Tidx] = max(coordsElem(:,2)); % Top

        if ( (Tidx == Ridx) && (Tidx ~= Lidx) ) || ( (Tidx == Lidx) && (Tidx ~= Ridx) )
            Cidx = Tidx;
        elseif ( (Bidx == Ridx) && (Bidx ~= Lidx) ) || ( (Bidx == Lidx) && (Bidx ~= Ridx) )
            Cidx = Bidx;
        else
            error('incorrectly selected triangular element')
        end

        Lc = coordsElem(Lidx,:);
        Rc = coordsElem(Ridx,:);
        Cc = coordsElem(Cidx,:);

        B  = sqrt( ( Rc(1) - Lc(1) )^2 + ( Rc(2) - Lc(2) )^2 );
        theta = atan( (Rc(2)-Lc(2)) / (Rc(1)-Lc(1)) );
        h1 = abs( Cc(2)-Lc(2) ) * cos(theta);

        Ae(i) = (h1*B + h2*B)/2; % elemental area

    elseif length(coordsElem) == 4 % quadrilaterial
        coordsElem = coords(elems(i,:),:);
        [~,Lidx] = min(coordsElem(:,1)); % Left
        [~,Ridx] = max(coordsElem(:,1)); % Right
        [~,Bidx] = min(coordsElem(:,2)); % Bottom
        [~,Tidx] = max(coordsElem(:,2)); % Top
              
        [Lidx,Ridx,Bidx,Tidx] = check_index(Lidx,Ridx,Bidx,Tidx);
        if mean([Lidx,Ridx,Bidx,Tidx]) == 7
            % perfect square
            W = abs(max(coordsElem(:,1))-min(coordsElem(:,1)));
            H = abs(max(coordsElem(:,2))-min(coordsElem(:,2)));
            Ae(i) = W*H;
            continue
        elseif mean([Lidx,Ridx,Bidx,Tidx]) == 8
            % trapezoidal, rotated 90deg:
            % https://cdn.tutors.com/assets/images/courses/math/geometry-help/area-of-a-trapezoid.jpg
            x = uniquetol(coordsElem(:,1),tol);
            left  = coordsElem(ismembertol(coordsElem(:,1),min(x),tol),:);
            right = coordsElem(ismembertol(coordsElem(:,1),max(x),tol),:);

            a = abs( left(1,1)- left(1,2));
            b = abs(right(1,1)-right(1,2));
            h = abs(x(1)-x(2));
            Ae(i) = h*(a+b)/2;
            continue
        elseif mean([Lidx,Ridx,Bidx,Tidx]) == 9
            % trapezoidal:
            % https://cdn.tutors.com/assets/images/courses/math/geometry-help/area-of-a-trapezoid.jpg
            y = uniquetol(coordsElem(:,2),tol);
            bottom = coordsElem(ismembertol(coordsElem(:,2),min(y),tol),:);
            top    = coordsElem(ismembertol(coordsElem(:,2),max(y),tol),:);
            
            % debug
            if ~isequal(size(top),[2,2]) || ~isequal(size(bottom),[2,2])
                keyboard
            end
            
            a = abs(bottom(1,1)-bottom(2,1));
            b = abs(   top(1,1)-   top(2,1));
            h = abs(y(1)-y(2));
            Ae(i) = h*(a+b)/2;
            continue
        end
        
        Lc = coordsElem(Lidx,:);
        Rc = coordsElem(Ridx,:);
        Bc = coordsElem(Bidx,:);
        Tc = coordsElem(Tidx,:);

        % Area computed by: https://www.archtoolbox.com/calculating-area/
        % Section: Area of a Quadrilateral
        B  = sqrt( ( Rc(1) - Lc(1) )^2 + ( Rc(2) - Lc(2) )^2 );
        theta = atan( (Rc(2)-Lc(2)) / (Rc(1)-Lc(1)) );
        h1 = ( Tc(2)-Lc(2) ) * cos(theta);
        h2 = ( Lc(2)-Bc(2) ) / cos(theta);

        Ae(i) = (h1+h2)*B/2; % elemental area

    else
        error('wrong element type detected')
    end
end    

A = sum(Ae); % total area
integ = 0;   % outputted integral

lenQ = length(Q);
% Quantity selector
if lenQ == max(parts)
    % homogeneous assumed in each heterogeneity
    for i = 1:lenQ
        if ismember(i,sParts)
            integ = integ + sum(Ae(parts==i)*Q(i));
        end
    end
elseif lenQ == length(elems)
    % assuming homogeneous quantity per element
    integ = sum(Ae(s).*Q(s));
elseif lenQ == length(coords)
    % assuming a more or less continous field
    % todo :)
end 
keyboard
integ = integ/A;

end


function [b,a] = swap(a,b)
% this function has no body!
% and is cheaper than deal!
% https://nl.mathworks.com/matlabcentral/answers/362680-how-to-swap-values-of-two-variables
end


function [Lidx,Ridx,Bidx,Tidx] = check_index(Lidx,Ridx,Bidx,Tidx)

    global coordsElem tol
    
    if length(uniquetol(abs(coordsElem(:,1)),tol))==2 && length(unique(abs(coordsElem(:,2))))==2
        % perfect square, output mean=7
        Lidx=7;Ridx=7;Bidx=7;Tidx=7; return
    elseif length(uniquetol(abs(coordsElem(:,1)),tol))==2
        % trapezoidal, rotated 90deg, output mean=8
        Lidx=8;Ridx=8;Bidx=8;Tidx=8; return
    elseif length(uniquetol(abs(coordsElem(:,2)),tol))==2
        % trapezoid, output mean=9
        Lidx=9;Ridx=9;Bidx=9;Tidx=9; return
    end
        
    if Lidx==Bidx
        x1=coordsElem(Lidx,:);
        x2=coordsElem(coordsElem(:,2)==x1(2),:);
        x2=x2(x2(:,1)~=x1(1),:);
        if x1(1)>x2(1)
            [x1,x2]=swap(x1,x2); % Bidx has to be inbetween
        end
        [~,Bidx]=ismember(coordsElem,x2,'rows');
        Bidx = find(Bidx==1);
    end
    if Ridx==Bidx
        x1=coordsElem(Ridx,:);
        x2=coordsElem(coordsElem(:,1)==x1(1),:);
        x2=x2(x2(:,2)~=x1(2),:);
        
        if isempty(x2); keyboard; end
        
        if x1(1)>x2(1)
            [x1,x2]=swap(x1,x2); % Bidx has to be inbetween
        end
        [~,Bidx]=ismember(coordsElem,x2,'rows');
        Bidx = find(Bidx==1);
    end
    
    if Lidx==Tidx
        x1=coordsElem(Lidx,:);
        x2=coordsElem(coordsElem(:,1)==x1(1),:);
        x2=x2(x2(:,2)~=x1(2),:);
        if x1(1)>x2(1)
            [x1,x2]=swap(x1,x2); % Bidx has to be inbetween
        end
        [~,Tidx]=ismember(coordsElem,x2,'rows');
        Tidx = find(Tidx==1);
    end
    if Ridx==Tidx
        x1=coordsElem(Ridx,:);
        x2=coordsElem(coordsElem(:,2)==x1(2),:);
        x2=x2(x2(:,1)~=x1(1),:);
        if x1(1)>x2(1)
            [x1,x2]=swap(x1,x2); % Bidx has to be inbetween
        end
        [~,Tidx]=ismember(coordsElem,x2,'rows');
        Tidx = find(Tidx==1);
    end
    
    if Lidx+Tidx+Ridx+Bidx ~= 10
        fprintf('Wrong element index used:\n%i\t%i\t%i\t%i\n',Lidx,Ridx,Bidx,Tidx)
%         keyboard
        Lidx=3;Ridx=1;Bidx=4;Tidx=2;
    end
    
end