%-------------------------------------------------------------------------%
% Background
% Function to add a resonator (type C: torsion resonator)
% four coupling nodes, one resonator
%|            top             |
%|             X              |
%| left- X resonator X-right |
%|             X              |
%|           bottom           |
% Created by Lei, TU/e, 2017
%-------------------------------------------------------------------------%

function [K,M,rDofs,extendElems,extendDofs,Alpha,Theta,coordsHin] = addResonatorC(Ko,Mo,KR,MR,coords,coupNodes,coordsRn,rs,r)

nNodes = length(coords);
I = MR(1) * r^2 /2;                 % rotational inertia
totDof = length(Ko);                % basic DOF number
rDofs = totDof + [1;2;3];           % resonator's DOFs, X, Y, gamma

% allocate 4 symmetric coupling nodes on the matrix
coupDofs = getDofs(coupNodes);
coordsCoupM = coords(coupNodes,:);  % left right bottom top

% 4 coupling nodes on the resonator
coordsCoupR = [0 rs; 0 -rs; -rs 0; rs 0]...
            + [coordsRn; coordsRn; coordsRn; coordsRn;]; % top bottom left right
coordsHin = coordsCoupR;

extendDofs = [rDofs;coupDofs];
extendElems = zeros(length(coupNodes),2);

% compute alpha and theta (right coupling node M as a reference)
alpha = funTheta(coordsCoupR(2,:),coordsRn);
alpha = pi/2 - abs(alpha);
theta = funTheta(coordsCoupM(2,:),coordsCoupR(2,:));

Alpha = alpha + pi/2 *[2 0 3 1];
Theta = theta + pi/2 *[2 0 3 1];

% extend K
K = zeros(totDof+3);
K(1:totDof,1:totDof) = Ko;

for i = 1:length(coupNodes)
    
    df = [ rDofs(1:2); coupDofs(2*i-1:2*i); rDofs(3)];
    
    % bar unit matrices, R-M coupling configuration as reference (0 degree)
    c = cos(Theta(i));
    s = sin(Theta(i));
    RT = [c*c c*s; c*s s*s]; 
    if i<=2
    Ku = KR(1) * RT;
    else
       Ku = 1*KR(1) * RT; 
    end
    
    % force and torque by the rotation
    fR = Ku * [cos(Alpha(i));sin(Alpha(i))] * rs ;
    tv = coordsCoupR(i,:) - coordsRn;
    TRv = cross([tv 0],[fR; 0]);
    TR = TRv(3);
    
    Kt = [Ku -Ku fR; -Ku Ku -fR; fR' -fR' TR];
    K(df,df) = K(df,df) + Kt;
    
    % collect the extra bar element
    extendElems(i,:) = [nNodes+1 coupNodes(i)];
     
end


% extend M
Mt = [MR(1) 0 0; 0 MR(2) 0; 0 0 I ];
M = zeros(totDof+3);
M(1:totDof,1:totDof) = Mo;
M(rDofs,rDofs) = M(rDofs,rDofs) + Mt;

end

function theta = funTheta(coordsCoup,coordsRn)

coordsRel = coordsCoup - coordsRn;       % relative coordinates
R = norm(coordsRel);
t = asin(coordsRel(2) / R);

theta = t*(coordsRel(1)>=0) + (pi-t)*(coordsRel(1)<0);

end