function PlotModesHM_new(coords, elems, Fai_full, scale_factor)
% Function to plot the deformed eigenmodes, i.e., displacement amplitudes
% input:
%           coords      nodal coordinates
%           elems       coords indices per element
%           Fai_full    displacement field, size(totDof,nModes)
%           scale_factor scaling factor for visualizing deformation
%
% Created by S.A. Verhoeven, TU/e, 2022
% Modified by X.Kuci, 2024

% Parameters for the reference geometry (square and circles)
L = 20e-3;           % Side length of the square
Dout = 15e-3;         % Diameter of the outer circle
Din = 10e-3;          % Diameter of the inner circle

% Number of currently open figures
h = findobj('type', 'figure');
n = length(h);

for iMode = 1:size(Fai_full, 2)
    % Calculate displacement amplitude at each node
    dispNode = zeros(size(coords, 1), 1);
    for iNode = 1:size(coords, 1)
        dispNode(iNode) = sqrt(Fai_full(iNode*2-1, iMode)^2 + Fai_full(iNode*2, iMode)^2);
    end
    dispNode = dispNode ./ max(dispNode); % normalize 0-1

    % Calculate deformed coordinates
    deformed_coords = coords;
    for iNode = 1:size(coords, 1)
        deformed_coords(iNode, 1) = coords(iNode, 1) + scale_factor * Fai_full(iNode*2-1, iMode);
        deformed_coords(iNode, 2) = coords(iNode, 2) + scale_factor * Fai_full(iNode*2, iMode);
    end

    % Create a new figure for each mode
    figure(n + iMode);
    clf;
    hold on

    % Plot the reference square and concentric circles
    % Plot square centered at (0,0)
    x_square = [-L/2, L/2, L/2, -L/2, -L/2]; % x-coordinates
    y_square = [-L/2, -L/2, L/2, L/2, -L/2]; % y-coordinates
    plot(x_square, y_square, 'k', 'LineWidth', 1.5); % Black line for the square perimeter

    % Plot outer circle centered at (0,0) with radius Dout/2
    theta = linspace(0, 2*pi, 100); % Angle for the circle
    x_outer = (Dout/2) * cos(theta); % x-coordinates of the outer circle
    y_outer = (Dout/2) * sin(theta); % y-coordinates of the outer circle
    plot(x_outer, y_outer, 'k', 'LineWidth', 1.5); % Black line for the outer circle perimeter

    % Plot inner circle centered at (0,0) with radius Din/2
    x_inner = (Din/2) * cos(theta); % x-coordinates of the inner circle
    y_inner = (Din/2) * sin(theta); % y-coordinates of the inner circle
    plot(x_inner, y_inner, 'k', 'LineWidth', 1.5); % Black line for the inner circle perimeter

    % Plot the deformed mesh with color representing displacement magnitude
    patch_faces = elems;
    patch('Faces', patch_faces, 'Vertices', deformed_coords, 'FaceVertexCData', dispNode, ...
          'FaceColor', 'interp', 'EdgeColor', 'none', 'LineWidth', 1.0);

    % Add a colorbar to represent the deformation magnitude
    colorbar;
    colormap(flipud(hot(256))); % Use the hot colormap, flipped for the colorbar

    % Axis settings
    daspect([1 1 1]);
    axis equal
    axis tight
    title("Mode shape of frequency " + num2str(iMode) + " Hz");
    hold off
end

end % function