%-------------------------------------------------------------------------%
% Background
% Convert the mesh file from Comsol to an appropriate format

% Notes
% 2D case
% quadrilateral element

% Created by Lei, TU/e, 2018
% Modified by X.Kuci, TU/e, 2024
%-------------------------------------------------------------------------%

clear, clc
keyM = 2;   % 1: coarse, 2: fine

fileComsol = ['mesh_UC1_New.mphtxt']; 
fileMesh = ['test.txt']; 


%% Read mesh data

fgmsh = fopen(fileComsol,'r');
nStrings = 0;         % mark the names, 1-MeshFormat, 3-Nodes, 5-Elements
nSelections = 0;
coords = [];
quads = [];
trias = [];
parts = [];
selections = {};

while ~feof(fgmsh)
    
    % read one line 
    tline = fgetl(fgmsh);
    ltline = length(tline);
    numtline = str2num(tline);
    
    % get the names
    if isempty(numtline) && ~isempty(tline)
        
        % detect the nodes
        if strcmpi(tline,'# Mesh point coordinates')
            nStrings = 1;
        end

        % detect the elements
        if strcmpi(tline,'# Elements')
            nStrings = 2;
        end
        
        % detect the parts
        if strcmpi(tline,'# Geometric entity indices')
            nStrings = 3;
        end
        
        % detect the selections
        if (ltline >=18) && (strcmpi(tline(1:18),'# --------- Object')) && ~isempty(coords)
            nStrings = 4; % to exist parts mode
        end
        if strcmpi(tline,'# Entities')
            nSelections = nSelections+1;
            selections{nSelections} = [];
        end
        
    end   
     
     % scan the data
     if ~isempty(numtline)  
            
         % read the nodes
         if nStrings == 1
                
             coords = [coords; numtline];   % normalization REMOVED SV
         end
            
         % read the elements
         if nStrings == 2
             if length(numtline)==3
                trias = [trias; numtline+1];
             elseif length(numtline)==4
                quads = [quads; numtline+1];
             else
                fprintf("Unknown element type detected.\n");
             end
                 
         end
         
         % read the part indexes
         if nStrings == 3
             parts = [parts; numtline];
         end
         
         % read the selectional indices
         if (nSelections ~= 0) && (length(numtline)==1)
             selections{nSelections} = [selections{nSelections}; numtline];
         end
            
     end
end

fclose(fgmsh);


%% Coordinate correction

cenX = (max(coords(:,1))+min(coords(:,1))) / 2;
cenY = (max(coords(:,2))+min(coords(:,2))) / 2;

coords(:,1) = coords(:,1) - cenX;
coords(:,2) = coords(:,2) - cenY;

%% Parts and selections correction
if ~isempty(selections)
    parts2 = zeros(size(parts));
    for i=1:nSelections
        idx = nSelections+1 -i; % reverse indexing
        s = ismember(parts,selections{i});
        parts2(s)=idx; % temporary storage
    end
    parts = parts2;
end


%% Boundary nodes

leftNodes = find(coords(:,1)<=0.9999*min(coords(:,1)));
[~,index] = sort(coords(leftNodes,2));
leftNodes = leftNodes(index);

rightNodes = find(coords(:,1)>=0.9999*max(coords(:,1)));
[~,index] = sort(coords(rightNodes,2));
rightNodes = rightNodes(index);

bottomNodes = find(coords(:,2)<=0.9999*min(coords(:,2)));
[~,index] = sort(coords(bottomNodes,1));
bottomNodes = bottomNodes(index);

topNodes = find(coords(:,2)>=0.9999*max(coords(:,2)));
[~,index] = sort(coords(topNodes,1));
topNodes = topNodes(index);


%% Corner nodes

cornerNodes = [leftNodes(1) rightNodes(1) rightNodes(end) leftNodes(end)];


%% Plot the matrix configuration

nNodes = size(coords,1);
nQuads = size(quads,1);
nTrias = size(trias,1);

X0 = min(coords(:,1));
Y0 = min(coords(:,2));
    
L = max(coords(:,1)) - X0;
H = max(coords(:,2)) - Y0;
    
figure
hold on

% mesh
for iElem =1:(nQuads+nTrias)
     if iElem<=nQuads
        elemNodes = quads(iElem,:);
     else
         elemNodes= trias(iElem-nQuads,:);
     end
     
     if length(elemNodes) == 4 % quad4
         elemNodes([3,4]) = elemNodes([4,3]);  % order correction

         if ispolycw(coords(elemNodes,1),coords(elemNodes,2))  % make sure nodes are numbered couterclockwise
             elemNodes = fliplr(elemNodes);
         end
         quads(iElem,:) = elemNodes;

         nodeZone = [elemNodes elemNodes(1)];
         
     elseif length(elemNodes) == 3 %tria3
         if ispolycw(coords(elemNodes,1),coords(elemNodes,2))  % make sure nodes are numbered couterclockwise
             elemNodes = fliplr(elemNodes);
         end
         trias(iElem-nQuads,:) = elemNodes;
         
         nodeZone = [elemNodes elemNodes(1)];
         
     else
         fprintf("Unknown element type detected.\n");
     end
     
     % identify the part
     switch parts(iElem)
         case 1
            Col = [ 78 110 241]./255;
         case 2
         	Col = [109 184 115]./255;  
         case 3
         	Col = [218  32  36]./255;
     end
     fill(coords(nodeZone,1),coords(nodeZone,2),Col,'edgecolor','k');
     
end

% transverse boundary nodes
plot(coords(leftNodes,1),coords(leftNodes,2),'-','color','c');
plot(coords(rightNodes,1),coords(rightNodes,2),'-','color','c');

% corner nodes
plot(coords(cornerNodes,1),coords(cornerNodes,2),'o','color',[0.5 0.5 0.5],'MarkerEdgeColor','none','MarkerFaceColor','b','MarkerSize',5);
    
hold off
xlim([X0 X0+L]);
ylim([Y0 Y0+H]);
axis('equal'); 
axis off
set(gca,'xtick',[])
set(gca,'ytick',[])
box on
    
title('Mesh','FontName','Times New Roman','FontWeight','Bold','FontSize',16,'interpreter','latex')
set(gcf,'Units','centimeters','Position',[5 5 20 15]);

%% Create a mesh file

fmesh = fopen(fileMesh,'wt');


% nodes
fprintf(fmesh,'NNODES');
fprintf(fmesh,' %i\n',nNodes);

for iNode =1:nNodes
    fprintf(fmesh,'%i %e %e \n',iNode,coords(iNode,1),coords(iNode,2));
    
end
fprintf(fmesh,' \n');

% elems (attached by the part indexes)
fprintf(fmesh,'NELEMS');
fprintf(fmesh,' %i \n',nQuads+nTrias);

for iElem =1:nQuads  
    fprintf(fmesh,'%i %i %i %i %i %i \n',iElem,quads(iElem,1),quads(iElem,2),quads(iElem,3),quads(iElem,4),parts(iElem));
    
end
for iElem =1:nTrias  
    fprintf(fmesh,'%i %i %i %i %i \n',iElem+nQuads,trias(iElem,1),trias(iElem,2),trias(iElem,3),parts(iElem+nQuads));
    
end
fprintf(fmesh,' \n');

% transverse boundary nodes

fprintf(fmesh,'LEFT\n');
for i = 1:length(leftNodes)
    fprintf(fmesh,'%i ',leftNodes(i));
    
end

fprintf(fmesh,' \n');
fprintf(fmesh,' \n');

fprintf(fmesh,'RIGHT\n');
for i = 1:length(rightNodes)
    fprintf(fmesh,'%i ',rightNodes(i));
    
end

fprintf(fmesh,' \n');
fprintf(fmesh,' \n');

fprintf(fmesh,'BOTTOM\n');
for i = 1:length(bottomNodes)
    fprintf(fmesh,'%i ',bottomNodes(i));
    
end

fprintf(fmesh,' \n');
fprintf(fmesh,' \n');

fprintf(fmesh,'TOP\n');
for i = 1:length(topNodes)
    fprintf(fmesh,'%i ',topNodes(i));
    
end

fprintf(fmesh,' \n');
fprintf(fmesh,' \n');

% corner nodes
fprintf(fmesh,'CORNERS\n');
for i = 1:length(cornerNodes)
    fprintf(fmesh,'%i ',cornerNodes(i));
    
end

fprintf(fmesh,' \n');
fprintf(fmesh,' \n');


fclose(fmesh);

curdir = pwd;
if ~strcmp(curdir(end-5:end),'Meshes')
    cd Meshes\
end
save_figure(get(gcf,'Number'),fileMesh(8:(end-4)))

cd(curdir)
