%-------------------------------------------------------------------------%
% Background
% Main function for the unit cell analysis
% Created by Lei, TU/e, 2017
% Modified by Stijn and Lei, TU/e, 2022
% Further modified and finalized by X.Kuci, TU/e, 2024

% Notes
% Model: Plane stress or Plane strain (linear elasticity)
% Boundary: periodic boundary conditions
% Element type: 3-node triangle or 4-node quadriangle
% Unit system (unit thickness): N,Pa,m,kg;
% Unit thickness
% sparse matrix solving method

% Key input
% material parameters, material model
% meshfile, result filenames

%-------------------------------------------------------------------------%

clc
%% material properties

cd ..
loadMats;
cd Dispersion_analysis\
matProp = LRAM;
Kc = matProp{1}.E/3/(1-2*matProp{1}.nu);
Gc = matProp{1}.E/2/(1+matProp{1}.nu);
matProp{1}.C = planeStrain(Kc,Gc);               % tangent elasticity

Kc = matProp{2}.E/3/(1-2*matProp{2}.nu);
Gc = matProp{2}.E/2/(1+matProp{2}.nu);
matProp{2}.C = planeStrain(Kc,Gc);      


Kc = matProp{3}.E/3/(1-2*matProp{3}.nu);
Gc = matProp{3}.E/2/(1+matProp{3}.nu);
matProp{3}.C = planeStrain(Kc,Gc); 

% Kc = matProp{4}.E/3/(1-2*matProp{4}.nu);
% Gc = matProp{4}.E/2/(1+matProp{4}.nu);
% matProp{4}.C = planeStrain(Kc,Gc);   

%% geometrical size reference
% W = 3.141e-6;                       	  % width
W = 20e-3;


%% Settings

% fImg = 0.25;                          % to scale the imaginary axisclc
coordsRef = [0.5 0.5] * W;            % reference postion, default at the center

% mesh file
  
meshFile = ['Meshes/mesh_UC1_New.txt']; 

Freqlim = 6e6;                        % Hz
RANGE = '-1 to 1';                    % k range

nModes = 6;                           % mode number
nSamples = 2e2+1;                       % for Bloch analysis
Nomega = 500*nSamples;                % for homogenization model 
freqTh = 1;                         % frequency threshold

Nset = [nModes nSamples Nomega freqTh];


%% Read the mesh and constraints
[coords,elems,parts,presNodes,cornerNodes,boundaries] = readMesh(meshFile);

%  coords = coords * W;% coords * W;                   % scaling

nNodes = length(coords);               % total node number
totDof = 2*nNodes;                     % total DOFs

nElems = length(elems);                % total element number
nElemNodes = length(elems(1,:));

presDofs = getDofs(presNodes);         % prescribed DOFs       
presDof = length(presDofs);

nNode = length(coords);         % number of nodes

if UseFigures
figure
SizeM = [5 5 15 15];
set(gcf,'Units','centimeters','Position',SizeM);

hold on

matrixNodes = zeros(length(coords),1);   % mark the matrix nodes
end


% plot the mesh
% plotMesh(coords,elems,parts);

%% Approximate volume

% X-Y sizes
maxcoordsX = max(coords(:,1));
mincoordsX = min(coords(:,1));
maxcoordsY = max(coords(:,2));
mincoordsY = min(coords(:,2));
rangeX = maxcoordsX - mincoordsX;
rangeY = maxcoordsY - mincoordsY;

V = rangeX*rangeY;


%% Select integration scheme

coordsElem = coords(elems(1,:),:);     % element sample
order = 0;                             % default value for the quadrilateral element

% get element properties
elemProp = getElemProp(coordsElem,order);
totInt = elemProp.nInts*nElems;              % total integration points


%% Assemble system stiffness and mass matrice

tkm1 = clock;
[K,M] = assembleKeMe(coords,elems,parts,matProp,elemProp,nElems,order);
if ~issymmetric(K)
    K = 0.5*K+0.5*K';
end
if ~issymmetric(M)
    M = 0.5*M+0.5*M';
end

%% Independent, dependent and free nodes
t1 = clock;
[IndNodes,DepNodes,freeNodes,Cdi] = relationInd(nNodes, presNodes, boundaries);

IndDofs = getDofs(IndNodes);           % basic independent DOFs
IndDof = length(IndDofs);              % number

DepDofs = getDofs(DepNodes); 
DepDof = length(DepDofs);

freeDofs =  getDofs( freeNodes );
freeDof = length(freeDofs);


%% partition K and M as [ii,id;di,dd] 
Kii = K(IndDofs,IndDofs);
Kid = K(IndDofs,DepDofs);
Kdi = K(DepDofs,IndDofs);
Kdd = K(DepDofs,DepDofs);

Mii = M(IndDofs,IndDofs);
Mid = M(IndDofs,DepDofs);
Mdi = M(DepDofs,IndDofs);
Mdd = M(DepDofs,DepDofs);

%%
% remove dependent nodes (KEff*ui + MEff*ai = fiEff)
KEff = sparse(Kii +Kid *Cdi + Cdi'*Kdi +Cdi'*(Kdd*Cdi));
MEff = sparse(Mii +Mid *Cdi + Cdi'*Mdi +Cdi'*(Mdd*Cdi));


%% Partition KEff and MEff as[pp,pf;fp,ff]

% partition KEff
Kpp = KEff(1:presDof,1:presDof);
Kpf = KEff(1:presDof,presDof+1:end);
Kfp = KEff(presDof+1:end,1:presDof);
Kff = KEff(presDof+1:end,presDof+1:end);

% partition MEff
Mpp = MEff(1:presDof,1:presDof);
Mpf = MEff(1:presDof,presDof+1:end);
Mfp = MEff(presDof+1:end,1:presDof);
Mff = MEff(presDof+1:end,presDof+1:end);

indIndices  = 1:IndDof;
presIndices = 1:presDof;
freeIndices = setdiff(indIndices, presIndices);

T.partition = etime(clock,t1);


%% Quasistatic response (Kqs*up + Mqs*ap = fiextEffp)

% get static condensate
S = -inv(Kff) * Kfp;

% get Kqs (i.e.KM) and Mqs (i.e.MM)
Kqs = Kpp + Kpf*S;
Mqs = S'*Mff*S + Mpf*S + S'*Mfp + Mpp;


%% Internal dynamics ([udynf; up]=[fai;0]*eta)

% eigenfrequency (high to low) and eigenmodes (vectors)
[Fai,Omega2] = eigs(Kff,Mff,nModes,'sm');
dispM = sqrt(real(Fai'*Mff*Fai));                     % modal displacement

% normalize
for i = 1:length(dispM)
    Fai(:,i) = Fai(:,i)/dispM(i,i); 
end

% sort (low to high frequency)
[Omega,index] = sort(abs(sqrt(diag(Omega2)))/2/pi); % Hz
Fai = Fai(:,index);

% filter too low frequency
[~,index] = find(Omega'>=freqTh);
Omega = Omega(index,:);
Fai = Fai(:,index);

% recover the full eigen modal shapes
Fai_full = zeros(2*nNode,nModes);

for i = 1:length(dispM)
    Fai_full(IndDofs(freeIndices),i) = Fai(:,i);
    Fai_full(DepDofs,i)  = Cdi* Fai_full(IndDofs,i);
    
end
HM.Fai_full = Fai_full;


%% Computational homogenization

% [HM,jh,NormkH, FreqH] = compHom(W,V,coords,coordsRef,presNodes,Kqs,Mqs,Mff,Mpf,S,Nset,Omega,Fai,DirWave,true);
[HM,jh] = compHom(W,V,coords,coordsRef,presNodes,Kqs,Mqs,Mff,Mpf,S,Nset,Omega,Fai);

%% Dispesion analysis

alpha = 1; %cosd(67.5)*pi/W;
beta = 0; %sind(67.5)*pi/W;
DirWave = [alpha beta]/sqrt(alpha^2+beta^2);

[NormkH, FreqH,HM.DynParas,HM.CS,HM.CS_LC,HM.DS,HM.rhoS,Polarization,V_modes_all] = getDisperHomK(HM,nSamples,Nomega,nModes,DirWave,W,true);

%% Plot dispersion curve
% figure
% % plot(NormkH,FreqH,'b.')
% plot(NormkH,FreqH,'bs', 'MarkerFaceColor', 'b', 'MarkerSize', 3)
% % xticklabels({'M','\Gamma','X','M'})
% % ylabel('Frequency [Hz]','FontName','Times New Roman','FontSize',14);

% figure;
% plot(NormkH,FreqH,'b','LineWidth',2)
hold on;
for mode = 1:nModes
    scatter(NormkH(:,1:end), FreqH(mode, 1:end), 15, Polarization(mode, 1:end), 'filled');
end
% % Customize colormap
% colormap(jet);
% c = colorbar;
% c.Label.String = 'Polarization (0 = Shear, 1 = Compressional)';
% xlabel('Normalized Wavenumber');
% ylabel('Frequency (Hz)');

% Get the current jet colormap with 256 colors (default)
cmap = jet(256);

% Brighten the whole colormap by blending all colors with white
% This will make the colors brighter overall
cmap = cmap + 0.3 * ones(size(cmap));  % Blend with white (adjust 0.3 to taste)

% Ensure RGB values stay within the [0,1] range
cmap(cmap > 1) = 1;

% Apply the brightened colormap
colormap(cmap);

% Update the colorbar to reflect changes
colorbar;
box on

hold on;

% ylim([0 2300])
ylabel('Frequency [Hz]','FontSize',14,'FontName','Times New Roman')
% xticks([0 1 2 3 4])
% xticklabels({'Y','\Gamma','X','M','Y'})
% 
% xline(1)
% xline(2)
% xline(3)
c = colorbar;
c.TickLabels = {};  % This removes the numerical labels but keeps the ticks
