%-------------------------------------------------------------------------%
% Background
% Main function for the unit cell analysis
% Created by Lei, TU/e, 2017
% Modified by Stijn and Lei, TU/e, 2022 :)

% Notes
% Model: Plane stress or Plane strain (linear elasticity)
% Boundary: periodic boundary conditions
% Element type: 3-node triangle or 4-node quadriangle
% Unit system (unit thickness): N,Pa,m,kg;
% Unit thickness
% sparse matrix solving method

% Key input
% material parameters, material model
% meshfile, result filenames

%-------------------------------------------------------------------------%
% clear all; close all;% clc
%clc
function [HM,W] = funcunitcellAnalysis(matProp,e_theta)



%% material

% matrix (part 1)
Kc = matProp{1}.E/3/(1-2*matProp{1}.nu);
Gc = matProp{1}.E/2/(1+matProp{1}.nu);
matProp{1}.C = planeStrain(Kc,Gc);               % tangent elasticity

% coating (part 2)
Kc = matProp{2}.E/3/(1-2*matProp{2}.nu);
Gc = matProp{2}.E/2/(1+matProp{2}.nu);
matProp{2}.C = planeStrain(Kc,Gc);      

% core (part 3)
Kc = matProp{3}.E/3/(1-2*matProp{3}.nu);
Gc = matProp{3}.E/2/(1+matProp{3}.nu);
matProp{3}.C = planeStrain(Kc,Gc);   

% % core (part 3)
% Kc = matProp{4}.E/3/(1-2*matProp{4}.nu);
% Gc = matProp{4}.E/2/(1+matProp{4}.nu);
% matProp{4}.C = planeStrain(Kc,Gc);   

%% geometrical size reference
% W = 3.141e-6;                       	  % width
W = 20e-3;%5e-6;


%% Settings
% resonator (case D is tested)
fImg = 0.25;                          % to scale the imaginary axis
coordsRef = [0.5 0.5] * W;            % reference postion, default at the center

% mesh file
keyCell = 2;    % 1: coarse, 2: fine
% meshFile = ['Meshes/meshUnitcell' num2str(keyCell) '.txt'];  % o is old with square in the middle showing correct dispersion diagram, without o: wrong dispersion diagram   
meshFile = ['Meshes/mesh_SoftRubber.txt']; 
% meshFile = ['Meshes/mesh_HardRubberRotated.txt'];
% meshFile = ['Meshes/freeQuad',num2str(keyCell),'.txt'];

% dispersion analysis
DirWave = e_theta; %[0 1];
Freqlim = 6e6;                        % Hz
RANGE = '-1 to 1';                    % k range

nModes = 6;                           % mode number
nSamples = 1e2+1;                       % for Bloch analysis
Nomega = 500*nSamples;                % for homogenization model 
freqTh = 100;                         % frequency threshold

Nset = [nModes nSamples Nomega freqTh];


%% Read the mesh and constraints
[coords,elems,parts,presNodes,cornerNodes,boundaries] = readMesh(meshFile);

%coords = coords * W;                   % scaling

nNodes = length(coords);               % total node number
totDof = 2*nNodes;                     % total DOFs

nElems = length(elems);                % total element number
nElemNodes = length(elems(1,:));

presDofs = getDofs(presNodes);         % prescribed DOFs       
presDof = length(presDofs);

nNode = length(coords);         % number of nodes

Rin = max(max(coords(elems(parts==3),:))); % inclusion
Rex = max(max(coords(elems(parts==2),:))); % coating

matrixNodes = zeros(length(coords),1);   % mark the matrix nodes

%% Approximate volume

% X-Y sizes
maxcoordsX = max(coords(:,1));
mincoordsX = min(coords(:,1));
maxcoordsY = max(coords(:,2));
mincoordsY = min(coords(:,2));
rangeX = maxcoordsX - mincoordsX;
rangeY = maxcoordsY - mincoordsY;

V = rangeX*rangeY;

% relV = elemArea(coords,elems,parts,[1,2,3]) / V;


%% Select integration scheme

coordsElem = coords(elems(1,:),:);     % element sample
order = 0;                             % default value for the quadrilateral element

% get element properties
elemProp = getElemProp(coordsElem,order);
totInt = elemProp.nInts*nElems;              % total integration points


%% Assemble system stiffness and mass matrice

tkm1 = clock;
[K,M] = assembleKeMe(coords,elems,parts,matProp,elemProp,nElems,order);
if ~issymmetric(K)
    fprintf('\nK matrix is not symmetric, but being made symmetric!\n');
    K = 0.5*K+0.5*K';
end
if ~issymmetric(M)
    fprintf('\nM matrix is not symmetric, but being made symmetric!\n');
    M = 0.5*M+0.5*M';
end

tkm2 = clock;
T.assembleKeMe = etime(tkm2,tkm1);

%% Independent, dependent and free nodes
t1 = clock;
[IndNodes,DepNodes,freeNodes,Cdi] = relationInd(nNodes, presNodes, boundaries);

IndDofs = getDofs(IndNodes);           % basic independent DOFs
IndDof = length(IndDofs);              % number

DepDofs = getDofs(DepNodes); 
DepDof = length(DepDofs);

freeDofs =  getDofs( freeNodes );
freeDof = length(freeDofs);


%% partition K and M as [ii,id;di,dd] 
Kii = K(IndDofs,IndDofs);
Kid = K(IndDofs,DepDofs);
Kdi = K(DepDofs,IndDofs);
Kdd = K(DepDofs,DepDofs);

Mii = M(IndDofs,IndDofs);
Mid = M(IndDofs,DepDofs);
Mdi = M(DepDofs,IndDofs);
Mdd = M(DepDofs,DepDofs);

% remove dependent nodes (KEff*ui + MEff*ai = fiEff)
KEff = sparse(Kii +Kid *Cdi + Cdi'*Kdi +Cdi'*(Kdd*Cdi));
MEff = sparse(Mii +Mid *Cdi + Cdi'*Mdi +Cdi'*(Mdd*Cdi));


%% Partition KEff and MEff as[pp,pf;fp,ff]

% partition KEff
Kpp = KEff(1:presDof,1:presDof);
Kpf = KEff(1:presDof,presDof+1:end);
Kfp = KEff(presDof+1:end,1:presDof);
Kff = KEff(presDof+1:end,presDof+1:end);

% partition MEff
Mpp = MEff(1:presDof,1:presDof);
Mpf = MEff(1:presDof,presDof+1:end);
Mfp = MEff(presDof+1:end,1:presDof);
Mff = MEff(presDof+1:end,presDof+1:end);

indIndices  = 1:IndDof;
presIndices = 1:presDof;
freeIndices = setdiff(indIndices, presIndices);

T.partition = etime(clock,t1);


%% Quasistatic response (Kqs*up + Mqs*ap = fiextEffp)
t1 = clock;
% get static condensate
S = -inv(Kff) * Kfp;

% get Kqs (i.e.KM) and Mqs (i.e.MM)
Kqs = Kpp + Kpf*S;
Mqs = S'*Mff*S + Mpf*S + S'*Mfp + Mpp;


%% Internal dynamics ([udynf; up]=[fai;0]*eta)

% eigenfrequency (high to low) and eigenmodes (vectors)
[Fai,Omega2] = eigs(Kff,Mff,nModes,'sm');
dispM = sqrt(real(Fai'*Mff*Fai));                     % modal displacement

% normalize
for i = 1:length(dispM)
    Fai(:,i) = Fai(:,i)/dispM(i,i); 
end

% sort (low to high frequency)
[Omega,index] = sort(real(sqrt(diag(Omega2)))/2/pi); % Hz
Fai = Fai(:,index);

% filter too low frequency
[~,index] = find(Omega'>=freqTh);
Omega = Omega(index,:);
Fai = Fai(:,index);

% recover the full eigen modal shapes
Fai_full = zeros(2*nNode,nModes);

for i = 1:length(dispM)
    Fai_full(IndDofs(freeIndices),i) = Fai(:,i);
    Fai_full(DepDofs,i)  = Cdi* Fai_full(IndDofs,i);
    
end
HM.Fai_full = Fai_full;


T.QS = etime(clock,t1);

%% Release some memory
%clear Kpp Kpf Kfp Kff Mpp Mfp Kii Kid Kdi Kdd Mii Mid Mdi Mdd



%% Computational homogenization

t1 = clock;
[HM,jh,NormkH, FreqH] = compHom(W,V,coords,coordsRef,presNodes,Kqs,Mqs,Mff,Mpf,S,Nset,Omega,Fai,DirWave,true);

T.HM = etime(clock,t1); % time for HM analysis

%% Mode selection criteria
fex = 0.4417864669110645; % volume fraction of total inclusion (i.e. Rex), fex = pi*R^2 / L^2
fin = 2/3; % = Rin/Rex (dimensionless)
Rex = sqrt(fex*W^2/pi);
Rin = fin*Rex;
M_all = matProp{3}.rho*pi*Rin^2 + matProp{2}.rho*pi*(Rex^2 - Rin^2) + matProp{1}.rho*(W*W-pi*Rex^2);
mu_inc = (matProp{3}.rho*pi*Rin^2 + matProp{2}.rho*pi*(Rex^2 - Rin^2)) / M_all;
mu_Jd = zeros(nModes,1); 

for p = 1:2
    ep = zeros(1,2);
    ep(p)=1;
    for s = 1:nModes
        mu_Jd(s) = ep*HM.Jd{s}*ep'; 
    end

    mu_Jd = mu_Jd./HM.rhoM(1,1);
    msc = sum(mu_Jd) / mu_inc; % mode selection criteria, msc>.95 is ok

    fprintf('\nMode selection criterion is ')
    if msc>.95 && msc<1.05
        fprintf('passed ')
    else
        fprintf('not fullfilled ')
    end
    fprintf('for p = %i.\nsum(mu_j) / mu_inc = %.2f.\n',p,msc)
end




end %function