%-------------------------------------------------------------------------%
% Background
% Dispersion analyis under the Bloch boundary condition
% Created by Lei, TU/e, 2017
%-------------------------------------------------------------------------%

function [Normk, Freq, U] = blochAnalysis(K,M,cornerNodes,boundaryNodes,W,nNodes,DirWave,Nset)

% Initialize
nModes = Nset(1);                                 % total number of selected modes
nSamples = Nset(2);                                % sampling number
freqTh = Nset(4);                                  % to filter low frequency

kmax = pi / W;
k = linspace(kmax,2*kmax,3);                     % wave number
Normk = k / kmax;

kX = DirWave(1) * k;
kY = DirWave(2) * k;

nNodesX = length(boundaryNodes{1});                % number of X-boundary nodes
nNodesY = length(boundaryNodes{3});                % number of Y-boundary nodes

% Alloctae the nodes
PresNodes = [cornerNodes(1) boundaryNodes{1} boundaryNodes{3}]'; % prescibed nodes----------------modified by lei, 2022
DepNodes = [cornerNodes([2 3 4]) boundaryNodes{2} boundaryNodes{4}]';  % dependent nodes
ReNodes = setdiff([1:nNodes]',[PresNodes;DepNodes]);                % independent nodes

IndNodes = [PresNodes; ReNodes];
IndDofs = getDofs(IndNodes);
IndDof = length(IndDofs);

DepDofs = getDofs(DepNodes);
DepDof = length(DepDofs);

totDof = IndDof+DepDof;

% Get the frequency
U = cell(1,nSamples);
Freq = zeros(nModes,nSamples);

Cm = zeros(IndDof,IndDof);
j = 1;
for i = 1:IndDof
    Cm(i,j) = 1;
    j = j+1;
    
end

% partition K and M as [ii,id;di,dd]
Kii = K(IndDofs,IndDofs);
Kid = K(IndDofs,DepDofs);
Kdi = K(DepDofs,IndDofs);
Kdd = K(DepDofs,DepDofs);

Mii = M(IndDofs,IndDofs);
Mid = M(IndDofs,DepDofs);
Mdi = M(DepDofs,IndDofs);
Mdd = M(DepDofs,DepDofs);

% 
% emptypool = isempty(gcp('nocreate'));
% if emptypool == 1
%   % no pool is found --> start a new one
%   parpool(maxNumCompThreads);
% end

fprintf('Progress:\n');
fprintf(['\n' repmat('.',1,nSamples) '\n\n']);
warning ('off','all');
for i = 1:nSamples    
    % form the dependency matrix Cdi
    kx = kX(i);
    ky = kY(i);
    Cdi = funCdi(IndDofs,DepDofs,nNodesX,nNodesY,kx,ky,W);

    % remove dependent nodes (KEff*ui + MEff*ai = fiEff)
    KEff = Kii +Kid *Cdi + Cdi'*Kdi +Cdi'*(Kdd*Cdi);
    MEff = Mii +Mid *Cdi + Cdi'*Mdi +Cdi'*(Mdd*Cdi);

    KFF = Cm' * KEff * Cm;                           % contrain only free DoFs
    MFF = Cm' * MEff * Cm;

    % eigenvalue analysis
    [freeU, Omega2] = eigs(KFF,MFF,nModes,'sm');      % results at free DOFs

    IndU = Cm * freeU;
    DepU = Cdi * IndU;
    
    % collect frequency
    [Freq(:,i), idFreq]= sort(abs(sqrt(diag(Omega2)))/2/pi);
    
    % collect results at all nodes
    totU = zeros(totDof,nModes);
    totU(IndDofs,:) = IndU(:,idFreq);
    totU(DepDofs,:) = DepU(:,idFreq);
    
    U{i} = totU; 

    % print progress
    fprintf('\b|\n');
            
end

% % closing paralell pool
% delete(gcp('nocreate'))

% filter too low frequencies
freq =  Freq(:,end);
[~,index] = find(freq'>=freqTh);
Freq = Freq(index,:);

for i = 1:nSamples 
    U{i} = U{i}(:,index);
end

end


function Cdi = funCdi(IndDofs,DepDofs,nNodesX,nNodesY,kx,ky,L)

% Ind: LB left bottom others, Dep: RB, RT, LT right top
IndDof = length(IndDofs);
DepDof = length(DepDofs);

% Initialize
Cdx = exp(1i*kx*L)*[ 1 0; 0 1];  
Cdy = exp(1i*ky*L)*[ 1 0; 0 1];  
Cdxy = exp(1i*(kx+ky)*L) * [1 0; 0 1];
Cdi = zeros(DepDof,IndDof);

% For corner nodes RB, RT, LT
Cdi(1:6,1:2) = [Cdx;Cdxy;Cdy];

% For right nodes
for iright = 1:nNodesX
    ncurrent = 3+iright;
    Cdi(2*ncurrent-1:2*ncurrent,2*(ncurrent-2)-1:2*(ncurrent-2)) = Cdx;

end

% For the top nodes
for itop = 1:nNodesY
    ncurrent = 3+itop+nNodesX;
    Cdi(2*ncurrent-1:2*ncurrent,2*(ncurrent-2)-1:2*(ncurrent-2)) = Cdy;
    
end

end