function [Nux_inf,Nux,Nux_av,fit_alpha,fit_beta,fit_delta,x,beta_local,x_anal,beta_anal,lambdaT,lambdaNu,xGz12,xGz13] = GRAETZpde(b,fig_on)
% SOLVING GREATZ EQUATION USING PDE SOLVER
% Syntax for this function: sol = pdepe(m,pdefun,icfun,bcfun,rmesh,xspan)
% [Nux_inf,Nux,Nux_av,fit_alpha,fit_beta,x] = GREATZPDE(b,fig_on)
% b = dimensionless slip length b/R
% Figures on or not: fig_on = 0 means NO, fig_on = 1 means YES

% clear all
close all

% figures on or not: 0 = NO, 1 = YES, defined by assigning value to fig_on
figure_on = fig_on;

% Define system parameters
m = 1; % cylindrical system, therefore m = 1
r_in = linspace(0,1,101);
r = 1 - (0.99.*(1-r_in).^2 + 0.01.*(1-r_in));   % Mesh refinement near r* = 1
x = [0 logspace(-7,0,71)];

% Alternative meshing
% x_in = linspace(0,1,101);
% x = 0.99999.*x_in.^4 + 0.00001.*x_in.^1;           % Mesh refinement near x* = 0

% Define physical parameters, IF Greatz equals 1, x* can be replaced by x*/Gz
% HERE, ALL LABELS ARE DEFINED ASSUMING Gz = 1!
% b = 0;
Gz = 1;      

%% Set filename properties

figurefile = ['GRAETZpde' '\' 'GRAETZpde_fig'];
nn = 0;

%% ------------------------------------------------------------------------
% Solving the PDE by calling nested functions, yields T*(x*,r*)
options = odeset('RelTol',1e-6,'AbsTol',1e-12);
sol = pdepe(m,@pdeGreatz,@icGreatz,@bdGreatz,r,x,options);
T = sol(:,:,1);                 % in this case, T = sol i.e. T - sol = 0
assignin('base','sol',sol);     % force sol to be visible in workspace
assignin('base','T',T);         % force u to be visible in workspace

% Plotting figure: surface plot of T
if figure_on == 1
    nn = nn + 1;
    figure(nn)
    surf(r,x,T);
    axis([0 1 0 1 0 1]);
    %title('Extended Greatz problem: $\tilde{T}$-profile where $\tilde{T}(0,r^*)=1$, $\tilde{T}(\tilde{x},1)=0$ and $\frac{\partial \tilde{T}}{\partial r^*}(\tilde{x},0)=0$','Interpreter','LaTex');
    xlabel('$\tilde{r}$','Interpreter','LaTex');
    ylabel('$\tilde{x}/\textrm{Gz}$','Interpreter','LaTex');
    zlabel('$\Theta$ [-]','Interpreter','LaTex');
    set(gcf,'Color','w');
    set(gca,'LineWidth',0.75,'GridLineStyle','--');
    alpha 0.6; % not supported by pdf output
    box off;
    grid on;
    view(135, 15);
    %CB = cbrewer('div','RdYlBu',length(u)); % Incompatibility with MATLAB?
    %colormap(CB);
    
    tikzfile = [figurefile num2str(nn) '.tikz'];
    matlab2tikz(tikzfile, 'height', '\figureheight', 'width', '\figurewidth');
    figfile = [figurefile num2str(nn) '.fig'];
    saveas(gcf, figfile, 'fig');
    
%     % Export figure(1)
%     saveas(gcf, 'GREATZPDE_fig1.fig', 'fig');
%     %print(gcf,'-dpdf','-r300','GREATZPDE_fig1');
%     matlab2tikz('GREATZPDE_fig1.tikz', 'height', '\figureheight', 'width', '\figurewidth');
else
end


%% ------------------------------------------------------------------------
% Calculation of dT*/dx* at r*=1 
nx = length(x);
domx = 2:nx;
for j = domx
  % At position x(j), where j>1, compute dT/dr at r* = 1.
  [T2(j),gradT(j)] = pdeval(m,r,T(j,:),1);
end

% force variables to be visible in workspace
assignin('base','x',x);
assignin('base','T2',T2);           % T2(j) is a dummy variable, should be 0 (defined BC)
assignin('base','gradT',gradT);    

if figure_on == 1
    % Plotting figure: gradT at r* = 1 versus x*/Gz
    nn = nn + 1;
    figure(nn)
    plot(x(domx),-gradT(domx),'-ok');
    set(gca,'XScale','log','YScale','linear');
    set(gcf,'Color','w');
    box on
    xlabel('$\tilde{x}/\textrm{Gz}$','Interpreter','LaTex');
    ylabel('$-\frac{\partial \Theta}{\partial \tilde{x}}$','Interpreter','LaTex');
    legend('$-\frac{\partial \Theta}{\partial \tilde{x}}$ at $\tilde{r}=1$');
    legend('boxoff');
    
    tikzfile = [figurefile num2str(nn) '.tikz'];
    matlab2tikz(tikzfile, 'height', '\figureheight', 'width', '\figurewidth');
    figfile = [figurefile num2str(nn) '.fig'];
    saveas(gcf, figfile, 'fig');
    
%     % Export figure(2)
%     saveas(gcf, 'GREATZPDE_fig2.fig', 'fig');
%     %print(gcf,'-dpdf','-r300','GREATZPDE_fig2');
%     matlab2tikz('GREATZPDE_fig2.tikz', 'height', '\figureheight', 'width', '\figurewidth');
else
end

%% ------------------------------------------------------------------------
% Calculation of dimensionless mixing cup temperature

% Calculate u*(r*) and T*u*(x*,r*)
nr = length(r);
domr = 1:nr;
for k = domr
    u(k) = (2.*(1-(r(k)).^2) + 4.*b) ./ (1 + 4.*b);
    for j = 1:length(x)
        Tu(j,k) = u(k).*T(j,k);
    end
end

% Calculate average temperature Tav(x*) and mixing cup temperature Tmc(x*)
for j = 1:length(x)
    Tav(j) = trapz(r,T(j,:).*r) ./ trapz(r,r);
    Tmc(j) = trapz(r,Tu(j,:).*r) ./ trapz(r,u.*r);
end

% force variables to be visible in workspace
assignin('base','r',r);
assignin('base','u',u);
assignin('base','Tu',Tu);
assignin('base','Tav',Tav);
assignin('base','Tmc',Tmc);

if figure_on == 1
    % Plotting figure: surface plot of T*u
    nn = nn + 1;
    figure(nn)
    surf(r,x,Tu);
    axis([0 1 0 1 0 ceil(max(max(Tu)))]);
    xlabel('$\tilde{r}$','Interpreter','LaTex');
    ylabel('$\tilde{x}/\textrm{Gz}$','Interpreter','LaTex');
    zlabel('$\Theta \tilde{u}$','Interpreter','LaTex');
    set(gcf,'Color','w');
    set(gca,'LineWidth',0.75,'GridLineStyle','--');
    alpha 0.6; % not supported by pdf output
    box off;
    grid on;
    view(135, 15);
    
    tikzfile = [figurefile num2str(nn) '.tikz'];
    matlab2tikz(tikzfile, 'height', '\figureheight', 'width', '\figurewidth');
    figfile = [figurefile num2str(nn) '.fig'];
    saveas(gcf, figfile, 'fig');
    
%     % Export figure(3)
%     saveas(gcf, 'GREATZPDE_fig3.fig', 'fig');
%     %print(gcf,'-dpdf','-r300','GREATZPDE_fig3');
%     matlab2tikz('GREATZPDE_fig3.tikz', 'height', '\figureheight', 'width', '\figurewidth');
    
    % Plotting figure: Tav and Tmc versus \tilde{x}/Gz
    nn = nn + 1;
    figure(nn)
    hold on
    box on
    set(gcf,'Color','w');
    plot(x(domx),Tav(domx),'-sr');
    %plot(x(domx),Tav2(domx),'-*b');
    plot(x(domx),Tmc(domx),'-ok');
    hold off
    xlabel('$\tilde{x}/\textrm{Gz}$','Interpreter','LaTex');
    ylabel('$\Theta_{av}$ and $\bar{\Theta}$','Interpreter','LaTex');
    legend('Average temp. $\Theta_{av}$','Mixing cup temp. $\bar{\Theta}$');
    legend('boxoff');

    tikzfile = [figurefile num2str(nn) '.tikz'];
    matlab2tikz(tikzfile, 'height', '\figureheight', 'width', '\figurewidth');
    figfile = [figurefile num2str(nn) '.fig'];
    saveas(gcf, figfile, 'fig');
    
%     % Export figure(4)
%     saveas(gcf, 'GREATZPDE_fig4.fig', 'fig');
%     %print(gcf,'-dpdf','-r300','GREATZPDE_fig4');
%     matlab2tikz('GREATZPDE_fig4.tikz', 'height', '\figureheight', 'width', '\figurewidth');
else
end

%% -----------------------------------------------------------------------
% Calculation of Nusselt numbers

% Calculate Nux(r*) and Nuav(x*)
for p = 1:length(x)
    Nux(p) = -2./Tmc(p).*gradT(p);
end

% Calculation of Nusselt number (inf.) for x*/Gz > 0.1
x_calcinf = x(find(x>=0.1));
Nux_calcinf = Nux(find(x>=0.1));
Nux_inf = mean(Nux_calcinf);
%Nux_inf_old = Nux(length(Nux));    % uses only last value of Nux_inf

% force variables to be visible in workspace
assignin('base','Nux',Nux);
assignin('base','Nux_inf',Nux_inf);
assignin('base','x_calcinf',x_calcinf);
assignin('base','Nux_calcinf',Nux_calcinf);

% Calculation of average Nusselt number
% You can only calculate from q >= 3, as trapz needs to numbers to
% calculate average Nux, and because Nux(1) = 0
% Simpson's rule should be more accurate, but currently I don't understand
% why for both functions the first point is so 'incorrect'
domx2 = 3:nx;
for q = domx2
    Nux_av(q) = trapz(x(2:q),Nux(2:q)) ./ trapz(x(2:q),ones(1,q-1));
    Nux_av2(q) = simps(x(2:q),Nux(2:q)) ./ simps(x(2:q),ones(1,q-1));
end

% Calculation of slope for thermally developing regime where 1e-8 <= x*/Gz <= 1e-5
% x_reg = x(find((x>=1e-8) & (x<=1e-5) & (x ~= 0)));
% Nux_reg = Nux(find((x>=1e-8) & (x<=1e-5) & (x ~= 0)));
x_reg = (x(find((x>=Gz*1e-7) & (x<=Gz*1e-4) & (x ~= 0))));
Nux_reg = Nux(find((x>=Gz*1e-7) & (x<=Gz*1e-4) & (x ~= 0)));
fit_reg = polyfit(log10(x_reg),log10(Nux_reg),1);
fit_beta = fit_reg(1);
fit_alpha = fit_reg(2);
line_reg = 10^(fit_alpha) .* x_reg.^(fit_beta);
Leveque = 10^(fit_alpha) .* x.^(fit_beta);
LevequeTheoretical = 1/sqrt(pi) .* x.^(0.5);

% force variables to be visible in workspace
assignin('base','Nux_av',Nux_av);
assignin('base','Nux_av2',Nux_av2);
assignin('base','x_reg',x_reg);
assignin('base','Nux_reg',Nux_reg);
assignin('base','line_reg',line_reg);

if figure_on == 1
    % Plotting figure: Nux versus \tilde{x}/Gz with values used for fitting
    % indicated
    nn = nn + 1;
    figure(nn)
    hold all
    box on
    set(gcf,'Color','w');
    plot(x(domx),Nux(domx),'-ok');
    plot(x_reg,line_reg,':+r');
    plot(x(domx),Leveque(domx),'-.db');
    plot(x(domx),Leveque(domx),'-.pg');
    set(gca,'XScale','log','YScale','log');
    hold off
    xlabel('$\tilde{x}/\textrm{Gz}$','Interpreter','LaTex');
    ylabel('$\textrm{Nu}_x$','Interpreter','LaTex');
    legend('$\textrm{Nu}_x$');
    legend('boxoff');
    text(5e-8,3.5,['$\textrm{Nu}_x (\infty) = ',num2str(Nux_inf),'$']);
    text(5e-8,2,['$\beta = ',num2str(-fit_beta),'$']);
    
    tikzfile = [figurefile num2str(nn) '.tikz'];
    matlab2tikz(tikzfile, 'height', '\figureheight', 'width', '\figurewidth');
    figfile = [figurefile num2str(nn) '.fig'];
    saveas(gcf, figfile, 'fig');
    
%     % Export figure(5)
%     saveas(gcf, 'GREATZPDE_fig5.fig', 'fig');
%     %print(gcf,'-dpdf','-r300','GREATZPDE_fig5');
%     matlab2tikz('GREATZPDE_fig5.tikz', 'height', '\figureheight', 'width', '\figurewidth');
    
    % Plotting figure: Nux and fit of Nux versus \tilde{x}/Gz
    nn = nn + 1;
    figure(nn)
    hold on
    box on
    set(gcf,'Color','w');
    plot(x_reg,Nux_reg,'-ok');
    plot(x_reg,line_reg,':+r');
    set(gca,'XScale','log','YScale','log');
    hold off
    xlabel('$\tilde{x}/\textrm{Gz}$','Interpreter','LaTex');
    ylabel('$\textrm{Nu}_x$','Interpreter','LaTex');
    legend('$\textrm{Nu}_x$','$\textrm{Nu}_x = 10^{\alpha} b^{\beta}$');
    legend('boxoff');
    text(1.2e-7,15,['$\beta $= $',num2str(-fit_beta)]);
    text(1.2e-7,12,['$\alpha $= $',num2str(fit_alpha)]);
    
    tikzfile = [figurefile num2str(nn) '.tikz'];
    matlab2tikz(tikzfile, 'height', '\figureheight', 'width', '\figurewidth');
    figfile = [figurefile num2str(nn) '.fig'];
    saveas(gcf, figfile, 'fig');
    
%     % Export figure(6)
%     saveas(gcf, 'GREATZPDE_fig6.fig', 'fig');
%     %print(gcf,'-dpdf','-r300','GREATZPDE_fig6');
%     matlab2tikz('GREATZPDE_fig6.tikz', 'height', '\figureheight', 'width', '\figurewidth');
    
    % Plotting figure: Nux and Nux_av versus \tilde{x}/Gz
    nn = nn + 1;
    figure(nn)
    hold all
    box on
    set(gcf,'Color','w');
    plot(x(domx),Nux(domx),'-ok');
    plot(x(domx2),Nux_av(domx2),'-sr');
    plot(x(domx2),Nux_av2(domx2),'-^b');
    set(gca,'XScale','log','YScale','log');
    hold off
    xlabel('$\tilde{x}/\textrm{Gz}$','Interpreter','LaTex');
    ylabel('$\textrm{Nu}_x$','Interpreter','LaTex');
    legend('$\textrm{Nu}_x$','$Nu_{x,av}$ (TRAPZ)','$Nu_{x,av}$ (SIMPS)');
    legend('boxoff');
    
    tikzfile = [figurefile num2str(nn) '.tikz'];
    matlab2tikz(tikzfile, 'height', '\figureheight', 'width', '\figurewidth');
    figfile = [figurefile num2str(nn) '.fig'];
    saveas(gcf, figfile, 'fig');
    
%     % Export figure(7)
%     saveas(gcf, 'GREATZPDE_fig7.fig', 'fig');
%     %print(gcf,'-dpdf','-r300','GREATZPDE_fig7');
%     matlab2tikz('GREATZPDE_fig7.tikz', 'height', '\figureheight', 'width', '\figurewidth','parseStrings',false);
else
end

% if figure_on == 1
%     % Plotting figure: log(Nux) versus log(\tilde{x}/Gz) to see how the linear
%     % fit of log(Nux) = \alpha - \beta log(x*/\textrm{Gz}) 'works', and why the
%     % cutoff is located around \alpha = 0
%     nn = nn + 1;
%     figure(nn)
%     hold all
%     box on
%     set(gcf,'Color','w');
%     plot(log10(x(domx)),log10(Nux(domx)),'-ok');
%     ylim([-.5 2.5]);
%     xlim([-8 1]);
%     set(gca,'XScale','linear','YScale','linear');
%     hold off
%     xlabel('$\log{\tilde{x}/\textrm{Gz}}$','Interpreter','LaTex');
%     ylabel('$\log{\textrm{Nu}_x}$','Interpreter','LaTex');
%     legend('$\log{\textrm{Nu}_x}$');
%     legend('boxoff');
% else
% end

%% Calculation of slope for dT/dr for thermally developing regime where 1e-7 <= x*/Gz <= 1e-4
x_reg2 = (x(find((x>=Gz*1e-7) & (x<=Gz*1e-4) & (x ~= 0))));
gradT_reg = -gradT(find((x>=Gz*1e-7) & (x<=Gz*1e-4) & (x ~= 0)));
fit_reg = polyfit(log10(x_reg2),log10(gradT_reg),1);
fit_delta = fit_reg(1);
fit_gamma = fit_reg(2);
linedTdr_reg = 10^(fit_gamma) .* x_reg2.^(fit_delta);

% Calculation of local beta-value
logx = log10(x(2:end));
logNux = log10(Nux(2:end));
beta_local = -gradient(logNux,logx);

% force variables to be visible in workspace
assignin('base','x_reg2',x_reg2);
assignin('base','gradT_reg',gradT_reg);
assignin('base','linedTdr_reg',linedTdr_reg);

% Calculation of analytical beta-value, which is based on the Leveque
% approximation. This is only valid up up to approximately x/Gz < 1e-3,
% therefore beta_a is only computed and plotted for 0 < x/Gz < 1e-3
x_anal = x(2:find(x==1e-3));
for k = 1:length(x_anal)
    %Z(k) = 2 .* (x_anal(k)/Gz) .* ((1+4.*b)./(4.*b^3));
    X(k) = (x_anal(k)/Gz) .* ((1+4.*b)./(b^3));
    G(k) = 0.546 ./ sqrt(X(k)) .* (X(k).^(1/2.420) + 1.058).^(2.420/6);
    
    Xint = X(k);
    funA1 = @(k,Xint) (airy(1,(k.*exp(-1i.*pi./3))).*exp(-k.^3*Xint))./(airy(k.*exp(-1i.*pi./3)));
    funB1 = @(k,Xint) (airy(1,(k.*exp(1i.*pi./3))).*exp(-k.^3*Xint))./(airy(k.*exp(1i.*pi./3)));
    qA1 = integral(@(k)funA1(k,Xint),Inf,0,'AbsTol',1e-12,'RelTol',1e-9);
    qB1 = integral(@(k)funB1(k,Xint),0,Inf,'AbsTol',1e-12,'RelTol',1e-9);
    gA1 = (-3*exp(-1i*pi/3))/(2*pi*1i) .* qA1;
    gB1 = (-3*exp(1i*pi/3))/(2*pi*1i) .* qB1;
    g(k) = gA1 + gB1;
end

logx_anal = log10(x_anal);
logG = log10(G);
logg = log10(g);
beta_anal_G = -gradient(logG,logx_anal);
beta_anal = -gradient(logg,logx_anal);

assignin('base','beta_anal_G',beta_anal_G);
assignin('base','X',X);
assignin('base','g',g);
assignin('base','G',G);

if figure_on == 1
    % Plotting figure: gradT versus \tilde{x}/Gz with values used for fitting
    % indicated
    nn = nn + 1;
    figure(nn)
    hold all
    box on
    set(gcf,'Color','w');
    plot(x(domx),-gradT(domx),'-ok');
    plot(x_reg2,linedTdr_reg,':+r');
    set(gca,'XScale','log','YScale','log');
    hold off
    xlabel('$\tilde{x}/\textrm{Gz}$','Interpreter','LaTex');
    ylabel('$-\frac{\partial \tilde{T}}{\partial \tilde{x}}$','Interpreter','LaTex');
    legend('$-\frac{\partial \tilde{T}}{\partial \tilde{x}}$');
    legend('boxoff');
    text(5e-8,2,['$\delta = ',num2str(-fit_delta),'$']);
    
    tikzfile = [figurefile num2str(nn) '.tikz'];
    matlab2tikz(tikzfile, 'height', '\figureheight', 'width', '\figurewidth');
    figfile = [figurefile num2str(nn) '.fig'];
    saveas(gcf, figfile, 'fig');
    
    nn = nn + 1;
    figure(nn)
    hold all
    box on
    set(gcf,'Color','w');
    plot(x(2:end),beta_local,'-ok');
    plot(x_anal,beta_anal,'--r');
    plot(x_anal,beta_anal_G,'--g');
    set(gca,'XScale','log','YScale','linear');
    hold off
    xlim([1e-7 1]);
    xlabel('$\tilde{x}/\textrm{Gz}$','Interpreter','LaTex');
    ylabel('$\beta_{l}$','Interpreter','LaTex');
    legend('$\beta_{l}$');
    legend('boxoff');
    
    tikzfile = [figurefile num2str(nn) '.tikz'];
    matlab2tikz(tikzfile, 'height', '\figureheight', 'width', '\figurewidth');
    figfile = [figurefile num2str(nn) '.fig'];
    saveas(gcf, figfile, 'fig');
    
else
end

%% Calculation of thermal and viscous boundary layer

% determine boundary layer thickness
delta = 0.99;
Tdiff = T - delta;

% length of vectors
nx = length(x);
lambdaT = ones(1,nx);
lambdaT(1) = 0;

% determine maximum x*-value for which T at r* = 1 > delta (otherwise BL
% thickness equals 1), as defined above
nx_max = max(find(T(:,1)>delta));

for ii=2:nx_max
    index = max(find(Tdiff(ii,:)>=0)); % find r* for which Tdiff >= 0
    % T(i,:) = aa*r + bb -- approximation for calculation r* = 1 - lambdaT,
    % where T = deltaT0
    aa = (T(ii,index+1) - T(ii,index)) ./ (r(index+1) - r(index)); 
    lambdaT(ii) = 1 - ((delta - T(ii,index)) ./ aa + r(index));
end

% calculation of viscous boundary layer
if b <= 0.5 * ( 1./(1-delta) - 1)
    lambdaNu = 1 - sqrt((1-delta).*(1+2*b));
else
    lambdaNu = 0;
end

% calculation of point where u_var = C_c*u_s and u = u_s + u_var
C_c_up = 10;
if b <= 1/(2*C_c_up)
    lambdaC_up = (1 - sqrt(1 - 2.*C_c_up.*b)) .* ones(1,length(x));
else
    lambdaC_up = 1 .* ones(1,length(x));
end
C_c_low = 0.1;
if b <= 1/(2*C_c_low)
    lambdaC_low = (1 - sqrt(1 - 2.*C_c_low.*b)) .* ones(1,length(x));
else
    lambdaC_low = 1 .* ones(1,length(x));
end
C_c_med = 1;
if b <= 1/(2*C_c_med)
    lambdaC_med = (1 - sqrt(1 - 2.*C_c_med.*b)) .* ones(1,length(x));
else
    lambdaC_med = 1 .* ones(1,length(x));
end

% determine point where lambdaC_up == lambdaT
if lambdaC_up(1) >= 1
    xGz13 = 1;
elseif lambdaT(2) >= lambdaC_up(1)
    xGz13 = 0;
else
    lambdadiff2 = lambdaT - lambdaC_up(1);
    index2 = max(find(lambdadiff2<=0)); % find r* for which Tdiff >= 0
    % T(i,:) = aa*r + bb -- approximation for calculation r* = 1 - lambdaT,
    % where T = deltaT0
    cc = (lambdaT(index2+1) - lambdaT(index2)) ./ (x(index2+1) - x(index2)); 
    xGz13 = ((lambdaC_up(1) - lambdaT(index2)) ./ cc + x(index2)); 
end

% determine point where lambdaC_low == lambdaT
if lambdaC_low(1) >= 1
    xGz12 = 1;
elseif lambdaT(2) >= lambdaC_low(1)
    xGz12 = 0;
else
    lambdadiff3 = lambdaT - lambdaC_low(1);
    index3 = max(find(lambdadiff3<=0)); % find r* for which Tdiff >= 0
    % T(i,:) = aa*r + bb -- approximation for calculation r* = 1 - lambdaT,
    % where T = deltaT0
    ee = (lambdaT(index3+1) - lambdaT(index3)) ./ (x(index3+1) - x(index3)); 
    xGz12 = ((lambdaC_low(1) - lambdaT(index3)) ./ ee + x(index3)); 
end

% figures
if figure_on == 1
    x_max = 0.01;
    n_max = find(x==x_max);
    nn = nn + 1;
    figure(nn)
    %     \definecolor{mycolor2}{rgb}{0.92157,0.25098,0.23137}%
    %     \definecolor{mycolor6}{rgb}{0.15294,0.26275,0.53725}%
    hold all
    box on
    set(gcf,'Color','w');
    plot(x(1:n_max),lambdaT(1:n_max),'-o','Color',[0.92157,0.25098,0.23137]);
    plot(x(1:n_max),lambdaC_med(1:n_max),'-','Color',[0.15294,0.26275,0.53725]);
    plot(x(1:n_max),lambdaC_up(1:n_max),'--','Color',[0.15294,0.26275,0.53725]);
    plot(x(1:n_max),lambdaC_low(1:n_max),'--','Color',[0.15294,0.26275,0.53725]);
    %plot(x,lambdaNu,':+r');
    %shadedErrorBar(x,lambdaC_med,lambdaC_range,'-r',1);
    bottom = [x(2) x(2) x(n_max) x(n_max)];
    top = [lambdaC_low(1) lambdaC_up(1) lambdaC_up(1) lambdaC_low(1)];
    fill(bottom, top,[0.15294,0.26275,0.53725], 'FaceAlpha', 0.1);
    set(gca,'XScale','log','YScale','log','XLim',[1e-7,x_max],'YLim',[1e-4,1]);
    hold off
    xlabel('$\tilde{x}/\textrm{Gz}$','Interpreter','LaTex');
    ylabel('$\lambda_{T}$ \& $\delta_{\nu}$','Interpreter','LaTex');
    legend(['$\lambda_{T}$ for $\tilde{b} = 10^{',num2str(log10(b)),'}$'],'$\delta_{\nu}$','Location','NorthWest');
    legend('boxoff');
    text(1.4e-2,lambdaC_med(1),'$\tilde{u}_v = \tilde{u}_s$');
    text(1.4e-2,lambdaC_up(1),'$\tilde{u}_v = 10\tilde{u}_s$');
    text(1.4e-2,lambdaC_low(1),'$\tilde{u}_v = 0.1\tilde{u}_s$');
   
    tikzfile = [figurefile num2str(nn) '.tikz'];
    matlab2tikz(tikzfile, 'height', '\figureheight', 'width', '\figurewidth');
    figfile = [figurefile num2str(nn) '.fig'];
    saveas(gcf, figfile, 'fig');
    
    nn = nn + 1;
    figure(nn)
    contourf(x(2:length(x)),r,((T(2:length(x),:)')-1).*-1,9);
    xlabel('$\tilde{x}/\textrm{Gz}$','Interpreter','LaTex');
    ylabel('$\tilde{r}$','Interpreter','LaTex');
    set(gca,'YScale','linear','YDir','reverse','XScale','log');
    set(gcf,'Color','w');
    set(gca,'LineWidth',0.75,'GridLineStyle','--');
    box off;
    
    tikzfile = [figurefile num2str(nn) '.tikz'];
    matlab2tikz(tikzfile, 'height', '\figureheight', 'width', '\figurewidth');
    figfile = [figurefile num2str(nn) '.fig'];
    saveas(gcf, figfile, 'fig');
else
end
    


% %% Calculation of derivatives
% 
% [dTdx,dTdr] = gradient(T(2:length(x),:),r,x(2:length(x)));
% 
% % Calculation of dT/dx at each grid point
% nx = length(x);
% domx = 2:nx;
% nr = length(r);
% domr = 1:nr;
% m = 1;
% % T(x,r) (first rows corresponding to x-direction, then columns
% % corresponding to r-direction)
% for k = domx
%     for l = domr
%         % At position x(k,l), where l>1, compute dT/dr at r* = 1.
%         [T3(k,l),gradT3(k,l)] = pdeval(m,x,T(:,l),x(k));
%         parabp(k,l) = Gz.*(1-r(l))./(1+4.*b).*gradT3(k,l);
%         plugfp(k,l) = Gz.*b./(1+4.*b).*gradT3(k,l);
%         delta(k,l) = parabp(k,l) - plugfp(k,l);
%     end
% end
% 
% assignin('base','dTdx',dTdx);
% assignin('base','dTdr',dTdr);
% assignin('base','T3',T3);
% assignin('base','gradT3',gradT3);
% assignin('base','parabp',parabp);
% assignin('base','plugfp',plugfp);
% assignin('base','delta',delta);
% 
% if figure_on == 1
% %     nn = nn + 1;
% %     figure(nn)
% %     hold on
% %     surf(r,x,parabp);
% %     surf(r,x,plugfp);
% %     set(gca,'XScale','log','YScale','log','ZScale','log')
% %     hold off
% %     %title('Extended Greatz problem: $\tilde{T}$-profile where $\tilde{T}(0,\tilde{r})=1$, $\tilde{T}(\tilde{x},1)=0$ and $\frac{\partial \tilde{T}}{\partial \tilde{r}}(\tilde{x},0)=0$','Interpreter','LaTex');
% %     xlabel('$\tilde{r}$ [-]','Interpreter','LaTex');
% %     ylabel('$\tilde{x}/\textrm{Gz}$ [-]','Interpreter','LaTex');
% %     zlabel('$\tilde{T}$ [-]','Interpreter','LaTex');
% %     set(gcf,'Color','w');
% %     set(gca,'LineWidth',0.75,'GridLineStyle','--');
% %     alpha 0.6; % not supported by pdf output
% %     box off;
% %     grid on;
% %     view(135, 15);
%     
%     nn = nn + 1;
%     figure(nn)
%     hold on
%     surf(r,x,delta);
%     set(gca,'XScale','linear','YScale','log','ZScale','linear','XLim',[.9,1],'YLim',[1e-8,1],'ZLim',[-20,20]);
%     caxis([-20,20])
%     hold off
%     %title('Extended Greatz problem: $\tilde{T}$-profile where $\tilde{T}(0,\tilde{r})=1$, $\tilde{T}(\tilde{x},1)=0$ and $\frac{\partial \tilde{T}}{\partial \tilde{r}}(\tilde{x},0)=0$','Interpreter','LaTex');
%     xlabel('$\tilde{r}$ [-]','Interpreter','LaTex');
%     ylabel('$\tilde{x}/\textrm{Gz}$ [-]','Interpreter','LaTex');
%     zlabel('$\tilde{T}$ [-]','Interpreter','LaTex');
%     set(gcf,'Color','w');
%     set(gca,'LineWidth',0.75,'GridLineStyle','--');
%     alpha 0.6; % not supported by pdf output
%     box off;
%     grid on;
%     view(135, 15);
% 
% %     tikzfile = [figurefile num2str(nn) '.tikz'];
% %     matlab2tikz(tikzfile, 'height', '\figureheight', 'width', '\figurewidth');
% %     figfile = [figurefile num2str(nn) '.fig'];
% %     saveas(gcf, figfile, 'fig');
% else
% end
% 
% % nx = length(x);
% % domx = 2:nx;
% % nr = length(r);
% % domr = 1:nr;
% % m = 1;
% % for k = domx
% %     for l = domr
% %         % At position x(k,l), where l>1, compute dT/dr at r* = 1.
% %         [T3(k,l),gradT3(k,l)] = pdeval(m,r,T(k,:),r(l));
% %     end
% % end

    

%% NESTED FUNCTIONS
% Nested function -------------------------------------------------------
    function [c,f,s] = pdeGreatz(r,x,T,dTdr)
        c = (Gz*(1-r.^2+2.*b)) ./ (2.*(1+4.*b));
        f = dTdr;
        s = 0;
    end

% Nested function -------------------------------------------------------
    function u0 = icGreatz(r)
        u0 = 1;
    end

% Nested function -------------------------------------------------------
    function [pl,ql,pr,qr] = bdGreatz(rl,Tl,rr,Tr,x)
        pl = 0;
        ql = 1;
        pr = Tr;
        qr = 0;
    end

end