from __future__ import division
import os.path as path
import numpy as np
import matplotlib.dates as dates
from scipy.io.netcdf import netcdf_file as nopen
import copy


# -------------------------------------------
# load datafiles
# -------------------------------------------
_base_dir = path.join(path.dirname(__file__), 'data')
has_const = path.exists(path.join(_base_dir, 't_constituents_const.nc'))
has_sat = path.exists(path.join(_base_dir, 't_constituents_sat.nc'))
has_shallow = path.exists(path.join(_base_dir, 't_constituents_shallow.nc'))


if (has_const and has_sat and has_shallow):
    _const = {}
    _sat = {}
    _shallow = {}

    ncid = nopen(path.join(_base_dir,
                           't_constituents_const.nc'), 'r', mmap=False)
    for key in ncid.variables.keys():
        _const[key] = ncid.variables[key].data
    ncid.close()

    ncid = nopen(path.join(_base_dir,
                           't_constituents_sat.nc'), 'r', mmap=False)
    for key in ncid.variables.keys():
        _sat[key] = ncid.variables[key].data
    ncid.close()

    ncid = nopen(path.join(_base_dir,
                           't_constituents_shallow.nc'), 'r', mmap=False)
    for key in ncid.variables.keys():
        _shallow[key] = ncid.variables[key].data
    ncid.close()

    # Correct issues with name strings
    _const['name'] = np.array([b''.join([s for s in arr])
                               for arr in _const['name']])

    _const['kmpr'] = np.array([b''.join([s for s in arr])
                               for arr in _const['kmpr']])

else:
    print('You do not have t_constituents_*.npy ' +
          'check that package installation is correct.')
    _const = {}
    _sat = {}
    _shallow = {}


# -------------------------------------------
# define functions
# -------------------------------------------
def compute_astronomical(jd):
    """T_ASTRON Computes astronomical Variables
     [A,ADER] = ASTRON(JD) computes the astronomical variables
                A=[tau,s,h,p,np,pp] (cycles)
      and their time derivatives
                ADER=[dtau,ds,dh,dp,dnp,dpp] (cycles/day)
      at the matlab time JD (UTC, but see code for details) where

        tau = lunar time
        s = mean longitude of the moon
        h = mean longitude of the sun
        p = mean longitude of the lunar perigee
        np = negative of the longitude of the mean ascending node
        pp = mean longitude of the perihelion (solar perigee)


        The formulae for calculating these ephemerides (other than tau)
        were taken from pages 98 and 107 of the Explanatory Supplement to
        the Astronomical Ephemeris and the American Ephemeris and Nautical
        Almanac (1961). They require EPHEMERIS TIME (ET), now TERRESTRIAL
        TIME (TT) and are based on observations made in the 1700/1800s.
        In a bizarre twist, the current definition of time is derived
        by reducing observations of planetary motions using these formulas.

        The current world master clock is INTERNATIONAL ATOMIC TIME (TAI).
        The length of the second is based on inverting the actual
        locations of the planets over the period 1956-65 into "time"
        using these formulas, and an offset added to keep the scale
        continuous with previous defns. Thus

                         TT = TAI + 32.184 seconds.

        Universal Time UT is a time scale that is 00:00 at midnight (i.e.,
        based on the earth's rotation rather than on planetary motions).
        Coordinated Universal Time (UTC) is kept by atomic clocks, the
        length of the second is the same as for TAI but leap seconds are
        inserted at intervals so that it provides UT to within 1 second.
        This is necessary because the period of the earth's rotation is
        slowly increasing (the day was exactly 86400 seconds around 1820,
        it is now about 2 ms longer). 22 leap seconds have been added in
        the last 27 years.

        As of 1/1/99,    TAI = UTC + 32 seconds.

        Thus,             TT = UTC + 62.184 seconds

        GPS time was synchronized with UTC 6/1/1980 ( = TAI - 19 secs),
        but is NOT adjusted for leap seconds. Your receiver might do this
        automatically...or it might not.

        Does any of this matter? The moon longitude is the fastest changing
        parameter at 13 deg/day. A time error of one minute implies a
        position error of less than 0.01 deg. This would almost always be
        unimportant for tidal work.

        The lunar time (tau) calculation requires UT as a base.  UTC is
        close enough - an error of 1 second, the biggest difference that
        can occur between UT and UTC, implies a Greenwich phase error of
        0.01 deg.  In Doodson's definition (Proc R. Soc. A, vol 100,
        reprinted in International Hydrographic Review, Appendix to
        Circular Letter 4-H, 1954) mean lunar time is taken to begin at
        "lunar midnight".
     B. Beardsley  12/29/98, 1/11/98
     R. Pawlowicz  9/1/01
     Version 1.0
     Compute number of days from epoch of 12:00 UT Dec 31, 1899.
     (January 0.5 1900 ET)
    """

    d = jd - dates.datestr2num('Dec 31 1899, 12:0:0')
    D = d / 10000

    # Compute astronomical constants at time d1.
    args = np.array([1, d, D*D, D**3])

    # These are the coefficients of the formulas in the Explan. Suppl.
    sc = np.array([270.434164, 13.1763965268, - 8.5e-05, 3.9e-08])
    hc = np.array([279.696678, 0.9856473354, 2.267e-05, 0.0])
    pc = np.array([334.329556, 0.1114040803, - 0.0007739, - 2.6e-07])
    npc = np.array([- 259.183275, 0.0529539222, - 0.0001557, - 5e-08])
    #  first coeff was 281.220833 in Foreman but Expl. Suppl. has 44.
    ppc = np.array([281.220844, 4.70684e-05, 3.39e-05, 7e-08])
    coef = np.vstack([sc, hc, pc, npc, ppc])

    # Compute the parameters;
    # we only need the factional part of the cycle.
    astro = np.fmod(np.dot(coef, args) / 360.0, 1)

    # Compute lunar time tau, based on fractional part of solar day.
    # We add the hour angle to the longitude of the sun and
    # subtract the longitude of the moon.
    tau = (np.fmod(jd, 1) + astro[1] - astro[0])
    astro = np.hstack([tau, astro])

    # Compute rates of change.
    dargs = np.array([0, 1, 0.0002 * D, 0.0003 * D * D])

    ader = np.dot(coef, dargs) / 360.0
    dtau = (1.0 + ader[1] - ader[0])
    ader = np.hstack([dtau, ader])

    return astro, ader


def consts_by_ctime(ctime):
    """
    t_getconsts - Gets constituent data structures holding
                  information for tidal analyses
    Variables are loaded from 't_constituents_*.npy'
    on init and a copy is made now.
    When ctime is specified t_getconsts recomputes the frequencies from
    the rates-of-change of astronomical parameters at the matlab TIME given.

     :Parameters:
        ctime: a datetime, the start time of the data input into t_tide.

    Note:
        Not sure if a copy has to be made here or if they can be used directly.
        For now a copy should be much fast then a load.
    """
    const = copy.deepcopy(_const)
    sat = copy.deepcopy(_sat)
    shallow = copy.deepcopy(_shallow)

    if ctime.size != 0:
        # If no time, just take the "standard" frequencies, otherwise
        # compute them from derivatives of astro parameters. This is
        # probably a real overkill - the diffs are in the
        # 10th decimal place (9th sig fig).
        astro, ader = compute_astronomical(ctime)

        ii = np.isfinite(const['ishallow'])
        const['freq'][~ii] = np.dot(const['doodson'][~ii, :], ader)/24

        for k in np.flatnonzero(ii):
            ik = ((const['ishallow'][k]-1 +
                   np.array(range(0, const['nshallow'][k]))).astype(int))
            const['freq'][k] = np.dot(const['freq'][shallow['iname'][ik]-1],
                                      shallow['coef'][ik])

    return const, sat, shallow