from __future__ import print_function
try:
    from PyQt5 import QtWidgets, QtGui, QtCore
except ImportError:
    from PyQt4 import QtGui, QtCore
    QtWidgets = QtGui
from .sql import DatabaseHandler, SQLiteDatabaseHandler
import logging
import datetime
from hashlib import md5
import os


class LogViewer(QtWidgets.QWidget):

    HEADERS = ['datetime', 'level', 'name', 'message', 'location', 'process', 'thread']
    LOGLEVELNAMES = ['DEBUG', 'INFO', 'WARNING', 'ERROR', 'CRITICAL']
    LEVELCOLORS = dict(
        DEBUG=QtGui.QColor(180, 255, 180),
        INFO=QtGui.QColor(255, 255, 255),
        WARNING=QtGui.QColor(255, 255, 0),
        ERROR=QtGui.QColor(255, 100, 0),
        CRITICAL=QtGui.QColor(255, 0, 0),
    )
    COLUMNWIDTHS = [150, 80, 150, 350, 250, 80]
    RELOAD_TIMEOUT = 5000  # ms

    def __init__(self, recorditer):
        super(LogViewer, self).__init__()
        self.recorditer = recorditer
        self.filters = dict()
        self.build()

    def build(self):
        layout = QtWidgets.QVBoxLayout(self)
        self.build_toolbar(layout)
        self.build_table(layout)
        self.fill()

        self.timer = QtCore.QTimer()
        self.timer.timeout.connect(self.check_reload)
        self.timer.start(self.RELOAD_TIMEOUT)

    def check_reload(self):
        try:
            if not self.recorditer.is_changed():
                self.set_msg('last update check: {}'.format(datetime.datetime.now().strftime('%Y-%m-%d %H:%M:%S')))
                return
        except AttributeError as e:
            print('VIEWER', e)
            pass
        return self.fill()

    def sizeHint(self):
        return QtCore.QSize(800, 400)

    def build_toolbar(self, layout):
        toolbar_layout = QtWidgets.QHBoxLayout()

        b = QtWidgets.QPushButton('reload')
        b.clicked.connect(self.fill)
        toolbar_layout.addWidget(b)

        b = QtWidgets.QPushButton('save')
        b.clicked.connect(self.save_current)
        toolbar_layout.addWidget(b)

        toolbar_layout.addItem(
            QtWidgets.QSpacerItem(0, 0, QtWidgets.QSizePolicy.Expanding, QtWidgets.QSizePolicy.Maximum))

        toolbar_layout.addWidget(QtWidgets.QLabel('threshold level'))
        self.filter_level_dropdown = QtWidgets.QComboBox()
        self.filter_level_dropdown.addItems(self.LOGLEVELNAMES)
        self.filter_level_dropdown.currentIndexChanged.connect(self.filter_level)
        toolbar_layout.addWidget(self.filter_level_dropdown)

        toolbar_layout.addItem(
            QtWidgets.QSpacerItem(0, 0, QtWidgets.QSizePolicy.Expanding, QtWidgets.QSizePolicy.Maximum))

        toolbar_layout.addWidget(QtWidgets.QLabel('filter name'))
        self.namesearch = QtWidgets.QLineEdit()
        self.namesearch.editingFinished.connect(self.filter_name)
        toolbar_layout.addWidget(self.namesearch)

        toolbar_layout.addItem(
            QtWidgets.QSpacerItem(0, 0, QtWidgets.QSizePolicy.Expanding, QtWidgets.QSizePolicy.Maximum))

        toolbar_layout.addWidget(QtWidgets.QLabel('filter message'))
        self.msgsearch = QtWidgets.QLineEdit()
        self.msgsearch.editingFinished.connect(self.filter_message)
        toolbar_layout.addWidget(self.msgsearch)

        self.msglabel = QtWidgets.QLabel()
        self.msglabel.hide()
        layout.addWidget(self.msglabel)

        layout.addLayout(toolbar_layout)

    def build_table(self, layout):
        self.table = QtWidgets.QTreeWidget()
        self.table.setColumnCount(len(self.HEADERS))
        self.table.setHeaderLabels(self.HEADERS)
        layout.addWidget(self.table)

    def set_msg(self, msg):
        self.msglabel.setText(msg)
        self.msglabel.show()

    def save_current(self):
        fileName = QtGui.QFileDialog.getSaveFileName(self,
                                                     'save logs to file',
                                                     '.',
                                                     selectedFilter='*.log')
        if not fileName:
            return

        h = logging.FileHandler(fileName, mode='w')
        with open(fileName, 'w') as f:
            for record in self.recorditer:
                if not self.is_enabled(record):
                    continue
                print(record.msg, record.args)
                h.emit(record)

    def fill(self):
        self.table.clear()
        for record in self.recorditer:
            if not self.is_enabled(record):
                continue

            msg_lines = record.msg.strip().splitlines()
            msg, other_lines = msg_lines[0], list(msg_lines[1:])

            if record.exc_text:
                other_lines += list(record.exc_text.splitlines())

            location = '{}:{}'.format(record.pathname, record.lineno)
            recorditem = QtWidgets.QTreeWidgetItem(
                [record.created.strftime('%Y-%m-%d %H:%M:%S'),
                 record.levelname,
                 record.name,
                 msg,
                 location,
                 str(record.processName),
                 record.threadName])

            brush = QtGui.QBrush(self.LEVELCOLORS.get(record.levelname, QtGui.QColor(255, 255, 255)))
            recorditem.setBackground(1, brush)

            for line in other_lines:
                content = [line.rstrip() if h == 'message' else '' for i, h in enumerate(self.HEADERS)]
                recorditem.addChild(QtWidgets.QTreeWidgetItem(content))

            self.table.addTopLevelItem(recorditem)

        for i, h in enumerate(self.HEADERS[:-1]):
            self.table.setColumnWidth(i, self.COLUMNWIDTHS[i])

        self.set_msg('last updated: {}'.format(datetime.datetime.now().strftime('%Y-%m-%d %H:%M:%S')))

    def is_enabled(self, record):
        for name, f in self.filters.items():
            if not f(record):
                return False
        return True

    def filter_level(self, i):
        level = getattr(logging, self.LOGLEVELNAMES[i])
        self.filters['minlevel'] = lambda x: x.levelno >= level
        self.fill()

    @classmethod
    def from_db(cls, dbname, **kwargs):
        return cls(DBLogLoader(dbname, **kwargs))

    def filter_name(self):
        s = str(self.namesearch.text())
        self.filters['name'] = lambda x: x.name.startswith(s)
        self.fill()

    def filter_message(self):
        s = str(self.msgsearch.text())
        items = s.split(' AND ')
        def filter(record):
            for item in items:
                if item.startswith('-'):
                    if item[1:] in record.msg:
                        return False
                elif item not in record.msg:
                    return False
            return True
        self.filters['message'] = filter
        self.fill()


class LogLoader(object):

    def __init__(self, location):
        self.location = location
        self.hash = self.__hash__()

    def __iter__(self):
        raise NotImplementedError('__iter__')

    def is_changed(self):
        hash = self.__hash__()
        try:
            return hash != self.hash
        finally:
            self.hash = hash

    def __hash__(self):
        if not os.path.isfile(self.location):
            import uuid
            return str(uuid.uuid4())
        hash_md5 = md5()
        with open(self.location, "rb") as f:
            hash_md5.update(f.read())

        return 'MD5_'+str(hash_md5.hexdigest())


class DBLogLoader(LogLoader):

    def __init__(self, location, sqlite=True):
        super(DBLogLoader, self).__init__(location)
        self.is_sqlite = sqlite
        self.hash = md5()

    def __iter__(self):
        if self.is_sqlite:
            handler = SQLiteDatabaseHandler(self.location)
        else:
            handler = DatabaseHandler(self.location)
        for record in handler.iter_records():
            yield record
