"""
Configuration of several important file locations is done in a file called:
    .sandwave_settings.json
which should be created in the user directory as defined by
    >>>os.path.expanduser('~')
or available on the python path (sys.path).
This file is in json format and may contain the following variables:
1. catalog_path        location of the netCDF catalog file (see sandwaves.datasets.iotools.sources.NCCatalog)
2. landboundary_NL     file that contains the landboundary file of the Netherlands
3. asc_metadata_path   location of the metadata file for bedform surveys
4. asc_grid_path       search directory of ascii grid data
5. nc_cachedir         location where netCDF files are cached for faster loading
"""
import os
import json
import sys


__all__ = []

_PATH_NAMES = ['catalog_path', 'landboundary_NL', 'asc_metadata_path', 'asc_grid_path', 'nc_cachedir']
_REQUIRE_PATH_EXISTS = ['catalog_path', 'landboundary_NL']
_OTHER_NAMES = []
_DEFAULTS = dict(
    nc_cachedir=None
)


def get_settings_filename():
    filename = '.sandwave_settings.json'
    localsettingsfile = os.path.join(os.path.expanduser('~'), filename)
    if os.path.isfile(localsettingsfile):
        return localsettingsfile
    for d in sys.path:
        path = os.path.join(d, filename)
        if os.path.isfile(path):
            return path
    raise IOError('settings file {!r} not found'.format(filename))


def load_locations():
    settingsfile = get_settings_filename()
    with open(settingsfile, 'r') as f:
        data = json.load(f)

    def _get_value(data, name, settingsfile):
        try:
            return data.get(name, _DEFAULTS.get(name))
        except KeyError:
            raise KeyError('{} not defined in settings file {!r}'.format(name, settingsfile))

    for pname in _PATH_NAMES:
        pval = _get_value(data, pname, settingsfile)
        if pval is None and pname in _DEFAULTS:
            pass
        elif not pval.startswith('http'):
            if not os.path.isabs(pval):
                pval = os.path.join(os.path.dirname(settingsfile), pval)
            if pname in _REQUIRE_PATH_EXISTS and not os.path.exists(pval):
                raise IOError('location {!r} as defined in {}:{} not found'.format(pval, settingsfile, pname))
        globals()[pname] = pval
        __all__.append(pname)

    for name in _OTHER_NAMES:
        val = _get_value(data, name, settingsfile)
        globals()[name] = val
        __all__.append(name)

load_locations()