from __future__ import print_function
import argparse
from argparse import RawTextHelpFormatter
import inspect
import textwrap
from collections import OrderedDict
from functools import partial
import sys
import subprocess
import re
import datetime
try:
    from PyQt5 import QtWidgets, QtGui, QtCore
except ImportError:
    from PyQt4 import QtGui, QtCore
    QtWidgets = QtGui


PARSER = None
SUBPARSERS = None


def setup_parser(*args, **kwargs):
    global PARSER
    global SUBPARSERS
    PARSER = argparse.ArgumentParser(*args, **kwargs)
    SUBPARSERS = PARSER.add_subparsers()
    return PARSER


def parse_args(argv):

    if argv and argv[0] == 'gui':
        try:
            name = argv[1]
        except IndexError:
            return CMD_choice_interface()
        else:
            p = SUBPARSERS.choices[name]
            if p._defaults.get('direct', False):
                return p._defaults['callback']()
            else:
                return CMD_interface(name)
    else:
        kwargs = vars(PARSER.parse_args(argv))
        kwargs.pop('nogui', None)
        callback = kwargs.pop('callback')
        if kwargs.pop('direct', False):
            return callback()
        return callback(**kwargs)


class Path(str): pass
class Directory(Path): pass
class File(Path): pass
class NewFile(Path): pass
def ChoiceList(*items):
    return type('Choices', (str,), dict(items=items))


def get_func_args(fn):
    args, varargs, varkw, defaults = inspect.getargspec(fn)
    if defaults is None:
        defaults = ()
    return args[:len(args)-len(defaults)], OrderedDict(zip(args[len(args)-len(defaults):], defaults))


def expose(fn=None, nogui=False, direct=False, **types):
    if fn is None:
        def wrapper(fn):
            return expose(fn, nogui=nogui, direct=direct, **types)
        return wrapper

    description, argdesc = parse_docstr(fn.__doc__ or '')
    p = SUBPARSERS.add_parser(fn.__name__.replace('_', '-'), description=description)

    if not direct:
        args, kwargs = get_func_args(fn)

        for a in args:
            p.add_argument(a.replace('_', '-'), type=types.get(a, None), help=argdesc.get(a, ''))

        used_short_names = ['h']
        for kw, default in kwargs.items():
            name = kw.replace('_', '-')
            short_name = kw[0].upper()

            if default is True:
                opts = dict(action='store_false')
                name = 'no-'+name
                assert types.get(kw, bool) == bool
            elif default is False:
                opts = dict(action='store_true')
                assert types.get(kw, bool) == bool
            else:
                opts = dict(default=default, type=types.get(kw, None))

            opts['dest'] = kw
            opts['help'] = argdesc.get(kw, '')

            if short_name in used_short_names:
                names = ['--'+name]
            else:
                names = ['-'+short_name, '--'+name]
                used_short_names.append(short_name)

            p.add_argument(*names, **opts)

    p.set_defaults(callback=fn, nogui=nogui, direct=direct)

    return fn


def parse_docstr(docstr):
    argdesc = {}
    param_pattern = re.compile(r':param (\w+): (.*)')
    return_pattern = re.compile(r':return: (.*)')

    description = ''
    in_header = True
    for line in textwrap.dedent(docstr).split('\n'):
        m1 = param_pattern.match(line)
        m2 = return_pattern.match(line)
        if m1:
            in_header = False
            argdesc[m1.group(1)] = m1.group(2)
        elif m2:
            continue
        elif in_header:
            description += line

    return description, argdesc


def hline():
    toto = QtWidgets.QFrame()
    toto.setFrameShape(QtWidgets.QFrame.HLine)
    toto.setFrameShadow(QtWidgets.QFrame.Sunken)
    return toto


class CMD_choice_interface(QtWidgets.QWidget):

    action_windows = dict()

    def __init__(self):
        self.choices = SUBPARSERS.choices
        super(CMD_choice_interface, self).__init__()
        self.build()

    def build(self):
        self.setWindowTitle('sandwaves toolbox - choose script')
        self.setFixedWidth(450)
        self.layout = QtWidgets.QVBoxLayout(self)
        for i, c in enumerate(self.choices):
            if self.choices[c]._defaults['nogui']:
                continue
            line_layout = QtWidgets.QHBoxLayout()

            button = QtWidgets.QPushButton(str(c))
            button.setMaximumWidth(150)
            button.clicked.connect(partial(self.select, c))
            line_layout.addWidget(button)

            # make label from first docstring line
            short_description = self.choices[c].description.strip().split('\n')[0].split('. ')[0]
            label = QtWidgets.QLabel(short_description)
            label.setWordWrap(True)
            line_layout.addWidget(label)
            line_layout.addItem(QtWidgets.QSpacerItem(0, 15, QtWidgets.QSizePolicy.Maximum, QtWidgets.QSizePolicy.Fixed))
            self.layout.addLayout(line_layout)

    def select(self, name):
        p = SUBPARSERS.choices[name]
        if p._defaults.get('direct', False):
            w = p._defaults['callback']()
        else:
            w = CMD_interface(name)
        self.action_windows[name] = w
        w.show()


class CMD_interface(QtWidgets.QWidget):

    def fill_with_new_filename(self, w):
        f = QtWidgets.QFileDialog.getSaveFileName()
        if isinstance(f, tuple):
            f = f[0]
        w.setText(f)

    def fill_with_filename(self, w):
        f = QtWidgets.QFileDialog.getOpenFileName()
        if isinstance(f, tuple):
            f = f[0]
        w.setText(f)

    def fill_with_directory(self, w):
        f = QtWidgets.QFileDialog.getExistingDirectory()
        if isinstance(f, tuple):
            f = f[0]
        w.setText(f)

    def __init__(self, name):
        parser = SUBPARSERS.choices[name]
        self.name = name
        self.parser = parser
        super(CMD_interface, self).__init__()
        self.build()

    def processor(self, srcfn, typeval):
        def process():
            val = srcfn()
            if isinstance(val, str):
                val = str(val)
            if typeval is not None:
                if val == '':
                    val = typeval()
                else:
                    val = typeval(val)
            return val
        return process

    def build(self):
        self.inputs = dict()

        title_font = QtGui.QFont()
        title_font.setPointSize(11)
        title_font.setFamily('Helvetica')

        self.setFixedWidth(750)
        self.layout = QtWidgets.QVBoxLayout(self)

        self.setWindowTitle('sandwaves toolbox - {}'.format(self.name))

        label = QtWidgets.QLabel(self.parser.description or 'no description available')
        label.setWordWrap(True)
        self.layout.addWidget(label)

        self.layout.addWidget(hline())

        arglayout = QtWidgets.QGridLayout()
        self.layout.addLayout(arglayout)

        for i, a in enumerate(self.parser._actions):
            if isinstance(a, (argparse._StoreAction, argparse._StoreTrueAction, argparse._StoreFalseAction)):
                argtitle = QtWidgets.QLabel(a.dest)
                argtitle.setFont(title_font)
                argtitle.setFixedWidth(100)
                arglayout.addWidget(argtitle, i, 0)

                if isinstance(a, (argparse._StoreTrueAction, argparse._StoreFalseAction)):
                    input_field = QtWidgets.QCheckBox()
                    input_field.setChecked(a.default)
                    arglayout.addWidget(input_field, i, 1)
                    self.inputs[a.dest] = self.processor(input_field.isChecked, a.type)

                elif a.type is not None and a.type.__name__ == 'Choices':
                    input_field = QtWidgets.QComboBox()
                    input_field.addItems(a.type.items)
                    arglayout.addWidget(input_field, i, 1)
                    self.inputs[a.dest] = self.processor(input_field.currentText, a.type)

                elif a.type is not None and issubclass(a.type, Path):

                    input_field = QtWidgets.QLineEdit()
                    input_field.setText(str(a.default or ''))
                    browse_button = QtWidgets.QPushButton('browse')
                    if issubclass(a.type, Directory):
                        browse_button.clicked.connect(partial(self.fill_with_directory, input_field))
                    elif issubclass(a.type, NewFile):
                        browse_button.clicked.connect(partial(self.fill_with_new_filename, input_field))
                    else:
                        browse_button.clicked.connect(partial(self.fill_with_filename, input_field))
                    l = QtWidgets.QHBoxLayout()
                    l.addWidget(input_field)
                    l.addWidget(browse_button)
                    arglayout.addLayout(l, i, 1)
                    self.inputs[a.dest] = self.processor(input_field.text, a.type)

                elif a.type is None or issubclass(a.type, (int, float, str)):
                    input_field = QtWidgets.QLineEdit()
                    input_field.setText(str(a.default or ''))
                    arglayout.addWidget(input_field, i, 1)
                    self.inputs[a.dest] = self.processor(input_field.text, a.type)

                arglayout.addWidget(QtWidgets.QLabel(a.help), i, 2)

        self.layout.addWidget(hline())

        button_layout = QtWidgets.QHBoxLayout()
        button_layout.addItem(QtWidgets.QSpacerItem(0, 0, QtWidgets.QSizePolicy.Expanding, QtWidgets.QSizePolicy.Maximum))
        submit_button = QtWidgets.QPushButton('execute')
        submit_button.setFixedWidth(150)
        submit_button.clicked.connect(self.execute)
        button_layout.addWidget(submit_button)
        button_layout.addItem(QtWidgets.QSpacerItem(0, 0, QtWidgets.QSizePolicy.Expanding, QtWidgets.QSizePolicy.Maximum))
        self.layout.addLayout(button_layout)

    def execute(self):
        kwargs = {}
        for k, v in self.inputs.items():
            try:
                kwargs[k] = v()
            except Exception as e:
                QtWidgets.QMessageBox.warning(self, 'could not parse arguments', 'invalid value for `{}`'.format(k))
                raise

        cmd = [sys.executable, '-m']+self.reconstruct(**kwargs)
        print('EXECUTING ::', ' '.join(cmd))
        try:
            subprocess.Popen(cmd, stdout=sys.stdout)
        except Exception as e:
            QtWidgets.QMessageBox.warning(self, 'could not execute function', str(e))
            raise

    def reconstruct(self, **data):
        from collections import OrderedDict
        args = ['sandwaves', self.name]
        actions = OrderedDict([(a.dest, a) for a in self.parser._actions])
        for a in self.parser._actions:
            k = a.dest
            v = data.get(k, None)
            if v is None:
                continue

            if v in (True, False) and not a.required:
                if a.const == v:
                    args.append(a.option_strings[-1])
                continue

            item_args = []
            try:
                item_args.append(a.option_strings[-1])
            except IndexError:
                pass
            vstr = str(v)
            if vstr == '':
                continue
            item_args.append(vstr)
            args += item_args

        return args


class stdoutPatch:

    def __init__(self, fn):
        self.fn = fn

    def __enter__(self):
        self.old_stdout, sys.stdout = sys.stdout, self

    def __exit__(self, exc_type, exc_val, exc_tb):
        sys.stdout = self.old_stdout

    def write(self, msg):
        self.fn(msg)
