"""
batchruns apply analysis functions to a large number of datasets using iteration
the most important batchrun functions are:
1. batchruns.shape        uses analysis.shape2D
2. batchruns.migration    uses analysis.migration
3. batchruns.growth       uses analysis.growth2D
"""

from .batchruns import BatchRun, BatchError, LoadingError, AnalysisError, SaveError
from .data import *
from .output import *
from .visualize import show_progress


ANALYSES = dict()
def register(fn):
    ANALYSES[fn.__name__] = fn
    return fn


_BASE_DOCSTRING = """:param location: output storage location (directory)
    :param xll: x coord of lower left corner of the area to analyse
    :param yll: y coord of lower left corner of the area to analyse
    :param xur: x coord of upper right corner of the area to analyse
    :param yur: y coord of upper right corner of the area to analyse
    :param blocksize: size of the blocks for subdividing the area
    :param margin: size of the margin to add to each block
    :param max_count: maximum number of iterations (0 for all)
    :param progresslog: filename of the log file to store progress
    :param continue_previous: continue from previous run"""


def _extend_docstring(fn):
    if fn.__doc__ is None:
        fn.__doc__ = _BASE_DOCSTRING
    else:
        fn.__doc__ = fn.__doc__.format(_BASE_DOCSTRING)
    return fn


def apply(fn, location, datatype,
          xll=450000, yll=5700000, xur=700000, yur=5950000,
          blocksize=5000, margin=0, max_count=0,
          progresslog=None, continue_previous=False, **kw):


    datatype = dict(single=GridDataBlock,
                    duo=DuoGridDataBlock,
                    stack=GridStackDataBlock,
                    raw=DataBlock)[datatype]

    with SeparateBlocksCSVOutput(location) as output:
        indata = DataIterable(
            xll=xll, yll=yll, xur=xur, yur=yur,
            blocksize=blocksize, margin=margin,
            blockcls=datatype, max_count=max_count)

        B = BatchRun(indata, fn, output,
            progresslog=progresslog,
            continue_previous=continue_previous,
            analysissettings=kw)
        B.run()

    if progresslog:
        show_progress(progresslog, xmin=xll, ymin=yll, xmax=xur, ymax=yur)

    return output.load()


@register
@_extend_docstring
def shape(location, minheight=0.8, lmin=100, lmax=1000, **kwargs):
    """
    calculate sand wave shape for a large area
    {}
    :param minheight: minimum sand wave height to take into account
    :param lmin: minimum sand wave length
    :param lmax: maximum sand wave length
    """
    from ..analysis import shape2D as blockshape
    return apply(blockshape, location, 'single',
                 minheight=minheight, lmin=lmin, lmax=lmax, **kwargs)


@register
@_extend_docstring
def migration(location, radius=100, points='c', **kwargs):
    """
    calculate migration rates for a large area
    {}
    """
    from ..analysis import migration as blockmigration
    return apply(blockmigration, location, 'stack', radius=radius, points=points, **kwargs)


@register
@_extend_docstring
def growth(location, **kwargs):
    """
    calculate growth rates for a large area
    {}
    """
    from ..analysis import growth2D
    apply(growth2D, location, 'stack', **kwargs)


def get_by_name(name):
    return ANALYSES[name]