__doc__ = """
package for the analysis of sand waves (and other bedforms) as developed for the SMARTSEA project
this package consists of several components:
1. analysis     contains the bedform analysis functions
2. batchruns    contains the analysis functions for large datasets
3. datasets     contains the data management functions
4. plotting     contains several plotting functions for uniform plotting
5. tools        contains various tools as used by the package
"""

from . import analysis
__doc__ += analysis.__doc__

from . import batchruns
__doc__ += batchruns.__doc__

from . import datasets
__doc__ += datasets.__doc__

from . import settings
__doc__ += settings.__doc__

from .plotting.cmaps import viridis_ext as viridis, discrete_viridis


__all__ = ['datasets', 'analysis', 'batchruns', 'settings'
           'notebook_logging', 'database_logging', 'show_database_logs']


def notebook_logging(level='INFO', fmt="%(levelname)s :: %(name)s :: %(message)s", names=['sandwaves']):
    """
    enable log messages in jupyter notebook
    :param level: threshold level for log messages
    :param fmt: log message format
    """
    from .tools.logs import NotebookHandler, setup
    return setup(names, level=level, fmt=fmt, handler=NotebookHandler())


def database_logging(level='INFO',
                     fmt="%(levelname)s :: %(name)s :: %(message)s",
                     names=['sandwaves'],
                     database='logging.db',
                     reset=True, viewer=False):
    """
    enable logging to database
    :param level: threshold level for log messages
    :param fmt: log message format
    :param database: sqlite database file for logging
    :param reset: clear possibly existing database
    :param viewer: open log viewer
    """
    from .tools.logs import SQLiteDatabaseHandler, setup
    out = setup(names, level=level, fmt=fmt, handler=SQLiteDatabaseHandler(database, reset=reset))
    if viewer:
        show_database_logs(database)
    return out


def basic_logging(level='INFO', fmt="%(levelname)s :: %(name)s :: %(message)s", names=['sandwaves']):
    """
    setup simple logging
    :param level: threshold level for log messages
    :param fmt: log message format
    """
    from .tools.logs import setup
    return setup(names, level=level, fmt=fmt, handler=None)


def file_logging(level='INFO', fmt="%(levelname)s :: %(name)s :: %(message)s", names=['sandwaves'], filename=None, reset=True):
    """
    setup logging to file
    :param level: threshold level for log messages
    :param fmt: log message format
    :param filename: filename for logging
    :param reset: start new log file
    """
    import datetime
    import logging
    from .tools.logs import setup
    if filename is None:
        filename = 'batch_{}.log'.format(datetime.datetime.now().strftime('%Y%m%d'))
    mode = 'w' if reset else 'a'
    return setup(names, level=level, fmt=fmt, handler=logging.FileHandler(filename, mode=mode))


def show_database_logs(database='logging.db'):
    """
    open PyQt window for viewing log messages
    also possible to use while running an analysis
    :param database: sqlite database file
    """
    import sys
    import os
    import subprocess

    if not os.path.isfile(database):
        raise IOError('database {!r} not found'.format(database))

    cwd = os.path.join(os.path.dirname(__file__), 'tools')
    cmd = [sys.executable, '-m', 'logs', os.path.abspath(database)]
    subprocess.Popen(cmd, cwd=cwd)


__author__ = "J.M. Damen"
__version__ = "2.1"
__email__ = "j.m.damen@utwente.nl"
