package DistanceMeasure;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.util.ArrayList;
//import java.util.LinkedList;

/*
 * Use this file as the main class to run and gather all distance measures which are then saved in MeasureResults
 */

public class ExperimentSimulationDistance {
	
	// The experimental data that is loaded for the basal state
	protected double[][] experimentalBasal;
	// The experimental data that is loaded for the transition state
	protected double[][] experimentalTransition;
	// The experimental data that is loaded for the Insulin state
	protected double[][] experimentalInsulin;
	// The below are the simulation arrays 
	protected double[][] simulationBasal;
	protected double[][] simulationTransition;
	protected double[][] simulationInsulin;
	protected ArrayList<double[][]> experimentalData;
	
	protected ArrayList<double[][]> normalisedSimulationData;
	
	protected MeasureResults results;
	
	public static final int BASAL = 0;
	public static final int TRANSITION = 1;
	public static final int INSULIN = 2;
	
	public ExperimentSimulationDistance(String BasalFile, String TransitionFile, String InsulinFile) {
		experimentalBasal = readExperimentalData(BasalFile);
		experimentalTransition = readExperimentalData(TransitionFile);
		experimentalInsulin = readExperimentalData(InsulinFile);
		
		experimentalData = new ArrayList<double[][]>();
		normalisedSimulationData = new ArrayList<double[][]>();
		
		results = new MeasureResults();
		
		experimentalData.add(experimentalBasal);
		experimentalData.add(experimentalTransition);
		experimentalData.add(experimentalInsulin);
	}
	
	/**
	 * Make the input file be a csv of the array
	 * Put the data into the right format externally to make life easier
	 * @param expDataFile
	 * @return
	 */
	public double[][] readExperimentalData(String expDataFile){
		final String delimiter = ",";
		ArrayList<String[]> data;
		try {
		   File file = new File(expDataFile);
		   FileReader fr = new FileReader(file);
		   BufferedReader br = new BufferedReader(fr);
		   String line = "";
		   String[] tempArr;
		   data = new ArrayList<String[]>();
		   while((line = br.readLine()) != null) {
		      tempArr = line.split(delimiter);
		      data.add(tempArr);
		      /*
		      for(String tempStr : tempArr) {
		         System.out.print(tempStr + " ");
		      }
		         System.out.println();
		         */
		   }
		   br.close();
		   double[][] finalData = new double[data.size()][data.get(0).length];
		   for (int i = 0; i < data.size(); i++) {
			   for (int j = 0; j < data.get(0).length; j++) {
				   finalData[i][j] = Double.parseDouble(data.get(i)[j]);
			   }
		   }
		   return finalData;
		   } catch(IOException ioe) {
		      ioe.printStackTrace();
		      return new double[0][0];
		   }
	}
	
	/**
	 * Reads in the simulation data results from the getArrangedArray method
	 * @param simulationBasal
	 * @param simulationTransition
	 * @param simulationInsulin
	 */
	public void setSimulationData(double[][] simulationBasal,
			double[][] simulationTransition, double[][] simulationInsulin) {
		this.simulationBasal = simulationBasal;
		this.simulationTransition = simulationTransition;
		this.simulationInsulin = simulationInsulin;
	}
	
	/**
	 * Returns the normalisation level for the transition experiment
	 * This is the mean of the final timepoint of the Transition Simulation Data
	 * @return
	 */
	public double getTransitionNormalisation() {
		return HelperFunctions.mean(simulationTransition[simulationTransition.length - 1]);
	}
	
	/**
	 * Returns the normalisation level for the Uptake experiments
	 * This is the mean of the final timepoint of the uptake insulin Simulation Data
	 * @return
	 */
	public double getUptakeNormalisation() {
		//System.out.println("Uptake Normalisation: " + HelperFunctions.mean(simulationInsulin[simulationInsulin.length - 1]));
		return HelperFunctions.mean(simulationInsulin[simulationInsulin.length - 1]);
	}
	
	/*
	public void normaliseSimData() {
		double transitionNormalisation = getTransitionNormalisation();
		double uptakeNormalisation = getUptakeNormalisation();
		
		double[][] normalisedTransition;
		
	}
	*/
	
	/**
	 * Note that this relies on basal being done before uptake or else the 
	 * wrong normalisation is given for insulin uptake. Should fix later
	 * @param exp
	 * @return
	 */
	public double[][] getNormalisedArray(int exp) {
		double[][] data;
		double normalisation;
		switch(exp) {
		case BASAL:
			//System.out.println("Entered Basal Switch");
			data = simulationBasal;
			normalisation = getUptakeNormalisation();
			break;
		case TRANSITION:
			//System.out.println("Entered Transition Switch");
			data = simulationTransition;
			normalisation = getTransitionNormalisation();
			break;
		case INSULIN: 
			//System.out.println("Entered Insulin Switch");
			data = simulationInsulin;
			normalisation = getUptakeNormalisation();
			break;
		default:
			//System.out.println("Entered Default Switch");
			data = simulationInsulin;
			normalisation = getUptakeNormalisation();
			break;
		}
		//System.out.println("Exp num: " + exp);
		//System.out.println("Normalisation: " + normalisation);
		// Normalise the array
		for (int i = 0; i < data.length; i++) {
			for (int j = 0; j < data[0].length; j++) {
				data[i][j] = data[i][j] / normalisation;
			}
		}
		return data;
	}
	
	public void outputResults(String filename) {
		results.writeToFile(filename);
	}
	
	/**
	 * Calculates all the distance measures and saves them to a MeasureResults type
	 * This method is the only one that should be called on the repeats of simulations
	 */
	public void performDistanceCalculations(double[][] simulationBasal,
			double[][] simulationTransition, double[][] simulationInsulin, String[] paramSet) {
		
		// Initialise ECDF class
		EmpiricalCDF ecdf;
		// Set the simulation Data
		setSimulationData(simulationBasal,
				simulationTransition, simulationInsulin);
		// First normalise the arrays
		// Save each into a new array to make looping easier
		int numArrays = 3; // The number of experiments / arrays given
		double[][] simData;
		double[] unsignedAreas = new double[numArrays];
		double[] signedAreas = new double[numArrays];
		double[] squaredAreas = new double[numArrays];
		double[] ksDistance = new double[numArrays];
		double[] unsignedAreasSup = new double[numArrays];
		double[] signedAreasSup = new double[numArrays];
		double[] squaredAreasSup = new double[numArrays];
		double[] ksDistanceSup = new double[numArrays];
		double[] unsignedAreas2Norm = new double[numArrays];
		double[] signedAreas2Norm = new double[numArrays];
		double[] squaredAreas2Norm = new double[numArrays];
		double[] ksDistance2Norm = new double[numArrays];
		double[] unsignedAreasMin = new double[numArrays];
		double[] signedAreasMin = new double[numArrays];
		double[] squaredAreasMin = new double[numArrays];
		double[] ksDistanceMin = new double[numArrays];
		double[] quadraticDistance = new double[numArrays];
		double[] quadraticDistanceSup = new double[numArrays];
		double[] quadraticDistance2Norm = new double[numArrays];
		double[] quadraticDistanceMin = new double[numArrays];
		double[] kuiperDistance = new double[numArrays];
		double[] kuiperDistanceSup = new double[numArrays];
		double[] kuiperDistance2Norm = new double[numArrays];
		double[] kuiperDistanceMin = new double[numArrays];
		double[] ADDistance = new double[numArrays];
		double[] ADDistanceSup = new double[numArrays];
		double[] ADDistance2Norm = new double[numArrays];
		double[] ADDistanceMin = new double[numArrays];
		double[] DTSDistance = new double[numArrays];
		double[] DTSDistanceSup = new double[numArrays];
		double[] DTSDistance2Norm = new double[numArrays];
		double[] DTSDistanceMin = new double[numArrays];
		
		double[] unsignedAreasWeighted = new double[numArrays];
		double[] signedAreasWeighted = new double[numArrays];
		double[] squaredAreasWeighted = new double[numArrays];
		double[] unsignedAreasWeightedSup = new double[numArrays];
		double[] signedAreasWeightedSup = new double[numArrays];
		double[] squaredAreasWeightedSup = new double[numArrays];
		double[] unsignedAreasWeighted2Norm = new double[numArrays];
		double[] signedAreasWeighted2Norm = new double[numArrays];
		double[] squaredAreasWeighted2Norm = new double[numArrays];
		double[] unsignedAreasWeightedMin = new double[numArrays];
		double[] signedAreasWeightedMin = new double[numArrays];
		double[] squaredAreasWeightedMin = new double[numArrays];		
		
		// Vectors where each element is the distance at a timepoint
		double[] unsignedAreaV;
		double[] signedAreaV;
		double[] squaredAreaV;
		double[] ksDistanceV;
		double[] quadraticDistanceV;
		double[] kuiperDistanceV;
		double[] ADDistanceV;
		double[] DTSDistanceV;
		double[] unsignedAreaWeightedV;
		double[] signedAreaWeightedV;
		double[] squaredAreaWeightedV;
		// loop through each experiment to get the distance measures for each
		for (int i = 0; i < numArrays; i++) {
			// get the sim data for exp
			simData = getNormalisedArray(i); // i = 0 Basal, 1 = Transition, 2 = Insulin
			// get the ecdf pair
			ecdf = new EmpiricalCDF(experimentalData.get(i), simData);
			// Calculate each distance measure
			unsignedAreaV = ecdf.getAreas();
			signedAreaV = ecdf.signedArea();
			squaredAreaV = ecdf.wasserstein2();
			ksDistanceV = ecdf.ksDistance();
			kuiperDistanceV = ecdf.kuiperDistance();
			ADDistanceV = ecdf.AndersonDarling();
			DTSDistanceV = ecdf.DTS_twoSample();
			quadraticDistanceV = ecdf.quadraticDistance();
			unsignedAreaWeightedV = ecdf.unsignedAreaWeighted();
			signedAreaWeightedV = ecdf.signedAreaWeighted();
			squaredAreaWeightedV = ecdf.L2Weighted();
			
			unsignedAreas[i] = HelperFunctions.mean(unsignedAreaV);
			signedAreas[i] = HelperFunctions.mean(signedAreaV);
			squaredAreas[i] = HelperFunctions.mean(squaredAreaV);
			ksDistance[i] = HelperFunctions.mean(ksDistanceV);
			quadraticDistance[i] = HelperFunctions.mean(quadraticDistanceV);
			kuiperDistance[i] = HelperFunctions.mean(kuiperDistanceV);
			ADDistance[i] = HelperFunctions.mean(ADDistanceV);
			DTSDistance[i] = HelperFunctions.mean(DTSDistanceV);
			unsignedAreasSup[i] = HelperFunctions.max(unsignedAreaV);
			signedAreasSup[i] = HelperFunctions.max(signedAreaV);
			squaredAreasSup[i] = HelperFunctions.max(squaredAreaV);
			ksDistanceSup[i] = HelperFunctions.max(ksDistanceV);
			quadraticDistanceSup[i] = HelperFunctions.max(quadraticDistanceV);
			kuiperDistanceSup[i] = HelperFunctions.max(kuiperDistanceV);
			ADDistanceSup[i] = HelperFunctions.max(ADDistanceV);
			DTSDistanceSup[i] = HelperFunctions.max(DTSDistanceV);
			unsignedAreas2Norm[i] = HelperFunctions.twoNorm(unsignedAreaV);
			signedAreas2Norm[i] = HelperFunctions.twoNorm(signedAreaV);
			squaredAreas2Norm[i] = HelperFunctions.twoNorm(squaredAreaV);
			ksDistance2Norm[i] = HelperFunctions.twoNorm(ksDistanceV);
			quadraticDistance2Norm[i] = HelperFunctions.twoNorm(quadraticDistanceV);
			kuiperDistance2Norm[i] = HelperFunctions.twoNorm(kuiperDistanceV);
			ADDistance2Norm[i] = HelperFunctions.twoNorm(ADDistanceV);
			DTSDistance2Norm[i] = HelperFunctions.twoNorm(DTSDistanceV);
			unsignedAreasMin[i] = HelperFunctions.min(unsignedAreaV);
			signedAreasMin[i] = HelperFunctions.min(signedAreaV);
			squaredAreasMin[i] = HelperFunctions.min(squaredAreaV);
			ksDistanceMin[i] = HelperFunctions.min(ksDistanceV);
			quadraticDistanceMin[i] = HelperFunctions.min(quadraticDistanceV);
			kuiperDistanceMin[i] = HelperFunctions.min(kuiperDistanceV);
			ADDistanceMin[i] = HelperFunctions.min(ADDistanceV);
			DTSDistanceMin[i] = HelperFunctions.min(DTSDistanceV);
			
			unsignedAreasWeighted[i] = HelperFunctions.mean(unsignedAreaWeightedV);
			signedAreasWeighted[i] = HelperFunctions.mean(signedAreaWeightedV);
			squaredAreasWeighted[i] = HelperFunctions.mean(squaredAreaWeightedV);
			unsignedAreasWeightedSup[i] = HelperFunctions.max(unsignedAreaWeightedV);
			signedAreasWeightedSup[i] = HelperFunctions.max(signedAreaWeightedV);
			squaredAreasWeightedSup[i] = HelperFunctions.max(squaredAreaWeightedV);
			unsignedAreasWeighted2Norm[i] = HelperFunctions.twoNorm(unsignedAreaWeightedV);
			signedAreasWeighted2Norm[i] = HelperFunctions.twoNorm(signedAreaWeightedV);
			squaredAreasWeighted2Norm[i] = HelperFunctions.twoNorm(squaredAreaWeightedV);
			unsignedAreasWeightedMin[i] = HelperFunctions.min(unsignedAreaWeightedV);
			signedAreasWeightedMin[i] = HelperFunctions.min(signedAreaWeightedV);
			squaredAreasWeightedMin[i] = HelperFunctions.min(squaredAreaWeightedV);	
		}
		// Save the measures to the results
		results.registerMeasures(paramSet, unsignedAreas, signedAreas,
				squaredAreas, unsignedAreasWeighted, signedAreasWeighted, squaredAreasWeighted,
				ksDistance, quadraticDistance, kuiperDistance, ADDistance, DTSDistance,
				unsignedAreasSup, signedAreasSup,
				squaredAreasSup, unsignedAreasWeightedSup, signedAreasWeightedSup, squaredAreasWeightedSup,
				ksDistanceSup, quadraticDistanceSup, kuiperDistanceSup, ADDistanceSup, DTSDistanceSup,
				unsignedAreas2Norm, signedAreas2Norm,
				squaredAreas2Norm, unsignedAreasWeighted2Norm, signedAreasWeighted2Norm, squaredAreasWeighted2Norm,
				ksDistance2Norm, kuiperDistance2Norm, ADDistance2Norm, DTSDistance2Norm,
				quadraticDistanceSup, unsignedAreasMin, signedAreasMin,
				squaredAreasMin, unsignedAreasWeightedMin, signedAreasWeightedMin, squaredAreasWeightedMin,
				ksDistanceMin, quadraticDistanceMin,kuiperDistanceMin, ADDistanceMin, DTSDistanceMin);
		
	}
}