package DiscreteEventSimulation;

//import java.util.Arrays;
import java.util.Random;
import java.io.*;



import dist.Distribution;
import dist.ExponentialDistribution;
import dist.DiscreteUniformDistribution;
import dist.DeterministicDistribution;

public class SimulationWithBlocking {
	
	/* The service rates for each station used to generate the distributions 
	 * First dimension moves through the different stages of service rates 
	 * Second dimension gives the service rates for a particular experiment
	 * Rows are service rates
	 * Columns are the experiment count. i.e., row 1 holds the storeRate for each experiment*/
	protected double[][] serviceRates;
	/* For each array element 0 is store, 1 is MT, 2 is fusion, 3 is membrane
	 * This is an array of length 4. Each element corresponds to a station number */
	protected Distribution[][] serviceTimeDist;
	/* The number of servers at each station */
	protected int[] numServers;
	/* The station capacity for each station
	 * This is equivalent to the length in the case of the Microtubules */
	protected int[] capacity;
	/* The number of customers in the system */
	protected int numCustomer;
	/* The number of microtubules / fusion sites in the network */
	protected int numMicrotubule;
	/* The times to switch the insulin level 
	 * At least one other switch time must correspond to when the insulin change occurs */
	protected int[] expSwitchTimes;
	/* Proportion of customers to begin in each station, must sum to 1*/
	//protected int[] startProportions;
	/* Array containing the active probabilities to be executed
	 * the first active probability must be at t = 0 and is the initial conditiion */
	protected double[] activeProbabilities;
	/* The times at which the active probabilities change
	 * Must be the same size as activeProbabilities
	 * First element must be 0
	 * All other elements must match an expSwitchTimes 
	 * Can also be an inhibitor event, just set changeType to false to close fusion sites 
	 * */
	protected int[] insulinTimes;
	/* Times to putput results */
	protected double[] outputTimes;
	// Time of stimulation events, can be insulin or inhibitor. Changes stimulation of fusion sites
	protected int[] stimulationTimes;
	/* Determines whether up and down events will interrupt service of customers at fusion sites
	 * true if interrupting
	 * false if customers permitted to finish service
	 */
	protected boolean interrupting;
	/* A boolean for whether to gather and output entire traces of data
	 * true indicates all data will be output, false only outputs data at the output times
	 * Only use true for runs of single parameter values, not for when fitting	 */
	protected boolean completeOutputs;
	/* A boolean for whether to gather and output microtubule stores and fusion events at outputTimes
	 * true indicates all data will be output, false only outputs data at the output times
	 * Only use true for runs of single parameter values, not for when fitting	 */
	protected boolean microtubuleOutputs;
	/* An array of the same size as activePRobabilites and stimulationTimes / insulinTimes
	 * Each element of the array dictates whether the associated event will increase or decrease fusion site activity
	 * False decreases activity, True increases activity */
	protected boolean[] eventChangeType;
	// Mean up times, first element of array is stimulated, second unstimulated
	protected double[] meanUpTimes;
	// Mean up times, first element of array is stimulated, second unstimulated
	protected double[] meanDownTimes;
	/* A uniform distribution to select which MT to go to from store */
	protected Distribution selectMicrotubuleDist;
	/* Distribution for fusion sites to become active after an INSULIN_CHANGE event */
	protected Distribution activateDistribution;
	/* Distribution for up time stimulated fusion sites */
	protected Distribution upStimulatedDistribution;
	/* Distribution for up time unstimulated fusion sites */
	protected Distribution upUnstimulatedDistribution;
	/* Distribution for down time stimulated fusion sites */
	protected Distribution downStimulatedDistribution;
	/* Distribution for down time unstimulated fusion sites */
	protected Distribution downUnstimulatedDistribution;
	
	
	/* Elements relating to each station */
	public static final int STORE = 0;
	public static final int MICROTUBULE = 1;
	public static final int FUSION = 2;
	public static final int MEMBRANE = 3;
	public static final int INCREMENT = 4; // only used to access the service distribution for increment time
	
	
	public SimulationWithBlocking() {
		
	}
	
	public SimulationWithBlocking(int numCustomers, double[][] serviceRates, int numMicrotubule, int[] servers, int[] capacity, double[] activeProbabilities,
			int[] insulinTimes, int[] expSwitchTimes, double[] outputTimes, Random rng, double meanActivationTime, int[] stimulationTimes,
			boolean[] eventChangeType, double[] meanUpTimes, double[] meanDownTimes, boolean interrupting, boolean microtubuleOutputs, boolean completeOutputs) {
		this.serviceRates = serviceRates;
		this.numServers = servers;
		this.capacity = capacity;
		this.numMicrotubule = numMicrotubule;
		this.activeProbabilities = activeProbabilities;
		this.insulinTimes = insulinTimes;
		this.expSwitchTimes = expSwitchTimes;
		this.numCustomer = numCustomers;
		this.outputTimes = outputTimes;
		this.stimulationTimes = stimulationTimes;
		this.eventChangeType = eventChangeType;
		//this.activateDistribution = null;
		this.meanUpTimes = meanUpTimes;
		this.meanDownTimes = meanDownTimes;
		this.interrupting = interrupting;
		this.microtubuleOutputs = microtubuleOutputs;
		this.completeOutputs = completeOutputs;
		if (meanActivationTime > 0) {
			this.activateDistribution = new ExponentialDistribution(1.0 / meanActivationTime, rng);
		} else {
			this.activateDistribution = new DeterministicDistribution(0);
		}
		
		// Assumes that all rows have the same number of elements
		this.serviceTimeDist = new Distribution[5][serviceRates[0].length];
		// Initialise an exponential distribution for each station service time
		// Loop through rows, the service rate for different stations
		for (int i = 0; i < serviceRates.length; i++) {
			// loop through rates for each experiment
			for (int j = 0; j < serviceRates[i].length; j++) {
				this.serviceTimeDist[i][j] = new ExponentialDistribution(serviceRates[i][j], rng);
			}
		}
		// Exponential distribution for increment time
		// Rate is the MT rate * the number of MT servers
		for (int i = 0; i < serviceRates[MICROTUBULE].length; i++) {
			this.serviceTimeDist[INCREMENT][i] = new ExponentialDistribution(serviceRates[MICROTUBULE][i]*numServers[MICROTUBULE], rng);
		}
		
		// A uniform distribution to select which MT to travel to
		this.selectMicrotubuleDist = new DiscreteUniformDistribution(0, numMicrotubule - 1, rng); // - 1 to give the indice of MT element
		
		// Distributions for up and down times
		this.upStimulatedDistribution = new ExponentialDistribution(1.0 / meanUpTimes[0], rng);
		
	}
	
	public static SimulationWithBlocking createInstance(int numCustomers, double[][] serviceRates, int numMicrotubule, int[] servers, int[] capacity, 
			double[] activeProbabilities, int[] insulinTimes, int[] expSwitchTimes, double[] outputTimes, double meanActivationTime, 
			int[] stimulationTimes,boolean[] eventChangeType, double[] meanUpTimes, double[] meanDownTimes, 
			boolean interrupting, boolean microtubuleOutputs, boolean completeOutputs) {
		return new SimulationWithBlocking(numCustomers, serviceRates, numMicrotubule, servers, capacity, activeProbabilities, 
				insulinTimes, expSwitchTimes, outputTimes, new Random(), meanActivationTime, stimulationTimes, eventChangeType, meanUpTimes,
				meanDownTimes, interrupting, microtubuleOutputs, completeOutputs);
	}
	
	public SimResults[] simulate(int maxTime) {
		
		Customer.resetNetworkState();
		// The number of stations
		// int numStations = serviceTimeDist.length;
		// create a queue for each of the microtubules
		// Create a queue for each fusion site - need to know if active or not
		Queue[] microtubule = new Queue[numMicrotubule];
		Queue[] fusion = new Queue[numMicrotubule];
		for (int i = 0; i < numMicrotubule; i++) {
			microtubule[i] = new Queue(numServers[MICROTUBULE], capacity[MICROTUBULE]);
			fusion[i] = new Queue(numServers[FUSION], capacity[FUSION]);
		}

		
		
		// Create an array of results, each element of the array holds the results for a different experiment type
		SimResults[] results = new SimResults[expSwitchTimes.length + 1];
		results[0] = new SimResults(0, numMicrotubule);
		for (int i = 1; i <=expSwitchTimes.length; i++) {
			results[i] = new SimResults(expSwitchTimes[i-1], numMicrotubule);
		}
		
		
		/* The proportions is a good idea to speed up the settling time for repeated simulations
		 * However need to think about implementation more. Not sure how to assign them to a substation
		 
		// Create an array of the customers in the system
		// Place customers proportionally in each station
		Customer[] customer = new Customer[numCustomer];
		int initProportion = 0;
		int endProportion = 0;
		for (int i = 0; i < startProportions.length; i++) {
			endProportion +=  numCustomer*startProportions[i]; // End the assignment at the previous number of customer proporiton + next proportion
			for (int j = initProportion; j < endProportion; j++) {
				customer[j] = new Customer(j, i, serviceTimeDist[i].nextRandom()); // serviceTimeDist[i] depending on station
			}
			initProportion = (int) Math.ceil(endProportion); // next customer to assign is the next integer after the last iteration stopped
		}
		
		// Generate a departure event for each customer in their respective starting station
		FES eventq = new FES();
		for (int i = 0; i < numCustomer; i++) {
			eventq.addEvent(new Event(Event.DEPARTURE, serviceTimeDist[customer[i].station].nextRandom(), customer[i]));
		}
		*/
		
		/* Create an array of customers
		 * Place each customer in the store 
		 * Generate a departure event for each customer originally in the store */
		Customer[] customer = new Customer[numCustomer];
		FES eventq = new FES();
		for (int i = 0; i < numCustomer; i++) {
			customer[i] = new Customer(i, STORE, serviceTimeDist[STORE][Event.experimentCount].nextRandom()); // serviceTimeDist[0] as initially placed in store
			eventq.addEvent(new Event(Event.DEPARTURESTORE, customer[i].getServiceTime(), customer[i]));
		}
		
		// Create the output time events
		/*
		int expSpacing = expSwitchTimes[0];
		for (int i = 0; i <= expSwitchTimes.length; i++) {
			for (int j = 0; j < outputTimes.length; j++) {
				eventq.addEvent(new Event(Event.DATATIME, outputTimes[j] + (i * expSpacing)));
			}
		}
		*/
		
		for (int j = 0; j < outputTimes.length; j++) {
			eventq.addEvent(new Event(Event.DATATIME, outputTimes[j]));
		}

		// Schedule the INSULINCHANGE events
		if (insulinTimes != null) {
			for (int i = 0; i < insulinTimes.length; i++) {
				eventq.addEvent(new Event(Event.INSULINCHANGE, insulinTimes[i], activeProbabilities[i], eventChangeType[i]));
			}
		}
		// Schedule the EXPCHANGE events
		for (int i = 0; i < expSwitchTimes.length; i++) {
			//System.out.println(expSwitchTimes[i]);
			eventq.addEvent(new Event(Event.EXPCHANGE, expSwitchTimes[i]));
		}

		// Schedule the STIMULATIONCHANGE events
		if (stimulationTimes != null) {
			for (int i = 0; i < stimulationTimes.length; i++) {
				eventq.addEvent(new Event(Event.STIMULATIONCHANGE, stimulationTimes[i], activeProbabilities[i], eventChangeType[i]));
			}
		}

		
		/* Initialising the start time for the simulation */
		double t = 0;
		// Log the initial state of the system
		//results[Event.experimentCount].registerQueueLengths
		//(t, Customer.networkState[STORE], Customer.networkState[MICROTUBULE], Customer.networkState[FUSION], Customer.networkState[MEMBRANE]);
		// Begin the actual simulation
		while(t < maxTime) {
			Event e = eventq.nextEvent();
			
			// If the event is not valid (it has been interrupted) skip this iteration and continue
			if (e.isValid() == false) {
				continue;
			}
			
			t = e.getTime();
			//System.out.println("Current Time: "+ t);
			//System.out.println("Event type: "+ e.getType());
			/*
			results[Event.experimentCount].registerQueueLengths
				(t, Customer.networkState[STORE], Customer.networkState[MICROTUBULE], Customer.networkState[FUSION], Customer.networkState[MEMBRANE]);
			*/
			// Event is Departure from STORE
			if (e.type == Event.DEPARTURESTORE) {
				Customer c = e.getCustomer();
				//int cID = c.getID();
				int nextSubstation = (int) selectMicrotubuleDist.nextRandom(); // select the next substation
				// Check if the chosen MT has capacity
				if (microtubule[nextSubstation].checkAvailability()) {
					// MT has capacity so move customer to MT
					c.moveTo(MICROTUBULE, nextSubstation, t, serviceTimeDist[MICROTUBULE][Event.experimentCount].nextRandom());
					eventq.addEvent(new Event(Event.ARRIVALMICROTUBULE, t, c));
				} else {
					c.redrawService(serviceTimeDist[STORE][Event.experimentCount].nextRandom());
					eventq.addEvent(new Event(Event.DEPARTURESTORE, t + c.getServiceTime(), c));
				}
			} else if (e.type == Event.DEPARTUREMICROTUBULE) {
				Customer c = e.getCustomer();
				//int cID = c.getID();
				int currSubstation = c.getSubStation();
				// First check if the customer is at the front of MT queue
				// System.out.println(microtubule[currSubstation].checkQueueFront(c));
				if (microtubule[currSubstation].checkQueueFront(c)) {
					// Check if the corresponding fusion site has capacity and active
					// System.out.println(fusion[currSubstation].checkAvailability() && fusion[currSubstation].checkActive());
					if (fusion[currSubstation].checkAvailability() && fusion[currSubstation].checkActive()) {
						// Fusion site has capacity so remove customer from MT and move
						microtubule[currSubstation].removeFirstCustomer();
						c.moveTo(FUSION, currSubstation, t, serviceTimeDist[FUSION][Event.experimentCount].nextRandom()); // fusion substation must match the MT substation
						eventq.addEvent(new Event(Event.ARRIVALFUSION, t, c));
						// Check if the new front customer of MT is blocked
						if (microtubule[currSubstation].checkFrontBlockage()) {
		                    // Unblock and schedule a departure after Increment time
		                    Customer frontCustomer = microtubule[currSubstation].getFirstCustomer();
		                    frontCustomer.unblock(serviceTimeDist[INCREMENT][Event.experimentCount].nextRandom()); // customer must get the INCREMENT service time
		                    // Schedule departure event
		                    eventq.addEvent(new Event(Event.DEPARTUREMICROTUBULE, t + frontCustomer.getServiceTime(), frontCustomer));
						}
					} else {
						// Fusion site is unavailable so block the customer
						microtubule[currSubstation].blockCustomer(c);
						c.block();
					}
				} else {
					// Customer is not at front of MT and is blocked
					microtubule[currSubstation].blockCustomer(c);
					c.block();
				}
			} else if (e.type == Event.DEPARTUREFUSION) {
				Customer c = e.getCustomer();
				int cID = c.getID();
				int currSubstation = c.getSubStation();
				// Always availability at membrane so move the customer
				customer[cID].moveTo(MEMBRANE, currSubstation, t, serviceTimeDist[MEMBRANE][Event.experimentCount].nextRandom()); // Substation doesn't matter so just leave as current value
				fusion[currSubstation].removeFirstCustomer();
				eventq.addEvent(new Event(Event.ARRIVALMEMBRANE, t, customer[cID]));
				// Remove the departure event associated to fusion site
				fusion[currSubstation].unassignDepartureEvent();
				// Check if there is a blocked customer at the start of the MT
				if (microtubule[currSubstation].checkFrontBlockage()) {
					// Unblock and schedule an immediate departure event
					Customer frontCustomer = microtubule[currSubstation].removeFirstCustomer();
					customer[frontCustomer.getID()].unblock(0); // zero service time as immediate departure from microtubule
					microtubule[currSubstation].addFrontCustomer(customer[frontCustomer.getID()]); // add the customer back to the start of the MT
					// Schedule an immediate departure event
					eventq.addEvent(new Event(Event.DEPARTUREMICROTUBULE, t + customer[frontCustomer.getID()].getServiceTime(), customer[frontCustomer.getID()]));
				}
			} else if (e.type == Event.DEPARTUREMEMBRANE) {
				Customer c = e.getCustomer();
				int cID = c.getID();
				// Always availability in the store so move customer to store
				customer[cID].moveTo(STORE, 0, t, serviceTimeDist[STORE][Event.experimentCount].nextRandom());
				eventq.addEvent(new Event(Event.ARRIVALSTORE, t, customer[cID]));
			} else if (e.type == Event.ARRIVALSTORE) {
				Customer c = e.getCustomer();
				// Log results
				//results[Event.experimentCount].registerQueueLengths
					//(t, Customer.networkState[STORE], Customer.networkState[MICROTUBULE], Customer.networkState[FUSION], Customer.networkState[MEMBRANE]);
				// Schedule a departure from the current station
				eventq.addEvent(new Event(Event.DEPARTURESTORE, t + c.getServiceTime(), c));
			} else if (e.type == Event.ARRIVALMICROTUBULE) {
				Customer c = e.getCustomer();
				// Log results
				//results[Event.experimentCount].registerQueueLengths
					//(t, Customer.networkState[STORE], Customer.networkState[MICROTUBULE], Customer.networkState[FUSION], Customer.networkState[MEMBRANE]);
				// Add customer to microtubule queue
				microtubule[c.getSubStation()].addCustomer(c);
				// Schedule a departure from the current station
				eventq.addEvent(new Event(Event.DEPARTUREMICROTUBULE, t + c.getServiceTime(), c));
			} else if (e.type == Event.ARRIVALFUSION) {
				Customer c = e.getCustomer();
				// Log results
				//results[Event.experimentCount].registerQueueLengths
					//(t, Customer.networkState[STORE], Customer.networkState[MICROTUBULE], Customer.networkState[FUSION], Customer.networkState[MEMBRANE]);
				// Get the current fusion site number
				int currSite = c.getSubStation();
				// Add customer to fusion queue
				fusion[currSite].addCustomer(c);
				// Create a departure event
				Event currEvent = new Event(Event.DEPARTUREFUSION, t + c.getServiceTime(), c);
				// Schedule a departure from the current station
				eventq.addEvent(currEvent);
				// Assign the departure event to the fusion site
				fusion[currSite].assignDepartureEvent(currEvent);
			} else if (e.type == Event.ARRIVALMEMBRANE) {
				Customer c = e.getCustomer();
				// Log results
				//results[Event.experimentCount].registerQueueLengths
					//(t, Customer.networkState[STORE], Customer.networkState[MICROTUBULE], Customer.networkState[FUSION], Customer.networkState[MEMBRANE]);
				// Schedule a departure from the current station
				eventq.addEvent(new Event(Event.DEPARTUREMEMBRANE, t + c.getServiceTime(), c));
			} else if (e.type == Event.INSULINCHANGE) {
				double ap = e.getActiveProbability();
				// for each fusion site draw the active probability
				// Customers currently in service are allowed to finish
				for (int i = 0; i < fusion.length; i++) {
					double activationTime = activateDistribution.nextRandom();
					// change type true so create activation events
					if (e.getChangeType() == true) { 
						eventq.addEvent(new Event(Event.ACTIVATE, t + activationTime, ap, i, false));
					} else { // change type is false and create closure events, these interrupt service
						eventq.addEvent(new Event(Event.CLOSEFUSION, t + activationTime, ap, i, interrupting));
					}
					
				}
			} else if (e.type == Event.ACTIVATE) {
				double ap = e.getActiveProbability();
				int i = e.getFusionSite();
				// Store the old active state
				boolean oldState = fusion[i].checkActive();
				// Change the state based on active probabilities
				if (!oldState) {
					fusion[i].drawActive(ap);
				}
				// Store the new state
				boolean newState = fusion[i].checkActive();
				// If the site changed from inactive to active then must unblock the MT
				if (oldState == false && newState == true) {
					//System.out.println("Activating "+i+ " at time "+t);
					// Check if there is a blocked customer at the start of the MT
					if (microtubule[i].checkFrontBlockage()) {
						// Unblock and schedule an immediate departure event
						Customer frontCustomer = microtubule[i].removeFirstCustomer();
						customer[frontCustomer.getID()].unblock(0); // zero service time as immediate departure from microtubule
						microtubule[i].addFrontCustomer(customer[frontCustomer.getID()]); // add the customer back to the start of the MT
						// Schedule an immediate departure event
						eventq.addEvent(new Event(Event.DEPARTUREMICROTUBULE, t + customer[frontCustomer.getID()].getServiceTime(), customer[frontCustomer.getID()]));
					}
				}
				// System.out.println("Active: "+ fusion[i].checkActive());
			} else if (e.type == Event.EXPCHANGE) {
				e.updateExperimentCount();
				/*
				if (t == 1000) {
					System.out.println("EXPCHANGE");
				}
				*/
				//System.out.println(t);
				//System.out.println(Event.experimentCount);
				// reset the count of visits to each station
				for (int i = 0; i < 4; i ++) {
					Customer.allVisits[i] = 0;
				}
				// Reset the internalisation count
				Customer.internalisationCount = 0;
				// Update the count of allVisits to be the current number of customers in each station
				// loop through each customer and see where they are and reset all the visits
				// This also updates the internalisation Markers of customers and internalisation count
				for (int i = 0; i < customer.length; i++) {
					customer[i].resetStationVisits();
				}
				
			} else if (e.type == Event.CLOSEFUSION) {
				double ap = e.getActiveProbability();
				int i = e.getFusionSite();
				// Store the old active state
				boolean oldState = fusion[i].checkActive();
				// If oldState is true -> fusion site is previously active. Draw probability of closure
				if (oldState) {
					fusion[i].drawActive(ap);
				}
				// If the site changed from active to inactive and the event is interrupting, then interrupt current departure event
				if (oldState == true && fusion[i].checkActive() == false && e.isInterrupting()) {
					fusion[i].getAssignedDepartureEvent().makeInvalid();
				}
			} else if (e.type == Event.FUSIONUP) { // fusion site starts up time
				int currFusionSite = e.getFusionSite();
				fusion[currFusionSite].activate();
				// Check if the fusion site is currently empty. If empty check for blockage at microtubule.
				// If not empty, check if current departure event is valid, if not valid, create a new departure event
				if (fusion[currFusionSite].checkAvailability() == false) {
					if (fusion[currFusionSite].getAssignedDepartureEvent().isValid() == false) {
						// Customer to create event for
						Customer c = fusion[currFusionSite].getFirstCustomer();
						// Create a departure event
						Event currEvent = new Event(Event.DEPARTUREFUSION, t + c.getServiceTime(), c);
						// Schedule a departure from the current station
						eventq.addEvent(currEvent);
						// Assign the departure event to the fusion site
						fusion[currFusionSite].assignDepartureEvent(currEvent);
					}
				} else { //fusion[currFusionSite].checkAvailability() == true
					microtubule[currFusionSite].getFirstCustomer().unblock(0);
					// Schedule an immediate departure event
					eventq.addEvent(new Event(Event.DEPARTUREMICROTUBULE, t + microtubule[currFusionSite].getFirstCustomer().getServiceTime(), microtubule[currFusionSite].getFirstCustomer()));
				}
			
				// Schedule next down time
				Event nextSwitchEvent;
				if (fusion[currFusionSite].stimulated == true) {
					nextSwitchEvent = new Event(Event.FUSIONDOWN, t + upStimulatedDistribution.nextRandom(), currFusionSite, interrupting);
				} else {
					nextSwitchEvent = new Event(Event.FUSIONDOWN, t + upUnstimulatedDistribution.nextRandom(), currFusionSite, interrupting);
				}
				// Add event to FES
				eventq.addEvent(nextSwitchEvent);
				// Assign the switch event to the fusion site
				fusion[currFusionSite].assignSwitchEvent(nextSwitchEvent);
			} else if (e.type == Event.FUSIONDOWN) {
				int currFusionSite = e.getFusionSite();
				fusion[currFusionSite].deactivate();
				// If interrupting, interrupt the departure of the next vesicle from fusion site
				if (e.isInterrupting() && fusion[currFusionSite].getAssignedDepartureEvent() != null) {
					fusion[currFusionSite].getAssignedDepartureEvent().makeInvalid();
				}
				// Schedule next up time
				Event nextUpTime;
				if (fusion[currFusionSite].stimulated == true) {
					nextUpTime = new Event(Event.FUSIONUP, t + downStimulatedDistribution.nextRandom(), currFusionSite, interrupting);
				} else {
					nextUpTime = new Event(Event.FUSIONUP, t + downUnstimulatedDistribution.nextRandom(), currFusionSite, interrupting);
				}
				// Add event to FES
				eventq.addEvent(nextUpTime);
				// Assign event to fusion site
				fusion[currFusionSite].assignSwitchEvent(nextUpTime);
			} else if (e.type == Event.STIMULATIONCHANGE) {
				double ap = e.getActiveProbability();
				// for each fusion site draw the active probability
				// Customers currently in service are allowed to finish
				for (int i = 0; i < fusion.length; i++) {
					double activationTime = activateDistribution.nextRandom();
					// change type true so create stimulation events.
					// Stimulation changes must be interrupting. They interrupt the next up or down time event
					if (e.getChangeType() == true) { 
						eventq.addEvent(new Event(Event.STIMULATEUP, t + activationTime, ap, i, true));
					} else { // change type is false and create unstimulate events (inhibitors)
						eventq.addEvent(new Event(Event.STIMULATEDOWN, t + activationTime, ap, i, true));
					}
					
				}
			} else if (e.type == Event.STIMULATEUP) {
				double ap = e.getActiveProbability();
				int i = e.getFusionSite();
				// Store the old active state
				boolean oldState = fusion[i].checkStimulated();
				// Change the state based on active probabilities
				if (!oldState) {
					fusion[i].drawStimulated(ap);
				}
				// Store the new state
				boolean newState = fusion[i].checkStimulated();
				// If site had become stimulated must interrupt the current up / down event and assign new event with new times
				if (oldState == false && newState == true && e.isInterrupting() == true) {
					// Replace the switch event
					if (fusion[i].getAssignedSwitchEvent() != null) {
						fusion[i].getAssignedSwitchEvent().makeInvalid();
					}
					// The interrupted event is a fusion up event
					Event replacementEvent;
					if (fusion[i].getAssignedSwitchEvent().type == Event.FUSIONUP) {
						// create new event for the next up time with stimulated state
						replacementEvent = new Event(Event.FUSIONUP, t + downStimulatedDistribution.nextRandom(), i, interrupting);
					} else { // if the interrupted event is a fusion down event
						replacementEvent = new Event(Event.FUSIONDOWN, t + upStimulatedDistribution.nextRandom(), i, interrupting);
					}
					// Add replacement event to FES
					eventq.addEvent(replacementEvent);
					// Assign replacement event to the fusion site
					fusion[i].assignSwitchEvent(replacementEvent);
				}
			} else if (e.type == Event.STIMULATEDOWN) {
				double ap = e.getActiveProbability();
				int i = e.getFusionSite();
				// Store the old active state
				boolean oldState = fusion[i].checkStimulated();
				// Change the state based on active probabilities
				// Only draw ap for previously stimulated sites to be come unstimulated
				if (oldState) {
					fusion[i].drawStimulated(ap);
				}
				// Store the new state
				boolean newState = fusion[i].checkStimulated();
				// If site had become unstimulated must interrupt the current up / down event and assign new event with new times
				if (oldState == true && newState == false && e.isInterrupting() == true) {
					// Replace the switch event
					fusion[i].getAssignedSwitchEvent().makeInvalid();
					// The interrupted event is a fusion up event
					Event replacementEvent;
					if (fusion[i].getAssignedSwitchEvent().type == Event.FUSIONUP) {
						// create new event for the next up time with stimulated state
						replacementEvent = new Event(Event.FUSIONUP, t + downUnstimulatedDistribution.nextRandom(), i, interrupting);
					} else { // if the interrupted event is a fusion down event
						replacementEvent = new Event(Event.FUSIONDOWN, t + upUnstimulatedDistribution.nextRandom(), i, interrupting);
					}
					// Add replacement event to FES
					eventq.addEvent(replacementEvent);
					// Assign replacement event to the fusion site
					fusion[i].assignSwitchEvent(replacementEvent);
				}
				
				// Log results
				//results[Event.experimentCount].registerQueueLengths
					//(t, Customer.networkState[STORE], Customer.networkState[MICROTUBULE], Customer.networkState[FUSION], Customer.networkState[MEMBRANE]);
			} else if (e.type == Event.DATATIME) {
				// Log results
				/*
				if (t == 1000) {
					System.out.println("Log time");
				}
				*/
				//System.out.println(t);
				results[Event.experimentCount].registerQueueLengths
					(t, Customer.networkState[STORE], Customer.networkState[MICROTUBULE], Customer.networkState[FUSION], Customer.networkState[MEMBRANE], Customer.allVisits[MEMBRANE], Customer.internalisationCount);
				// Register microtubule and fusion data if microtubuleOutput == true
				if (microtubuleOutputs) {
					results[Event.experimentCount].registerFusionEvents(Customer.fusionCount, microtubule);
				}
			}
			
			// If taking complete outputs, register updates to the system
			if (completeOutputs) {
				// If an arrival has occurred, log the queue lengths
				if (e.type == Event.ARRIVALSTORE || e.type == Event.ARRIVALMICROTUBULE 
						|| e.type == Event.ARRIVALFUSION || e.type == Event.ARRIVALMEMBRANE) {
					int station = e.getCustomer().getStation();
					results[Event.experimentCount].registerTraceQueueLengths(station, t);
				}
				// If arrival or attempted departure from MT, log microtubule states
				if (e.type == Event.ARRIVALMICROTUBULE || e.type == Event.DEPARTUREMICROTUBULE) {
					results[Event.experimentCount].registerMicrotubuleCustomerStatus(t, customer, fusion);
				}
			}
		}
		
		return results;
	}
	

	
	public static void main(String[] arg) {
		
		/* Input Order:
		 * storeRate, microtubuleRate, fusionRate, membraneRate, activeProb1, activeProb2, numCustomers, 
		 * numMicrotubules, microtubuleLength, numberOfRepeats, eventChangeType
		 * Need to find a way to alter the number of active probabilities input. For now it must take two and only two ap
		 */
		
		int numCustomer;
		int[] numServers;
		double storeRate;
		double microtubuleRate;
		double fusionRate;
		double membraneRate;
		double[][] serviceRates;
		int numMicrotubule;
		int microtubuleLength;
		int[] capacity;
		int[] expSwitchTimes;
		int maxTime;
		double[] finalActiveProbabilities ;
		double[] activeProbabilities;
		int[] insulinTimes;
		double meanActivationTime;
		int numberOfRepeats;
		int[] stimulationTimes;
		boolean[] eventChangeType;
		double[] meanUpTimes;
		double[] meanDownTimes;
		boolean interrupting;
		boolean microtubuleOutputs;
		boolean completeOutputs;
		
		
		//double[] outputTimes = {0, 0.5, 1, 2, 5, 10, 15, 20, 25, 30, 45, 60, 90, 120, 180, 240, 300};

		// Set some default values if no input given
		if (arg.length == 0) {
			storeRate = 0.12875d; // service rate of the store
			microtubuleRate = 0.3d; // service rate (travel rate) of microtubules
			fusionRate = 0.1d; // fusion site service rate
			membraneRate = 0.37625d; // membrane service rate
			serviceRates = new double[][] {{storeRate,storeRate,storeRate,storeRate},
				{microtubuleRate,microtubuleRate,microtubuleRate,microtubuleRate},
				{fusionRate,fusionRate,fusionRate,fusionRate},
				{membraneRate,membraneRate,membraneRate,membraneRate}};
			numMicrotubule = 200; // total number of microtubules
			numCustomer = 25000; // total customers in system
			microtubuleLength = 50; // the length (capacity) of microtubules
			numServers = new int[] {numCustomer + 1, microtubuleLength, 1, numCustomer + 1};
			capacity = numServers; 
			expSwitchTimes = new int[] {500, 801, 1300}; // Times to switch between experiments / data recording
			maxTime = 1601; // maxTime of all experiments
			finalActiveProbabilities = new double[] {0.1d, 0.9d}; // active probabilities for fusion sites
			// Careful, the formula below ONLY works for 2 sets 
			activeProbabilities = new double[]
				{finalActiveProbabilities[0], 1 - (1 - finalActiveProbabilities[1])/(1-finalActiveProbabilities[0])};
			
			//insulinTimes = new int[] {0, 801}; // times for active probabilities to be applied
			meanActivationTime = 0;  // mean activation time after an INSULIN_CHANGE event
			//numberOfRepeats = 1;
			numberOfRepeats = 1;
			stimulationTimes = new int[] {};
			eventChangeType = new boolean[] {true, true};
			meanUpTimes = new double[] {0d, 0d};
			meanDownTimes = new double[] {0d, 0d};
			interrupting = false;
			microtubuleOutputs = true;
			completeOutputs = true;
		} else {
			storeRate = Double.parseDouble(arg[0]); // service rate of the store
			microtubuleRate = Double.parseDouble(arg[1]); // service rate (travel rate) of microtubules
			fusionRate = Double.parseDouble(arg[2]); // fusion site service rate
			membraneRate = Double.parseDouble(arg[3]); // membrane service rate
			serviceRates = new double[][] {{storeRate,storeRate,storeRate,storeRate},
				{microtubuleRate,microtubuleRate,microtubuleRate,microtubuleRate},
				{fusionRate,fusionRate,fusionRate,fusionRate},
				{membraneRate,membraneRate,membraneRate,membraneRate}};
			numMicrotubule = Integer.parseInt(arg[7]);; // total number of microtubules
			numCustomer = Integer.parseInt(arg[6]); // total customers in system
			microtubuleLength = Integer.parseInt(arg[8]);; // the length (capacity) of microtubules
			numServers = new int[] {numCustomer + 1, microtubuleLength, 1, numCustomer + 1};
			capacity = numServers; 
			expSwitchTimes = new int[] {500, 801, 1500}; // Times to switch between experiments / data recording
			maxTime = 2000; // maxTime of all experiments
			finalActiveProbabilities = new double[] {Double.parseDouble(arg[4]), Double.parseDouble(arg[5])}; // active probabilities for fusion sites
			insulinTimes = new int[] {0, 1000}; // times for active probabilities to be applied
			meanActivationTime = Double.parseDouble(arg[9]); // mean activation time after an INSULIN_CHANGE event
			numberOfRepeats = Integer.parseInt(arg[10]);
			stimulationTimes = new int[0];
			eventChangeType = new boolean[] {Boolean.parseBoolean(arg[11]), Boolean.parseBoolean(arg[12])};
			meanUpTimes = new double[] {Double.parseDouble(arg[13]), Double.parseDouble(arg[14])};
			meanDownTimes = new double[] {Double.parseDouble(arg[15]), Double.parseDouble(arg[16])};
			interrupting = Boolean.parseBoolean(arg[17]);
			completeOutputs = Boolean.parseBoolean(arg[18]);
			microtubuleOutputs = Boolean.parseBoolean(arg[19]);
		}
		
		//expSwitchTimes = new int[] {500, 1000, 1500}; // Times to switch between experiments / data recording
		//maxTime = 2000; // maxTime of all experiments
		insulinTimes = new int[] {0, expSwitchTimes[1]}; // times for active probabilities to be applied
		// Combined
		//double[] outputTimes = {0, 0.5, 1, 2, 5, 10, 15, 20, 25, 30, 45, 60, 90, 120, 180, 240, 300, 360, 420, 480};
		// Per Experiment
		int basalStartTime = expSwitchTimes[0];
		int transitionStartTime = expSwitchTimes[1];
		int insulinStartTime = expSwitchTimes[2];
		double[] outputTimes = {basalStartTime+0, basalStartTime+2, basalStartTime+5, basalStartTime+10, 
				basalStartTime+20, basalStartTime+30, basalStartTime+60, basalStartTime+90,
				basalStartTime+120, basalStartTime+180, basalStartTime+240, basalStartTime+300, // Basal Uptake
				transitionStartTime+0, transitionStartTime+0.5, transitionStartTime+1,
				transitionStartTime+2, transitionStartTime+5, transitionStartTime+10,
				transitionStartTime+15, transitionStartTime+20, transitionStartTime+25,
				transitionStartTime+30, transitionStartTime+45, transitionStartTime+60, //Transition
				insulinStartTime+0, insulinStartTime+2, insulinStartTime+5, insulinStartTime+10, 
				insulinStartTime+20, insulinStartTime+30, insulinStartTime+60, insulinStartTime+90,
				insulinStartTime+120, insulinStartTime+180, insulinStartTime+240, insulinStartTime+300}; // insulin
		
		Random rng = new Random();
		// Careful, the formula below ONLY works for 2 sets 
		activeProbabilities = new double[]
			{finalActiveProbabilities[0], 1 - (1 - finalActiveProbabilities[1])/(1-finalActiveProbabilities[0])};
		
		
		/*
	    System.out.println("Simulation time: "+dt+" seconds.");
	    
	    System.out.println("Basal Steady state");
	    System.out.println("STORE: "+results[1].getLastValue(STORE));
	    System.out.println("MT: "+results[1].getLastValue(MICROTUBULE));
	    System.out.println("FUSION: "+results[1].getLastValue(FUSION));
	    System.out.println("PM: "+results[1].getLastValue(MEMBRANE));
	    
	    System.out.println("Insulin Steady state");
	    System.out.println("STORE: "+results[3].getLastValue(STORE));
	    System.out.println("MT: "+results[3].getLastValue(MICROTUBULE));
	    System.out.println("FUSION: "+results[3].getLastValue(FUSION));
	    System.out.println("PM: "+results[3].getLastValue(MEMBRANE));	    
	    */
		
	    /*
	    System.out.println("Test of Transition Array");
	    System.out.println(Arrays.deepToString(results[2].getArrangedArray(MEMBRANE, 1)));
	    
	    System.out.println("Length of Array Should be " + outputTimes.length +", it actually is:");
	    System.out.println(results[2].getArrangedArray(MEMBRANE, 1).length);
	    */
	    
	    // Try to write some results to a file
	    String saveFile = "GroundTruth_Set1.csv";
	    String filePath = "../Simulation Outputs/";
	    String fileName = filePath + saveFile;
	    
	    //t1 = System.currentTimeMillis();
	    File csvOutputFile = new File(fileName);
	    // Print headers
	    String headers = "storeRate,microtubuleRate,fusionRate,membraneRate,"
	    		+ "numCustomer,numMicrotubule,microtubuleLength,activeProbability1,"
	    		+ "activeProbability2,repeat,experimentName,time,storeQueue,microtubuleQueue,fusionQueue,membraneQueue,membraneVisits";
	    try (PrintWriter pw = new PrintWriter(csvOutputFile)) {
	        pw.println(headers);
	    } catch (Exception e) {
			System.out.println("File "+fileName+" has not been created");
		}
	    
	    String[] paramString = {""+storeRate, ""+microtubuleRate, ""+fusionRate, ""+membraneRate,
				""+numCustomer,""+numMicrotubule, ""+microtubuleLength, ""+finalActiveProbabilities[0],
				""+finalActiveProbabilities[1]};
	    
	    // Experiment names for printing
	    String[] experiment = {"Settling", "Basal", "Transition", "Insulin"};
	    
	    // Append each simResults to the file
	    //System.out.println(results[3].getNumberOfTimePoints());
	    double t1 = System.currentTimeMillis();
	    for(int repeat = 0; repeat < numberOfRepeats; repeat ++) {
			
			
			SimulationWithBlocking sim = new SimulationWithBlocking(numCustomer, serviceRates, numMicrotubule, numServers, capacity, activeProbabilities, 
					insulinTimes, expSwitchTimes, outputTimes, rng, meanActivationTime, stimulationTimes,
					eventChangeType, meanUpTimes, meanDownTimes, interrupting, microtubuleOutputs, completeOutputs);
			
			//double t1 = System.currentTimeMillis();
		    SimResults[] results = sim.simulate(maxTime);
		    //double t2 = System.currentTimeMillis();
		    //double dt = (t2-t1)/1000d;
		    for (int i = 0; i < results.length; i++) {
		    	results[i].appendQueueLengthData(csvOutputFile, paramString, 1, experiment[i]);
		    	//System.out.println(Arrays.toString(results[i].getQueueLengths(3)));
		    }
		    
		    
		}
	    
	    
	    System.out.println("Simulation Complete");
	    
	    
	    double t2 = System.currentTimeMillis();
	    double dt = (t2-t1)/1000d;
	    System.out.println("Runtime: "+dt+" seconds.");
	    
	}
}