package DiscreteEventSimulation;

// import java.util.LinkedList;
import java.io.*;
import java.util.stream.*;
import java.util.*;
/* LinkedList is used over ArrayList as elements are only added to the end of the list
 * In this case LinkedList is faster O(1) compared to ArrayList O(n) worst case if
 * the array needs to be resized
 */

public class SimResults{
	/* Get the start timeQueue of an experiment so we can reset the results to be relative to start timeQueue.
	 * i.e. an experiment may not start at t = 0 but we want the output read from 0 */
	protected double startTime;
	/* The time points for gathering time series data to be output. The event times or the experimental data times 
	 * This variable stores the times for queue lengths being gathered */
	protected LinkedList<Double>[] times;
	// Time data for when registering data at specified times
	protected LinkedList<Double> timeQueue;
	/* The queue lengths for all queues */
	protected LinkedList<Integer>[] queueLengths;
	/* Below are the lists to store the queue lengths at various timeQueues. 
	 * The microtubule and fusion queue lengths are really the sum of all stations of the type

	The store queue length */
	protected LinkedList<Integer> storeQueue;
	/* The MT queue length */
	protected LinkedList<Integer> microtubuleQueue;
	/* The fusion queue length */
	protected LinkedList<Integer> fusionQueue;
	/* The PM queue length */
	protected LinkedList<Integer> membraneQueue;
	/* The number of unique visits to each station */
	protected LinkedList<Integer> membraneVisits;
	/* The number of internalisation marked customers at the PM */
	protected LinkedList<Integer> internalisationCount;
	// Times that the state of customers in microtubules are checked
	// i.e. arrival times to MT and attempted departure times from MT
	protected LinkedList<Double> microtubuleStateTimes;
	// Number of customers in microtubules at microtubuleStateTimes
	protected LinkedList<Integer> totalMicrotubuleCustomers;
	// Number of blocked customers in microtubules at microtubuleStateTimes
	protected LinkedList<Integer> blockedMicrotubuleCustomers;
	// Number of customers in a blocked microtubule
	protected LinkedList<Integer> customersInBlockedMicrotubule;
	// Number of inactive fusionsites
	protected LinkedList<Integer> numBlockedFusionSites;
	// Count of fusion events in previous time interval
	protected LinkedList<Integer> numFusionEvents;
	// Queue lengths of individual microtubules
	protected LinkedList<Integer>[] individualMicrotubuleOccupancy;
	
	/* Elements relating to each station */
	public static final int STORE = 0;
	public static final int MICROTUBULE = 1;
	public static final int FUSION = 2;
	public static final int MEMBRANE = 3;
	public static final int UPTAKE = 4; // only used to access the service distribution for increment time
	public static final int INTERNALISATION = 5; // Used ot access the inernalisation experiment data
	public static final int TIME = -1;
	
	
	public SimResults(double startTime, int numMicrotubule) {
		this.startTime = startTime;
		timeQueue = new LinkedList<Double>();
		times = new LinkedList[5];
		queueLengths = new LinkedList[5];
		for (int i = 0; i < 5; i++) {
			times[i] = new LinkedList<Double>();
			queueLengths[i] = new LinkedList<Integer>();
		}
		storeQueue = new LinkedList<Integer>();
		microtubuleQueue = new LinkedList<Integer>();
		fusionQueue = new LinkedList<Integer>();
		membraneQueue = new LinkedList<Integer>();
		membraneVisits = new LinkedList<Integer>();
		internalisationCount = new LinkedList<Integer>();
		microtubuleStateTimes = new LinkedList<Double>();
		totalMicrotubuleCustomers = new LinkedList<Integer>();
		blockedMicrotubuleCustomers = new LinkedList<Integer>();
		customersInBlockedMicrotubule = new LinkedList<Integer>();
		numBlockedFusionSites = new LinkedList<Integer>();
		numFusionEvents = new LinkedList<Integer>();
		
		individualMicrotubuleOccupancy = new LinkedList[numMicrotubule];
		for (int i = 0; i < numMicrotubule; i++) {
			individualMicrotubuleOccupancy[i] = new LinkedList<Integer>();
		}
	}
	
	// For use when registering data for specified times
	public void registerQueueLengths(double t, int store, int microtubule, int fusion, int membrane, int visits, int internalisation) {
		timeQueue.add(t - startTime);
		storeQueue.add(store);
		microtubuleQueue.add(microtubule);
		fusionQueue.add(fusion);
		membraneQueue.add(membrane);
		membraneVisits.add(visits);
		internalisationCount.add(internalisation);
	}
	
	// Used for registering data for all event times
	// Use only for plasma membrane to have uptake and internalisation recorded at correct times
	// internalisation and uptake times match times membraneQueue changes
	// Call at the end of the loop after the system has updated customer states
	public void registerTraceQueueLengths(int station, double t) {
		// Queue lengths for the station arrived at
		times[station].add(t - startTime);
		queueLengths[station].add(Customer.networkState[station]);
		// Queue lengths for departed station
		int prevStation = (((station - 1) % 4) + 4) % 4; // Needed to get positive modulus
		//System.out.println("prevStation = " + prevStation);
		times[prevStation].add(t - startTime);
		queueLengths[prevStation].add(Customer.networkState[prevStation]); 
		
		// Update uptake and internalisation if there is an arrival at the PM or store
		if (station == 0 || station == 3) {
			times[4].add(t-startTime);
			membraneVisits.add(Customer.allVisits[MEMBRANE]);
			internalisationCount.add(Customer.internalisationCount);
		}
	}
	
	// Log the number of fusion events in previous time interval.
	// Only to be called at data output times. Count at time n is the number of fusions in period from time n-1 to n
	public void registerFusionEvents(int fusionCount, Queue[] microtubules) {
		numFusionEvents.add(fusionCount);
		// After logging results reset counter to zero.
		Customer.fusionCount = 0;
		// log the queue lengths in each microtubule
		for (int i = 0; i < microtubules.length; i++) {
			individualMicrotubuleOccupancy[i].add(microtubules[i].getSize());
		}
	}
	
	/**
	 * 
	 * @param t - current time
	 * @param customers - array of customers to check state of
	 */
	public void registerMicrotubuleCustomerStatus(double t, Customer[] customers, Queue[] fusionSites) {
		int totalMTCustomers = 0;
		int blockedMTCustomers = 0;
		int customersInBlockedMT = 0;
		int numBlockedFusion = 0;
		
		Customer currCustomer;
		// loop through all customer to count total customer in microtubules and total blocked in microtubules
		for (int i = 0; i < customers.length; i++) {
			// Check if customer is in microtubule
			currCustomer = customers[i];
			if (currCustomer.getStation() == MICROTUBULE) {
				totalMTCustomers++;
				// Check if the customer is blocked
				if (currCustomer.checkBlock()) {
					blockedMTCustomers++;
					// If blocked, check if it is because they are in an MT with blocked fusion site
					if (!fusionSites[currCustomer.getSubStation()].checkActive()) {
						customersInBlockedMT++;
					}
				}
			}
		}
		// Count the number of inactive fusion servers
		for (int i = 0; i < fusionSites.length; i++) {
			if (!fusionSites[i].checkActive()) {
				numBlockedFusion++;
			}
		}
		// Add counts to linked lists
		microtubuleStateTimes.add(t - startTime);
		totalMicrotubuleCustomers.add(totalMTCustomers);
		blockedMicrotubuleCustomers.add(blockedMTCustomers);
		customersInBlockedMicrotubule.add(customersInBlockedMT);
		numBlockedFusionSites.add(numBlockedFusion);
	}

	
	// Get number of time points in nth queue event times
	public int getNumberOfTimePoints() {
		return timeQueue.size();
	}
	
	
	public int getLastValue(int station) {
		if (station == 0) {
			return storeQueue.getLast();
		} else if (station == 1) {
			return microtubuleQueue.getLast();
		} else if (station == 2) {
			return fusionQueue.getLast();
		} else {
			return membraneQueue.getLast();
		}
	}
	
	public String convertToCSV(String[] data) {
	    return Stream.of(data)
	      .collect(Collectors.joining(","));
	}
	
	public String[] concat(String[]... arrays) {
	    int length = 0;
	    for (String[] array : arrays) {
	        length += array.length;
	    }
	    String[] result = new String[length];
	    int pos = 0;
	    for (String[] array : arrays) {
	        for (String element : array) {
	            result[pos] = element;
	            pos++;
	        }
	    }
	    return result;
	}
	
	
	public int[] getDataArray(int station) {
		switch (station) {
		case STORE:
			return storeQueue.stream().mapToInt(Integer::intValue).toArray();
		case MICROTUBULE:
			return microtubuleQueue.stream().mapToInt(Integer::intValue).toArray();
		case FUSION:
			return fusionQueue.stream().mapToInt(Integer::intValue).toArray();
		case UPTAKE:
			return membraneVisits.stream().mapToInt(Integer::intValue).toArray();
		default:
			return membraneQueue.stream().mapToInt(Integer::intValue).toArray();
		}
	}
	
	public double[] getTimeArray() {
		return timeQueue.stream().mapToDouble(Double::doubleValue).toArray();
	}
	
	/**
	 * 
	 * @param station - which queue to get the event times for
	 * 5 is the event times of microtubule arrival sand departures
	 * @return
	 */
	public double[] getTimes(int station) {
		return times[station].stream().mapToDouble(Double::doubleValue).toArray();
	}
	
	/**
	 * 
	 * @param station - which queue to get the event times for
	 * 5 is the event times of microtubule arrival sand departures
	 * @return
	 */
	public int[] getQueueLengths(int station) {
		return queueLengths[station].stream().mapToInt(Integer::intValue).toArray();
	}
	
	public int[] getFusionCounts() {
		return numFusionEvents.stream().mapToInt(Integer::intValue).toArray();
	}
	
	public int[][] getMicrotubuleOccupancy() {
		int[][] mtOccupancy = new int[individualMicrotubuleOccupancy.length][];
		for (int i = 0; i < individualMicrotubuleOccupancy.length; i++) {
			mtOccupancy[i] = individualMicrotubuleOccupancy[i].stream().mapToInt(Integer::intValue).toArray();
		}
		return mtOccupancy;
	}
	
	/**
	 * 
	 * @param option: 1 for totalMicrotubuleCustomers
	 * 				  2 for blockedMicrotubuleCustomers
	 * 				  3 for customersInBlockedMicrotubule
	 * @return
	 */
	public int[] getMicrotubuleStatus(int option) {
		if (option == 1) {
			return totalMicrotubuleCustomers.stream().mapToInt(Integer::intValue).toArray();
		}
		else if (option == 2) {
			return blockedMicrotubuleCustomers.stream().mapToInt(Integer::intValue).toArray();
		}
		else if (option == 3) {
			return customersInBlockedMicrotubule.stream().mapToInt(Integer::intValue).toArray();
		}
		else {
			return numBlockedFusionSites.stream().mapToInt(Integer::intValue).toArray();
		}
	}
	
	public double[] getMicrotubuleStatusTimes() {
		return microtubuleStateTimes.stream().mapToDouble(Double::doubleValue).toArray();
	}
	
	public double[][] getArrangedArray(int station, int repeats){
		int[] data = getDataArray(station);
		// Rearrrange the 1D array to make a 2d array with each column a single timeseries
		double[][] arrangedData = new double[data.length / repeats][repeats];
		int column = 0;
		int row = 0;
		int numRows = data.length / repeats;
		// loop through each item in data and rearrange
		for (int i = 0; i < data.length; i++) {
			arrangedData[row][column] = data[i];
			row = (row + 1) % numRows;
			if (row == 0) column++;
		}
		return arrangedData;
	}
	
	public void appendQueueLengthData(File filename, String[] ParamString, int repeat, String experiment) {
		List<String[]> data = new ArrayList<>();
		//double t1 = System.currentTimeMillis();
		for (int i = 0; i < timeQueue.size(); i++) {
			/*data.add(new String[] {Double.toString(timeQueue.get(i)), 
					Integer.toString(storeQueue.poll()), Integer.toString(microtubuleQueue.poll()), 
					Integer.toString(fusionQueue.poll()), Integer.toString(membraneQueue.poll())});*/
			data.add(concat(ParamString, new String[] {"" + repeat, experiment, "" + timeQueue.get(i), 
					"" + storeQueue.poll(), "" + microtubuleQueue.poll(), 
					"" + fusionQueue.poll(), "" + membraneQueue.poll(), "" + membraneVisits.poll()}));
		}
		//double t2 = System.currentTimeMillis();
		//double dt = (t2-t1)/1000d;
	    //System.out.println("Time to make data structure: "+dt+" seconds.");
	    
	    //t1 = System.currentTimeMillis();
		File csvOutputFile = filename;
		try (PrintWriter pw = new PrintWriter(new FileOutputStream(csvOutputFile, true))) {
	        data.stream()
	          .map(this::convertToCSV)
	          .forEach(pw::println);
	    }
		catch (Exception e) {
			System.out.println("File "+filename+" has not been created");
		}
		//t2 = System.currentTimeMillis();
		//dt = (t2-t1)/1000d;
	    //System.out.println("Actual Write time: "+dt+" seconds.");
	}
}