package DiscreteEventSimulation;

import java.util.LinkedList;

import java.util.Random;

/**
 * An efficient implementation of a queue, using the Java LinkedList
 * class, which implements a doubly-linked list, the most efficient data structure for
 * lists where elements are mostly added to and removed from the beginning or end.
 * @author MBoon
 *
 */
 


public class Queue {

  /* List of customers in the queue */
  protected LinkedList<Customer> customers;  
  /* Number of servers for this queue */
  protected int nrOfServers;
  // The capacity of this station
  protected int capacity;
  /* Whether the station is active or not 
   * true if active, false if inactive */
  protected boolean active;
  /* Whether the station is insulin stimulated or not
   * this will only impact fusion sites. No other queues should check condition
   * False by default, must be stimulated at time zero */
  protected boolean stimulated;
  /* create a random number between 0 and 1 to determine active or inactive */
  protected Random rng = new Random();
  /* The current departure event associated with the queue
   * This is used for the fusion sites to change the validity of interrupted events
   */
  protected Event associatedDepartureEvent;
  /* The event associated with the queue for the next up or down time switch
   * This is used for the fusion sites to change the validity of interrupted events
   */
  protected Event associatedSwitchEvent;
  
  
  /**
   * Constructs an empty single-server Queue with single capacity
   */
  public Queue() {
    this(1,1);
  }
  
  /**
   * Constructs an empty Queue with the specified number of servers
   */
   
  public Queue(int nrOfServers, int capacity) {
    customers = new LinkedList<Customer>();
    this.nrOfServers = nrOfServers;
    this.capacity = capacity;
    this.active = false;
    this.stimulated = false;
    this.associatedDepartureEvent = null;
    this.associatedSwitchEvent = null;
  }
  
  /** 
   * Returns the number of customers in the queue (including the person in service)
   * @return the number of customers
   */
   
  public int getSize() {
    return customers.size();
  }
  
  /* Check if the station has capacity to accept a new customer 
   * Returns true if there is at least one available space false otherwise */
  public boolean checkAvailability() {
	  if (getSize() < capacity) {
		  return true;			  
	  } else {
		  return false;
	  }
  }
  
  /**
   * Replace the first customer in the queue with customer with new event times and unblocked
   * The replacement is done to save having to update values twice
   * Update the values in customer array in simulation class and then substitute in
   */
  
  public void updateFirstCustomer(Customer c) {
	  removeFirstCustomer();
	  customers.addFirst(c);
  }
  
  /**
   * Adds a customer to this queue
   * @param c the new customer
   */
   
  public void addCustomer(Customer c) {
    customers.add(c);
  }
  
  /**
   * Adds a customer to the front of queue
   * To be used when the front customer becomes unblocked
   */
  
  public void addFrontCustomer(Customer c) {
	  customers.addFirst(c);
  }
  
  // Assign a departure event to this queue. Used to find events to interrupt
  public void assignDepartureEvent(Event e) {
	  this.associatedDepartureEvent = e;
  }
  
  // Remove associated events when the departure has occured
  public void unassignDepartureEvent() {
	  this.associatedDepartureEvent = null;
  }
  
  // Get get the currently assigned departure event
  public Event getAssignedDepartureEvent() {
	  return this.associatedDepartureEvent;
  }
  
  // Assign a switch event to this queue. Used to find events to interrupt
  public void assignSwitchEvent(Event e) {
	  this.associatedSwitchEvent = e;
  }
  
  // Remove associated switch when the switch has occured
  public void unassignSwitchEvent() {
	  this.associatedSwitchEvent = null;
  }
  
  // Get the currently assigned switch event
  public Event getAssignedSwitchEvent() {
	  return this.associatedSwitchEvent;
  }
  
  /**
   * Returns true if the given customer is at the front of the queue
   */
  
  public boolean checkQueueFront(Customer c) {
	  if (c.getID() == getFirstCustomer().getID()) {
		  return true;
	  } else {
		  return false;
	  }
  }
  
  /**
   * Returns true if the station is active
   */
  
  public boolean checkActive() {
	  return active;
  }
  
  /**
   * Returns true if the station is stimulated
   */
  
  public boolean checkStimulated() {
	  return stimulated;
  }
  
  /**
   * Returns true if the customer at front of queue is blocked
   */
  
  public boolean checkFrontBlockage() {
	  if (getSize() > 0) {
		  return getFirstCustomer().checkBlock();
	  } else {
		  return false; // If there is no one in the queue then it is not blocked
	  }
  }
  
  /** 
   * Returns the first customer in the queue.
   * @return the first customer
   */
  
  public Customer getFirstCustomer() {
      return customers.peek();
  }
  
  /**
   * Removes the first customer from the queue at time <i>t</i>
   * @param t the time that the customer is removed
   * @return the customer that is removed from the queue
   */
   
  public Customer removeFirstCustomer() {
	  // Check the ordering of the customers in queue
	  //checkCustomerOrdering();
	  return customers.poll();
  }
  
  /** 
   * Removes the specified customer.
   * @param customer the customer that should be removed
   */
  
  public void removeCustomer(Customer c) {
      customers.remove(c);
  }
  
  /**
   * Makes the station active with probability ap otherwise made inactive
   * Should be called when an insulin change event occurs to decide on if the station is active or not
   */
  
  public void drawActive(double ap) {
	  if (rng.nextDouble() <= ap) {
		  active = true;
	  } else {
		  active = false;
	  }
  }
  
  /**
   * Makes the station stimulated with probability ap otherwise made unstimulated
   * Should be called when an insulin stimulation event occurs to decide on if the station is stimulated or not
   */
  
  public void drawStimulated(double ap) {
	  if (rng.nextDouble() <= ap) {
		  stimulated = true;
	  } else {
		  stimulated = false;
	  }
  }
  
  public void activate() {
	  active = true;
  }
  
  public void deactivate() {
	  active = false;
  }

  /**
   * Returns the position of the customer in this queue, or -1 if the customer cannot be found.
   * @param customer the customer whose position should be determined.
   * @return the index of the customer in the queue, or -1 if not found.
   */
  
  public int getCustomerPosition(Customer c) {
      return customers.indexOf(c);
  }
  
  /**
   * Blocks the given customer 
   */
  
  public void blockCustomer(Customer c) {
	  int indice = getCustomerPosition(c);
	  Customer cust = getCustomerAtPosition(indice);
	  cust.block();
	  customers.set(indice, cust);
  }
  
  /**
   * Returns the customer at the specified position
   * @param position the position in the queue
   * @return the customer at the specified position
   */
  
  public Customer getCustomerAtPosition(int position) {
      return customers.get(position);
  }

  /**
   * Returns the number of servers for this queue
   * @return the number of servers for this queue
   */
  
  public int getNumberOfServers() {
      return nrOfServers;
  }
  
  /**
   * Check the ordering of the queue to ensure the front of queue has
   * the smallest (earliest) arrival time
   */
  
  public void checkCustomerOrdering() {
	  if (customers.size() > 1) {
		  for (int i = 0; i < customers.size() - 1; i++) {
			  // Check each customer arrived before the customer directly behind them
			  Customer thisC = getCustomerAtPosition(i);
			  Customer nextC = getCustomerAtPosition(i+1);
			  if (thisC.getArrivalTime() > nextC.getArrivalTime()) {
				  System.out.println("Customer is out of order.");
			  }
		  }
	  }
  }
} 