package DiscreteEventSimulation;

/**
 * An event implementation which is compatible with the  more efficient implementation of the Future Event Set, 
 * using the Java PriorityQueue class. In order to use the PriorityQueue, Event must implement the method
 * compareTo(anotherEvent), so the priority queue knows how to sort the events.
 * @author MBoon
 *
 */

public class Event implements Comparable<Event> {

  /* Departure Event from the Store */
  public static final int DEPARTURESTORE = 0;
  /* Departure Event from a microtubule */
  public static final int DEPARTUREMICROTUBULE = 1;
  /* Departure Event from a fusion site */
  public static final int DEPARTUREFUSION = 2;
  /* Departure Event from the plasma membrane */
  public static final int DEPARTUREMEMBRANE = 3;
  /* Arrival Event */
  public static final int ARRIVALMEMBRANE = 4;
  // Arrival at STORE 
  public static final int ARRIVALSTORE = 5;
  // Arrival at a MT 
  public static final int ARRIVALMICROTUBULE = 6;
  // Arrival at a FUSION 
  public static final int ARRIVALFUSION = 7;
  /* Change in Active Probabilities */
  public static final int INSULINCHANGE = 8;
  /* Change in the experiment being recorded */
  public static final int EXPCHANGE = 9;
  /* Times that the experimental data must be collected */
  public static final int DATATIME = 10;
  /* Change active/inactive state of a Fusion Site */
  public static final int ACTIVATE = 11;
  /* Ensure there is at least one active microtubule */
  
  // Start a microtubule up time - micrtobule begins service - make sure not to start new
  public static final int FUSIONUP = 12;
  // Start microtubule down time without interrupting service of current customer
  public static final int FUSIONDOWN = 13;
  // Start microtubule down time - interrupt the service of current customer and make them wait for next up time
  // public static final int FUSIONDOWNINTERRUPT = 14;
  // Start a microtubule up time from an interrupted down time
  // public static final int FUSIONUPINTERRUPT = 15;
  
  // Close fusion sites when inhibitor applied
  public static final int CLOSEFUSION = 16;
  /* Stimulation event can be either insulin application 
   * or inhibitor application
   * Changes the stimulation state of fusion sites */
  public static final int STIMULATIONCHANGE = 17;
  /* Change stimulation state of a Fusion Site to stimulated */
  public static final int STIMULATEUP = 19;
  /* Change stimulation state of a Fusion Site to unstimulated */
  public static final int STIMULATEDOWN = 20;
  /* Change the rate parameters of the distributions
   * This event is an inhibitor or insulin type event 
   * Use to change state of system between experiments */
  public static final int RATECHANGE = 18;
  
  /* Event type */
  protected int type;
  /* Event time */
  protected double time;
  /* Customer */
  protected Customer customer;
  /* A counter to keep track of which results to store in */
  protected static int experimentCount = 0;
  /* If the event is a change in insulin level then store the activeProbability */
  protected double activeProbability;
  /* Fusion site number */
  protected int fusionSite = -1;
  /* True for more stimulated/open, False for less stimulation/open
   * This is used in insulinchange and stimulation change events to determine
   * if fusion sites are being opened or closed, or if they become more or less stimulated
   * Determines whether event creates up or down events */
  protected boolean changeType;
  /* Whether the current event is valid or not
   * True of the current event is valid, false otherwise
   * Created events should be valid by default
   */
  protected boolean validity;
  /* whether this event will be interrupting another event
   * True means to check for another event to interrupt,
   * False otherwise
   * Only fusion down events and insulin change events can be interrupting
   */
  protected boolean interrupting;
  
  /**
   * Constructs an Event
   * @param type the Event type (ARRIVAL or DEPARTURE)
   * @param time the Event time
   * @param customer the customer that this event is associated with. Specify 'null' if no customer is provided.
   */

  
  public Event(int type, double time, Customer customer, double activeProb, int fusionSite, boolean changeType, boolean validity, boolean interrupting) {
	    this.type = type;
	    this.time = time;
	    this.customer = customer;
	    this.fusionSite = fusionSite;
	    this.activeProbability = activeProb;
	    this.changeType = changeType;
	    this.validity = validity;
	    this.interrupting = interrupting;
  }
	  
  
  // Use to schedule arrivals and departures
  public Event(int type, double time, Customer customer) {
    this(type, time, customer, 0, -1, false, true, false);
  }
  
  // Use to change activity of a fusion site
  // Closure events may want to be interrupting?
  public Event(int type, double time, double activeProb, int fusionSite, boolean interrupting) {
	  this(type, time, null, activeProb, fusionSite, false, true, interrupting);
  }
  
  // Use to schedule insulin change
  // Create the event to also know which insulin level it should change to
  public Event(int type, double time, double activeProb, boolean changeType) {
	    this(type, time, null, activeProb, -1, changeType, true, false);
  }
  
  // Use to schedule a change in experiment
  // The event is merely changing the experiment
  // Also used for the DATATIME events
  public Event(int type, double time) {
	  this(type, time, null, 0, -1, false, true, false);
	  // This should only be called at start of a simulation
	  experimentCount = 0;
  }
  
  // Use to schedule microtubule up and down times
  public Event(int type, double time, int fusionSite, boolean interrupting) {
	  this(type, time, null, 0, fusionSite, false, true, interrupting);
  }
  
  /**
   * Returns the active probability associated with the event
   */
  
  public double getActiveProbability() {
	  return activeProbability;
  }
  
  
  /**
   * Returns the event type.
   * @return event type
   */
  
  public int getType() {
    return type;
  }
  
  /**
   * Return the fusion site number.
   * @return the fusion site
   */
  
  public int getFusionSite() {
	  return fusionSite;
  }
  
  public void updateExperimentCount() {
	  experimentCount ++;
  }
  
  /**
   * Returns the event time.
   * @return event time
   */

  public double getTime() {
    return time;
  }
  
  public boolean getChangeType() {
	  return changeType;
  }
  
  public Customer getCustomer() {
      return customer;
  }
  
  // Check if the event is currently valid
  public boolean isValid() {
	  return validity;
  }
  
  // Make the event invalid
  public void makeInvalid() {
	  this.validity = false;
  }
  
  // Check if the event is an interrupting event
  public boolean isInterrupting() {
	  return interrupting;
  }
  
  public String toString() {
      String[] typeStr = {"", "ARRIVAL", "DEPARTURE", "INSULINCHANGE"};
      return typeStr[type]+" at t = "+time+" for customer "+customer;
  }

  /**
   * The method below indicates how events should be sorted.
   * We use the built-in compare() function in Double, which compares to doubles.
   * It returns -1, 0, or 1 (smaller, equal, larger).
   */
  
  @Override
  public int compareTo(Event e) {
	  int currType = e.getType();
	  if (currType < INSULINCHANGE) {
		  return Double.compare(time, e.getTime());
	  }
	  if (Double.compare(time, e.getTime()) != 0) {
		  return Double.compare(time, e.getTime());
	  } else {
		  //System.out.println(e.getTime());
		  //System.out.println(e.getType());
		  if (e.getType() == EXPCHANGE) { // If EXPCHANGE force it to occur first
			  return 1;
		  } else if (e.getType() == INSULINCHANGE){ // Force INSULINCHANGE to occur last
			  return -1;
		  } else if (e.getType() == DATATIME && type == EXPCHANGE) {
			  return -1;
		  } else if (e.getType() == DATATIME && type == INSULINCHANGE) {
			  return 1;
		  } else { // The DATATIME report should occur in the middle
			  return 0;
		  }
	  }
  }
  
} 