package DiscreteEventSimulation;

public class Customer {

  /* The arrival time of this customer at the current queue*/
  protected double arrivalTime;
  /* The service time of this customer at the current queue */
  protected double serviceTime;
  /* The current station of the customer (queue number, or -1 if he has already left the system */
  protected int station;
  /* The current substation of the customer. i.e. which microtubule or fusion site it is in */
  protected int subStation;
  /* Whether the customer is currently blocked or not. true if blocked*/
  protected boolean blocked;
  /* Number of visits to each station. This value is specific to an individual customer */
  protected int[] individualVisits = {0, 0, 0, 0};
  /* A counter for the number of unique visits to a station. It is common to all customers and is not a count for the individual customer but for all customers.
   * This value is to be used for the uptake experiment */
  protected static int[] allVisits = {0, 0, 0, 0};
  /* A customer ID to uniquely identify different customers */
  protected int id;
  /* The customers also store the current state of the network
   * Easiest to implement here because it can be updated when a customer moves 
   * Each element is the queue length of each station*/
  protected static int[] networkState = {0, 0, 0, 0};
  /* Marker for the internalisation experiment. This is given to each customer.
   * This marker is true if the customer is at the plasma membrane when the experiment starts.
   * This marker is then used to track the number of customers with this marker on the PM at any given time
   */
  protected boolean internalisationMarker;
  /* Count of customers on the surface with the internalisation marker
   * Add and subtract from the count as customers move into and out of the PM
   */
  protected static int internalisationCount;
  // A count of the fusion events
  // Should be reset to zero each time the events are logged
  protected static int fusionCount = 0;
  
  /* Elements relating to each station */
  // These are also defined in the simulation file so be careful changing
  public static final int STORE = 0;
  public static final int MICROTUBULE = 1;
  public static final int FUSION = 2;
  public static final int MEMBRANE = 3;
  public static final int INCREMENT = 4; // only used to access the service distribution for increment time

  /**
   * Constructs a Customer with specified arrival time, service time, and patience. The location is set to 0, because each customer arrives at the first queue.
   * The queue arrival time and the system arrival time are both set to the specified arrival time.
   * @param arrivalTime the arrival time
   * @param serviceTime the service time at the first queue
   * @param endPatienceTime the time  at which this customer's patience runs out
   */

  public Customer(int id, int station, double serviceTime) {
    this.id = id;
	  this.arrivalTime = 0; // Can be used to check ordering, all customer originally arrive at t = 0
    this.serviceTime = serviceTime;
    this.station = station; // allows for initialising customers in whichever station we choose
    this.subStation = 0;
    this.blocked = false;
    this.internalisationMarker = false;
    
    // Increase the queue length for each customer created
    networkState[station] ++;
  }

  /**
   * Returns the arrival time of this customer at the current queue.
   * @return arrival time
   */

  public double getArrivalTime() {
    return arrivalTime;
  }
  

  /**
   * Returns the service time of this customer.
   * @return service time
   */

  public double getServiceTime() {
    return serviceTime;
  }

  /**
   * Changes the location of this customer and specifies the service time of this new location.
   * The location is the index of the queue in which the customer is waiting/being served,
   * or -1 if the customer has left the system.
   * If the customer has left the system, the value of the service time is irrelevant.
   * In this case, it is recommended, however, to use the function <tt>leaveSystem()</tt> instead.
   * @param location the new location of the customer
   * @param time the time at which he is transferred to the next queue
   * @param newServiceTime the service time at this next queue
   */
  
  public void moveTo(int station, int subStation, double time, double newServiceTime) {
      // Decrease count in current station
	  networkState[this.station] --;
	  // If leaving the PM decrease internalisationCount if customer is marked
	  if (this.station == MEMBRANE && this.internalisationMarker == true) {
		  internalisationCount --;
	  // If entering the PM, increase the internalisationCount
	  } else if (station == MEMBRANE && this.internalisationMarker == true) {
		  internalisationCount ++;
	  }
	  
      // Increase the count in new station
      networkState[station] ++;
      
	  this.station = station;
      this.subStation = subStation;
      this.arrivalTime = time;
      this.serviceTime = newServiceTime;
      this.individualVisits[station] ++;
      if (this.individualVisits[station] == 1) {
    	  allVisits[station] ++;
      }
      // if customer moved to PM increase fusion count
      if (station == MEMBRANE) {
    	  fusionCount ++;
      }
      
  }
  
  public void resetFusionCount() {
	  fusionCount = 0;
  }
  
  public void resetStationVisits() {
	  int currStation = getStation();
	  //System.out.println(station);
	  for (int i = 0; i < 4; i++) {	  
		  if (currStation == i) {
			  individualVisits[i] = 1;
			  allVisits[i] ++;
			  //System.out.println(individualVisits[station]);
		  } else {
			  individualVisits[i] = 0;
			  //System.out.println("Entered Else Statement: " + individualVisits[station]);
		  }
		  // Reset apply the internalisation marker if at PM, increase the count on PM
		  if (currStation == MEMBRANE) {
			  internalisationMarker = true;
			  internalisationCount ++;
		  } else { // if not at PM then set marker to false
			  internalisationMarker = false;
		  }
	  }
	  //System.out.println("Station: " + station);
	  //System.out.println(individualVisits[3]);
  }
  
  /**
   * Gives the customer a new service time without moving station
   * Used for increment time increases and repetitive service after blocking
   */
  
  public void redrawService(double newServiceTime) {
	  this.serviceTime = newServiceTime;
  }
  /**
   * Changes the location (and service time) of this customer to -1, indicating that he has left the system.
   */
  
  /**
   * Returns the location of this customer. The location is the index of the queue in which the customer is waiting/being served,
   * or -1 if the customer has left the system.
   * @return location the location of the customer
   */
  
  public int getStation() {
      return station;
  }
  
  /**
   * Returns the location of this customer in substation. The location is the index of the queue in which the customer is waiting/being served,
   * or -1 if the customer has left the system.
   * @return location the location of the customer
   */
  
  public int getSubStation() {
      return subStation;
  }
  
  /**
   * Returns the customer ID
   */
  
  public int getID() {
	  return id;
  }

  
  /**
   * Changes the customer to a blocked state
   */
  
  public void block() {
	  this.blocked = true;
  }
  
  /**
   * Changes the customer to an unblocked state
   * If unblocked they must also get a new service time
   */
  
  public void unblock(double serviceTime) {
	  this.blocked = false;
	  this.serviceTime = serviceTime;
  }
  
  /**
   * Returns true if customer is blocked
   */
  
  public boolean checkBlock() {
	  return blocked;
  }
  
  /**
   * Need to restart state at state of each simulation run
   */
  public static void resetNetworkState() {
	  for (int i = 0; i < 4; i++) {
		  networkState[i] = 0;
		  allVisits[i] = 0;
	  }
  }
  
} 