# Discrete Event Simulation for Queuing Model of Intracellular GLUT4 Translocation

The files contained in the folder folder include the main code for a discrete event simulation of the queuing model as well as the associated class files. The simulation can be run using *NetworkSimulation_Inhibitor.jar*, which has all required files packaged. Below, we outline the source code contained.

## Discrete Event Simulation

The folder titled *DisreteEventSimulation* contains all java files for the main simulation. The mathematical details of the model are given in 

Brock D. Sherlock, Marko A. A. Boon, Maria Vlasiou & Adelle C. F. Coster (2024).  
*The Distance Between: An Algorithmic Approach to Comparing Stochastic Models to Time-Series Data.*  
Bulletin of Mathematical Biology, 86:111.  
[https://doi.org/10.1007/s11538-024-01331-y](https://doi.org/10.1007/s11538-024-01331-y)


### SimulationWithBlocking.java

The `SimulationWithBlocking` class models a discrete‑event simulation of vesicle transport through a network of stations (store, microtubule, fusion site, membrane). It supports blocking, capacity limits, and insulin‑dependent activation events. The `main` method will run default parameter values, or read a comma separated input of parameters. Recommended use is to construct the simulation object in another program using `createInstance`. Then, use the method `simulate` to run the simulation.

To construct a simulation, use the method `createInstance`. This method wraps the full constructor and automatically provides a random number generator.

```java
public static SimulationWithBlocking createInstance(
    int numCustomers,
    double[][] serviceRates,
    int numMicrotubule,
    int[] servers,
    int[] capacity,
    double[] activeProbabilities,
    int[] insulinTimes,
    int[] expSwitchTimes,
    double[] outputTimes,
    double meanActivationTime,
    int[] stimulationTimes,
    boolean[] eventChangeType,
    double[] meanUpTimes,
    double[] meanDownTimes,
    boolean interrupting,
    boolean microtubuleOutputs,
    boolean completeOutputs
)
```

#### Parameters

- **numCustomers**  
  Total number of customers (vesicles) in the system.

- **serviceRates**  
  2D array of service rates for each station across experiments.  
  - Row 0 → Store  
  - Row 1 → Microtubule  
  - Row 2 → Fusion site  
  - Row 3 → Membrane  
  - Row 4 → Increment (derived from microtubule rate × number of servers)

- **numMicrotubule**  
  Number of microtubules in the network.

- **servers**  
  Number of servers at each station.

- **capacity**  
  Capacity of each station (e.g., microtubule length).

- **activeProbabilities**  
  Array of fusion site activation probabilities over time.

- **insulinTimes**  
  Times at which insulin levels change.

- **expSwitchTimes**  
  Experiment switch times (used to reset counters and results).

- **outputTimes**  
  Times at which simulation results are recorded.

- **meanActivationTime**  
  Mean time for fusion sites to activate after insulin change.  
  If ≤ 0, activation is deterministic (immediate).

- **stimulationTimes**  
  Times of stimulation events (insulin or inhibitor).

- **eventChangeType**  
  Boolean array indicating whether each event increases (`true`) or decreases (`false`) fusion site activity.

- **meanUpTimes**  
  Mean up times for fusion sites: `[stimulated, unstimulated]`.

- **meanDownTimes**  
  Mean down times for fusion sites: `[stimulated, unstimulated]`.

- **interrupting**  
  If `true`, up/down events interrupt service at fusion sites.

- **microtubuleOutputs**  
  If `true`, output microtubule states and fusion events at output times.

- **completeOutputs**  
  If `true`, output full traces of data (use only for single parameter runs).


#### The `simulate` Method

##### Use
Call `simulate(int maxTime)` to run the discrete‑event simulation up to the given maximum time.  
It schedules and processes events (arrivals, departures, insulin/stimulation changes) across the system.

Example:
```java
SimResults[] results = sim.simulate(1000);
```

##### Output
Returns an array of `SimResults` objects, one per experiment phase, containing:
- Queue length histories at output times  
- Fusion/microtubule event data (if enabled)  
- Full trace logs (if `completeOutputs` is true)

It is recommended to only output data at required times. Using `completeOutputs` generally results in long run times and high memory use. However, the option is included as a complete history of events is needed for some analyses (such as the rate of fusion events).

### SimResults.java

The `SimResults` class stores and organizes output data from the discrete‑event simulation.  
It records queue lengths, fusion events, microtubule states, and internalisation markers at specified times or continuously, depending on simulation settings.  
It also provides accessor methods to retrieve results in array form and utilities to export data to CSV.

#### Purpose
- Collects time series data during simulation runs.  
- Tracks queue lengths for each station (store, microtubule, fusion site, membrane).  
- Records uptake and internalisation counts for biological experiments.  
- Logs fusion events and microtubule occupancy for detailed analysis.  
- Provides methods to convert results into arrays or CSV for downstream analysis.

#### Key Features
- **Fields**
  - `timeQueue`: list of output times.  
  - `storeQueue`, `microtubuleQueue`, `fusionQueue`, `membraneQueue`: queue length histories.  
  - `membraneVisits`: number of unique visits to the plasma membrane.  
  - `internalisationCount`: count of internalisation‑marked customers at the membrane.  
  - `numFusionEvents`: fusion event counts per interval.  
  - `individualMicrotubuleOccupancy`: occupancy per microtubule.  
  - `microtubuleStateTimes`, `totalMicrotubuleCustomers`, `blockedMicrotubuleCustomers`: detailed microtubule state information.

- **Data Access Methods**
  - `getTimeArray()`: returns all output times as a `double[]`.  
  - `getTimes(int station)`: returns event times for a given station.  
  - `getQueueLengths(int station)`: returns queue length history for a given station.  
  - `getFusionCounts()`: returns fusion event counts per interval.  
  - `getMicrotubuleOccupancy()`: returns occupancy of each microtubule as a 2D array.  
  - `getMicrotubuleStatus(int option)`: returns counts of total, blocked, or customers in blocked microtubules.  
  - `getMicrotubuleStatusTimes()`: returns times at which microtubule states were logged.  
  - `getArrangedArray(int station, int repeats)`: reshapes 1D data into a 2D array for repeated experiments.  

- **Export**
  - `appendQueueLengthData(File filename, String[] ParamString, int repeat, String experiment)`: appends queue length data to a CSV file, including parameters and experiment metadata.

#### Output
The `SimResults` class produces structured data for analysis, including:
- Queue length histories over time.  
- Fusion event counts and microtubule occupancy.  
- Uptake and internalisation experiment metrics.  



### Customer.java

The `Customer` class represents an individual vesicle in the discrete‑event simulation.  
It tracks the customer’s state, movement between stations, and contributes to global counters used in experiments.

#### Purpose
- Models a single vesicle (customer) moving through the system.
- Maintains both **individual state** (arrival time, service time, station, blocked status) and **global state** (queue lengths, visit counts, fusion events).

#### Key Features
- **Fields**
  - `arrivalTime`, `serviceTime`: timing information for the current queue.
  - `station`, `subStation`: current location in the system.
  - `blocked`: whether the customer is blocked at a microtubule.
  - `individualVisits`: per‑customer visit counts.
  - `allVisits`: static counter of unique visits across all customers.
  - `networkState`: static array of queue lengths at each station.
  - `internalisationMarker`, `internalisationCount`: track plasma membrane internalisation experiments.
  - `fusionCount`: static counter of fusion events.

- **Stations**
  - `STORE = 0`, `MICROTUBULE = 1`, `FUSION = 2`, `MEMBRANE = 3`, `INCREMENT = 4`.

- **Core Methods**
  - `moveTo(station, subStation, time, newServiceTime)`: move customer to a new station and update counters.
  - `redrawService(newServiceTime)`: assign a new service time without moving.
  - `block()`, `unblock(serviceTime)`, `checkBlock()`: manage blocked state.
  - `resetFusionCount()`, `resetStationVisits()`, `resetNetworkState()`: reset counters for new experiments.
  - Getters: `getArrivalTime()`, `getServiceTime()`, `getStation()`, `getSubStation()`, `getID()`.

#### Output
The `Customer` class does not produce direct output, but updates shared counters (`networkState`, `allVisits`, `fusionCount`, `internalisationCount`) that are logged by the simulation for analysis.

### Event.java

The `Event` class represents a scheduled action in the discrete‑event simulation.  
It defines all possible event types (arrivals, departures, insulin/stimulation changes, experiment switches, data logging) and provides ordering logic for the Future Event Set using Java’s `PriorityQueue`.

#### Purpose
- Encapsulates simulation events with type, time, and associated customer or fusion site.
- Supports insulin/stimulation changes, experiment transitions, and logging events.
- Implements `Comparable<Event>` so events are sorted chronologically and by priority.

#### Key Features
- **Event Types**  
  - Departures: `DEPARTURESTORE`, `DEPARTUREMICROTUBULE`, `DEPARTUREFUSION`, `DEPARTUREMEMBRANE`  
  - Arrivals: `ARRIVALSTORE`, `ARRIVALMICROTUBULE`, `ARRIVALFUSION`, `ARRIVALMEMBRANE`  
  - State changes: `INSULINCHANGE`, `EXPCHANGE`, `DATATIME`, `ACTIVATE`, `CLOSEFUSION`, `STIMULATIONCHANGE`, `STIMULATEUP`, `STIMULATEDOWN`, `FUSIONUP`, `FUSIONDOWN`, `RATECHANGE`  

- **Fields**
  - `type`: event type constant.  
  - `time`: scheduled time of the event.  
  - `customer`: associated `Customer` object (if applicable).  
  - `fusionSite`: fusion site index for site‑specific events.  
  - `activeProbability`: probability used for activation/stimulation events.  
  - `changeType`: indicates increase (`true`) or decrease (`false`) in activity.  
  - `validity`: whether the event is still valid (can be invalidated).  
  - `interrupting`: whether the event interrupts another ongoing event.  
  - `experimentCount`: static counter for experiment phases.

- **Core Methods**
  - Constructors for different event types (arrivals, departures, insulin changes, stimulation changes, experiment changes).  
  - `getType()`, `getTime()`, `getCustomer()`, `getFusionSite()`, `getActiveProbability()`: accessors.  
  - `isValid()`, `makeInvalid()`: manage event validity.  
  - `isInterrupting()`: check if event interrupts another.  
  - `updateExperimentCount()`: increment experiment phase counter.  
  - `compareTo(Event e)`: defines ordering in the event queue (by time, with special rules for `EXPCHANGE`, `INSULINCHANGE`, and `DATATIME`).

#### Output
The `Event` class does not produce direct output. Instead, it drives the simulation by scheduling and ordering actions in the Future Event Set, ensuring correct timing and priority of arrivals, departures, and state changes.

### FES.java

The `FES` (Future Event Set) class manages the event queue for the discrete‑event simulation.  
It uses Java’s `PriorityQueue` to keep events sorted by time and priority, ensuring the simulation processes them in the correct order.

#### Purpose
- Provides an efficient event scheduling mechanism.
- Stores all upcoming events and returns them in chronological order.

#### Key Features
- **Fields**
  - `events`: a `PriorityQueue<Event>` holding all scheduled events.

- **Core Methods**
  - `FES()`: constructs an empty event queue.
  - `addEvent(Event newEvent)`: adds a new event to the queue.
  - `nextEvent()`: retrieves and removes the next scheduled event.
  - `numberOfEvents()`: returns the number of events currently in the queue.

#### Output
The `FES` class does not produce direct output.  
It serves as the backbone of the simulation loop, supplying the next event to be processed until the simulation ends.

### Queue.java

The `Queue` class implements a station queue in the discrete‑event simulation, using Java’s `LinkedList` for efficient insertion and removal.  
It manages customers, capacity, server counts, and state changes (active/inactive, stimulated/unstimulated), and supports blocking logic for microtubules and fusion sites.

#### Purpose
- Represents a single station (store, microtubule, fusion site, or membrane) as a queue.  
- Tracks customers, capacity limits, and whether the station is active or stimulated.  
- Provides methods to add, remove, and block/unblock customers.  
- Stores associated events for departures and state switches, allowing interruption when conditions change.

#### Key Features
- **Fields**
  - `customers`: list of customers in the queue.  
  - `nrOfServers`: number of servers at the station.  
  - `capacity`: maximum number of customers allowed.  
  - `active`, `stimulated`: booleans for station state.  
  - `associatedDepartureEvent`, `associatedSwitchEvent`: track events tied to the queue.  

- **Core Methods**
  - `addCustomer(Customer c)`, `addFrontCustomer(Customer c)`: add customers to queue.  
  - `removeFirstCustomer()`, `removeCustomer(Customer c)`: remove customers.  
  - `checkAvailability()`: test if queue has space.  
  - `checkQueueFront(Customer c)`: verify if a customer is at the front.  
  - `checkFrontBlockage()`: check if the front customer is blocked.  
  - `drawActive(double ap)`, `drawStimulated(double ap)`: probabilistically set active/stimulated state.  
  - `blockCustomer(Customer c)`, `activate()`, `deactivate()`: manage state changes.  
  - `getFirstCustomer()`, `getCustomerAtPosition(int pos)`, `getCustomerPosition(Customer c)`: access customers.  
  - `assignDepartureEvent(Event e)`, `assignSwitchEvent(Event e)`: link events to the queue.  

#### Output
The `Queue` class does not produce direct output.  
It maintains queue state and customer ordering, which the simulation uses to determine arrivals, departures, blocking, and fusion/microtubule activity.

## DistanceMeasure

The folder *DistanceMeasure* contains code for calculating distances between experimental data and the simulation outputs. The methods contained in these files match the methods given in 

Brock D. Sherlock, Marko A. A. Boon, Maria Vlasiou & Adelle C. F. Coster (2024).  
*The Distance Between: An Algorithmic Approach to Comparing Stochastic Models to Time-Series Data.*  
Bulletin of Mathematical Biology, 86:111.  
[https://doi.org/10.1007/s11538-024-01331-y](https://doi.org/10.1007/s11538-024-01331-y)

It is not required to use these methods. Simulation outputs can be exported and distance calculations implemented in external software. Indeed, if minimising a distance, it is recommended to calculate the distance in the same environment as the minimisation function.

### ExperimentSimulationDistance.java

The `ExperimentSimulationDistance` class compares simulation outputs with experimental data for basal, transition, and insulin states.  
It calculates statistical distance measures between experimental and simulated distributions and stores results in `MeasureResults`.

#### Purpose
- Loads experimental datasets (CSV files) for basal, transition, and insulin conditions.  
- Accepts simulation outputs for the same conditions.  
- Normalises simulation data for consistent comparison.  
- Computes multiple distance measures (e.g., KS, Kuiper, Anderson–Darling, Wasserstein, quadratic distances).  
- Saves results for later analysis.

#### Key Features
- **Data Handling**
  - `readExperimentalData(file)`: loads experimental data from CSV.  
  - `setSimulationData(...)`: sets simulation arrays for each condition.  
  - `getNormalisedArray(exp)`: normalises simulation data by uptake or transition levels.  

- **Distance Calculations**
  - `performDistanceCalculations(...)`: calculates all distance measures between experimental and simulation data, storing them in `MeasureResults`.  
  - Measures include unsigned/signed/squared areas, KS distance, Kuiper distance, Anderson–Darling, DTS two‑sample, quadratic distance, and weighted variants.  

- **Output**
  - `outputResults(filename)`: writes calculated measures to file.  

#### Output
Produces a `MeasureResults` object containing:
- Statistical distances between experimental and simulated distributions.  
- Normalised comparisons across basal, transition, and insulin states.  
- Exportable results for downstream analysis of GLUT4 translocation experiments.

### Helper functions

The remaining files in *DistanceMeasure*: `EmpiricalCDF.java`, `HelperFunctions.java`, `MeasureResults.java`, and `TimeSeriesData.java`, each contain helper functions that are called within `ExperimentSimulationDistance`. The methods are used solely for reading, formatting, and exporting results.

## dist

The *dist* folder contains a number of classes. Each is used for sampling a random number from a distribution. The distribution sampled is as the class is named.