#!/usr/bin/env bash


seed_file=""
partitions=3
debug="False"
start_iteration=0
max_time="12:00:00"
do_scheduling="True"


# Parse command line options
while getopts ":s:n:i:t:d" opt; do
    case $opt in
        d)
            do_scheduling='False'
            ;;
        t)
	          max_time="$OPTARG"
	          ;;
        s)
            seed_file="$OPTARG"
            ;;
        n)
            partitions="$OPTARG"
            ;;
        i)
            start_iteration="$OPTARG"
            ;;
        \?)
            echo "Invalid option: -$OPTARG" >&2
            exit 1
            ;;
        :)
            echo "Option -$OPTARG requires an argument." >&2
            exit 1
            ;;
    esac
done

# Shift the parsed options out of the positional parameters
shift $((OPTIND - 1))

# Check if required positional arguments are provided
if [ $# -lt 2 ]; then
  echo "Usage: $0 [-s seed_file] [-d (sets do_scheduling to false)] [-t max_time] [-n partitions] [-i start_iteration] <config file> <output folder>"
  exit 1
fi


if [ -z "$1" ]; then
    echo "Usage: $0 [-s seed_file] [-d (sets do_scheduling to false)] [-t max_time] [-n partitions] [-i start_iteration] <config file> <output folder>"
    exit 1
fi

if [ ! -d "$2" ]; then
    echo "Usage: $0 [-s seed_file] [-d (sets do_scheduling to false)] [-t max_time] [-n partitions] [-i start_iteration] <config file> <output folder>"
    exit 1
fi

network_config=''
scheduler_config=''
session_config=''

session_rates=()
packet_rates=()
simulation_length=''
iterations=''
single_session_per_flow=''
save_sessions='False'

output_dir=$2


# Read YAML file using Python and export values to Bash
eval "$(python3 <<EOF
import yaml
import sys

yaml_file = '$1'

with open(yaml_file) as f:
    data = yaml.load(f, Loader=yaml.FullLoader)


# Export each key as a Bash variable or array
print(f"network_config='{data['configuration files']['network']}'")
print(f"scheduler_config='{data['configuration files']['scheduler']}'")
print(f"session_config='{data['configuration files']['sessions']}'")
print(f"session_rates=({' '.join([str(x) for x in data['session rates']])})")
print(f"packet_rates=({' '.join([str(x) for x in data['packet rates']])})")
print(f"simulation_length={data['simulation length']}")
print(f"iterations={data['iterations']}")
print(f"debug={data['debug'] if 'debug' in data else False}")
print(f"single_session_per_flow={data['single session per flow'] if 'single session per flow' in data else False}")
print(f"save_sessions={data['save sessions'] if 'save sessions' in data else False}")
EOF
)"

if [ $debug == 'True' ]; then
    echo "$network_config"
    echo "$scheduler_config"
    echo "$session_config"
    echo "${session_rates[@]}"
    echo "${packet_rates[@]}"
    echo "$simulation_length"
    echo "$iterations"
    echo "$debug"
    echo "$single_session_per_flow"
fi

start_time=$(date)
SECONDS=0

dataset_uuid=$(uuidgen)

source_repo=$(git config --get remote.origin.url)  # Get remote URL using git command
code_version=$(git rev-parse --abbrev-ref HEAD)
code_commit=$(git rev-parse HEAD)

config_file_name=$(basename "$1" .yaml)

export source_repo code_version code_commitcat


# Define function to generate seed
generate_seed() {
    random_hex=$(od -An -N8 -tx1 /dev/urandom | tr -d ' \n')
    random_hex=$(printf "%016x" $((0x$random_hex & 0x7FFFFFFFFFFFFFFF)))
    random_hex1=$(od -An -N8 -tx1 /dev/urandom | tr -d ' \n')
    random_hex1=$(printf "%016x" $((0x$random_hex1 & 0x7FFFFFFFFFFFFFFF)))
    hex_seed="$random_hex$random_hex1"
    seed=$((0x$hex_seed))
    echo "$seed"
}

# Function to create PBS script for each combination partition
create_partition_script() {
    local partition="$1"
    local seed_file="$2"
    local network_config="$3"
    local session_config="$4"
    local scheduler_config="$5"
    local single_session_per_flow="$6"
    local simulation_length="$7"
    local debug="$8"
    local output_dir="$9"
    local save_sessions="${10}"
    local combinations=("${@:11}")

#    for combination in "${combinations[@]}"; do
#      read -r session_rate packet_rate iteration seed <<< "$combination"
#      echo "$session_rate" "$packet_rate" "$iteration" "$seed"
#    done
    if [ ! -d "$output_dir/pbs" ]; then
      mkdir -p "$output_dir/pbs"
    fi
    if [ ! -d "$output_dir/metadata" ]; then
      mkdir -p "$output_dir/metadata"
    fi
    if [ ! -d "$output_dir/raw-data" ]; then
      mkdir -p "$output_dir/raw-data"
    fi

    partition_file="$output_dir/pbs/pbs_script_partition_${partition}.pbs"

    cat <<EOF > "$partition_file"
#!/bin/bash
#PBS -l nodes=1:ppn=20
#PBS -l walltime=${max_time}
#PBS -o $output_dir/partition_${partition}.out
#PBS -e $output_dir/partition_${partition}.err
#PBS -N QNA-partition-${partition}

# Activate venv
source ~/.bashrc
conda activate quantum-arx

# Copy dir to node and change directory to the directory of the script

TMP=/var/tmp/\${PBS_JOBID}
mkdir -p \${TMP}
/usr/bin/rsync -vax "\${PBS_O_WORKDIR}/" \${TMP}/
cd \${TMP}

function process_combination(){
  local combination=\$1
  read -r session_rate packet_rate iteration seed <<< "\$combination"

  python3 -c "from Quantum_Network_Architecture.simulation.runners import run_network_simulation_homogeneous_demands;run_network_simulation_homogeneous_demands(configs={'network':'$network_config', 'sessions':'$session_config', 'scheduler': '$scheduler_config'},iteration = \$iteration,session_renewal_rate = \$session_rate,target_packet_generation_rate = \$packet_rate,single_session_per_flow=$single_session_per_flow,simulation_length=$simulation_length,debug=$debug,seed=\$seed,do_qoala_sim=False,output_dir='$output_dir', save_sessions=$save_sessions, do_actual_scheduling=${do_scheduling})"

}
export -f process_combination

start_time=\$(date)
SECONDS=0

echo "${combinations[@]}" | xargs -n 1 -P 20 bash -c 'process_combination "\$@"' _

end_time=\$(date)
secs=$SECONDS
hrs=\$(( secs/3600 )); mins=\$(( (secs-hrs*3600)/60 )); secs=\$(( secs-hrs*3600-mins*60 ))

elapsed_time=\$(printf '%02d:%02d:%02d\n' "\$hrs" "\$mins" "\$secs")





# Set variables
name="Quantum Network Architecture Simulation"
dataset_type="simulation"
uid="quantum arx simulation - ${config_file_name} - \${start_time} - partition $partition"
uuid=$dataset_uuid  # Generate UUID using uuidgen
ranking=2
setup=\$(hostname)
simulatedVariables=("proportion minimal service" "average queuing time")
code_name="quantum-network-architecture-simulation"




# Construct JSON
output=\$(cat <<EOM
{
  "name": "\$name",
  "dataset_type": "\$dataset_type",
  "ran": "\$start_time - \$end_time",
  "uid": "\$uid",
  "uuid": "\$uuid",
  "ranking": \$ranking,
  "metadata": {
    "setup": "\$setup",
    "simulatedVariables": [ "\${simulatedVariables[@]}" ],
    "Code": {
      "name": "\$code_name",
      "source_repo": "$source_repo",
      "version": "$code_version",
      "commit": "$code_commit"
    },
    "optional_metadata": {
      "elapsed_time": "\$elapsed_time",
      "sim config file": "$1"
      "seed file": "$seed_file"
    }
  }
}
EOM
)

echo "\$output" > "$output_dir/metadata/metadata_partition_$partition.json"


/usr/bin/rsync -vax \${TMP}/ "\${PBS_O_WORKDIR}/"
[ \$? -eq 0 ] && /bin/rm -rf \${TMP}

EOF

    qsub "$partition_file"
    echo "Submitted $partition_file"
}

# Export functions to be used with xargs
export -f generate_seed
export -f create_partition_script

#Calculate Iteration Numbers

# Calculate the number of combinations per partition
total_combinations=$(( ${#session_rates[@]} * ${#packet_rates[@]} * iterations ))
combinations_per_partition=$(( (total_combinations + partitions - 1) / partitions ))

# Partition the combinations
partition=0
count=0
for iteration in $(seq "$start_iteration" $((start_iteration + iterations - 1))); do
    for session_rate in "${session_rates[@]}"; do
        for packet_rate in "${packet_rates[@]}"; do


          if [ -f "$seed_file" ]; then
            seed=$(awk -v session_rate="$session_rate" -v packet_rate="$packet_rate" -v i="$iteration" '$1 == session_rate && $2 == packet_rate && $3 == i {print $4; exit}' "$seed_file")
            if [ -z "$seed" ]; then
              seed=$(generate_seed)
              echo "$session_rate $packet_rate $iteration $seed" >> "$seed_file"
            fi
          else
              seed_file="$output_dir/seeds"
              seed=$(generate_seed)
              echo "$session_rate $packet_rate $iteration $seed" >> "$seed_file"
          fi

          combinations[$count]="\\\"$session_rate $packet_rate $iteration $seed\\\""
          ((count++))

          if [ "$count" -eq "$combinations_per_partition" ]; then
              create_partition_script "$partition" "$seed_file" "$network_config" "$session_config" "$scheduler_config" "$single_session_per_flow" "$simulation_length" "$debug" "$output_dir" "$save_sessions" "${combinations[@]}"
#              echo "$partition |"
#              echo "${combinations[0]}"


              ((partition++))
              count=0
              combinations=()
          fi
        done
    done
done


# If there are any remaining combinations, assign them to the last partition
if [ "${#combinations[@]}" -gt 0 ]; then
    create_partition_script "$partition" "$seed_file" "$network_config" "$session_config" "$scheduler_config" "$single_session_per_flow" "$simulation_length" "$debug" "$output_dir" "$save_sessions" "${combinations[@]}"
fi


### Email Completion ###




