#!/usr/bin/env bash


seed_file=""
max_child_processes=3
debug="False"
start_iteration=0

# Parse command line options
while getopts ":s:m:i:" opt; do
    case $opt in
        s)
            seed_file="$OPTARG"
            ;;
        m)
            max_child_processes="$OPTARG"
            ;;
        i)
            start_iteration="$OPTARG"
            ;;
        \?)
            echo "Invalid option: -$OPTARG" >&2
            exit 1
            ;;
        :)
            echo "Option -$OPTARG requires an argument." >&2
            exit 1
            ;;
    esac
done

# Shift the parsed options out of the positional parameters
shift $((OPTIND - 1))

# Check if required positional arguments are provided
if [ $# -lt 2 ]; then
  echo "Usage: $0 [-s seed_file] [-m max_child_processes] <config file> <output folder>"
  exit 1
fi


if [ -z "$1" ]; then
    echo "Usage: $0 <yaml_file> <network_config> <output_directory>"
    exit 1
fi

if [ ! -d "$2" ]; then
    echo "Usage: $0 <yaml_file> <network_config> <output_directory>"
    exit 1
fi

network_config=''
scheduler_config=''
session_config=''

session_rates=()
packet_rates=()
simulation_length=''
iterations=''
single_session_per_flow=''
save_sessions='False'

output_dir=$2


# Read YAML file using Python and export values to Bash
eval "$(python3 <<EOF
import yaml
import sys

yaml_file = '$1'

with open(yaml_file) as f:
    data = yaml.load(f, Loader=yaml.FullLoader)


# Export each key as a Bash variable or array
print(f"network_config='{data['configuration files']['network']}'")
print(f"scheduler_config='{data['configuration files']['scheduler']}'")
print(f"session_config='{data['configuration files']['sessions']}'")
print(f"session_rates=({' '.join([str(x) for x in data['session rates']])})")
print(f"packet_rates=({' '.join([str(x) for x in data['packet rates']])})")
print(f"simulation_length={data['simulation length']}")
print(f"iterations={data['iterations']}")
print(f"debug={data['debug'] if 'debug' in data else False}")
print(f"single_session_per_flow={data['single session per flow'] if 'single session per flow' in data else False}")
print(f"save_sessions={data['save sessions'] if 'save sessions' in data else False}")
EOF
)"

if [ $debug == 'True' ]; then
    echo "$network_config"
    echo "$scheduler_config"
    echo "$session_config"
    echo "${session_rates[@]}"
    echo "${packet_rates[@]}"
    echo "$simulation_length"
    echo "$iterations"
    echo "$debug"
    echo "$single_session_per_flow"
fi

start_time=$(date)
SECONDS=0



# Define function to generate seed
generate_seed() {
    random_hex=$(od -An -N8 -tx1 /dev/urandom | tr -d ' \n')
    random_hex=$(printf "%016x" $((0x$random_hex & 0x7FFFFFFFFFFFFFFF)))
    random_hex1=$(od -An -N8 -tx1 /dev/urandom | tr -d ' \n')
    random_hex1=$(printf "%016x" $((0x$random_hex1 & 0x7FFFFFFFFFFFFFFF)))
    hex_seed="$random_hex$random_hex1"
    seed=$((0x$hex_seed))
    echo "$seed"
}

# Function to process each combination of session_rate and packet_rate
process_combination() {
    local seed_file="$1"
    local network_config="$2"
    local session_config="$3"
    local scheduler_config="$4"
    local single_session_per_flow="$5"
    local simulation_length="$6"
    local debug="$7"
    local output_dir="$8"
    local session_rate="${10}"
    local packet_rate="${11}"
    local iteration="${12}"
    local save_sessions="${9}"

    if [ -f "$seed_file" ]; then
        seed=$(awk -v session_rate="$session_rate" -v packet_rate="$packet_rate" -v i="$iteration" '$1 == session_rate && $2 == packet_rate && $3 == i {print $4; exit}' "$seed_file")
        if [ -z "$seed" ]; then
            seed=$(generate_seed)
            echo "$session_rate $packet_rate $iteration $seed" >> "$seed_file"
        fi
    else
        seed_file="$output_dir/seeds"
        seed=$(generate_seed)
        echo "$session_rate $packet_rate $iteration $seed" >> "$seed_file"
    fi

    python3 -c "from Quantum_Network_Architecture.simulation.runners import run_network_simulation_homogeneous_demands;run_network_simulation_homogeneous_demands(configs={'network':'$network_config', 'sessions':'$session_config', 'scheduler': '$scheduler_config'},iteration = $iteration,session_renewal_rate = $session_rate,target_packet_generation_rate = $packet_rate,single_session_per_flow=$single_session_per_flow,simulation_length=$simulation_length,debug=$debug,seed=$seed,do_qoala_sim=False,output_dir='$output_dir', save_sessions=$save_sessions, do_actual_scheduling=False)" 
}

# Export functions to be used with xargs
export -f generate_seed
export -f process_combination

#printf '<%q>\n' "$start_iteration"
for a in "${session_rates[@]}"; do for b in "${packet_rates[@]}"; do for c in $(seq $start_iteration $((start_iteration + iterations - 1))); do echo "$a $b $c"; done; done; done | xargs -L 1 -P "$max_child_processes" bash -c 'process_combination "$@"' _ "$seed_file" "$network_config" "$session_config" "$scheduler_config" "$single_session_per_flow" "$simulation_length" "$debug" "$output_dir" "$save_sessions"

end_time=$(date)
secs=$SECONDS
hrs=$(( secs/3600 )); mins=$(( (secs-hrs*3600)/60 )); secs=$(( secs-hrs*3600-mins*60 ))

elapsed_time=$(printf '%02d:%02d:%02d\n' "$hrs" "$mins" "$secs")



#### Make metadata  ####

config_file_name=$(basename "$1" .yaml)

# Set variables
name="Quantum Network Architecture Simulation"
dataset_type="simulation"
uid="quantum arx simulation - ${config_file_name} - ${start_time}"
uuid=$(uuidgen)  # Generate UUID using uuidgen
ranking=2
setup=$(hostname)
simulatedVariables=("proportion_minimal_service" "proportion expired pgt minimal service" "average latency to service" "average sojourn time" "average service time")
code_name="quantum-network-architecture-simulation"
source_repo=$(git config --get remote.origin.url)  # Get remote URL using git command
# Get branch name and HEAD commit hash using git commands
code_version=$(git rev-parse --abbrev-ref HEAD)
code_commit=$(git rev-parse HEAD)


# Construct JSON
output=$(cat <<EOF
{
  "name": "$name",
  "dataset_type": "$dataset_type",
  "ran": "$start_time - $end_time",
  "uid": "$uid",
  "uuid": "$uuid",
  "ranking": $ranking,
  "metadata": {
    "setup": "$setup",
    "simulatedVariables": [ "${simulatedVariables[@]}" ],
    "Code": {
      "name": "$code_name",
      "source_repo": "$source_repo",
      "version": "$code_version",
      "commit": "$code_commit"
    },
    "optional_metadata": {
      "elapsed_time": "$elapsed_time",
      "sim config file": "$1",
    }
  }
}
EOF
)

echo "$output" > "$2/metadata.json"







### Email Completion ###
