from __future__ import annotations

import yaml

from Quantum_Network_Architecture.schedules import NetworkSchedule

from Quantum_Network_Architecture.tasks import PacketGenerationTask, Taskset, PacketGenerationAttempt
from Quantum_Network_Architecture.networks import Network
from Quantum_Network_Architecture.utils.logging import LogManager
from Quantum_Network_Architecture.schedulers.priority_functions import edf_priority, dropped_pga_biased_priority, \
    expiry_time_priority
from Quantum_Network_Architecture.demands import NetworkDemand
from Quantum_Network_Architecture.exceptions import ExpiryError

from math import lcm, inf

from typing import List, Callable, Optional, Dict, Tuple

import random

from time import time

import heapq

from datetime import timedelta

from tqdm import tqdm


class NetworkScheduler:

    def __init__(
            self,
            tasks_to_schedule: Taskset | list,
            network: Network,
            scheduling_interval: int = None,
            ignore_resources: bool = False,
            start_time: int = 0,
            priority_function: Optional[Callable[[PacketGenerationTask, float], float]] = None,
            update_pga_cap_from_computation_time: bool = False,
            debug: bool = False,
            utilisation_bound: float = 0.85,
            individual_pgt_utilisation_bound: float | None = None,
            initial_pga_cap: int | None = None,
            allow_skipping_of_pgas: bool = False,
            pga_success_probability: float = 0.2,
            strict_fifo_queue: bool = True
    ):

        self.all_tasks: Taskset[PacketGenerationTask] = Taskset(tasks_to_schedule)

        adjusted_start_time = start_time * 1e9 // network.timeslot_duration

        self.decision_time: int = adjusted_start_time

        self._start_time = adjusted_start_time

        self._ignore_resources: bool = ignore_resources

        self._scheduling_interval = scheduling_interval

        self._end_of_schedule = self.length_of_schedule + adjusted_start_time

        self._links = network.link_ids

        self._branches_explored = 1

        self._network = network

        self._logger = LogManager.get_scheduler_logger()

        self._priority_function = priority_function

        self._demand_queue: List[NetworkDemand] = []
        heapq.heapify(self._demand_queue)

        self._resource_utilisation: dict[str,float] = {r: 0 for r in self.resource_availability()}

        task: PacketGenerationTask

        self._debug: bool = debug

        self._utilisation_bound: float = utilisation_bound

        self._pgt_utilisation_bound = individual_pgt_utilisation_bound if individual_pgt_utilisation_bound is not None else utilisation_bound

        self._resource_utilisations: Dict[str,List[float]] = {r:[] for r in self._resource_utilisation}

        self._branches_explored = 1

        self._computation_times: List[Tuple[int,float]] = []

        self._update_max_number_of_pgas_from_computation_time: bool = update_pga_cap_from_computation_time

        self._max_total_number_of_pgas_scheduled_in_period = inf if initial_pga_cap is None else initial_pga_cap

        self._allow_skipping_of_pgas = allow_skipping_of_pgas

        self._pga_success_probability: float = pga_success_probability

        self._strict_fifo_queue: bool = strict_fifo_queue

    @property
    def demand_queue(self):
        return self._demand_queue

    @classmethod
    def edf_scheduler(
            cls,
            network: Network,
            tasks_to_schedule: Optional[Taskset | list] = None,
            scheduling_interval: int = None,
            ignore_resources: bool = False,
            start_time: int = 0,
            debug: bool = False,
            update_pga_cap_from_computation_time: bool = True,
            utilisation_bound: float = 0.85,
            initial_pga_cap: int | None = None
    ) -> NetworkScheduler:

        return NetworkScheduler(tasks_to_schedule if tasks_to_schedule is not None else [], network, scheduling_interval,
                                ignore_resources, start_time, priority_function=edf_priority,
                                update_pga_cap_from_computation_time=update_pga_cap_from_computation_time,
                                debug=debug, utilisation_bound=utilisation_bound, initial_pga_cap=initial_pga_cap)

    @classmethod
    def biased_scheduler(
            cls,
            tasks_to_schedule: Taskset | list,
            network: Network,
            scheduling_interval: int = None,
            ignore_resources: bool = False,
            start_time: int = 0,
            debug: bool = False,
            update_pga_cap_from_computation_time: bool = True,
            utilisation_bound: float = 0.85,
            initial_pga_cap: int | None = None
    ) -> NetworkScheduler:

        return NetworkScheduler(tasks_to_schedule, network, scheduling_interval, ignore_resources, start_time,
                                priority_function=dropped_pga_biased_priority,
                                update_pga_cap_from_computation_time=update_pga_cap_from_computation_time,
                                debug=debug, utilisation_bound=utilisation_bound, initial_pga_cap=initial_pga_cap)

    @classmethod
    def from_yaml(
            cls,
            config: str,
            network: Network,
            start_time: int = 0,
            debug: bool = False,
    ) -> NetworkScheduler:

        with open(config,'r') as F:
            config = yaml.safe_load(F)

        # match config["priority function"].lower():
        #     case 'edf':
        #         priority_function = edf_priority
        #     case 'biased':
        #         priority_function = dropped_pga_biased_priority
        #     case _:
        #         raise ValueError("Unknown priority function in config, please use either 'edf' or 'biased' ")

        if config["priority function"].lower() == 'edf':
            priority_function = edf_priority
        elif config["priority function"].lower() == 'biased':
            priority_function = dropped_pga_biased_priority
        else:
            raise ValueError("Unknown priority function in config, please use either 'edf' or 'biased' ")

        if config['initial cap on pgas per schedule'] == -1:
            config['initial cap on pgas per schedule'] = inf

        if "pgt utilisation bound" not in config:
            pgt_utilisation_bound = None
        else:
            pgt_utilisation_bound = float(config['pgt utilisation bound'])

        return cls(
            tasks_to_schedule=[],
            network=network,
            scheduling_interval=config["scheduling interval"] * 1e9 // network.timeslot_duration,  # needs adjusting to timeslots
            start_time=start_time,
            priority_function=priority_function,
            update_pga_cap_from_computation_time=config["update cap from computation time"],
            utilisation_bound=config["utilisation bound"],
            individual_pgt_utilisation_bound=pgt_utilisation_bound,
            initial_pga_cap=config['initial cap on pgas per schedule'],
            debug=debug,
            allow_skipping_of_pgas=config["allow skipping of pgas"],
            pga_success_probability=config["pga success probability"]
        )


    def submit_demand(self, new_demand: NetworkDemand):
        heapq.heappush(self._demand_queue, new_demand)

    def pop_next_demand(self) -> NetworkDemand:
        return heapq.heappop(self._demand_queue)

    def remove_demand(self, _demand_to_remove: NetworkDemand):
        if _demand_to_remove.packet_generation_task is not None:
            self.remove_task(_demand_to_remove.packet_generation_task)

        if _demand_to_remove in self._demand_queue:
            self._demand_queue.remove(_demand_to_remove)

    def accept_task(self, _new_task: PacketGenerationTask):
        self.all_tasks.append(_new_task)
        for link in _new_task.links:
            self._resource_utilisation[link] += _new_task.utilisation

    def remove_task(self, _task_to_remove: PacketGenerationTask):
        _task_to_remove.terminate()
        if _task_to_remove in self.all_tasks:
            self.all_tasks.remove(_task_to_remove)
            for link in _task_to_remove.links:
                self._resource_utilisation[link] -= _task_to_remove.utilisation

    def get_tasks_to_add_to_scheduler(self):

        while len(self._demand_queue) > 0 and (
        self.projected_number_of_pgas_to_schedule < self._max_total_number_of_pgas_scheduled_in_period):

            _next_demand = self.pop_next_demand()
            try:
                _new_task: PacketGenerationTask = _next_demand.create_packet_generation_task(self._network, self.current_real_time/1e9, pga_success_probability=self.pga_success_probability)
            except ExpiryError:
                self.logger.warning(f"Demand {_next_demand.identifier} has expired, removing from queue.")
                _next_demand.queue_exit_time = self.current_real_time / 1e9 - self.real_scheduling_interval / 1e9
                continue

            if _new_task.utilisation > self._pgt_utilisation_bound:
                del _next_demand.packet_generation_task
                self.logger.warning(f"PGT for demand {_next_demand.identifier} exceeds utilisation bound, so removing from queue.")
                _next_demand.queue_exit_time = self.current_real_time / 1e9 - self.real_scheduling_interval / 1e9
                continue  # As already popped the demand from the queue this will remove it entirely

            elif all(self._resource_utilisation[link] + _new_task.utilisation < self._utilisation_bound for link in _new_task.links):
                self.accept_task(_new_task)
                self.logger.info(f"Accepted task {_new_task.identifier} for scheduling at time {timedelta(seconds=self.current_real_time / 1e9)} with QoS option '{_new_task.accepted_qos_option}'")
                _next_demand.queue_exit_time = self.current_real_time / 1e9 - self.real_scheduling_interval / 1e9

            else:
                del _next_demand.packet_generation_task
                self.submit_demand(_next_demand)
                break # puts the demand back into the queue and exits

    @property
    def pga_success_probability(self):
        return self._pga_success_probability

    @property
    def computed_schedule(self):
        return NetworkSchedule(
            self.all_tasks,
            "FEASIBLE" if not self.is_infeasible else "INFEASIBLE",
            schedule_length=self.end_time - self._start_time
        )

    @property
    def start_time(self):
        return self._start_time

    @start_time.setter
    def start_time(self, value):
        if not isinstance(value, int):
            raise ValueError

        self._start_time = value * 1e9 // self._network.timeslot_duration
        if self._start_time > self.decision_time:
            self.decision_time = 0 + self.start_time


    @property
    def last_computation_time(self):
        return self._computation_times[-1][1]

    @property
    def schedule_computation_times(self):
        return self._computation_times

    @property
    def schedule_computation_times_proportion_of_scheduling_interval(self):
        return [(x[0],x[1] / self.real_scheduling_interval / 1e9) for x in self._computation_times]

    @property
    def average_est_pgas_in_schedule(self):
        _values = [x[0] for x in self._computation_times]
        return sum(_values) / len(_values) if _values else 0

    @property
    def average_resource_utilisations(self) -> Dict[str,float]:
        return {r: (sum(self._resource_utilisations[r])/len(self._resource_utilisations[r]) if self._resource_utilisations[r] else 0) for r in self._resource_utilisations.keys()}

    @property
    def end_time(self):
        return self._end_of_schedule

    @property
    def logger(self):
        return self._logger

    @property
    def current_real_time(self):
        return self.decision_time * self._network.timeslot_duration

    @property
    def real_end_time(self):
        return self.end_time * self._network.timeslot_duration

    @property
    def real_scheduling_interval(self):
        return self._scheduling_interval * self._network.timeslot_duration

    @property
    def did_branch(self) -> bool:
        return self._branches_explored > 1  # bool indicating whether branching has occurred

    @property
    def branches_explored(self) -> int:
        return self._branches_explored  # Counter of number of branches explored.

    def running_tasks(self, t: Optional[int] = None) -> [PacketGenerationTask]:

        _target_time = self.decision_time if t is None else t
        _rt = [T for T in self.all_tasks if T.is_active_at(_target_time)]
        _rt.sort(key=lambda x: x.next_complete())
        return _rt

    def resource_availability(self, t: Optional[int] = None) -> {str: bool}:
        if self._ignore_resources:
            return {N: True for N in self._links}
        else:
            T: PacketGenerationTask
            return {L: not (True in [L in T.links for T in self.running_tasks(t)]) for L in
                    self._links}  # Removes resources currently being used by running tasks

    @property
    def requested_length_of_schedule(self) -> int:
        return lcm(*[round(1 / t.requested_rate) for t in
                     self.all_tasks])  # Returns the hyper-period. For comparison when doing rate reduction

    @property
    def length_of_schedule(self) -> int:
        if self._scheduling_interval is None:
            return lcm(*[T.period for T in self.all_tasks])  # If length of schedule unspecified return hyper-period,
        else:
            return self._scheduling_interval

    @property
    def eligible_tasks(self) -> List[PacketGenerationTask]:
        T: PacketGenerationTask
        _et = [T for T in self.all_tasks if T.next_release <= self.decision_time and all(
            self.resource_availability()[link] for link in
            T.links) and not T.is_expired]  # Tasks are eligible if they have been released, have available resources and are not expired
        _et.sort(key=lambda x: x.priority(self.decision_time),
                 reverse=True)  # Sorted so the [0] element has the highest priority
        return _et

    def next_task_to_complete(self, t: Optional[int] = None) -> PacketGenerationTask:
        return self.running_tasks(t)[0] if self.running_tasks(t) else None

    def next_completion_time(self, t: Optional[int] = None) -> int:
        _next_completed_task = self.next_task_to_complete(t)
        return _next_completed_task.next_complete(
            t) if _next_completed_task is not None else inf  # defaults to ∞ if no running tasks

    @property
    def is_infeasible(self) -> bool:
        return any(self.decision_time > T.next_start_deadline for T in
                   self.all_tasks)  # Schedule becomes infeasible if a start deadline is missed.

    @property
    def is_complete(self) -> bool:
        return self._end_of_schedule <= self.decision_time  # Completed schedule when get to the end of it.

    @property
    def tasks_pending_release(self) -> [PacketGenerationTask]:
        return [T for T in self.all_tasks if
                T.next_release > self.decision_time]  # Used to check if the next change is a task being released versus a running task completing

    @property
    def next_release_time(self) -> int:
        return min(
            T.next_release for T in self.tasks_pending_release) if self.tasks_pending_release else inf  # see above

    def next_expiry_time(self, t: Optional[int] = None) -> int:
        T: PacketGenerationTask
        _target_time = t if t is not None else self.decision_time
        _active_tasks: List[PacketGenerationTask] = [T for T in self.all_tasks if T.expiry_time > _target_time]
        _active_tasks.sort(key=lambda x: x.expiry_time)

        return _active_tasks[0].expiry_time if _active_tasks else inf

    @property
    def zero_service(self):
        return True in [0.01 >= t.rate_scaling for t in
                        self.all_tasks]  # Checks if rate reduction has resulted in no service.

    @property
    def projected_number_of_pgas_to_schedule(self):
        return sum(self._scheduling_interval // t.period for t in self.all_tasks)

    @property
    def ave_max_utilisation(self):
        transpose = [[self._resource_utilisations[x][y] for x in self._resource_utilisations]
                        for y in range(len(self._resource_utilisations[list(self._resource_utilisations.keys())[0]]))]
        return sum(max(x) for x in transpose) / len(transpose)

    def compute_schedule(self, random_tie_breaks: bool = False):



        while True:
            self.logger.info(f"Computing schedule starting at time {timedelta(seconds=self.current_real_time / 1e9)}")
            # self.logger.info(f"Current Utilisations: {self._resource_utilisation}")

            for r in self._resource_utilisation.keys():
                self._resource_utilisations[r].append(self._resource_utilisation[r])  # Records resource utilisation for this schedule


            new_tasks = self.all_tasks.initialise_tasks(time=self.decision_time)
            if new_tasks is not None:
                self._logger.warning(
                    f"{timedelta(seconds=self.decision_time * self._network.timeslot_duration / 1e9)}: Initialised tasks: {', '.join(new_tasks.ids)}")
                # new_tasks.get_full_link_ids(network=self._network)
                if self._priority_function is not None:
                    new_tasks.set_priority_functions(self._priority_function)

            if self._debug:
                prog_bar = tqdm(total=self.projected_number_of_pgas_to_schedule)

            _tic = time()
            while not self.is_complete:
                while self.eligible_tasks:
                    # self._logger.info(f"{timedelta(seconds = self.current_real_time / 1e9)}: Scheduled task {self.eligible_tasks[0].identifier}")
                    if not random_tie_breaks:
                        self.eligible_tasks[0].add_to_schedule(self.decision_time)
                        if self._debug:
                            prog_bar.update()
                    else:
                        tied_tasks = [t for t in self.eligible_tasks if
                                      t.next_deadline == self.eligible_tasks[0].next_deadline]
                        random.shuffle(tied_tasks)
                        tied_tasks[0].add_to_schedule(self.decision_time)
                        if self._debug:
                            prog_bar.update()

                self.decision_time = min(self.next_completion_time(), self.next_release_time, self._end_of_schedule)

                if self.is_infeasible and self._allow_skipping_of_pgas:
                    t: PacketGenerationTask
                    for task in [t for t in self.all_tasks if
                                 self.decision_time > t.next_start_deadline and not t.is_expired]:
                        task: PacketGenerationTask
                        self._logger.warning(
                            f"{timedelta(seconds=self.decision_time * self._network.timeslot_duration / 1e9)}: Dropped packet for task {task.identifier}")
                        task.drop_packet()
                        if self._debug:
                            prog_bar.update()

            self._computation_times.append((self.projected_number_of_pgas_to_schedule,time() - _tic))
            self._logger.info(f"Computed schedule in time {timedelta(seconds=self.last_computation_time)}")
            if self.last_computation_time > self._scheduling_interval * self._network.timeslot_duration * 1e-9:
                self._logger.warning(f"Schedule computation time exceeded scheduling interval.")


            elif self.last_computation_time > 0.75 * self._scheduling_interval * self._network.timeslot_duration * 1e-9:
                self._logger.warning(f"Schedule computation time exceeded 75\% of scheduling interval.")

            elif self.last_computation_time > 0.5 * self._scheduling_interval * self._network.timeslot_duration * 1e-9:

                self._logger.warning(f"Schedule computation time exceeded 50\% of scheduling interval.")


            if self._update_max_number_of_pgas_from_computation_time:
                if (
                        self.last_computation_time > 0.5 * self._scheduling_interval * self._network.timeslot_duration * 1e-9 and
                        self.projected_number_of_pgas_to_schedule < self._max_total_number_of_pgas_scheduled_in_period

                ):
                    self._max_total_number_of_pgas_scheduled_in_period = 0 + self.projected_number_of_pgas_to_schedule

                if self.last_computation_time > 0.75 * self._scheduling_interval * self._network.timeslot_duration * 1e-9:
                    self._max_total_number_of_pgas_scheduled_in_period -= min(self._scheduling_interval // t.period for t in self.all_tasks)


            self.logger.info(f"Maximum number of PGAs to schedule set to {self._max_total_number_of_pgas_scheduled_in_period}")

            if self._debug:
                prog_bar.close()


            yield
            self._end_of_schedule += self._scheduling_interval

    def compute_branching_schedule(self, time_remaining: float | int = inf):

        branching_points = {}

        _start_time = time()

        while time() - _start_time < time_remaining:
            while self.eligible_tasks:
                alternatives = [x for x in self.eligible_tasks if x.next_deadline == self.eligible_tasks[
                    0].next_deadline]  # List of tasks which share a common deadline
                active_task: PacketGenerationTask = alternatives.pop(0)
                active_task.add_to_schedule(self.decision_time)
                if alternatives:
                    branching_points[self.decision_time] = alternatives

            if self.next_completion_time() <= self.next_release_time:
                self.decision_time = self.next_completion_time()
            else:
                self.decision_time = self.next_release_time

            if self.is_complete:
                return

            if self.is_infeasible:
                if not branching_points:
                    return  # If explored whole tree then fail
                else:
                    self._branches_explored += 1

                    branch_time = max(branching_points.keys())  # Find the last time branching occurred
                    for T in self.all_tasks:
                        T.clear_schedule(t=branch_time)  # Reset to that time

                    self.decision_time = branch_time

                    active_task: PacketGenerationTask = branching_points[branch_time].pop(
                        0)  # Start a different choice of task
                    active_task.add_to_schedule(self.decision_time)
                    if not branching_points[
                        branch_time]:  # If all alternatives at a given time explored, remove from list.
                        branching_points.pop(branch_time)


class ExactDemandsNetworkScheduler(NetworkScheduler):

    def __init__(
            self,
            tasks_to_schedule: Taskset | list,
            network: Network,
            scheduling_interval: int = None,
            ignore_resources: bool = False,
            start_time: int = 0,
            priority_function: Optional[Callable[[PacketGenerationTask, float], float]] = None,
            single_queue: bool = True
    ):
        super().__init__(tasks_to_schedule, network, scheduling_interval, ignore_resources, start_time,
                         expiry_time_priority if priority_function is None else priority_function)

        if single_queue:
            self._queues: Dict[str, Taskset[PacketGenerationTask]] = {'global': Taskset(T for T in self.all_tasks)}

        else:
            t: PacketGenerationTask
            self._queues: Dict[str, Taskset[PacketGenerationTask]] = {
                l: Taskset(t for t in self.all_tasks if l in t.links) for l in self.resource_availability().keys()}

        for Q in self._queues.values():
            Q.sort(key=lambda x: x.priority(self.decision_time), reverse=True)

        self._tried_to_schedule: Dict[PacketGenerationTask:bool] = {T: False for T in self.all_tasks}

        self._resource_schedule: Dict[str, Dict[int, List[PacketGenerationAttempt]]] = {r: {} for r in
                                                                                        self.resource_availability().keys()}

        self._tasks_awaiting_additional_pgas = Taskset()

    @property
    def eligible_tasks(self) -> List[PacketGenerationTask]:
        T: PacketGenerationTask

        _et = [T for T in self.all_tasks if (not T.is_expired and not self._tried_to_schedule[T] and all(
            l.index(T) == 0 if T in l else True for l in self._queues.values()))]
        _et.sort(key=lambda x: x.priority(self.decision_time), reverse=True)

        return _et

    @property
    def tasks_in_queues(self):
        return Taskset(T for T in self.all_tasks if not T.is_expired)

    def reset_tried_to_schedule(self):
        self._tried_to_schedule = {T: False for T in self.all_tasks}

    def remove_scheduled_tasks_from_queues(self):
        for q in self._queues.values():
            q.remove_expired_tasks()

    def add_pga_to_resource_schedule(self, r: str, pga: PacketGenerationAttempt):
        for t in range(pga.start_time, pga.end_time):
            if t in self._resource_schedule[r]:
                self._resource_schedule[r][t].append(pga)
            else:
                self._resource_schedule[r][t] = [pga]

    def resource_conflict(self, max_quantity: Dict[str, int] = None) -> List[str]:
        if max_quantity is None:
            max_quantity = {r: 1 for r in self._resource_schedule.keys()}

        return [r for r in self._resource_schedule.keys() if
                any(len(L) > max_quantity[r] for L in self._resource_schedule[r].values())]

    def compute_exact_demands_schedule(self):

        while True:

            self.logger.info(f"Computing schedule starting at time {timedelta(seconds=self.current_real_time / 1e9)}s")

            new_tasks = self.all_tasks.initialise_tasks(self.decision_time)
            if new_tasks is not None:
                self._logger.warning(
                    f"{timedelta(seconds=self.decision_time * self._network.timeslot_duration / 1e9)}: Initialised tasks: {', '.join(new_tasks.ids)}")
                # new_tasks.get_full_link_ids(network=self._network)
                if self._priority_function is not None:
                    new_tasks.set_priority_functions(self._priority_function)

            _tic = time()
            self.reset_tried_to_schedule()

            while not self.is_complete:

                while self.eligible_tasks:

                    _task_to_add = self.eligible_tasks[0]
                    _pointer = self.decision_time

                    while not _task_to_add.is_expired and (
                            _task_to_add.number_scheduled < _task_to_add.number_to_schedule or _pointer < self.end_time):
                        if all(self.resource_availability(t)[link] for link in _task_to_add.links for t in
                               range(_pointer, _pointer + _task_to_add.execution_time)):
                            _task_to_add.add_to_schedule(_pointer)
                            _pointer = _task_to_add.next_release

                        else:
                            _pointer = self.next_completion_time(_pointer)

                        if _pointer > _task_to_add.next_start_deadline:
                            self._tried_to_schedule[_task_to_add] = True
                            _task_to_add.clear_schedule(_task_to_add.initialisation_time)
                            break

                    if not _task_to_add.is_expired:
                        self._tasks_awaiting_additional_pgas.append(_task_to_add)

                    self.remove_scheduled_tasks_from_queues()

                self.decision_time = min(self.next_expiry_time(), self._end_of_schedule)

                for task in self.tasks_in_queues:
                    task.next_release = self.decision_time

                self.reset_tried_to_schedule()

                if not self.eligible_tasks:
                    self.decision_time = self._end_of_schedule

            yield
            self._end_of_schedule += self._scheduling_interval

    def compute_schedule_minimal_trials_exact_rate(self):

        while True:

            self.logger.info(f"Computing schedule starting at time {timedelta(seconds=self.current_real_time / 1e9)}s")

            new_tasks = self.all_tasks.initialise_tasks(self.decision_time)
            if new_tasks is not None:
                self._logger.warning(
                    f"{timedelta(seconds=self.decision_time * self._network.timeslot_duration / 1e9)}: Initialised tasks: {', '.join(new_tasks.ids)}")
                # new_tasks.get_full_link_ids(network=self._network)
                if self._priority_function is not None:
                    new_tasks.set_priority_functions(self._priority_function)

            _tic = time()
            self.reset_tried_to_schedule()

            while not self.is_complete:

                while self.eligible_tasks:

                    _task_to_add = self.eligible_tasks[0]
                    _pointer = self.decision_time

                    while not _task_to_add.is_expired and (
                            _task_to_add.number_scheduled < _task_to_add.number_to_schedule or _pointer < self.end_time):
                        if all(self.resource_availability(t)[link] for link in _task_to_add.links for t in
                               range(_pointer, _pointer + _task_to_add.execution_time)):
                            _task_to_add.add_to_schedule(_pointer)
                            _pointer = _task_to_add.next_release

                        else:
                            _pointer = self.next_completion_time(_pointer)

                        if _pointer > _task_to_add.next_start_deadline:
                            self._tried_to_schedule[_task_to_add] = True
                            _task_to_add.clear_schedule(_task_to_add.initialisation_time)
                            break

                    if not _task_to_add.is_expired:
                        self._tasks_awaiting_additional_pgas.append(_task_to_add)

                    self.remove_scheduled_tasks_from_queues()

                self.decision_time = min(self.next_expiry_time(), self._end_of_schedule)

                for task in self.tasks_in_queues:
                    task.next_release = self.decision_time

                self.reset_tried_to_schedule()

                if not self.eligible_tasks:
                    self.decision_time = self._end_of_schedule

            yield
            self._end_of_schedule += self._scheduling_interval

class DummyNetworkScheduler(NetworkScheduler):

    def get_tasks_to_add_to_scheduler(self, skipping = True):

        demands_attempted_to_accept = []
        while len(self._demand_queue) > 0 and (
        self.projected_number_of_pgas_to_schedule < self._max_total_number_of_pgas_scheduled_in_period):

            _next_demand = self.pop_next_demand()
            try:
                _new_task: PacketGenerationTask = _next_demand.create_packet_generation_task(self._network, self.current_real_time/1e9, pga_success_probability=self.pga_success_probability)
            except ExpiryError:
                self.logger.warning(f"Demand {_next_demand.identifier} has expired, removing from queue.")
                continue

            if _new_task.utilisation > self._pgt_utilisation_bound:
                del _next_demand.packet_generation_task
                self.logger.warning(f"PGT for demand {_next_demand.identifier} exceeds utilisation bound, so removing from queue.")
                continue  # As already popped the demand from the queue this will remove it entirely

            elif all(self._resource_utilisation[link] + _new_task.utilisation < self._utilisation_bound for link in _new_task.links):
                self.accept_task(_new_task)
                self.logger.info(f"Accepted task {_new_task.identifier} for scheduling at time {timedelta(seconds=self.current_real_time / 1e9)} with QoS option '{_new_task.accepted_qos_option}'")

            else:
                del _next_demand.packet_generation_task
                if self._strict_fifo_queue:
                    self.submit_demand(_next_demand)
                    break # puts the demand back into the queue and exits
                else:
                    demands_attempted_to_accept.append(_next_demand)

        for demand in demands_attempted_to_accept:
            self.submit_demand(demand)





    def compute_schedule(self):
        while True:
            self.logger.info(f"Adding dummy start times for schedule starting at time {timedelta(seconds=self.current_real_time / 1e9)}")
            for r in self._resource_utilisation.keys():
                self._resource_utilisations[r].append(self._resource_utilisation[r])

            new_tasks = self.all_tasks.initialise_tasks(time=self.decision_time)
            if new_tasks is not None:
                self._logger.warning(
                    f"{timedelta(seconds=self.decision_time * self._network.timeslot_duration / 1e9)}: Initialised tasks: {', '.join(new_tasks.ids)}")


            for task in self.all_tasks:
                for t in range(int(task.next_release), min(int(self.end_time), int(task.expiry_time)), task.period):
                    task.add_to_schedule(t)  # assumes PGAs are scheduled at exactly the requested frequency.

            self.decision_time += self._scheduling_interval
            self._computation_times.append((self.projected_number_of_pgas_to_schedule, 0))

            yield
            self._end_of_schedule += self._scheduling_interval




class ResourceSchedule(Dict[str, Dict[int, list]]):

    def has_conflict(self):
        return any(len(l) > 1 for x in self.values() for l in x.values())

        pass
